/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "access/martian/martian_resources.h"
#include "access/access.h"

namespace Access {

namespace Martian {

static const int RMOUSE[10][2] = {
	{ 7, 36 },    { 38, 68 },   { 70, 99 },   { 102, 125 }, { 128, 152 },
	{ 155, 185 }, { 188, 216 }, { 219, 260 }, { 263, 293 }, { 295, 314 }
};

// MM defines the last range twice in the original.
static const int BTN_RANGES[6][2] = {
	{ 0, 60 }, { 64, 124 }, { 129, 192 }, { 194, 227 }, { 233, 292 }, { 297, 319 }
};


MartianResources::~MartianResources() {
	delete _font1;
	delete _font2;
	delete _bitFont;
}

int MartianResources::getRMouse(int i, int j) const {
	return RMOUSE[i][j];
}

int MartianResources::inButtonXRange(int x) const {
	for (int i = 0; i < ARRAYSIZE(BTN_RANGES); i++) {
		if ((x >= BTN_RANGES[i][0]) && (x < BTN_RANGES[i][1]))
			return i;
	}
	return -1;
}

/*------------------------------------------------------------------------*/

const int SIDEOFFR[] = {  4, 0, 7, 10,  3, 1, 2, 13, 0, 0, 0, 0 };
const int SIDEOFFL[] = { 11, 6, 1,  4, 10, 6, 1,  4, 0, 0, 0, 0 };
const int SIDEOFFU[] = {  1, 2, 0,  2,  2, 1, 1,  0, 0, 0, 0, 0 };
const int SIDEOFFD[] = {  2, 0, 1,  1,  0, 1, 1,  1, 2, 0, 0, 0 };

const byte CREDIT_DATA[] = {
	0x1F, 0x00, 0x49, 0x00, 0x00, 0x00, 0xB7, 0x00, 0x49, 0x00,
	0x01, 0x00, 0x79, 0x00, 0x6F, 0x00, 0x02, 0x00, 0xFF, 0xFF,
	0xEA, 0x01, 0x75, 0x00, 0x46, 0x00, 0x03, 0x00, 0x46, 0x00,
	0x5E, 0x00, 0x04, 0x00, 0xFF, 0xFF, 0xEA, 0x01, 0x72, 0x00,
	0x3E, 0x00, 0x05, 0x00, 0x46, 0x00, 0x57, 0x00, 0x04, 0x00,
	0x5C, 0x00, 0x6E, 0x00, 0x06, 0x00, 0xFF, 0xFF, 0xEA, 0x01,
	0x63, 0x00, 0x48, 0x00, 0x07, 0x00, 0x2A, 0x00, 0x65, 0x00,
	0x08, 0x00, 0xFF, 0xFF, 0xEA, 0x01, 0x7E, 0x00, 0x39, 0x00,
	0x09, 0x00, 0x5C, 0x00, 0x57, 0x00, 0x06, 0x00, 0x45, 0x00,
	0x6B, 0x00, 0x04, 0x00, 0xFF, 0xFF, 0xEA, 0x01, 0x5F, 0x00,
	0x46, 0x00, 0x0A, 0x00, 0x67, 0x00, 0x62, 0x00, 0x0B, 0x00,
	0x47, 0x00, 0x76, 0x00, 0x0C, 0x00, 0xFF, 0xFF, 0xEA, 0x01,
	0x62, 0x00, 0x38, 0x00, 0x0D, 0x00, 0x47, 0x00, 0x55, 0x00,
	0x0E, 0x00, 0x49, 0x00, 0x6A, 0x00, 0x0F, 0x00, 0xFF, 0xFF,
	0xEA, 0x01, 0x18, 0x00, 0x22, 0x00, 0x10, 0x00, 0x17, 0x00,
	0x3E, 0x00, 0x11, 0x00, 0x16, 0x00, 0x52, 0x00, 0x12, 0x00,
	0xEE, 0x00, 0x7B, 0x00, 0x13, 0x00, 0xB5, 0x00, 0x93, 0x00,
	0x0B, 0x00, 0xFF, 0xFF, 0xF4, 0x01, 0xFF, 0xFF, 0xFF, 0xFF
};

const byte ICON_PALETTE[] = {
	0x3F, 0x3F, 0x00, 0x00, 0x07, 0x16,
	0x00, 0x0A, 0x1A, 0x00, 0x0D, 0x1F,
	0x00, 0x11, 0x28, 0x00, 0x15, 0x30,
	0x00, 0x19, 0x39, 0x00, 0x1B, 0x3F,
	0x00, 0x2D, 0x3A
};

const char *const TRAVDATA[] = {
	"GALACTIC PICTURES", "TERRAFORM", "WASHROOM", "MR. BIG", "ALEXIS' HOME",
	"JOHNNY FEDORA", "JUNKYARD IN", "TEX'S OFFICE", "MURDER SCENE", "PLAZA HOTEL",
	"RESTAURANT", "GIFT SHOP", "LOVE SCENE", "RICK LOGAN", "HUT",
	"SMUGGLERS BASE", "PYRAMID", "CASINO", "CAS LOBBY", "BAR",
	"DUCTWORK", "RESTROOM", "OFFICE", "SAFE", "ALLEY",
	"POWER PLANT", "PLANT OFFICE", "PLANT ROOM", "TEMPLE", "IN TEMPLE",
	"JANE MANSFIELD'S HOME", "AEROBICS ACADEMY", "DR. LAWRENCE BARKLEY", "COLONISTS CAMP", "IN SLUM",
	"REMOTE OUTPOST", "WALK", "CAVE", "PRISON", "ORACLE",
	"JOCQUES SPARROW", "MAC MALDEN", "CHANTAL VARGAS", "GUY CALLABERO", "ROCKWELL BACHE",
	"FERRIS COLLETTE", "NORA DESMOND ALEXANDER", "LOWELL PERCIVAL", "MICHELE BLOODWORTH", "BRADLEY ERICSON",
	"COOPER BRADBURY", nullptr
};

const char *const ASK_TBL[] = {
	"NONE", "MARSHALL ALEXANDER", "TERRAFORM CORP.", "COLLIER STANTON", "ROCKWELL BACHE",
	"JOCQUES SPARROW", "NORA DESMOND ALEXANDER", "GALACTIC PICTURES", "LAWRENCE BARKLEY", "TMS",
	"MAC MALDEN", "STANTON EXPEDITION", "LOWELL PERCIVAL", "CHANTAL VARGAS", "RICK LOGAN",
	"ALEXIS ALEXANDER", "FERRIS COLLETT", "GUY CALLABERO", "ORACLE STONE", "THOMAS DANGERFIELD",
	"JANE MANSFIELD", "STACY CRAWFORD", "DICK CASTRO", "ROCKY BULLWINKEL", "DEACON HAWKE",
	"NATHAN BLOODWORTH", "MICHELLE BLOODWORTH", "BRADLEY ERICSON", "COOPER BRADBURY", "MARTIAN MEMORANDUM",
	"JOHNNY FEDORA", "RHONDA FOXWORTH", "ANGELO ANDRETTI", "TEX MURPHY", "ROBERT BLOODWORTH",
	"LARRY HAMMOND", nullptr
};

byte HELP[] = {
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1
};

const char *const SPEC7MESSAGE = {
	"THOMAS DANGERFIELD'S MAD EXPERIMENT OF ATTEMPTING TO HARNESS THE STONE'S POWER, ENDED HIS LIFE. DANGERFIELD WAS A DECENT HUMAN " \
	"BEING ONCE, BUT WAS DRIVEN INSANE BY HIS QUEST FOR THE ULTIMATE POWER. ALEXIS AND I DECIDE THAT DEACON HAWKE IS THE ONLY " \
	"LOGICAL CHOICE FOR THE STONE. WE ARRIVE AT THE TEMPLE WHERE SHE IS WAITING FOR US. SHE TURNS AND WHISPERS; 'YOU HAVE RETURNED " \
	"THE STONE TO THE MISTRESS OF THE LIGHT. YOU HAVE SURELY SAVED THE WORLD FROM ANNIHILATION BUT YOU MUST CONTINUE TO BE DILIGENT. " \
	"MANKIND MAY YET PROVE TO BE THE AUTHOR OF HIS OWN DEMISE. REVERENCE LIFE. PROTECT THE LIVING THINGS AND RECYCLE. AND NOW FOR " \
	"THE SAFETY OF MANKIND, I MUST TAKE THE STONE. PERHAPS SOMEDAY, WHEN THE HUMAN RACE IS READY, IT WILL BE RETURNED. UNTIL THEN "\
	"FAREWELL...'"
};

const byte CAN_TRAVEL_MATRIX[] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
	1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
	0, 0, 0, 0, 0, 1, 0, 1, 1, 1,
	1
};

const int16 PICTURE_RANGE[][2] = {
// { min X, max X}, {min Y, max Y}
	{ 20, 30 },  { 82, 87 },
	{ 20, 30 },  { 105, 110 },
	{ 0, 8 },    { 92, 100 },
	{ 42, 46 },  { 92, 100 },
	{ 9, 41 },   { 88, 104 },
	{ 9, 41 },   { 117, 133 },
	{ -1, -1 }
};

const int16 DUCT_ARROW_BUTTON_RANGE[][2] = {
// { min X, max X}, {min Y, max Y}
	{ 12, 23 }, { 80, 87 }, // up button
	{ 4, 15 },  { 87, 93 }, // left button
	{ 14, 26 }, { 93, 104 },// down button
	{ 25, 36 }, { 87, 93 }, // right button
	{ -1, -1 }
};

const DuctMapPoint DUCT_MAP_DATA[] = {
	// type //shapetype // x    // y
	{  0x0,     0xB,     0x0,   0x4B0, },
	{  0xD,    0x10,    0x64,   0x4B0, },
	{  0xA,     0xD,   0x12C,   0x4B0, },
	{  0xE,     0xA,    0xC8,   0x44C, },
	{  0xE,     0xF,    0xC8,   0x3E8, },
	{  0x5,     0xD,    0xC8,   0x384, },
	{  0xE,     0xA,    0xC8,   0x320, },
	{  0xC,     0xC,    0xC8,   0x2BC, },
	{  0xD,     0xB,    0x64,   0x2BC, },
	{  0x0,    0x10,     0x0,   0x2BC, },
	{  0xD,     0xB,   0x12C,   0x384, },
	{  0xD,    0x10,   0x190,   0x384, },
	{  0x3,     0xA,   0x1F4,   0x514, },
	{  0xE,     0xF,   0x1F4,   0x4B0, },
	{  0xE,     0xA,   0x1F4,   0x44C, },
	{  0xE,     0xF,   0x1F4,   0x3E8, },
	{ 0xFF,     0xD,     0x0,   0x4B0, },
	{  0xE,     0xA,   0x1F4,   0x320, },
	{  0x2,     0xF,   0x1F4,   0x2BC, },
	{  0xD,     0xB,   0x258,   0x384, },
	{  0xD,    0x10,   0x2BC,   0x384, },
	{  0xD,     0xB,   0x320,   0x384, },
	{  0xD,    0x10,   0x384,   0x384, },
	{  0x1,     0xB,   0x578,   0x514, },
	{  0xD,    0x10,   0x514,   0x514, },
	{  0xD,     0xB,   0x4B0,   0x514, },
	{  0xD,    0x10,   0x44C,   0x514, },
	{  0x9,     0xD,   0x3E8,   0x514, },
	{  0xE,     0xA,   0x3E8,   0x4B0, },
	{  0xE,     0xF,   0x3E8,   0x44C, },
	{  0xE,     0xA,   0x3E8,   0x3E8, },
	{ 0xFF,     0xD,     0x0,   0x2BC, },
	{  0xE,     0xA,   0x3E8,   0x320, },
	{  0xE,     0xF,   0x3E8,   0x2BC, },
	{  0xE,     0xA,   0x3E8,   0x258, },
	{  0x5,     0xD,   0x3E8,   0x1F4, },
	{  0xE,     0xA,   0x3E8,   0x190, },
	{  0xE,     0xF,   0x3E8,   0x12C, },
	{  0xE,     0xA,   0x3E8,    0xC8, },
	{  0xB,     0xD,   0x3E8,    0x64, },
	{  0xD,     0xB,   0x44C,    0x64, },
	{  0x8,     0xC,   0x4B0,    0xC8, },
	{  0x2,     0xA,   0x4B0,     0x0, },
	{  0xD,     0xB,   0x514,    0x64, },
	{  0xD,    0x10,   0x578,    0x64, },
	{  0xD,     0xB,   0x5DC,    0x64, },
	{  0xD,    0x10,   0x640,    0x64, },
	{  0x8,     0xC,   0x6A4,    0xC8, },
	{  0xC,     0xA,   0x6A4,     0x0, },
	{  0xD,     0xB,   0x708,    0x64, },
	{  0xD,    0x10,   0x76C,    0x64, },
	{  0x7,     0xC,   0x7D0,    0x64, },
	{  0xD,     0xB,   0x834,    0x64, },
	{  0x8,     0xC,   0x898,    0xC8, },
	{  0x2,     0xA,   0x898,     0x0, },
	{  0xD,     0xC,   0x8FC,    0x64, },
	{  0x7,     0xB,   0x960,    0x64, },
	{  0xD,    0x10,   0x9C4,    0x64, },
	{  0xD,     0xB,   0xA28,    0x64, },
	{  0x1,    0x10,   0xA8C,    0x64, },
	{  0xE,     0xA,   0x960,    0xC8, },
	{  0x9,     0xD,   0x960,   0x12C, },
	{  0xD,     0xB,   0x9C4,   0x12C, },
	{  0xD,    0x10,   0xA28,   0x12C, },
	{  0xD,     0xB,   0xA8C,   0x12C, },
	{  0x1,    0x10,   0x8F0,   0x17C, },
	{  0xE,     0xA,   0x7D0,    0xC8, },
	{  0x6,     0xD,   0x834,   0x12C, },
	{  0xD,     0xA,   0x76C,   0x12C, },
	{  0x0,     0xF,   0x708,   0x12C, },
	{  0xE,     0xA,   0x7D0,   0x190, },
	{  0x9,     0xD,   0x7D0,   0x1F4, },
	{  0xD,     0xB,   0x834,   0x1F4, },
	{  0xD,    0x10,   0x898,   0x1F4, },
	{  0xD,     0xB,   0x8FC,   0x1F4, },
	{  0xD,    0x10,   0x960,   0x1F4, },
	{  0xD,     0xB,   0x9C4,   0x1F4, },
	{  0xD,    0x10,   0xA28,   0x1F4, },
	{  0xC,     0xC,   0xA8C,   0x1F4, },
	{  0xE,     0xA,   0xA8C,   0x258, },
	{  0xE,     0xF,   0xA8C,   0x2BC, },
	{  0xE,     0xA,   0xA8C,   0x320, },
	{  0xE,     0xF,   0xA8C,   0x384, },
	{  0xA,     0xD,   0xAF0,   0x3E8, },
	{  0xD,     0xB,   0xA28,   0x3E8, },
	{  0xD,    0x10,   0x44C,   0x1F4, },
	{  0xD,     0xB,   0x4B0,   0x1F4, },
	{  0xD,    0x10,   0x514,   0x1F4, },
	{  0x8,     0xC,   0x578,   0x258, },
	{  0xE,     0xA,   0x578,   0x190, },
	{  0x2,     0xF,   0x578,   0x12C, },
	{  0xD,     0xB,   0x5DC,   0x1F4, },
	{  0x7,     0xC,   0x640,   0x1F4, },
	{  0xD,     0xB,   0x6A4,   0x1F4, },
	{  0x1,    0x10,   0x708,   0x1F4, },
	{  0xE,     0xA,   0x640,   0x258, },
	{  0xE,     0xF,   0x640,   0x2BC, },
	{  0xE,     0xA,   0x640,   0x320, },
	{  0xD,     0xB,   0x44C,   0x384, },
	{  0xD,    0x10,   0x4B0,   0x384, },
	{  0x7,     0xC,   0x514,   0x384, },
	{  0xE,     0xA,   0x514,   0x3E8, },
	{  0x3,     0xF,   0x514,   0x44C, },
	{  0xD,     0xB,   0x578,   0x384, },
	{  0xD,    0x10,   0x5DC,   0x384, },
	{ 0xFF,     0xC,   0x3E8,   0x578, },
	{  0xE,     0xB,   0x640,   0x3E8, },
	{  0xE,    0x10,   0x640,   0x44C, },
	{  0xE,     0xB,   0x640,   0x4B0, },
	{  0x3,    0x10,   0x640,   0x514, },
	{  0xD,     0xB,   0x6A4,   0x384, },
	{  0x1,    0x10,   0x708,   0x384, },
	{ 0xFF,     0xC,   0x1F4,   0x578, },
	{ 0xFF,     0xC,    0xC8,   0x514, },
	{ 0xFF,     0xD,   0x12C,   0x2BC, },
	{ 0xFF,     0xD,   0x5DC,   0x514, },
	{ 0xFF,     0xC,   0x3E8,    0x64, },
	{ 0xFF,     0xD,   0xAF0,    0x64, },
	{ 0xFF,     0xC,   0x960,   0x190, },
	{ 0xFF,     0xD,   0x708,   0x12C, },
	{ 0xFF,     0xC,   0x7D0,   0x258, },
	{ 0xFF,     0xD,   0xAF0,   0x1F4, },
	{ 0xFF,     0xC,   0xA8C,   0x44C, },
	{ 0xFF,     0xC,   0x578,   0x12C, },
	{ 0xFF,     0xD,   0x76C,   0x1F4, },
	{ 0xFF,     0xC,   0x514,   0x4B0, },
	{ 0xFF,     0xC,   0x640,   0x578, },
	{ 0xFF,     0xD,   0x76C,   0x384, },
	{  0x4,     0xE,   0x1F4,   0x384, },
	{  0x4,     0xE,   0x3E8,   0x384, },
	{  0x4,     0xE,   0x640,   0x384, },
	{ 0xFF,     0xE,    0xC8,   0x384, },
	{ 0xFF,     0xE,    0xC8,   0x2BC, },
	{ 0xFF,     0xE,    0xC8,   0x4B0, },
	{ 0xFF,     0xE,   0x3E8,   0x514, },
	{ 0xFF,     0xE,   0x3E8,    0x64, },
	{ 0xFF,     0xE,   0x514,   0x384, },
	{ 0xFF,     0xE,   0x4B0,     0x0, },
	{ 0xFF,     0xE,   0x578,   0x1F4, },
	{ 0xFF,     0xE,   0x640,   0x1F4, },
	{ 0xFF,     0xE,   0x6A4,     0x0, },
	{ 0xFF,     0xE,   0x7D0,   0x12C, },
	{ 0xFF,     0xE,   0x7D0,   0x1F4, },
	{ 0xFF,     0xE,   0x898,     0x0, },
	{ 0xFF,     0xE,   0x960,    0x64, },
	{ 0xFF,     0xE,   0x960,   0x12C, },
	{ 0xFF,     0xE,   0xA8C,   0x1F4, },
	{ 0xFF,     0xE,   0xA8C,   0x3E8, },
	{ 0xFF,     0xE,   0x3E8,   0x1F4, },
    { -1,  -1,  -1,  -1, },
};

static const Point3 DUCT_SHAPE_0_POINTS[] = {
	{    0,  300,    0 }, {    0,  100,    0 },
	{  100,  400,    0 }, {  100,    0,    0 },
	{  300,  400,    0 }, {  300,    0,    0 },
	{  400,  300,    0 }, {  400,  100,    0 },
	{    0,  300,  500 }, {    0,  100,  500 },
	{  100,  400,  500 }, {  100,    0,  500 },
	{  300,  400,  500 }, {  300,    0,  500 },
	{  400,  300,  500 }, {  400,  100,  500 },
};
static const Point3 DUCT_SHAPE_1_POINTS[] = {
	{    0,  300,  400 }, {    0,  100,  400 },
	{    0,  400,  300 }, {    0,    0,  300 },
	{    0,  400,  100 }, {    0,    0,  100 },
	{    0,  300,    0 }, {    0,  100,    0 },
	{  500,  300,  400 }, {  500,  100,  400 },
	{  500,  400,  300 }, {  500,    0,  300 },
	{  500,  400,  100 }, {  500,    0,  100 },
	{  500,  300,    0 }, {  500,  100,    0 },
};
static const Point3 DUCT_SHAPE_2_POINTS[] = {
	{    0,  300,  500 }, {    0,  100,  500 },
	{  100,  400,  500 }, {  100,    0,  500 },
	{  300,  400,  500 }, {  300,    0,  500 },
	{  400,  300,  500 }, {  400,  100,  500 },
	{   40,  300,  300 }, {   40,  100,  300 },
	{  130,  400,  340 }, {  130,    0,  340 },
	{  320,  400,  400 }, {  320,    0,  400 },
	{  410,  300,  450 }, {  410,  100,  450 },
	{  170,  300,  120 }, {  170,  100,  120 },
	{  230,  400,  200 }, {  230,    0,  200 },
	{  380,  400,  340 }, {  380,    0,  340 },
	{  440,  300,  420 }, {  440,  100,  420 },
	{  330,  300,   20 }, {  330,  100,   20 },
	{  360,  400,  120 }, {  360,    0,  120 },
	{  500,  300,  400 }, {  500,  100,  400 },
	{  500,  400,  300 }, {  500,    0,  300 },
	{  500,  400,  100 }, {  500,    0,  100 },
	{  500,  300,    0 }, {  500,  100,    0 },
};
static const Point3 DUCT_SHAPE_3_POINTS[] = {
	{  500,  300,  500 }, {  500,  100,  500 },
	{  400,  400,  500 }, {  400,    0,  500 },
	{  200,  400,  500 }, {  200,    0,  500 },
	{  100,  300,  500 }, {  100,  100,  500 },
	{  460,  300,  300 }, {  460,  100,  300 },
	{  370,  400,  340 }, {  370,    0,  340 },
	{  180,  400,  400 }, {  180,    0,  400 },
	{   90,  300,  450 }, {   90,  100,  450 },
	{  330,  300,  120 }, {  330,  100,  120 },
	{  270,  400,  200 }, {  270,    0,  200 },
	{  120,  400,  340 }, {  120,    0,  340 },
	{   60,  300,  420 }, {   60,  100,  420 },
	{  170,  300,   20 }, {  170,  100,   20 },
	{  140,  400,  120 }, {  140,    0,  120 },
	{    0,  300,  400 }, {    0,  100,  400 },
	{    0,  400,  300 }, {    0,    0,  300 },
	{    0,  400,  100 }, {    0,    0,  100 },
	{    0,  300,    0 }, {    0,  100,    0 },
};
static const Point3 DUCT_SHAPE_4_POINTS[] = {
	{    0,  300,  500 }, {    0,  100,  500 },
	{    0,  400,  400 }, {    0,    0,  400 },
	{    0,  400,  200 }, {    0,    0,  200 },
	{    0,  300,  100 }, {    0,  100,  100 },
	{  170,  300,  480 }, {  170,  100,  480 },
	{  140,  400,  380 }, {  140,    0,  380 },
	{  120,  400,  160 }, {  120,    0,  160 },
	{   60,  300,   80 }, {   60,  100,   80 },
	{  330,  300,  380 }, {  330,  100,  380 },
	{  270,  400,  300 }, {  270,    0,  300 },
	{  180,  400,  200 }, {  180,    0,  200 },
	{   90,  300,   50 }, {   90,  100,   50 },
	{  460,  300,  200 }, {  460,  100,  200 },
	{  370,  400,  160 }, {  370,    0,  160 },
	{  100,  300,    0 }, {  100,  100,    0 },
	{  200,  400,    0 }, {  200,    0,    0 },
	{  400,  400,    0 }, {  400,    0,    0 },
	{  500,  300,    0 }, {  500,  100,    0 },
};
static const Point3 DUCT_SHAPE_5_6_POINTS[] = {
	{  500,  300,  500 }, {  500,  100,  500 },
	{  500,  400,  400 }, {  500,    0,  400 },
	{  500,  400,  200 }, {  500,    0,  200 },
	{  500,  300,  100 }, {  500,  100,  100 },
	{  330,  300,  480 }, {  330,  100,  480 },
	{  360,  400,  380 }, {  360,    0,  380 },
	{  380,  400,  160 }, {  380,    0,  160 },
	{  440,  300,   80 }, {  440,  100,   80 },
	{  170,  300,  380 }, {  170,  100,  380 },
	{  230,  400,  300 }, {  230,    0,  300 },
	{  320,  400,  200 }, {  320,    0,  200 },
	{  410,  300,   50 }, {  410,  100,   50 },
	{   40,  300,  200 }, {   40,  100,  200 },
	{  130,  400,  160 }, {  130,    0,  160 },
	{  400,  300,    0 }, {  400,  100,    0 },
	{  300,  400,    0 }, {  300,    0,    0 },
	{  100,  400,    0 }, {  100,    0,    0 },
	{    0,  300,    0 }, {    0,  100,    0 },
};
static const Point3 DUCT_SHAPE_7_POINTS[] = {
	{    0,  300,    0 }, {    0,  100,    0 },
	{  100,  400,    0 }, {  100,    0,    0 },
	{  300,  400,    0 }, {  300,    0,    0 },
	{  400,  300,    0 }, {  400,  100,    0 },
	{    0,  300,  500 }, {    0,  100,  500 },
	{  100,  400,  500 }, {  100,    0,  500 },
	{  300,  400,  500 }, {  300,    0,  500 },
	{  400,  300,  500 }, {  400,  100,  500 },
};
static const Point3 DUCT_SHAPE_8_9_POINTS[] = {
	{    0,  300,  400 }, {    0,  100,  400 },
	{    0,  400,  300 }, {    0,    0,  300 },
	{    0,  400,  100 }, {    0,    0,  100 },
	{    0,  300,    0 }, {    0,  100,    0 },
	{    0,  300,  400 }, {    0,  100,  400 },
	{    0,  400,  300 }, {    0,    0,  300 },
	{    0,  400,  100 }, {    0,    0,  100 },
	{    0,  300,    0 }, {    0,  100,    0 },
};
static const Point3 DUCT_SHAPE_10_POINTS[] = {
	{    0,  100,    0 }, {    0,    0,    0 },
	{    0,  100,  100 }, {    0,    0,  100 },
	{  100,  100,  100 }, {  100,    0,  100 },
	{  100,  100,    0 }, {  100,    0,    0 },
};
static const Point3 DUCT_SHAPE_11_POINTS[] = {
	{    0,  100,    0 }, {    0,    0,    0 },
	{    0,  100,  100 }, {    0,    0,  100 },
	{  100,  100,  100 }, {  100,    0,  100 },
	{  100,  100,    0 }, {  100,    0,    0 },
};
static const Point3 DUCT_SHAPE_12_POINTS[] = {
	{    0,  100,    0 }, {    0,    0,    0 },
	{  100,  100,    0 }, {  100,    0,    0 },
};
static const Point3 DUCT_SHAPE_13_POINTS[] = {
	{    0,  100,    0 }, {    0,    0,    0 },
	{    0,  100,  100 }, {    0,    0,  100 },
};
static const Point3 DUCT_SHAPE_14_POINTS[] = {
	{    0,  100,    0 }, {    0,    0,    0 },
	{    0,  100,  100 }, {    0,    0,  100 },
	{  100,  100,  100 }, {  100,    0,  100 },
	{  100,  100,    0 }, {  100,    0,    0 },
};
static const Point3 DUCT_SHAPE_15_POINTS[] = {
	{    0,  100,    0 }, {    0,    0,    0 },
	{    0,  100,  100 }, {    0,    0,  100 },
	{  100,  100,  100 }, {  100,    0,  100 },
	{  100,  100,    0 }, {  100,    0,    0 },
};
static const Point3 DUCT_SHAPE_16_POINTS[] = {
	{    0,  100,    0 }, {    0,    0,    0 },
	{    0,  100,  100 }, {    0,    0,  100 },
	{  100,  100,  100 }, {  100,    0,  100 },
	{  100,  100,    0 }, {  100,    0,    0 },
};

static const uint16 DUCT_SHAPE_0_DATA[] = {
    1, 4, 8, 0xA, 2, 0, 2, 4, 1, 9, 8, 0, 3, 4, 0xB, 3,
    1, 9, 4, 4, 0xC, 0xA, 2, 4, 5, 4, 0xD, 0xB, 3, 5, 6, 4,
    0xE, 0xC, 4, 6, 7, 4, 0xF, 0xE, 6, 7, 8, 4, 0xF, 0xD, 5, 7
};
static const uint16 DUCT_SHAPE_1_DATA[] = {
    1, 4, 8, 0xA, 2, 0, 2, 4, 1, 9, 8, 0, 3, 4, 0xB, 3, 1, 9,
	4, 4, 0xC, 0xA, 2, 4, 5, 4, 0xD, 0xB, 3, 5, 6, 4, 0xE, 0xC,
	4, 6, 7, 4, 0xF, 0xE, 6, 7, 8, 4, 0xF, 0xD, 5, 7
};
static const uint16 DUCT_SHAPE_2_DATA[] = {
	0x1, 0x4, 0x8, 0xA, 0x2, 0x0, 0x2, 0x4, 0x9, 0xB, 0x3, 0x1,
	0x3, 0x4, 0x8, 0x9, 0x1, 0x0, 0x4, 0x4, 0x10, 0x12, 0xA, 0x8,
	0x5, 0x4, 0x11, 0x13, 0xB, 0x9, 0x6, 0x4, 0x11, 0x10, 0x8, 0x9,
	0x7, 0x4, 0x18, 0x1A, 0x12, 0x10, 0x8, 0x4, 0x19, 0x1B, 0x13, 0x11,
	0x1, 0x4, 0x19, 0x18, 0x10, 0x11, 0x2, 0x4, 0x22, 0x20, 0x1A, 0x18,
	0x3, 0x4, 0x23, 0x21, 0x1B, 0x19, 0x4, 0x4, 0x23, 0x22, 0x18, 0x19,
	0x5, 0x9, 0x2, 0xA, 0x12, 0x1A, 0x20, 0x1E, 0x14, 0xC, 0x4, 0x6,
	0x9, 0x3, 0xB, 0x13, 0x1B, 0x21, 0x1F, 0x15, 0xD, 0x5, 0x7, 0x4,
	0x1C, 0x16, 0x14, 0x1E, 0x8, 0x4, 0x1F, 0x1D, 0x17, 0x15, 0x1, 0x4,
	0x1D, 0x1C, 0x16, 0x17, 0x2, 0x4, 0xC, 0x14, 0x16, 0xE, 0x3, 0x4,
	0xD, 0x15, 0x17, 0xF, 0x4, 0x4, 0x17, 0x16, 0xE, 0xF, 0x5, 0x4,
	0x4, 0xC, 0xE, 0x6, 0x6, 0x4, 0xD, 0xF, 0x7, 0x5, 0x7, 0x4,
	0xF, 0xE, 0x6, 0x7
};
static const uint16 DUCT_SHAPE_3_DATA[] = {
	0x1, 0x4, 0x8, 0xA, 0x2, 0x0, 0x2, 0x4, 0x9, 0xB, 0x3, 0x1,
	0x3, 0x4, 0x8, 0x9, 0x1, 0x0, 0x4, 0x4, 0x10, 0x12, 0xA, 0x8,
	0x5, 0x4, 0x11, 0x13, 0xB, 0x9, 0x6, 0x4, 0x11, 0x10, 0x8, 0x9,
	0x7, 0x4, 0x18, 0x1A, 0x12, 0x10, 0x8, 0x4, 0x19, 0x1B, 0x13, 0x11,
	0x1, 0x4, 0x19, 0x18, 0x10, 0x11, 0x2, 0x4, 0x22, 0x20, 0x1A, 0x18,
	0x3, 0x4, 0x23, 0x21, 0x1B, 0x19, 0x4, 0x4, 0x23, 0x22, 0x18, 0x19,
	0x5, 0x9, 0x2, 0xA, 0x12, 0x1A, 0x20, 0x1E, 0x14, 0xC, 0x4, 0x6,
	0x9, 0x3, 0xB, 0x13, 0x1B, 0x21, 0x1F, 0x15, 0xD, 0x5, 0x7, 0x4,
	0x1C, 0x16, 0x14, 0x1E, 0x8, 0x4, 0x1F, 0x1D, 0x17, 0x15, 0x1, 0x4,
	0x1D, 0x1C, 0x16, 0x17, 0x2, 0x4, 0xC, 0x14, 0x16, 0xE, 0x3, 0x4,
	0xD, 0x15, 0x17, 0xF, 0x4, 0x4, 0x17, 0x16, 0xE, 0xF, 0x5, 0x4,
	0x4, 0xC, 0xE, 0x6, 0x6, 0x4, 0xD, 0xF, 0x7, 0x5, 0x7, 0x4,
	0xF, 0xE, 0x6, 0x7
};
static const uint16 DUCT_SHAPE_4_DATA[] = {
	0x1, 0x4, 0x8, 0xA, 0x2, 0x0, 0x2, 0x4, 0x9, 0xB, 0x3, 0x1,
	0x3, 0x4, 0x8, 0x9, 0x1, 0x0, 0x4, 0x4, 0x10, 0x12, 0xA, 0x8,
	0x5, 0x4, 0x11, 0x13, 0xB, 0x9, 0x6, 0x4, 0x11, 0x10, 0x8, 0x9,
	0x7, 0x4, 0x18, 0x1A, 0x12, 0x10, 0x8, 0x4, 0x19, 0x1B, 0x13, 0x11,
	0x1, 0x4, 0x19, 0x18, 0x10, 0x11, 0x2, 0x4, 0x22, 0x20, 0x1A, 0x18,
	0x3, 0x4, 0x23, 0x21, 0x1B, 0x19, 0x4, 0x4, 0x23, 0x22, 0x18, 0x19,
	0x5, 0x9, 0x2, 0xA, 0x12, 0x1A, 0x20, 0x1E, 0x14, 0xC, 0x4, 0x6,
	0x9, 0x3, 0xB, 0x13, 0x1B, 0x21, 0x1F, 0x15, 0xD, 0x5, 0x7, 0x4,
	0x1C, 0x16, 0x14, 0x1E, 0x8, 0x4, 0x1F, 0x1D, 0x17, 0x15, 0x1, 0x4,
	0x1D, 0x1C, 0x16, 0x17, 0x2, 0x4, 0xC, 0x14, 0x16, 0xE, 0x3, 0x4,
	0xD, 0x15, 0x17, 0xF, 0x4, 0x4, 0x17, 0x16, 0xE, 0xF, 0x5, 0x4,
	0x4, 0xC, 0xE, 0x6, 0x6, 0x4, 0xD, 0xF, 0x7, 0x5, 0x7, 0x4,
	0xF, 0xE, 0x6, 0x7
};
static const uint16 DUCT_SHAPE_5_6_DATA[] = {
	0x1, 0x4, 0x8, 0xA, 0x2, 0x0, 0x2, 0x4, 0x9, 0xB, 0x3, 0x1,
	0x3, 0x4, 0x8, 0x9, 0x1, 0x0, 0x4, 0x4, 0x10, 0x12, 0xA, 0x8,
	0x5, 0x4, 0x11, 0x13, 0xB, 0x9, 0x6, 0x4, 0x11, 0x10, 0x8, 0x9,
	0x7, 0x4, 0x18, 0x1A, 0x12, 0x10, 0x8, 0x4, 0x19, 0x1B, 0x13, 0x11,
	0x1, 0x4, 0x19, 0x18, 0x10, 0x11, 0x2, 0x4, 0x22, 0x20, 0x1A, 0x18,
	0x3, 0x4, 0x23, 0x21, 0x1B, 0x19, 0x4, 0x4, 0x23, 0x22, 0x18, 0x19,
	0x5, 0x9, 0x2, 0xA, 0x12, 0x1A, 0x20, 0x1E, 0x14, 0xC, 0x4, 0x6,
	0x9, 0x3, 0xB, 0x13, 0x1B, 0x21, 0x1F, 0x15, 0xD, 0x5, 0x7, 0x4,
	0x1C, 0x16, 0x14, 0x1E, 0x8, 0x4, 0x1F, 0x1D, 0x17, 0x15, 0x1, 0x4,
	0x1D, 0x1C, 0x16, 0x17, 0x2, 0x4, 0xC, 0x14, 0x16, 0xE, 0x3, 0x4,
	0xD, 0x15, 0x17, 0xF, 0x4, 0x4, 0x17, 0x16, 0xE, 0xF, 0x5, 0x4,
	0x4, 0xC, 0xE, 0x6, 0x6, 0x4, 0xD, 0xF, 0x7, 0x5, 0x7, 0x4,
	0xF, 0xE, 0x6, 0x7
};
static const uint16 DUCT_SHAPE_7_DATA[] = {
	0x2, 0x4, 0x8, 0xA, 0x2, 0x0, 0x3, 0x4, 0x1, 0x9, 0x8, 0x0,
	0x4, 0x4, 0xB, 0x3, 0x1, 0x9, 0x5, 0x4, 0xC, 0xA, 0x2, 0x4,
	0x6, 0x4, 0xD, 0xB, 0x3, 0x5, 0x7, 0x4, 0xE, 0xC, 0x4, 0x6,
	0x8, 0x4, 0xF, 0xE, 0x6, 0x7, 0x1, 0x4, 0xF, 0xD, 0x5, 0x7
};
static const uint16 DUCT_SHAPE_8_9_DATA[] = {
	0x2, 0x4, 0x8, 0xA, 0x2, 0x0, 0x3, 0x4, 0x1, 0x9, 0x8, 0x0,
	0x4, 0x4, 0xB, 0x3, 0x1, 0x9, 0x5, 0x4, 0xC, 0xA, 0x2, 0x4,
	0x6, 0x4, 0xD, 0xB, 0x3, 0x5, 0x7, 0x4, 0xE, 0xC, 0x4, 0x6,
	0x8, 0x4, 0xF, 0xE, 0x6, 0x7, 0x1, 0x4, 0xF, 0xD, 0x5, 0x7
};
static const uint16 DUCT_SHAPE_10_DATA[] = {
	5, 4, 0, 2, 3, 1, 3, 4, 0, 6, 4, 2,
	4, 4, 1, 7, 5, 3, 6, 4, 6, 7, 5, 4
};
static const uint16 DUCT_SHAPE_11_DATA[] = {
	3, 4, 0, 6, 4, 2, 4, 4, 1, 7, 5, 3,
	5, 4, 0, 6, 7, 1, 6, 4, 2, 4, 5, 3
};
static const uint16 DUCT_SHAPE_12_DATA[] = {
	7, 4, 0, 1, 3, 2
};
static const uint16 DUCT_SHAPE_13_DATA[] = {
	7, 4, 0, 1, 3, 2
};
static const uint16 DUCT_SHAPE_14_DATA[] = {
	1, 4, 0, 6, 4, 2, 2, 4, 1, 7, 5, 3
};
static const uint16 DUCT_SHAPE_15_DATA[] = {
    6, 4, 0, 2, 3, 1, 4, 4, 0, 6, 4, 2,
    3, 4, 1, 7, 5, 3, 5, 4, 6, 7, 5, 4
};
static const uint16 DUCT_SHAPE_16_DATA[] = {
	4, 4, 0, 6, 4, 2, 3, 4, 1, 7, 5, 3,
	6, 4, 0, 6, 7, 1, 5, 4, 2, 4, 5, 3
};

static const DuctShape DUCT_SHAPE_0[] = {
	{ 16, 8, DUCT_SHAPE_0_POINTS, DUCT_SHAPE_0_DATA }
};

static const DuctShape DUCT_SHAPE_1[] = {
	{ 16, 8, DUCT_SHAPE_1_POINTS, DUCT_SHAPE_1_DATA }
};

static const DuctShape DUCT_SHAPE_2[] = {
	{ 36, 23, DUCT_SHAPE_2_POINTS, DUCT_SHAPE_2_DATA }
};

static const DuctShape DUCT_SHAPE_3[] = {
	{ 36, 23, DUCT_SHAPE_3_POINTS, DUCT_SHAPE_3_DATA }
};

static const DuctShape DUCT_SHAPE_4[] = {
	{ 36, 23, DUCT_SHAPE_4_POINTS, DUCT_SHAPE_4_DATA }
};

static const DuctShape DUCT_SHAPE_5_6[] = {
	{ 36, 23, DUCT_SHAPE_5_6_POINTS, DUCT_SHAPE_5_6_DATA }
};

static const DuctShape DUCT_SHAPE_7[] = {
	{ 16, 8, DUCT_SHAPE_7_POINTS, DUCT_SHAPE_7_DATA }
};

static const DuctShape DUCT_SHAPE_8_9[] = {
	{ 16, 8, DUCT_SHAPE_8_9_POINTS, DUCT_SHAPE_8_9_DATA }
};

static const DuctShape DUCT_SHAPE_10[] = {
	{ 8, 4, DUCT_SHAPE_10_POINTS, DUCT_SHAPE_10_DATA }
};

static const DuctShape DUCT_SHAPE_11[] = {
	{ 8, 4, DUCT_SHAPE_11_POINTS, DUCT_SHAPE_11_DATA }
};

static const DuctShape DUCT_SHAPE_12[] = {
	{ 4, 1, DUCT_SHAPE_12_POINTS, DUCT_SHAPE_12_DATA }
};

static const DuctShape DUCT_SHAPE_13[] = {
	{ 4, 1, DUCT_SHAPE_13_POINTS, DUCT_SHAPE_13_DATA }
};

static const DuctShape DUCT_SHAPE_14[] = {
	{ 8, 2, DUCT_SHAPE_14_POINTS, DUCT_SHAPE_14_DATA }
};

static const DuctShape DUCT_SHAPE_15[] = {
	{ 8, 4, DUCT_SHAPE_15_POINTS, DUCT_SHAPE_15_DATA }
};

static const DuctShape DUCT_SHAPE_16[] = {
	{ 8, 4, DUCT_SHAPE_16_POINTS, DUCT_SHAPE_16_DATA }
};


const DuctShape *DUCT_SHAPE_DATA[17] {
	DUCT_SHAPE_0, DUCT_SHAPE_1, DUCT_SHAPE_2, DUCT_SHAPE_3,
	DUCT_SHAPE_4, DUCT_SHAPE_5_6, DUCT_SHAPE_5_6, DUCT_SHAPE_7,
	DUCT_SHAPE_8_9, DUCT_SHAPE_8_9, DUCT_SHAPE_10, DUCT_SHAPE_11,
	DUCT_SHAPE_12, DUCT_SHAPE_13, DUCT_SHAPE_14, DUCT_SHAPE_15,
	DUCT_SHAPE_16
};

static const byte FONT1_WIDTHS[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0x5,  0x3,  0x4,  0x6,  0x5,  0x6,  0x6,  0x3,
	0x4,  0x4,  0x5,  0x6,  0x3,  0x5,  0x3,  0x6,
	0x6,  0x4,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,
	0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,
	0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,
	0x6,  0x4,  0x6,  0x5,  0x5,  0x6,  0x6,  0x6,
	0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,
	0x6,  0x6,  0x6,  0x4,  0x6,  0x4,  0x4,  0x6,
	0x3, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

static const byte FONT2_WIDTHS[] = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF,  0x5, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0x5,  0x3,  0x5,  0x6,  0x6,  0x5,  0x6,  0x3,
	0x5,  0x5,  0x6,  0x5,  0x3,  0x5,  0x3,  0x5,
	0x5,  0x5,  0x5,  0x5,  0x5,  0x5,  0x5,  0x5,
	0x5,  0x5,  0x5,  0x5,  0x5,  0x5,  0x5,  0x5,
	0x5,  0x5,  0x5,  0x5,  0x5,  0x5,  0x5,  0x5,
	0x5,  0x5,  0x5,  0x5,  0x5,  0x7,  0x6,  0x5,
	0x5,  0x5,  0x5,  0x5,  0x5,  0x5,  0x5,  0x7,
	0x5,  0x5,  0x5,  0x4,  0x5,  0x4,  0x4,  0x5,
	0x3, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
};

// Offsets into the data array for each char (starts at 0x4d13)
static const int FONT1_OFFSETS[] = {
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x2, 0xa, 0x12, 0x20, 0x37, 0x4e, 0x65,
	0x6a, 0x75, 0x80, 0x97, 0x9f, 0xa4, 0xa9, 0xae,
	0xbf, 0xd6, 0xe1, 0xf2, 0x103, 0x10e, 0x11f, 0x130,
	0x13e, 0x155, 0x166, 0x16e, 0x176, 0x187, 0x18f, 0x1a0,
	0x1b1, 0x1c2, 0x1d0, 0x1e4, 0x1ef, 0x1fd, 0x20b, 0x216,
	0x227, 0x232, 0x23d, 0x24b, 0x25f, 0x267, 0x278, 0x289,
	0x297, 0x2a5, 0x2b6, 0x2ca, 0x2db, 0x2e3, 0x2ee, 0x2ff,
	0x310, 0x32d, 0x33e, 0x34f, 0x35a, 0x36b, 0x376, 0x381,
	0x386, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
};

// Offsets into the data array for each char (starts at 0x485d)
static const int FONT2_OFFSETS[] = {
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x0, 0x2, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x13, 0x1b, 0x23, 0x31, 0x48, 0x59, 0x70,
	0x75, 0x80, 0x8b, 0xa2, 0xaa, 0xaf, 0xb4, 0xb9,
	0xc4, 0xd2, 0xdd, 0xee, 0xff, 0x10a, 0x11b, 0x12c,
	0x137, 0x14e, 0x15f, 0x167, 0x16f, 0x180, 0x188, 0x199,
	0x1a7, 0x1b5, 0x1c3, 0x1d7, 0x1e2, 0x1f0, 0x1fe, 0x209,
	0x217, 0x222, 0x22d, 0x238, 0x246,0x24e, 0x25f, 0x26d,
	0x27b, 0x289, 0x29a, 0x2ab,0x2bc, 0x2c4, 0x2cf, 0x2dd,
	0x2ee, 0x2ff, 0x30a,0x31b, 0x326, 0x331, 0x33c, 0x347,
	0x34c, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0
};

static const byte FONT1_DATA[] = {
	0xFF, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x01, 0x01, 0x03, 0xFF, 0xFF, 0x01, 0x01, 0x02, 0x03, 0x01, 0x02, 0xFF, 0x01, 0x05, 0x02, 0x01, 0x05, 0x04, 0xFF, 0x02, 0x01, 0x05, 0x04, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x00, 0x02, 0x04, 0x01, 0x01, 0x01, 0x02, 0x02, 0x03, 0x03, 0x04, 0x04, 0x04, 0x01, 0x03, 0x05, 0x02, 0x02, 0x06, 0xFF, 0xFF, 0x01, 0x01, 0x01, 0x05, 0x05, 0x01, 0x04, 0x04, 0x02, 0x03, 0x03, 0x03, 0x02, 0x02, 0x04, 0x01, 0x01, 0x05, 0x05, 0x05, 0x05, 0xFF, 0xFF, 0x02, 0x03, 0x01, 0x02, 0x03, 0x02, 0x04, 0x04, 0x02, 0x03, 0x03, 0x04, 0x02, 0x04, 0x05, 0xFF, 0x01, 0x03, 0x04, 0x04, 0x03, 0x04, 0xFF, 0xFF, 0x01, 0x01, 0x02, 0xFF, 0x02, 0x02, 0x01, 0x02, 0x02, 0x05, 0xFF, 0x01, 0x02, 0x04, 0xFF, 0x01, 0x01, 0x01, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x04, 0xFF, 0x01, 0x01, 0x01, 0x04, 0x04, 0x01, 0x01, 0x04, 0x03, 0x01, 0x01, 0x05, 0x04, 0x04, 0x05, 0xFF, 0x02, 0x02, 0x04, 0x03, 0x02, 0x04, 0xFF, 0x01, 0x05, 0x03, 0xFF, 0x03, 0x01, 0x05, 0xFF, 0xFF, 0x01, 0x05, 0x06, 0xFF, 0x02, 0x04, 0x03, 0xFF, 0xFF, 0x01, 0x02, 0x05, 0xFF, 0xFF, 0x01, 0x01, 0x05, 0x02, 0x02, 0x04, 0x03, 0x03, 0x03, 0x04, 0x04, 0x02, 0x05, 0x05, 0x01, 0xFF, 0xFF, 0x02, 0x04, 0x01, 0x02, 0x04, 0x05, 0x02, 0x02, 0x04, 0x03, 0x03, 0x03, 0x04, 0x04, 0x02, 0xFF, 0x01, 0x02, 0x04, 0x05, 0x02, 0x04, 0xFF, 0x01, 0x01, 0x02, 0x01, 0x03, 0x05, 0xFF, 0x02, 0x01, 0x04, 0xFF, 0x01, 0x04, 0x01, 0x05, 0x05, 0x02, 0x03, 0x04, 0x03, 0x02, 0x02, 0x04, 0x01, 0x05, 0x05, 0xFF, 0xFF, 0x01, 0x04, 0x01, 0x05, 0x05, 0x02, 0x02, 0x04, 0x03, 0x05, 0x05, 0x04, 0x01, 0x04, 0x05, 0xFF, 0xFF, 0x01, 0x01, 0x02, 0x01, 0x05, 0x03, 0xFF, 0x04, 0x01, 0x05, 0xFF, 0x02, 0x05, 0x01, 0x02, 0x04, 0x03, 0x05, 0x05, 0x04, 0x01, 0x04, 0x05, 0xFF, 0x01, 0x01, 0x03, 0xFF, 0x02, 0x05, 0x01, 0x02, 0x04, 0x03, 0x05, 0x05, 0x04, 0x02, 0x04, 0x05, 0xFF, 0x01, 0x02, 0x04, 0xFF, 0x01, 0x05, 0x01, 0x05, 0x05, 0x02, 0x04, 0x04, 0x03, 0xFF, 0x03, 0x04, 0x05, 0xFF, 0x02, 0x04, 0x01, 0x01, 0x01, 0x02, 0x05, 0x05, 0x02, 0x02, 0x04, 0x03, 0x01, 0x01, 0x04, 0x05, 0x05, 0x04, 0x02, 0x04, 0x05, 0xFF, 0xFF, 0x02, 0x04, 0x01, 0x01, 0x01, 0x02, 0x02, 0x04, 0x03, 0x01, 0x04, 0x05, 0xFF, 0x05, 0x02, 0x04, 0xFF, 0x01, 0x01, 0x02, 0x01, 0x01, 0x04, 0xFF, 0xFF, 0x01, 0x01, 0x02, 0xFF, 0x01, 0x04, 0x05, 0xFF, 0x03, 0x03, 0x01, 0x02, 0x02, 0x02, 0x01, 0x01, 0x03, 0x02, 0x02, 0x04, 0x03, 0x03, 0x05, 0xFF, 0xFF, 0x01, 0x04, 0x02, 0x01, 0x04, 0x04, 0xFF, 0xFF, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x02, 0x02, 0x04, 0x01, 0x01, 0x05, 0xFF, 0xFF, 0x02, 0x03, 0x01, 0x01, 0x01, 0x02, 0x03, 0x03, 0x04, 0x03, 0x03, 0x06, 0xFF, 0x04, 0x02, 0x03, 0xFF, 0x02, 0x04, 0x01, 0x05, 0x05, 0x02, 0x04, 0x05, 0x03, 0x02, 0x04, 0x05, 0xFF, 0x01, 0x02, 0x04, 0xFF, 0x02, 0x04, 0x01, 0x02, 0x04, 0x04, 0xFF, 0x01, 0x02, 0x05, 0x05, 0x02, 0x05, 0xFF, 0x02, 0x04, 0x01, 0x05, 0x05, 0x02, 0x02, 0x04, 0x03, 0x05, 0x05, 0x04, 0x02, 0x04, 0x05, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x05, 0x01, 0x02, 0x05, 0x05, 0xFF, 0x01, 0x02, 0x04, 0xFF, 0x02, 0x04, 0x01, 0x02, 0x04, 0x05, 0xFF, 0x01, 0x01, 0x05, 0x05, 0x02, 0x04, 0xFF, 0x02, 0x05, 0x01, 0x02, 0x03, 0x03, 0x02, 0x05, 0x05, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x05, 0x01, 0x02, 0x03, 0x03, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x05, 0x01, 0x04, 0x05, 0x03, 0x02, 0x04, 0x05, 0xFF, 0x01, 0x02, 0x04, 0x05, 0x03, 0x04, 0xFF, 0x02, 0x04, 0x03, 0xFF, 0x01, 0x01, 0x05, 0x05, 0x01, 0x05, 0xFF, 0x01, 0x03, 0x01, 0x01, 0x03, 0x05, 0xFF, 0x02, 0x04, 0x02, 0xFF, 0x03, 0x05, 0x01, 0x01, 0x01, 0x04, 0x02, 0x03, 0x05, 0xFF, 0x04, 0x02, 0x04, 0xFF, 0x04, 0x04, 0x01, 0x03, 0x03, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x05, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x04, 0x05, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x02, 0x04, 0x04, 0x02, 0x03, 0x03, 0x03, 0xFF, 0x01, 0x01, 0x05, 0x05, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0xFF, 0x01, 0x01, 0x05, 0x05, 0x01, 0x05, 0xFF, 0x02, 0x04, 0x01, 0x02, 0x04, 0x05, 0xFF, 0x01, 0x02, 0x04, 0x05, 0x02, 0x04, 0xFF, 0x02, 0x04, 0x01, 0x05, 0x05, 0x02, 0x02, 0x04, 0x03, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x04, 0x01, 0x03, 0x03, 0x04, 0x02, 0x04, 0x05, 0xFF, 0x01, 0x02, 0x04, 0x05, 0x02, 0x06, 0xFF, 0x02, 0x04, 0x01, 0x05, 0x05, 0x02, 0x02, 0x04, 0x03, 0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x04, 0x01, 0x01, 0x01, 0x02, 0x02, 0x04, 0x03, 0x05, 0x05, 0x04, 0x02, 0x04, 0x05, 0xFF, 0xFF, 0x01, 0x05, 0x01, 0xFF, 0x03, 0x02, 0x05, 0xFF, 0x02, 0x04, 0x05, 0xFF, 0x01, 0x01, 0x04, 0x05, 0x01, 0x04, 0xFF, 0x02, 0x02, 0x04, 0x04, 0x04, 0x04, 0x03, 0x03, 0x05, 0xFF, 0x01, 0x01, 0x03, 0x05, 0x01, 0x03, 0xFF, 0x03, 0x03, 0x03, 0x02, 0x02, 0x04, 0x04, 0x04, 0x04, 0xFF, 0x01, 0x01, 0x05, 0x05, 0x01, 0x05, 0xFF, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x01, 0x01, 0x05, 0x02, 0x02, 0x04, 0x04, 0x04, 0x02, 0x05, 0x05, 0x01, 0xFF, 0xFF, 0x02, 0x02, 0x03, 0x04, 0x04, 0x03, 0xFF, 0x01, 0x01, 0x02, 0x05, 0x01, 0x02, 0x03, 0x04, 0x05, 0xFF, 0x01, 0x05, 0x01, 0x04, 0x04, 0x02, 0x03, 0x03, 0x03, 0x02, 0x02, 0x04, 0x01, 0x05, 0x05, 0xFF, 0xFF, 0x02, 0x03, 0x01, 0x02, 0x03, 0x05, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0xFF, 0xFF, 0x01, 0x02, 0x01, 0x01, 0x02, 0x05, 0xFF, 0x02, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x01, 0x01, 0x01, 0x02, 0x03, 0x03, 0x02, 0xFF, 0xFF, 0x01, 0x05, 0x05, 0xFF, 0xFF, 0xFF, 0x01, 0x01, 0x02, 0xFF
};

static const byte FONT2_DATA[] = {
	0xFF, 0xFF, 0x01, 0x03, 0x01, 0x00, 0x04, 0x02, 0x00, 0x04, 0x06, 0x01, 0x03, 0x07, 0xFF, 0x02, 0x00, 0x08, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x01, 0x01, 0x03, 0xFF, 0xFF, 0x01, 0x01, 0x02, 0x03, 0x01, 0x02, 0xFF, 0x01, 0x05, 0x02, 0x01, 0x05, 0x04, 0xFF, 0x02, 0x01, 0x05, 0x04, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x00, 0x02, 0x04, 0x01, 0x01, 0x01, 0x02, 0x02, 0x03, 0x03, 0x04, 0x04, 0x04, 0x01, 0x03, 0x05, 0x02, 0x02, 0x06, 0xFF, 0xFF, 0x01, 0x01, 0x01, 0x02, 0x02, 0x03, 0x03, 0x03, 0x05, 0xFF, 0x01, 0x04, 0x05, 0x03, 0x01, 0x02, 0xFF, 0x02, 0x03, 0x01, 0x02, 0x03, 0x02, 0x04, 0x04, 0x02, 0x03, 0x03, 0x04, 0x02, 0x04, 0x05, 0xFF, 0x01, 0x03, 0x04, 0x04, 0x03, 0x04, 0xFF, 0xFF, 0x01, 0x01, 0x02, 0xFF, 0x02, 0x02, 0x01, 0x02, 0x02, 0x05, 0xFF, 0x01, 0x02, 0x04, 0xFF, 0x01, 0x01, 0x01, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x04, 0xFF, 0x01, 0x01, 0x01, 0x04, 0x04, 0x01, 0x01, 0x04, 0x03, 0x01, 0x01, 0x05, 0x04, 0x04, 0x05, 0xFF, 0x02, 0x02, 0x04, 0x03, 0x02, 0x04, 0xFF, 0x01, 0x03, 0x03, 0xFF, 0x02, 0x02, 0x04, 0xFF, 0xFF, 0x01, 0x05, 0x06, 0xFF, 0x01, 0x03, 0x03, 0xFF, 0xFF, 0x01, 0x02, 0x05, 0xFF, 0xFF, 0x02, 0x02, 0x03, 0xFF, 0x01, 0x04, 0x05, 0x03, 0x01, 0x02, 0xFF, 0x02, 0x02, 0x01, 0x02, 0x02, 0x05, 0xFF, 0x01, 0x02, 0x04, 0x03, 0x02, 0x04, 0xFF, 0x01, 0x01, 0x02, 0x01, 0x03, 0x05, 0xFF, 0x02, 0x01, 0x04, 0xFF, 0x01, 0x02, 0x01, 0x03, 0x03, 0x02, 0x02, 0x02, 0x03, 0x01, 0x01, 0x04, 0x01, 0x03, 0x05, 0xFF, 0xFF, 0x01, 0x02, 0x01, 0x03, 0x03, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x04, 0x01, 0x02, 0x05, 0xFF, 0xFF, 0x01, 0x01, 0x02, 0x01, 0x03, 0x03, 0xFF, 0x03, 0x01, 0x05, 0xFF, 0x01, 0x03, 0x01, 0x01, 0x01, 0x02, 0x01, 0x02, 0x03, 0x03, 0x03, 0x04, 0x01, 0x02, 0x05, 0xFF, 0xFF, 0x02, 0x03, 0x01, 0x02, 0x02, 0x03, 0x03, 0x03, 0x04, 0x02, 0x02, 0x05, 0xFF, 0x01, 0x02, 0x04, 0xFF, 0x01, 0x03, 0x01, 0x03, 0x03, 0x02, 0xFF, 0x02, 0x03, 0x05, 0xFF, 0x02, 0x02, 0x01, 0x01, 0x01, 0x02, 0x03, 0x03, 0x02, 0x02, 0x02, 0x03, 0x01, 0x01, 0x04, 0x03, 0x03, 0x04, 0x02, 0x02, 0x05, 0xFF, 0xFF, 0x02, 0x02, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x01, 0x02, 0x05, 0xFF, 0x03, 0x02, 0x04, 0xFF, 0x01, 0x01, 0x02, 0x01, 0x01, 0x04, 0xFF, 0xFF, 0x01, 0x01, 0x02, 0xFF, 0x01, 0x04, 0x05, 0xFF, 0x03, 0x03, 0x01, 0x02, 0x02, 0x02, 0x01, 0x01, 0x03, 0x02, 0x02, 0x04, 0x03, 0x03, 0x05, 0xFF, 0xFF, 0x01, 0x03, 0x02, 0x01, 0x03, 0x04, 0xFF, 0xFF, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x02, 0x02, 0x04, 0x01, 0x01, 0x05, 0xFF, 0xFF, 0x01, 0x02, 0x01, 0x03, 0x03, 0x02, 0x02, 0x02, 0x03, 0x02, 0x02, 0x05, 0xFF, 0xFF, 0x02, 0x02, 0x01, 0x02, 0x03, 0x05, 0xFF, 0x01, 0x02, 0x04, 0x03, 0x02, 0x03, 0xFF, 0x02, 0x02, 0x01, 0x02, 0x02, 0x04, 0xFF, 0x01, 0x02, 0x05, 0x03, 0x02, 0x05, 0xFF, 0x02, 0x02, 0x01, 0x03, 0x03, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x04, 0x02, 0x02, 0x05, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x03, 0x01, 0x02, 0x03, 0x05, 0xFF, 0x01, 0x02, 0x04, 0xFF, 0x02, 0x02, 0x01, 0x02, 0x02, 0x05, 0xFF, 0x01, 0x01, 0x05, 0x03, 0x02, 0x04, 0xFF, 0x02, 0x03, 0x01, 0x02, 0x02, 0x03, 0x02, 0x03, 0x05, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x03, 0x01, 0x02, 0x02, 0x03, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x03, 0x01, 0x03, 0x03, 0x04, 0x02, 0x03, 0x05, 0xFF, 0x01, 0x02, 0x04, 0xFF, 0x02, 0x02, 0x03, 0xFF, 0x01, 0x01, 0x05, 0x03, 0x01, 0x05, 0xFF, 0x01, 0x03, 0x01, 0x01, 0x03, 0x05, 0xFF, 0x02, 0x04, 0x02, 0xFF, 0x01, 0x01, 0x04, 0x02, 0x02, 0x05, 0xFF, 0x03, 0x01, 0x04, 0xFF, 0x02, 0x02, 0x03, 0xFF, 0x01, 0x01, 0x05, 0x03, 0x01, 0x02, 0x03, 0x04, 0x05, 0xFF, 0x02, 0x03, 0x05, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x02, 0x04, 0x04, 0x02, 0x03, 0x03, 0x03, 0xFF, 0x01, 0x01, 0x05, 0x05, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0xFF, 0x01, 0x01, 0x05, 0x04, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x01, 0x02, 0x02, 0x05, 0xFF, 0x01, 0x02, 0x04, 0x03, 0x02, 0x04, 0xFF, 0x02, 0x02, 0x01, 0x03, 0x03, 0x02, 0x02, 0x02, 0x03, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x01, 0x01, 0x03, 0x04, 0x02, 0x03, 0x05, 0xFF, 0x01, 0x02, 0x03, 0x03, 0x02, 0x03, 0xFF, 0x02, 0x02, 0x01, 0x03, 0x03, 0x02, 0x02, 0x02, 0x03, 0xFF, 0x01, 0x01, 0x05, 0x03, 0x04, 0x05, 0xFF, 0x02, 0x03, 0x01, 0x01, 0x01, 0x02, 0x02, 0x02, 0x03, 0x03, 0x03, 0x04, 0x02, 0x03, 0x05, 0xFF, 0xFF, 0x01, 0x03, 0x01, 0xFF, 0x02, 0x02, 0x05, 0xFF, 0x02, 0x02, 0x05, 0xFF, 0x01, 0x01, 0x04, 0x03, 0x01, 0x04, 0xFF, 0x02, 0x02, 0x04, 0x02, 0x02, 0x05, 0xFF, 0x01, 0x01, 0x03, 0x03, 0x01, 0x03, 0xFF, 0x03, 0x03, 0x03, 0x02, 0x02, 0x04, 0x04, 0x04, 0x04, 0xFF, 0x01, 0x01, 0x05, 0x05, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x03, 0xFF, 0x01, 0x01, 0x02, 0x01, 0x04, 0x05, 0x03, 0x01, 0x02, 0x03, 0x04, 0x05, 0xFF, 0xFF, 0x01, 0x01, 0x02, 0x03, 0x01, 0x02, 0x02, 0x03, 0x05, 0xFF, 0x01, 0x03, 0x01, 0x03, 0x03, 0x02, 0x02, 0x02, 0x03, 0x01, 0x01, 0x04, 0x01, 0x03, 0x05, 0xFF, 0xFF, 0x02, 0x03, 0x01, 0x02, 0x03, 0x05, 0xFF, 0x01, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x03, 0xFF, 0x01, 0x01, 0x02, 0x03, 0x04, 0x05, 0xFF, 0x01, 0x02, 0x01, 0x01, 0x02, 0x05, 0xFF, 0x02, 0x01, 0x05, 0xFF, 0x02, 0x02, 0x01, 0x01, 0x01, 0x02, 0x03, 0x03, 0x02, 0xFF, 0xFF, 0x01, 0x05, 0x05, 0xFF, 0xFF, 0xFF, 0x01, 0x01, 0x02, 0xFF
};


static const byte CURSOR_1_DATA[] = {
	0x0,  0x0,  0x0,  0x0,  0x0,  0x2,  0xF7, 0x5,
	0x0,  0x3,  0xF7, 0xF7, 0x5,  0x0,  0x3,  0xF7,
	0xF7, 0x5,  0x0,  0x4,  0xF7, 0xF7, 0xF7, 0x5,
	0x0,  0x4,  0xF7, 0xF7, 0xF7, 0x5,  0x0,  0x5,
	0xF7, 0xF7, 0xF7, 0xF7, 0x5,  0x0,  0x5,  0xF7,
	0xF7, 0xF7, 0xF7, 0x5,  0x0,  0x6,  0xF7, 0xF7,
	0xF7, 0xF7, 0xF7, 0x5,  0x0,  0x6,  0xF7, 0xF7,
	0xF7, 0xF7, 0xF7, 0x5,  0x0,  0x7,  0xF7, 0xF7,
	0xF7, 0xF7, 0xF7, 0xF7, 0x5,  0x0,  0x6,  0xF7,
	0xF7, 0xF7, 0xF7, 0xF7, 0x5,  0x0,  0x5,  0xF7,
	0xF7, 0xF7, 0xF7, 0x5,  0x2,  0x3,  0xF7, 0xF7,
	0x5,  0x3,  0x3,  0xF7, 0xF7, 0x5,  0x3,  0x3,
	0xF7, 0xF7, 0x5,  0x4,  0x2,  0xF7, 0x5
};

static const byte CURSOR_2_DATA[] = {
	0x7,  0x0,  0x7,  0x0,  0x6,  0x1,  0xF7, 0x4,
	0x5,  0xFF, 0xFF, 0x0,  0xFF, 0xFF, 0x3,  0x7,
	0xFF, 0x0,  0x0,  0x0,  0x0,  0x0,  0xFF, 0x2,
	0x9,  0xFF, 0x0,  0x0,  0x0,  0xF7, 0x0,  0x0,
	0x0,  0xFF, 0x1,  0xB,  0xFF, 0x0,  0x0,  0x0,
	0x0,  0x0,  0x0,  0x0,  0x0,  0x0,  0xFF, 0x1,
	0xB,  0xFF, 0x0,  0x0,  0x0,  0x0,  0xF7, 0x0,
	0x0,  0x0,  0x0,  0xFF, 0x0,  0xD,  0xF7, 0x0,
	0x0,  0xF7, 0x0,  0xF7, 0x0,  0xF7, 0x0,  0xF7,
	0x0,  0x0,  0xF7, 0x1,  0xB,  0xFF, 0x0,  0x0,
	0x0,  0x0,  0xF7, 0x0,  0x0,  0x0,  0x0,  0xFF,
	0x1,  0xB,  0xFF, 0x0,  0x0,  0x0,  0x0,  0x0,
	0x0,  0x0,  0x0,  0x0,  0xFF, 0x2,  0x9,  0xFF,
	0x0,  0x0,  0x0,  0xF7, 0x0,  0x0,  0x0,  0xFF,
	0x3,  0x7,  0xFF, 0x0,  0x0,  0x0,  0x0,  0x0,
	0xFF, 0x4,  0x5,  0xFF, 0xFF, 0x0,  0xFF, 0xFF,
	0x6,  0x1,  0xF7, 0x0,  0x0,  0x0,  0x0,  0x0,
	0x0
};

static const byte BITFONT_DATA[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x08, 0x08, 0x08, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x24, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x14, 0x3E, 0x14, 0x3E, 0x14, 0x00, 0x00,
    0x3C, 0x50, 0x38, 0x14, 0x78, 0x10, 0x00, 0x00,
    0x00, 0x22, 0x04, 0x08, 0x10, 0x22, 0x00, 0x00,
    0x00, 0x30, 0x48, 0x30, 0x2A, 0x4C, 0x32, 0x00,
    0x00, 0x18, 0x08, 0x08, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0C, 0x18, 0x10, 0x18, 0x0C, 0x00, 0x00,
    0x00, 0x30, 0x18, 0x08, 0x18, 0x30, 0x00, 0x00,
    0x00, 0x2A, 0x1C, 0x3E, 0x1C, 0x2A, 0x00, 0x00,
    0x00, 0x00, 0x08, 0x1C, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x18, 0x08, 0x10, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x3C, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00,
    0x00, 0x02, 0x04, 0x08, 0x10, 0x20, 0x00, 0x00,
    0x00, 0x3C, 0x4A, 0x52, 0x62, 0x3C, 0x00, 0x00,
    0x00, 0x08, 0x18, 0x08, 0x08, 0x1C, 0x00, 0x00,
    0x00, 0x3C, 0x42, 0x0C, 0x30, 0x7E, 0x00, 0x00,
    0x00, 0x3C, 0x42, 0x1C, 0x42, 0x3C, 0x00, 0x00,
    0x00, 0x42, 0x42, 0x7E, 0x02, 0x02, 0x00, 0x00,
    0x00, 0x7E, 0x40, 0x7C, 0x02, 0x7C, 0x00, 0x00,
    0x00, 0x3C, 0x40, 0x7C, 0x42, 0x3C, 0x00, 0x00,
    0x00, 0x7E, 0x02, 0x04, 0x08, 0x08, 0x00, 0x00,
    0x00, 0x3C, 0x42, 0x3C, 0x42, 0x3C, 0x00, 0x00,
    0x00, 0x3C, 0x42, 0x3E, 0x02, 0x3C, 0x00, 0x00,
    0x00, 0x00, 0x08, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x08, 0x00, 0x18, 0x08, 0x10, 0x00,
    0x00, 0x08, 0x10, 0x20, 0x10, 0x08, 0x00, 0x00,
    0x00, 0x00, 0x3C, 0x00, 0x3C, 0x00, 0x00, 0x00,
    0x00, 0x10, 0x08, 0x04, 0x08, 0x10, 0x00, 0x00,
    0x00, 0x3C, 0x24, 0x0C, 0x00, 0x08, 0x00, 0x00,
    0x00, 0x3C, 0x42, 0x5E, 0x40, 0x3E, 0x00, 0x00,
    0x00, 0x3C, 0x42, 0x42, 0x7E, 0x42, 0x00, 0x00,
    0x00, 0x7C, 0x42, 0x7C, 0x42, 0x7C, 0x00, 0x00,
    0x00, 0x3C, 0x42, 0x40, 0x42, 0x3C, 0x00, 0x00,
    0x00, 0x7C, 0x42, 0x42, 0x42, 0x7C, 0x00, 0x00,
    0x00, 0x7E, 0x40, 0x7C, 0x40, 0x7E, 0x00, 0x00,
    0x00, 0x7E, 0x40, 0x7C, 0x40, 0x40, 0x00, 0x00,
    0x00, 0x3C, 0x40, 0x4E, 0x42, 0x3C, 0x00, 0x00,
    0x00, 0x42, 0x42, 0x7E, 0x42, 0x42, 0x00, 0x00,
    0x00, 0x7C, 0x10, 0x10, 0x10, 0x7C, 0x00, 0x00,
    0x00, 0x1E, 0x04, 0x04, 0x44, 0x38, 0x00, 0x00,
    0x00, 0x42, 0x4C, 0x70, 0x4C, 0x42, 0x00, 0x00,
    0x00, 0x40, 0x40, 0x40, 0x40, 0x7E, 0x00, 0x00,
    0x00, 0x41, 0x63, 0x55, 0x49, 0x41, 0x00, 0x00,
    0x00, 0x42, 0x62, 0x5A, 0x46, 0x42, 0x00, 0x00,
    0x00, 0x3C, 0x42, 0x42, 0x42, 0x3C, 0x00, 0x00,
    0x00, 0x7C, 0x42, 0x7C, 0x40, 0x40, 0x00, 0x00,
    0x00, 0x3C, 0x42, 0x42, 0x44, 0x3A, 0x00, 0x00,
    0x00, 0x7C, 0x42, 0x7C, 0x44, 0x42, 0x00, 0x00,
    0x00, 0x3C, 0x40, 0x3C, 0x02, 0x3C, 0x00, 0x00,
    0x00, 0x7C, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00,
    0x00, 0x42, 0x42, 0x42, 0x42, 0x3C, 0x00, 0x00,
    0x00, 0x42, 0x42, 0x42, 0x24, 0x18, 0x00, 0x00,
    0x00, 0x41, 0x49, 0x55, 0x63, 0x41, 0x00, 0x00,
    0x00, 0x42, 0x24, 0x18, 0x24, 0x42, 0x00, 0x00,
    0x00, 0x42, 0x42, 0x24, 0x18, 0x18, 0x00, 0x00,
    0x00, 0x7E, 0x04, 0x18, 0x20, 0x7E, 0x00, 0x00,
    0x00, 0x10, 0x38, 0x7C, 0x38, 0x38, 0x00, 0x00,
    0x00, 0x38, 0x38, 0x7C, 0x38, 0x10, 0x00, 0x00,
    0x00, 0x7F, 0x41, 0x41, 0x41, 0x7F, 0x00, 0x00,
    0x00, 0x7F, 0x55, 0x49, 0x55, 0x7F, 0x00, 0x00,
    0xFF, 0x91, 0xBD, 0xBD, 0x91, 0xFF, 0x00, 0x00,
    0xFF, 0x89, 0xBD, 0xBD, 0x89, 0xFF, 0x00, 0x00
};

static const byte CURSOR_3_DATA[] = {
	0x8,  0x0,  0x8,  0x0,  0x0,  0x0,  0x0,  0x0,
	0x7,  0x2,  0x4,  0x5,  0x7,  0x2,  0x4,  0x5,
	0x7,  0x2,  0x4,  0x5,  0x7,  0x2,  0x4,  0x5,
	0x7,  0x2,  0x4,  0x5,  0x2,  0xC,  0x4,  0x4,
	0x4,  0x4,  0x4,  0x0,  0x4,  0x4,  0x4,  0x4,
	0x4,  0x5,  0x7,  0x2,  0x4,  0x5,  0x7,  0x2,
	0x4,  0x5,  0x7,  0x2,  0x4,  0x5,  0x7,  0x2,
	0x4,  0x5,  0x7,  0x2,  0x4,  0x5,  0x0,  0x0,
	0x0,  0x0,  0x0,  0x0
};

static const byte CURSOR_4_DATA[] = {
	0x0,  0x0,  0x0,  0x0,  0x0,  0xB,  0x6,  0x6,
	0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,
	0x6,  0x0,  0xC,  0x6,  0x7,  0x7,  0x7,  0x7,
	0x7,  0x7,  0x7,  0x7,  0x6,  0x6,  0x5,  0x0,
	0xC,  0x6,  0x7,  0x7,  0x7,  0x7,  0x7,  0x7,
	0x7,  0x7,  0x6,  0x5,  0x5,  0x0,  0xC,  0x6,
	0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,
	0x6,  0x6,  0x5,  0x0,  0xC,  0x6,  0x6,  0x6,
	0x6,  0x6,  0x5,  0x6,  0x6,  0x6,  0x6,  0x6,
	0x5,  0x0,  0xC,  0x6,  0x6,  0x6,  0x6,  0x5,
	0x0,  0x0,  0x6,  0x6,  0x6,  0x6,  0x5,  0x0,
	0xC,  0x6,  0x6,  0x6,  0x6,  0x6,  0x0,  0x6,
	0x6,  0x6,  0x6,  0x6,  0x5,  0x0,  0xC,  0x6,
	0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,
	0x6,  0x6,  0x5,  0x0,  0xC,  0x6,  0x6,  0x6,
	0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,
	0x5,  0x0,  0xC,  0x6,  0x6,  0x6,  0x6,  0x6,
	0x5,  0x6,  0x6,  0x6,  0x6,  0x6,  0x5,  0x0,
	0xC,  0x6,  0x6,  0x6,  0x6,  0x6,  0x5,  0x6,
	0x6,  0x6,  0x6,  0x6,  0x5,  0x0,  0xC,  0x6,
	0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,  0x6,
	0x6,  0x6,  0x5,  0x1,  0xB,  0x5,  0x5,  0x5,
	0x5,  0x5,  0x5,  0x5,  0x5,  0x5,  0x5,  0x5,
	0x0,  0x0,  0x0,  0x0,  0x0,  0x0
};


static const byte *CURSOR_DATA[] =
{
	CURSOR_1_DATA,
	CURSOR_2_DATA,
	CURSOR_3_DATA,
	CURSOR_4_DATA,
};


void MartianResources::load(Common::SeekableReadStream &s) {
	Resources::load(s);

	// Create the fonts from static data.
	_font1 = new MartianFont(6, ARRAYSIZE(FONT1_WIDTHS), FONT1_WIDTHS, FONT1_OFFSETS, FONT1_DATA);
	_font2 = new MartianFont(6, ARRAYSIZE(FONT2_WIDTHS), FONT2_WIDTHS, FONT2_OFFSETS, FONT2_DATA);
	_bitFont = new MartianBitFont(ARRAYSIZE(BITFONT_DATA) / 8, BITFONT_DATA);

	CANT_GET_THERE = "YOU CAN'T GET THERE FROM HERE.";

	// The EXE doesn't have full data for the last few room table entries (49~51), as they are only
	// talk locations.  That means the DAT file doesn't include them, so we add them manually.
	ROOMTBL.push_back(RoomEntry({"MICHELE BLOODWORTH", Common::Point(-1, 18), Common::Array<byte>()}));
	ROOMTBL.push_back(RoomEntry({"BRADLEY ERICSON", Common::Point(-1, 19), Common::Array<byte>()}));
	ROOMTBL.push_back(RoomEntry({"COOPER BRADBURY", Common::Point(-1, 21), Common::Array<byte>()}));
}

const byte *MartianResources::getCursor(int num) const {
	assert(num < ARRAYSIZE(CURSOR_DATA) && num >= 0);

	return CURSOR_DATA[num];
}




} // End of namespace Martian
} // End of namespace Access
