// license:BSD-3-Clause
// copyright-holders:Barry Rodewald
// Convergent AWS keyboard device

#include "emu.h"
#include "aws_kb.h"


aws_keyboard_device::aws_keyboard_device(const machine_config& mconfig, const char* tag, device_t* owner, uint32_t clock)
	: device_t(mconfig, AWS_KEYBOARD, tag, owner, 0)
	, device_buffered_serial_interface(mconfig, *this)
	, device_rs232_port_interface(mconfig, *this)
	, m_rows(*this, "kbd_row%u", 0U)
	, m_last_reset(0U)
{
}

static INPUT_PORTS_START( aws_keyboard )

	PORT_START("kbd_row0")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Help")
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Up")  PORT_CHAR(UCHAR_MAMEKEY(UP))
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Mark")
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Bound")
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Finish")
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Prev Page")
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("1/2  1/4")
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Cancel")

	PORT_START("kbd_row1")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Backspace")  PORT_CODE(KEYCODE_BACKSPACE)  PORT_CHAR(0x08U)
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Tab")  PORT_CODE(KEYCODE_TAB)  PORT_CHAR('\t')
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Return")  PORT_CODE(KEYCODE_ENTER) PORT_CHAR(0x0dU)
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Down") PORT_CODE(KEYCODE_DOWN)  PORT_CHAR(UCHAR_MAMEKEY(DOWN))
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Next Page")
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Next")
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Left") PORT_CODE(KEYCODE_LEFT)  PORT_CHAR(UCHAR_MAMEKEY(LEFT))
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Right") PORT_CODE(KEYCODE_RIGHT)  PORT_CHAR(UCHAR_MAMEKEY(LEFT))

	PORT_START("kbd_row2")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("(SH-L\')")
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Scroll Up")  PORT_CODE(KEYCODE_PGUP)  PORT_CHAR(UCHAR_MAMEKEY(PGUP))
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Move")
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Scroll Down") PORT_CODE(KEYCODE_PGDN)  PORT_CHAR(UCHAR_MAMEKEY(PGDN))
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Copy")
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("F1") PORT_CODE(KEYCODE_F1)  PORT_CHAR(UCHAR_MAMEKEY(F1))
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("F2") PORT_CODE(KEYCODE_F2)  PORT_CHAR(UCHAR_MAMEKEY(F2))
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("F3") PORT_CODE(KEYCODE_F3)  PORT_CHAR(UCHAR_MAMEKEY(F3))

	PORT_START("kbd_row3")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("F4") PORT_CODE(KEYCODE_F4)  PORT_CHAR(UCHAR_MAMEKEY(F4))
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("F5") PORT_CODE(KEYCODE_F5)  PORT_CHAR(UCHAR_MAMEKEY(F5))
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("F6") PORT_CODE(KEYCODE_F6)  PORT_CHAR(UCHAR_MAMEKEY(F6))
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Go")
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("F7") PORT_CODE(KEYCODE_F7)  PORT_CHAR(UCHAR_MAMEKEY(F7))
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("F8") PORT_CODE(KEYCODE_F8)  PORT_CHAR(UCHAR_MAMEKEY(F8))
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("F9") PORT_CODE(KEYCODE_F9)  PORT_CHAR(UCHAR_MAMEKEY(F9))
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("F10") PORT_CODE(KEYCODE_F10)  PORT_CHAR(UCHAR_MAMEKEY(F10))

	PORT_START("kbd_row4")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Space") PORT_CODE(KEYCODE_SPACE)  PORT_CHAR(' ')
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Numpad 9") PORT_CODE(KEYCODE_9_PAD)  PORT_CHAR(UCHAR_MAMEKEY(9_PAD))
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("(SH-R\')")
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("(0\')")
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("(Next\')")
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("\' \"") PORT_CODE(KEYCODE_QUOTE) PORT_CHAR('\'')

	PORT_START("kbd_row5")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("= +") PORT_CODE(KEYCODE_EQUALS) PORT_CHAR('=') PORT_CHAR('+')
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME (",") PORT_CODE(KEYCODE_COMMA)  PORT_CHAR(',')
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("- _") PORT_CODE(KEYCODE_MINUS)  PORT_CHAR('-') PORT_CHAR('_')
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME (".") PORT_CODE(KEYCODE_STOP)  PORT_CHAR('.')
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("/ ?") PORT_CODE(KEYCODE_SLASH)  PORT_CHAR('/') PORT_CHAR('?')

	PORT_START("kbd_row6")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("0 )") PORT_CODE(KEYCODE_0) PORT_CHAR('0') PORT_CHAR(')')
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("1 !") PORT_CODE(KEYCODE_1) PORT_CHAR('1') PORT_CHAR('!')
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("2 @") PORT_CODE(KEYCODE_2) PORT_CHAR('2') PORT_CHAR('@')
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("3 #") PORT_CODE(KEYCODE_3) PORT_CHAR('3') PORT_CHAR('#')
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("4 $") PORT_CODE(KEYCODE_4) PORT_CHAR('4') PORT_CHAR('$')
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("5 %") PORT_CODE(KEYCODE_5) PORT_CHAR('5') PORT_CHAR('%')
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("6 \u00a2") PORT_CODE(KEYCODE_6) PORT_CHAR('6')
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("7 &") PORT_CODE(KEYCODE_7) PORT_CHAR('7') PORT_CHAR('&')

	PORT_START("kbd_row7")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("8 *") PORT_CODE(KEYCODE_8) PORT_CHAR('8') PORT_CHAR('*')
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("9 (") PORT_CODE(KEYCODE_9) PORT_CHAR('9') PORT_CHAR('(')
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("; :") PORT_CODE(KEYCODE_QUOTE) PORT_CHAR(';') PORT_CHAR(':')
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_UNUSED)  // "invalid"

	PORT_START("kbd_row8")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_UNUSED) // key up code
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Numpad 6") PORT_CODE(KEYCODE_6_PAD) PORT_CHAR(UCHAR_MAMEKEY(9_PAD))
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Numpad -") PORT_CODE(KEYCODE_MINUS_PAD) PORT_CHAR(UCHAR_MAMEKEY(MINUS_PAD))
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Action")
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Overtype")
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Numpad 2") PORT_CODE(KEYCODE_2_PAD) PORT_CHAR(UCHAR_MAMEKEY(2_PAD))
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Numpad 3") PORT_CODE(KEYCODE_3_PAD) PORT_CHAR(UCHAR_MAMEKEY(3_PAD))

	PORT_START("kbd_row9")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Left Shift") PORT_CODE(KEYCODE_LSHIFT) PORT_CHAR(UCHAR_SHIFT_1)
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Right Shift") PORT_CODE(KEYCODE_RSHIFT) PORT_CHAR(UCHAR_SHIFT_2)
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("kbd_row10")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("kbd_row11")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("[ <") PORT_CODE(KEYCODE_OPENBRACE) PORT_CHAR('[') PORT_CHAR('<')
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME ("Numpad 7 |") PORT_CODE(KEYCODE_7_PAD) PORT_CHAR(UCHAR_MAMEKEY(7_PAD)) PORT_CHAR('|')
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("] >") PORT_CODE(KEYCODE_CLOSEBRACE) PORT_CHAR(']') PORT_CHAR('>')
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("^ ~") PORT_CODE(KEYCODE_TILDE) PORT_CHAR('^') PORT_CHAR('~')
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_UNUSED)

	PORT_START("kbd_row12")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Numpad 1 \\") PORT_CODE(KEYCODE_1_PAD)  PORT_CHAR(UCHAR_MAMEKEY(1_PAD)) PORT_CHAR('\\')
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("A") PORT_CODE(KEYCODE_A) PORT_CHAR('a') PORT_CHAR('A')
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("B") PORT_CODE(KEYCODE_B) PORT_CHAR('b') PORT_CHAR('B')
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("C") PORT_CODE(KEYCODE_C) PORT_CHAR('c') PORT_CHAR('C')
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("D") PORT_CODE(KEYCODE_D) PORT_CHAR('d') PORT_CHAR('D')
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("E") PORT_CODE(KEYCODE_E) PORT_CHAR('e') PORT_CHAR('E')
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("F") PORT_CODE(KEYCODE_F) PORT_CHAR('f') PORT_CHAR('F')
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("G") PORT_CODE(KEYCODE_G) PORT_CHAR('g') PORT_CHAR('G')

	PORT_START("kbd_row13")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("H") PORT_CODE(KEYCODE_H) PORT_CHAR('h') PORT_CHAR('H')
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("I") PORT_CODE(KEYCODE_I) PORT_CHAR('i') PORT_CHAR('I')
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("J") PORT_CODE(KEYCODE_J) PORT_CHAR('j') PORT_CHAR('J')
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("K") PORT_CODE(KEYCODE_K) PORT_CHAR('k') PORT_CHAR('K')
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("L") PORT_CODE(KEYCODE_L) PORT_CHAR('l') PORT_CHAR('L')
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("M") PORT_CODE(KEYCODE_M) PORT_CHAR('m') PORT_CHAR('M')
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("N") PORT_CODE(KEYCODE_N) PORT_CHAR('n') PORT_CHAR('N')
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("O") PORT_CODE(KEYCODE_O) PORT_CHAR('o') PORT_CHAR('O')

	PORT_START("kbd_row14")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("P") PORT_CODE(KEYCODE_P) PORT_CHAR('p') PORT_CHAR('P')
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Q") PORT_CODE(KEYCODE_Q) PORT_CHAR('q') PORT_CHAR('Q')
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("R") PORT_CODE(KEYCODE_R) PORT_CHAR('r') PORT_CHAR('R')
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("S") PORT_CODE(KEYCODE_S) PORT_CHAR('s') PORT_CHAR('S')
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("T") PORT_CODE(KEYCODE_T) PORT_CHAR('t') PORT_CHAR('T')
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("U") PORT_CODE(KEYCODE_U) PORT_CHAR('u') PORT_CHAR('U')
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("V") PORT_CODE(KEYCODE_V) PORT_CHAR('v') PORT_CHAR('V')
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("W") PORT_CODE(KEYCODE_W) PORT_CHAR('w') PORT_CHAR('W')

	PORT_START("kbd_row15")
	PORT_BIT(0x01U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("X") PORT_CODE(KEYCODE_X) PORT_CHAR('x') PORT_CHAR('X')
	PORT_BIT(0x02U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Y") PORT_CODE(KEYCODE_Y) PORT_CHAR('y') PORT_CHAR('Y')
	PORT_BIT(0x04U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Z") PORT_CODE(KEYCODE_Z) PORT_CHAR('z') PORT_CHAR('Z')
	PORT_BIT(0x08U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Numpad 4 {") PORT_CODE(KEYCODE_4_PAD)  PORT_CHAR(UCHAR_MAMEKEY(4_PAD))
	PORT_BIT(0x10U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Numpad 8 \\") PORT_CODE(KEYCODE_8_PAD)  PORT_CHAR(UCHAR_MAMEKEY(8_PAD))
	PORT_BIT(0x20U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Numpad 5 }") PORT_CODE(KEYCODE_5_PAD)  PORT_CHAR(UCHAR_MAMEKEY(5_PAD))
	PORT_BIT(0x40U, IP_ACTIVE_HIGH, IPT_UNUSED)
	PORT_BIT(0x80U, IP_ACTIVE_HIGH, IPT_KEYBOARD) PORT_NAME("Delete") PORT_CODE(KEYCODE_DEL) PORT_CHAR(UCHAR_MAMEKEY(DEL))

	PORT_START("RS232_TXBAUD")
	PORT_CONFNAME(0xff, RS232_BAUD_1200, "TX Baud") // PORT_WRITE_LINE_DEVICE_MEMBER(DEVICE_SELF, FUNC(serial_keyboard_device::update_serial))
	PORT_CONFSETTING( RS232_BAUD_1200, "1200") // one part of the hardware manual says the uPD7201 used 1200 baud, another says the keyboard 8048 uses 1221 baud

	PORT_START("RS232_DATABITS")
	PORT_CONFNAME(0xff, RS232_DATABITS_8, "Data Bits")
	PORT_CONFSETTING( RS232_DATABITS_8, "8")

	PORT_START("RS232_PARITY")
	PORT_CONFNAME(0xff, RS232_PARITY_NONE, "Parity")
	PORT_CONFSETTING( RS232_PARITY_NONE, "None")

	PORT_START("RS232_STOPBITS")
	PORT_CONFNAME(0xff, RS232_STOPBITS_1, "Stop Bits")
	PORT_CONFSETTING( RS232_STOPBITS_1, "1")
INPUT_PORTS_END

void aws_keyboard_device::write(uint8_t data)
{
	logerror("KB: received character %02x\n",data);
	switch(data)
	{
	case 0x92U:  // reset(?)
		m_last_reset = 0x01U;
		break;
	}
}

void aws_keyboard_device::device_start()
{
	save_item(NAME(m_keys_down));
	save_item(NAME(m_last_reset));

	set_data_frame(1,8,PARITY_NONE,STOP_BITS_1);
	set_rate(1200);

	m_scan_timer = timer_alloc(FUNC(aws_keyboard_device::scan_keys), this);
}

void aws_keyboard_device::device_reset()
{
	memset(m_keys_down, 0, sizeof(m_keys_down));
	m_last_reset = 0U;
	clear_fifo();
	m_scan_timer->adjust(attotime::from_msec(15), 0, attotime::from_msec(15));
}

TIMER_CALLBACK_MEMBER(aws_keyboard_device::scan_keys)
{
	bool bit_changed = false;
	std::vector<int> codes;

	// check if any bits have changed
	for(int x=0;x<16;x++)
	{
		for(int y=0;y<8;y++)
		{
			int bit = BIT(m_rows[x]->read(), y);
			if(bit ^ m_keys_down[x][y])
				bit_changed = true;
			m_keys_down[x][y] = bit;
		}
	}

	// if any key state has changed, send all keys that are pressed down, or 0xc0 if none.
	if(bit_changed)
	{
		for(int x=0;x<16;x++)
		{
			for(int y=0;y<8;y++)
			{
				if(m_keys_down[x][y])
					codes.push_back((x << 3) | y);
			}
		}
		if(codes.empty())
			send_key(0xc0);
		else
		{
			int data;
			// The last byte sent will have its MSB set.
			while(!codes.empty())
			{
				data = codes.back();
				codes.pop_back();
				if(codes.empty())
					data |= 0x80;
				send_key(data);
			}
		}
	}
}

void aws_keyboard_device::send_key(uint8_t code)
{
	transmit_byte(code);
	logerror("KB: sent code 0x%02x\n", code);
}

void aws_keyboard_device::rcv_complete()
{
	receive_register_extract();
	write(get_received_char());
}

void aws_keyboard_device::input_txd(int state)
{
	device_buffered_serial_interface::rx_w(state);
}

void aws_keyboard_device::tra_callback()
{
	output_rxd(transmit_register_get_data_bit());
}

void aws_keyboard_device::received_byte(uint8_t byte)
{
	logerror("KB: recieved 0x%02x\n",byte);

}

ioport_constructor aws_keyboard_device::device_input_ports() const
{
	return INPUT_PORTS_NAME(aws_keyboard);
}


DEFINE_DEVICE_TYPE(AWS_KEYBOARD, aws_keyboard_device, "aws_kb", "AWS Keyboard")
