//===- LoongArchSubtarget.h - Define Subtarget for the LoongArch -*- C++ -*-==//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
//
// This file declares the LoongArch specific subclass of TargetSubtargetInfo.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIB_TARGET_LOONGARCH_LOONGARCHSUBTARGET_H
#define LLVM_LIB_TARGET_LOONGARCH_LOONGARCHSUBTARGET_H

#include "LoongArchFrameLowering.h"
#include "LoongArchISelLowering.h"
#include "LoongArchInstrInfo.h"
#include "LoongArchRegisterInfo.h"
#include "MCTargetDesc/LoongArchBaseInfo.h"
#include "llvm/CodeGen/TargetSubtargetInfo.h"
#include "llvm/IR/DataLayout.h"
#include "llvm/Target/TargetMachine.h"

#define GET_SUBTARGETINFO_HEADER
#include "LoongArchGenSubtargetInfo.inc"

namespace llvm {
class StringRef;

class LoongArchSubtarget : public LoongArchGenSubtargetInfo {
  virtual void anchor();

#define GET_SUBTARGETINFO_MACRO(ATTRIBUTE, DEFAULT, GETTER)                    \
  bool ATTRIBUTE = DEFAULT;
#include "LoongArchGenSubtargetInfo.inc"

  unsigned GRLen = 32;
  // TODO: The default value is empirical and conservative. Override the
  // default in initializeProperties once we support optimizing for more
  // uarches.
  uint8_t MaxInterleaveFactor = 2;
  MVT GRLenVT = MVT::i32;
  LoongArchABI::ABI TargetABI = LoongArchABI::ABI_Unknown;
  LoongArchFrameLowering FrameLowering;
  LoongArchInstrInfo InstrInfo;
  LoongArchTargetLowering TLInfo;
  std::unique_ptr<const SelectionDAGTargetInfo> TSInfo;

  Align PrefFunctionAlignment;
  Align PrefLoopAlignment;
  unsigned MaxBytesForAlignment;

  /// Initializes using the passed in CPU and feature strings so that we can
  /// use initializer lists for subtarget initialization.
  LoongArchSubtarget &initializeSubtargetDependencies(const Triple &TT,
                                                      StringRef CPU,
                                                      StringRef TuneCPU,
                                                      StringRef FS,
                                                      StringRef ABIName);

  /// Initialize properties based on the selected processor family.
  void initializeProperties(StringRef TuneCPU);

public:
  // Initializes the data members to match that of the specified triple.
  LoongArchSubtarget(const Triple &TT, StringRef CPU, StringRef TuneCPU,
                     StringRef FS, StringRef ABIName, const TargetMachine &TM);

  ~LoongArchSubtarget() override;

  // Parses features string setting specified subtarget options. The
  // definition of this function is auto-generated by tblgen.
  void ParseSubtargetFeatures(StringRef CPU, StringRef TuneCPU, StringRef FS);

  const LoongArchFrameLowering *getFrameLowering() const override {
    return &FrameLowering;
  }
  const LoongArchInstrInfo *getInstrInfo() const override { return &InstrInfo; }
  const LoongArchRegisterInfo *getRegisterInfo() const override {
    return &InstrInfo.getRegisterInfo();
  }
  const LoongArchTargetLowering *getTargetLowering() const override {
    return &TLInfo;
  }

  const SelectionDAGTargetInfo *getSelectionDAGInfo() const override;

#define GET_SUBTARGETINFO_MACRO(ATTRIBUTE, DEFAULT, GETTER)                    \
  bool GETTER() const { return ATTRIBUTE; }
#include "LoongArchGenSubtargetInfo.inc"

  bool is64Bit() const { return HasLA64; }
  MVT getGRLenVT() const { return GRLenVT; }
  unsigned getGRLen() const { return GRLen; }
  LoongArchABI::ABI getTargetABI() const { return TargetABI; }
  bool isSoftFPABI() const {
    return TargetABI == LoongArchABI::ABI_LP64S ||
           TargetABI == LoongArchABI::ABI_ILP32S;
  }
  bool isXRaySupported() const override { return is64Bit(); }
  Align getPrefFunctionAlignment() const { return PrefFunctionAlignment; }
  Align getPrefLoopAlignment() const { return PrefLoopAlignment; }
  unsigned getMaxBytesForAlignment() const { return MaxBytesForAlignment; }
  unsigned getMaxInterleaveFactor() const { return MaxInterleaveFactor; }
  bool enableMachineScheduler() const override { return true; }
  bool useAA() const override;
};
} // end namespace llvm

#endif // LLVM_LIB_TARGET_LOONGARCH_LOONGARCHSUBTARGET_H
