// bibiman - a TUI for managing BibLaTeX databases
// Copyright (C) 2024  lukeflo
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.
/////

use ratatui::{
    style::Stylize,
    text::{Line, Span, Text},
    widgets::ListState,
};

use crate::config::BibiConfig;

#[derive(Debug)]
pub enum PopupKind {
    Help,
    /// use for a confirmation message
    MessageConfirm,
    /// use for a warning message
    MessageError,
    /// open a resource connected to the entry
    OpenRes,
    /// select file to append entry to
    AppendToFile,
    /// append entry to a bibfile (selected in `AppendToFile` popup)
    AddEntry,
    /// select an item of the current entry to yank to clipboard
    YankItem,
    /// Create a new note, select extension
    CreateNote,
    /// Select field to be searched
    SearchField,
}

#[derive(Debug)]
pub enum PopupItem {
    Bibfile,
    Entryfile,
    Notefile,
    Citekey,
    Link,
    SearchField(crate::bibiman::search::SearchField),
    Default,
    None,
}

#[derive(Debug, Default)]
pub struct PopupArea {
    pub is_popup: bool,
    pub popup_kind: Option<PopupKind>,
    pub popup_message: String,
    pub popup_scroll_pos: u16,
    pub popup_list: Vec<(String, String, PopupItem)>,
    pub popup_state: ListState,
    pub popup_sel_item: String,
    // pub add_entry_input: String,
    // pub add_entry_cursor_position: usize,
}

impl PopupArea {
    pub fn popup_help<'a>(cfg: &BibiConfig) -> Text<'a> {
        let help = [
            ("General", "first"),
            ("TAB: ", "Toggle areas (Entries, Keywords)"),
            ("/|Ctrl+f: ", "Enter search mode"),
            ("q|Ctrl+c: ", "Quit bibiman"),
            ("?: ", "Show help"),
            ("Entry Table", "sub"),
            ("j,k|↓,↑: ", "Select next/previous entry"),
            ("Ctrl+d,Ctrl+u: ", "Select fifth-next/previous entry"),
            ("h,l|←,→: ", "Select next/previous column"),
            ("g|Home: ", "Go to first entry"),
            ("G|End: ", "Go to last entry"),
            ("Alt+/|Alt+f: ", "Search specific fields"),
            ("s: ", "sort entries by selected column (toggles reversed)"),
            (
                "S: ",
                "sort entries by position in/of file (toggles reversed)",
            ),
            ("y: ", "yank/copy citekey of selected entry to clipboard"),
            ("e: ", "Open editor at selected entry"),
            ("o: ", "Open resource attached to selected entry"),
            ("n: ", "Create note for selected entry"),
            ("a: ", "Add new entry"),
            ("ESC: ", "Reset all lists"),
            ("Keyword List", "sub"),
            ("j,k|↓,↑: ", "Select next/previous item"),
            ("Ctrl+d,Ctrl+u: ", "Select fifth-next/previous entry"),
            ("g|Home: ", "Go to first keyword"),
            ("G|End: ", "Go to last keyword"),
            ("ENTER: ", "Filter by selected keyword"),
            ("Search", "sub"),
            ("↓,↑,←,→: ", "Move cursor"),
            ("BACKSPACE: ", "Delete Character"),
            ("ENTER: ", "Confirm search"),
            ("ESC: ", "Abort search"),
        ];

        let mut helptext: Vec<Line<'_>> = vec![];

        for (keys, help) in help {
            if help == "first" {
                helptext.push(Line::from(
                    Span::raw(keys).bold().fg(cfg.colors.main_text_color),
                ))
            } else if help == "sub" {
                helptext.push(Line::from(""));
                helptext.push(Line::from(
                    Span::raw(keys).bold().fg(cfg.colors.main_text_color),
                ))
            } else {
                helptext.push(Line::from(vec![
                    Span::raw(keys).bold().fg(cfg.colors.main_text_color),
                    Span::raw(help),
                ]))
            }
        }

        Text::from(helptext)
    }

    /// Opens a popup with a selectable list
    ///
    /// The list items are passed as argument of the kind `Vec<(String, String, PopupItem)>`.
    pub fn popup_selection(&mut self, items: Vec<(String, String, PopupItem)>) {
        self.popup_list = items;
        // self.popup_kind = Some(PopupKind::SelectRes);
        self.is_popup = true;
    }

    /// Returns the `PopupItem` of the selected entry. If no entry is selected,
    /// the value is always `PopupItem::None`
    pub(crate) fn get_sel_popup_item(&self) -> &PopupItem {
        if let Some(idx) = self.popup_state.selected() {
            &self.popup_list[idx].2
        } else {
            &PopupItem::None
        }
    }

    pub fn popup_scroll_down(&mut self) {
        self.popup_scroll_pos = self.popup_scroll_pos.saturating_add(1)
    }

    pub fn popup_scroll_up(&mut self) {
        self.popup_scroll_pos = self.popup_scroll_pos.saturating_sub(1)
    }
}
