/**************************************************************************
 **                                                                      **
 ** Copyright (C) 2011-2026 Lukas Spies                                  **
 ** Contact: https://photoqt.org                                         **
 **                                                                      **
 ** This file is part of PhotoQt.                                        **
 **                                                                      **
 ** PhotoQt is free software: you can redistribute it and/or modify      **
 ** it under the terms of the GNU General Public License as published by **
 ** the Free Software Foundation, either version 2 of the License, or    **
 ** (at your option) any later version.                                  **
 **                                                                      **
 ** PhotoQt is distributed in the hope that it will be useful,           **
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of       **
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        **
 ** GNU General Public License for more details.                         **
 **                                                                      **
 ** You should have received a copy of the GNU General Public License    **
 ** along with PhotoQt. If not, see <http://www.gnu.org/licenses/>.      **
 **                                                                      **
 **************************************************************************/
pragma ComponentBehavior: Bound

import QtQuick
import QtQuick.Controls
import PhotoQt

Item {

    id: bartop

    anchors.fill: parent

    /*******************************************/
    // these values are READONLY

    property real loaderImageScale
    property bool isMainImage
    property string imageSource

    /*******************************************/

    property list<var> barcodes: []
    property int barcode_z: 0

    signal barcodeClicked()

    Connections {
        target: PQCConstants
        function onCurrentImageSourceChanged() {
            if(!bartop.isMainImage) {
                bartop.barcodes = []
            }
        }

    }

    Connections {

        target: PQCNotify

        function onCurrentImageDetectBarCodes() {
            if(bartop.isMainImage) {
                if(!PQCConstants.barcodeDisplayed) {
                    bartop.barcodes = PQCScriptsImages.getZXingData(bartop.imageSource)
                    if(bartop.barcodes.length === 0) {
                        PQCNotify.showNotificationMessage(qsTranslate("image", "Nothing found"), qsTranslate("image", "No bar/QR codes found."))
                    } else if(bartop.barcodes.length/3 == 1) {
                        PQCNotify.showNotificationMessage(qsTranslate("image", "Success"),  qsTranslate("image", "1 bar/QR code found."))
                    } else if(bartop.barcodes.length/3 > 1) {
                        PQCNotify.showNotificationMessage(qsTranslate("image", "Success"),  qsTranslate("image", "%1 bar/QR codes found.").arg(bartop.barcodes.length/3))
                    }
                    PQCConstants.barcodeDisplayed = bartop.barcodes.length>0
                } else {
                    PQCConstants.barcodeDisplayed = false
                    bartop.barcodes = []
                }
            }
        }
    }

    Loader {

        active: bartop.barcodes.length>0
        asynchronous: true
        anchors.fill: parent

        Repeater {
            model: bartop.barcodes.length/3

            Rectangle {

                id: bardeleg

                required property int modelData

                property string val: bartop.barcodes[3*modelData]

                x: bartop.barcodes[3*modelData+1].x
                y: bartop.barcodes[3*modelData+1].y
                width: bartop.barcodes[3*modelData+2].x
                height: bartop.barcodes[3*modelData+2].y

                color: "#8800ff00"
                radius: 5

                property bool overrideCursorSet: false
                property bool hovered: false

                Column {

                    x: (parent.width-width)/2
                    y: (parent.height-height)/2

                    spacing: 1

                    scale: 1/bartop.loaderImageScale
                    Behavior on scale { enabled: !PQCSettings.generalDisableAllAnimations; NumberAnimation { duration: 200 } }

                    Rectangle {
                        id: txtcont
                        x: (parent.width-width)/2
                        width: valtxt.width+10
                        height: valtxt.height+10
                        color: palette.base
                        border.width: 1
                        border.color: PQCLook.baseBorder
                        radius: 5
                        opacity: bardeleg.hovered||copycont.hovered||linkcont.hovered||(bartop.barcodes.length<4) ? 1 : 0.4
                        Behavior on opacity { enabled: !PQCSettings.generalDisableAllAnimations; NumberAnimation { duration: 200 } }
                        PQText {
                            id: valtxt
                            x: 5
                            y: 5
                            wrapMode: Text.WrapAtWordBoundaryOrAnywhere
                            font.weight: PQCLook.fontWeightBold
                            text: bardeleg.val
                        }

                    }

                    Row {

                        x: (parent.width-width)/2

                        spacing: 1

                        Item {
                            id: copycont
                            width: 32
                            height: 32
                            Rectangle {
                                anchors.fill: parent
                                color: palette.base
                                border.width: 1
                                border.color: PQCLook.baseBorder
                                radius: 5
                            }
                            property bool hovered: false
                            opacity: hovered ? 1 : 0.4
                            Behavior on opacity { enabled: !PQCSettings.generalDisableAllAnimations; NumberAnimation { duration: 200 } }
                            Image {
                                anchors.fill: parent
                                anchors.margins: 5
                                sourceSize: Qt.size(width, height)
                                fillMode: Image.Pad
                                source: "image://svg/:/" + PQCLook.iconShade + "/copy.svg"
                            }
                        }

                        Item {
                            id: linkcont
                            width: 32
                            height: 32
                            Rectangle {
                                anchors.fill: parent
                                color: palette.base
                                border.width: 1
                                border.color: PQCLook.baseBorder
                                radius: 5
                            }
                            property bool hovered: false
                            opacity: hovered ? 1 : 0.4
                            Behavior on opacity { enabled: !PQCSettings.generalDisableAllAnimations; NumberAnimation { duration: 200 } }
                            visible: PQCScriptsFilesPaths.isUrl(bardeleg.val)
                            enabled: visible
                            Image {
                                anchors.fill: parent
                                anchors.margins: 5
                                sourceSize: Qt.size(width, height)
                                fillMode: Image.Pad
                                source: "image://svg/:/" + PQCLook.iconShade + "/globe.svg"
                            }
                        }

                        Connections {

                            target: bartop

                            function onBarcodeClicked() {
                                if(copycont.hovered)
                                    PQCScriptsClipboard.copyTextToClipboard(bardeleg.val)
                                else if(linkcont.hovered)
                                    Qt.openUrlExternally(bardeleg.val)
                            }

                        }

                        Connections {

                            target: PQCNotify
                            enabled: bartop.isMainImage

                            function onMouseMove(x : int, y : int) {

                                var local = copycont.mapFromItem(fullscreenitem, Qt.point(x,y))
                                copycont.hovered = (local.x > 0 && local.y > 0 && local.x < copycont.width && local.y < copycont.height)

                                local = linkcont.mapFromItem(fullscreenitem, Qt.point(x,y))
                                linkcont.hovered = (local.x > 0 && local.y > 0 && local.x < linkcont.width && local.y < linkcont.height)

                                local = bardeleg.mapFromItem(fullscreenitem, Qt.point(x,y))
                                bardeleg.hovered = (local.x > 0 && local.y > 0 && local.x < bardeleg.width && local.y < bardeleg.height)

                                if(copycont.hovered || linkcont.hovered || bardeleg.hovered) {
                                    bartop.barcode_z += 1
                                    bardeleg.z = bartop.barcode_z
                                    if(copycont.hovered || linkcont.hovered) {
                                        bardeleg.overrideCursorSet = true
                                        PQCScriptsOther.setPointingHandCursor()
                                    }
                                } else if(bardeleg.overrideCursorSet) {
                                    bardeleg.overrideCursorSet = false
                                    PQCScriptsOther.restoreOverrideCursor()
                                }

                            }

                        }

                    }

                }

            }

        }

    }

}
