// Copyright 2012-2015 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

// NOTE: The following code was generated by "scripts/unicode.py", do not edit directly

#![allow(missing_docs, non_upper_case_globals, non_snake_case)]

/// The version of [Unicode](http://www.unicode.org/)
/// that this version of unicode-security is based on.
pub const UNICODE_VERSION: (u64, u64, u64) = (16, 0, 0);


pub mod util {
    use core::result::Result::{Ok, Err};

    #[inline]
    pub fn bsearch_table(c: char, r: &'static [char]) -> bool {
        r.binary_search(&c).is_ok()
    }

    #[inline]
    pub fn bsearch_value_table<T: Copy>(c: char, r: &'static [(char, T)]) -> Option<T> {
        match r.binary_search_by_key(&c, |&(k, _)| k) {
            Ok(idx) => {
                let (_, v) = r[idx];
                Some(v)
            }
            Err(_) => None
        }
    }

    #[inline]
    pub fn bsearch_range_table(c: char, r: &'static [(char,char)]) -> bool {
        use core::cmp::Ordering::{Equal, Less, Greater};
        r.binary_search_by(|&(lo,hi)| {
            if lo <= c && c <= hi { Equal }
            else if hi < c { Less }
            else { Greater }
        }).is_ok()
    }

    pub fn bsearch_range_value_table<T: Copy>(c: char, r: &'static [(char, char, T)]) -> Option<T> {
        use core::cmp::Ordering::{Equal, Less, Greater};
        match r.binary_search_by(|&(lo, hi, _)| {
            if lo <= c && c <= hi { Equal }
            else if hi < c { Less }
            else { Greater }
        }) {
            Ok(idx) => {
                let (_, _, cat) = r[idx];
                Some(cat)
            }
            Err(_) => None
        }
    }

}

pub mod identifier {

    #[derive(Copy, Clone, Hash, Eq, PartialEq, Ord, PartialOrd, Debug)]
    #[allow(non_camel_case_types)]
    /// https://www.unicode.org/reports/tr39/#Identifier_Status_and_Type
    pub enum IdentifierType {
        // Restricted
        Not_Character,
        Deprecated,
        Default_Ignorable,
        Not_NFKC,
        Not_XID,
        Exclusion,
        Obsolete,
        Technical,
        Uncommon_Use,
        Limited_Use,

        // Allowed
        Inclusion,
        Recommended
    }
    #[inline]
    pub fn identifier_status_allowed(c: char) -> bool {
        // FIXME: do we want to special case ASCII here?
        match c as usize {
            _ => super::util::bsearch_range_table(c, IDENTIFIER_STATUS)
        }
    }

    #[inline]
    pub fn identifier_type(c: char) -> Option<IdentifierType> {
        // FIXME: do we want to special case ASCII here?
        match c as usize {
            _ => super::util::bsearch_range_value_table(c, IDENTIFIER_TYPE)
        }
    }
    // Identifier status table:
    const IDENTIFIER_STATUS: &'static [(char, char)] = &[
        ('\u{27}', '\u{27}'), ('\u{2d}', '\u{2e}'), ('\u{30}', '\u{3a}'), ('\u{41}', '\u{5a}'),
        ('\u{5f}', '\u{5f}'), ('\u{61}', '\u{7a}'), ('\u{b7}', '\u{b7}'), ('\u{c0}', '\u{d6}'),
        ('\u{d8}', '\u{f6}'), ('\u{f8}', '\u{131}'), ('\u{134}', '\u{13e}'), ('\u{141}', '\u{148}'),
        ('\u{14a}', '\u{17e}'), ('\u{18f}', '\u{18f}'), ('\u{1a0}', '\u{1a1}'), ('\u{1af}',
        '\u{1b0}'), ('\u{1cd}', '\u{1dc}'), ('\u{1de}', '\u{1e3}'), ('\u{1e6}', '\u{1f0}'),
        ('\u{1f4}', '\u{1f5}'), ('\u{1f8}', '\u{1f9}'), ('\u{1fa}', '\u{217}'), ('\u{218}',
        '\u{21b}'), ('\u{21e}', '\u{21f}'), ('\u{226}', '\u{233}'), ('\u{259}', '\u{259}'),
        ('\u{2bb}', '\u{2bc}'), ('\u{2ec}', '\u{2ec}'), ('\u{300}', '\u{304}'), ('\u{306}',
        '\u{30c}'), ('\u{30f}', '\u{311}'), ('\u{313}', '\u{314}'), ('\u{31b}', '\u{31b}'),
        ('\u{323}', '\u{328}'), ('\u{32d}', '\u{32e}'), ('\u{330}', '\u{331}'), ('\u{335}',
        '\u{335}'), ('\u{338}', '\u{339}'), ('\u{342}', '\u{342}'), ('\u{345}', '\u{345}'),
        ('\u{375}', '\u{375}'), ('\u{37b}', '\u{37d}'), ('\u{386}', '\u{386}'), ('\u{388}',
        '\u{38a}'), ('\u{38c}', '\u{38c}'), ('\u{38e}', '\u{3a1}'), ('\u{3a3}', '\u{3ce}'),
        ('\u{3fc}', '\u{3ff}'), ('\u{400}', '\u{400}'), ('\u{401}', '\u{40c}'), ('\u{40d}',
        '\u{40d}'), ('\u{40e}', '\u{44f}'), ('\u{450}', '\u{450}'), ('\u{451}', '\u{45c}'),
        ('\u{45d}', '\u{45d}'), ('\u{45e}', '\u{45f}'), ('\u{48a}', '\u{48b}'), ('\u{48c}',
        '\u{48f}'), ('\u{490}', '\u{4c4}'), ('\u{4c5}', '\u{4c6}'), ('\u{4c7}', '\u{4c8}'),
        ('\u{4c9}', '\u{4ca}'), ('\u{4cb}', '\u{4cc}'), ('\u{4cd}', '\u{4ce}'), ('\u{4cf}',
        '\u{4cf}'), ('\u{4d0}', '\u{4eb}'), ('\u{4ec}', '\u{4ed}'), ('\u{4ee}', '\u{4f5}'),
        ('\u{4f6}', '\u{4f7}'), ('\u{4f8}', '\u{4f9}'), ('\u{4fa}', '\u{4ff}'), ('\u{510}',
        '\u{513}'), ('\u{514}', '\u{523}'), ('\u{524}', '\u{525}'), ('\u{526}', '\u{527}'),
        ('\u{528}', '\u{529}'), ('\u{52e}', '\u{52f}'), ('\u{531}', '\u{556}'), ('\u{559}',
        '\u{559}'), ('\u{561}', '\u{586}'), ('\u{58a}', '\u{58a}'), ('\u{5b4}', '\u{5b4}'),
        ('\u{5d0}', '\u{5ea}'), ('\u{5ef}', '\u{5ef}'), ('\u{5f0}', '\u{5f4}'), ('\u{620}',
        '\u{620}'), ('\u{621}', '\u{63a}'), ('\u{63b}', '\u{63f}'), ('\u{641}', '\u{652}'),
        ('\u{653}', '\u{655}'), ('\u{660}', '\u{669}'), ('\u{670}', '\u{672}'), ('\u{674}',
        '\u{674}'), ('\u{679}', '\u{68d}'), ('\u{68f}', '\u{6a0}'), ('\u{6a2}', '\u{6b7}'),
        ('\u{6b8}', '\u{6b9}'), ('\u{6ba}', '\u{6be}'), ('\u{6bf}', '\u{6bf}'), ('\u{6c0}',
        '\u{6ce}'), ('\u{6cf}', '\u{6cf}'), ('\u{6d0}', '\u{6d3}'), ('\u{6d5}', '\u{6d5}'),
        ('\u{6e5}', '\u{6e6}'), ('\u{6ee}', '\u{6ef}'), ('\u{6f0}', '\u{6f9}'), ('\u{6fa}',
        '\u{6fe}'), ('\u{6ff}', '\u{6ff}'), ('\u{750}', '\u{76d}'), ('\u{76e}', '\u{77f}'),
        ('\u{780}', '\u{7b0}'), ('\u{7b1}', '\u{7b1}'), ('\u{870}', '\u{887}'), ('\u{889}',
        '\u{88e}'), ('\u{8a0}', '\u{8a0}'), ('\u{8a1}', '\u{8a1}'), ('\u{8a2}', '\u{8ac}'),
        ('\u{8b2}', '\u{8b2}'), ('\u{8b5}', '\u{8b5}'), ('\u{8b6}', '\u{8bd}'), ('\u{8be}',
        '\u{8c7}'), ('\u{8c8}', '\u{8c9}'), ('\u{901}', '\u{903}'), ('\u{904}', '\u{904}'),
        ('\u{905}', '\u{939}'), ('\u{93a}', '\u{93b}'), ('\u{93c}', '\u{94d}'), ('\u{94f}',
        '\u{94f}'), ('\u{950}', '\u{950}'), ('\u{956}', '\u{957}'), ('\u{960}', '\u{963}'),
        ('\u{966}', '\u{96f}'), ('\u{971}', '\u{972}'), ('\u{973}', '\u{977}'), ('\u{979}',
        '\u{97a}'), ('\u{97b}', '\u{97c}'), ('\u{97d}', '\u{97d}'), ('\u{97e}', '\u{97f}'),
        ('\u{981}', '\u{983}'), ('\u{985}', '\u{98c}'), ('\u{98f}', '\u{990}'), ('\u{993}',
        '\u{9a8}'), ('\u{9aa}', '\u{9b0}'), ('\u{9b2}', '\u{9b2}'), ('\u{9b6}', '\u{9b9}'),
        ('\u{9bc}', '\u{9bc}'), ('\u{9bd}', '\u{9bd}'), ('\u{9be}', '\u{9c4}'), ('\u{9c7}',
        '\u{9c8}'), ('\u{9cb}', '\u{9cd}'), ('\u{9ce}', '\u{9ce}'), ('\u{9d7}', '\u{9d7}'),
        ('\u{9e0}', '\u{9e3}'), ('\u{9e6}', '\u{9f1}'), ('\u{9fe}', '\u{9fe}'), ('\u{a01}',
        '\u{a01}'), ('\u{a02}', '\u{a02}'), ('\u{a03}', '\u{a03}'), ('\u{a05}', '\u{a0a}'),
        ('\u{a0f}', '\u{a10}'), ('\u{a13}', '\u{a28}'), ('\u{a2a}', '\u{a30}'), ('\u{a32}',
        '\u{a32}'), ('\u{a35}', '\u{a35}'), ('\u{a38}', '\u{a39}'), ('\u{a3c}', '\u{a3c}'),
        ('\u{a3e}', '\u{a42}'), ('\u{a47}', '\u{a48}'), ('\u{a4b}', '\u{a4d}'), ('\u{a5c}',
        '\u{a5c}'), ('\u{a66}', '\u{a74}'), ('\u{a81}', '\u{a83}'), ('\u{a85}', '\u{a8b}'),
        ('\u{a8c}', '\u{a8c}'), ('\u{a8d}', '\u{a8d}'), ('\u{a8f}', '\u{a91}'), ('\u{a93}',
        '\u{aa8}'), ('\u{aaa}', '\u{ab0}'), ('\u{ab2}', '\u{ab3}'), ('\u{ab5}', '\u{ab9}'),
        ('\u{abc}', '\u{ac5}'), ('\u{ac7}', '\u{ac9}'), ('\u{acb}', '\u{acd}'), ('\u{ad0}',
        '\u{ad0}'), ('\u{ae0}', '\u{ae0}'), ('\u{ae1}', '\u{ae3}'), ('\u{ae6}', '\u{aef}'),
        ('\u{afa}', '\u{aff}'), ('\u{b01}', '\u{b03}'), ('\u{b05}', '\u{b0c}'), ('\u{b0f}',
        '\u{b10}'), ('\u{b13}', '\u{b28}'), ('\u{b2a}', '\u{b30}'), ('\u{b32}', '\u{b33}'),
        ('\u{b35}', '\u{b35}'), ('\u{b36}', '\u{b39}'), ('\u{b3c}', '\u{b43}'), ('\u{b47}',
        '\u{b48}'), ('\u{b4b}', '\u{b4d}'), ('\u{b55}', '\u{b55}'), ('\u{b56}', '\u{b57}'),
        ('\u{b5f}', '\u{b61}'), ('\u{b66}', '\u{b6f}'), ('\u{b71}', '\u{b71}'), ('\u{b82}',
        '\u{b83}'), ('\u{b85}', '\u{b8a}'), ('\u{b8e}', '\u{b90}'), ('\u{b92}', '\u{b95}'),
        ('\u{b99}', '\u{b9a}'), ('\u{b9c}', '\u{b9c}'), ('\u{b9e}', '\u{b9f}'), ('\u{ba3}',
        '\u{ba4}'), ('\u{ba8}', '\u{baa}'), ('\u{bae}', '\u{bb5}'), ('\u{bb6}', '\u{bb6}'),
        ('\u{bb7}', '\u{bb9}'), ('\u{bbe}', '\u{bc2}'), ('\u{bc6}', '\u{bc8}'), ('\u{bca}',
        '\u{bcd}'), ('\u{bd0}', '\u{bd0}'), ('\u{bd7}', '\u{bd7}'), ('\u{be6}', '\u{be6}'),
        ('\u{be7}', '\u{bef}'), ('\u{c01}', '\u{c03}'), ('\u{c04}', '\u{c04}'), ('\u{c05}',
        '\u{c0c}'), ('\u{c0e}', '\u{c10}'), ('\u{c12}', '\u{c28}'), ('\u{c2a}', '\u{c33}'),
        ('\u{c35}', '\u{c39}'), ('\u{c3c}', '\u{c3c}'), ('\u{c3d}', '\u{c3d}'), ('\u{c3e}',
        '\u{c44}'), ('\u{c46}', '\u{c48}'), ('\u{c4a}', '\u{c4d}'), ('\u{c55}', '\u{c56}'),
        ('\u{c5d}', '\u{c5d}'), ('\u{c60}', '\u{c61}'), ('\u{c66}', '\u{c6f}'), ('\u{c80}',
        '\u{c80}'), ('\u{c82}', '\u{c83}'), ('\u{c85}', '\u{c8c}'), ('\u{c8e}', '\u{c90}'),
        ('\u{c92}', '\u{ca8}'), ('\u{caa}', '\u{cb3}'), ('\u{cb5}', '\u{cb9}'), ('\u{cbc}',
        '\u{cbd}'), ('\u{cbe}', '\u{cc4}'), ('\u{cc6}', '\u{cc8}'), ('\u{cca}', '\u{ccd}'),
        ('\u{cd5}', '\u{cd6}'), ('\u{cdd}', '\u{cdd}'), ('\u{ce0}', '\u{ce1}'), ('\u{ce2}',
        '\u{ce3}'), ('\u{ce6}', '\u{cef}'), ('\u{cf1}', '\u{cf2}'), ('\u{cf3}', '\u{cf3}'),
        ('\u{d00}', '\u{d00}'), ('\u{d02}', '\u{d03}'), ('\u{d05}', '\u{d0c}'), ('\u{d0e}',
        '\u{d10}'), ('\u{d12}', '\u{d28}'), ('\u{d29}', '\u{d29}'), ('\u{d2a}', '\u{d39}'),
        ('\u{d3a}', '\u{d3a}'), ('\u{d3d}', '\u{d3d}'), ('\u{d3e}', '\u{d43}'), ('\u{d46}',
        '\u{d48}'), ('\u{d4a}', '\u{d4d}'), ('\u{d4e}', '\u{d4e}'), ('\u{d54}', '\u{d56}'),
        ('\u{d57}', '\u{d57}'), ('\u{d60}', '\u{d61}'), ('\u{d66}', '\u{d6f}'), ('\u{d7a}',
        '\u{d7f}'), ('\u{d82}', '\u{d83}'), ('\u{d85}', '\u{d8e}'), ('\u{d91}', '\u{d96}'),
        ('\u{d9a}', '\u{da5}'), ('\u{da7}', '\u{db1}'), ('\u{db3}', '\u{dbb}'), ('\u{dbd}',
        '\u{dbd}'), ('\u{dc0}', '\u{dc6}'), ('\u{dca}', '\u{dca}'), ('\u{dcf}', '\u{dd4}'),
        ('\u{dd6}', '\u{dd6}'), ('\u{dd8}', '\u{dde}'), ('\u{df2}', '\u{df2}'), ('\u{e01}',
        '\u{e32}'), ('\u{e34}', '\u{e3a}'), ('\u{e40}', '\u{e4e}'), ('\u{e50}', '\u{e59}'),
        ('\u{e81}', '\u{e82}'), ('\u{e84}', '\u{e84}'), ('\u{e86}', '\u{e86}'), ('\u{e87}',
        '\u{e88}'), ('\u{e89}', '\u{e89}'), ('\u{e8a}', '\u{e8a}'), ('\u{e8c}', '\u{e8c}'),
        ('\u{e8d}', '\u{e8d}'), ('\u{e8e}', '\u{e93}'), ('\u{e94}', '\u{e97}'), ('\u{e98}',
        '\u{e98}'), ('\u{e99}', '\u{e9f}'), ('\u{ea0}', '\u{ea0}'), ('\u{ea1}', '\u{ea3}'),
        ('\u{ea5}', '\u{ea5}'), ('\u{ea7}', '\u{ea7}'), ('\u{ea8}', '\u{ea9}'), ('\u{eaa}',
        '\u{eab}'), ('\u{eac}', '\u{eac}'), ('\u{ead}', '\u{eb2}'), ('\u{eb4}', '\u{eb9}'),
        ('\u{eba}', '\u{eba}'), ('\u{ebb}', '\u{ebd}'), ('\u{ec0}', '\u{ec4}'), ('\u{ec6}',
        '\u{ec6}'), ('\u{ec8}', '\u{ecd}'), ('\u{ece}', '\u{ece}'), ('\u{ed0}', '\u{ed9}'),
        ('\u{ede}', '\u{edf}'), ('\u{f00}', '\u{f00}'), ('\u{f0b}', '\u{f0b}'), ('\u{f20}',
        '\u{f29}'), ('\u{f35}', '\u{f35}'), ('\u{f37}', '\u{f37}'), ('\u{f3e}', '\u{f42}'),
        ('\u{f44}', '\u{f47}'), ('\u{f49}', '\u{f4c}'), ('\u{f4e}', '\u{f51}'), ('\u{f53}',
        '\u{f56}'), ('\u{f58}', '\u{f5b}'), ('\u{f5d}', '\u{f68}'), ('\u{f6a}', '\u{f6a}'),
        ('\u{f6b}', '\u{f6c}'), ('\u{f71}', '\u{f72}'), ('\u{f74}', '\u{f74}'), ('\u{f7a}',
        '\u{f80}'), ('\u{f82}', '\u{f84}'), ('\u{f86}', '\u{f8b}'), ('\u{f8c}', '\u{f8f}'),
        ('\u{f90}', '\u{f92}'), ('\u{f94}', '\u{f95}'), ('\u{f96}', '\u{f96}'), ('\u{f97}',
        '\u{f97}'), ('\u{f99}', '\u{f9c}'), ('\u{f9e}', '\u{fa1}'), ('\u{fa3}', '\u{fa6}'),
        ('\u{fa8}', '\u{fab}'), ('\u{fad}', '\u{fad}'), ('\u{fae}', '\u{fb0}'), ('\u{fb1}',
        '\u{fb7}'), ('\u{fb8}', '\u{fb8}'), ('\u{fba}', '\u{fbc}'), ('\u{fc6}', '\u{fc6}'),
        ('\u{1000}', '\u{1021}'), ('\u{1022}', '\u{1022}'), ('\u{1023}', '\u{1027}'), ('\u{1028}',
        '\u{1028}'), ('\u{1029}', '\u{102a}'), ('\u{102b}', '\u{102b}'), ('\u{102c}', '\u{1032}'),
        ('\u{1033}', '\u{1035}'), ('\u{1036}', '\u{1039}'), ('\u{103a}', '\u{103f}'), ('\u{1040}',
        '\u{1049}'), ('\u{1050}', '\u{1059}'), ('\u{105a}', '\u{1099}'), ('\u{109a}', '\u{109d}'),
        ('\u{10c7}', '\u{10c7}'), ('\u{10cd}', '\u{10cd}'), ('\u{10d0}', '\u{10f0}'), ('\u{10f7}',
        '\u{10f8}'), ('\u{10f9}', '\u{10fa}'), ('\u{10fd}', '\u{10ff}'), ('\u{1200}', '\u{1206}'),
        ('\u{1207}', '\u{1207}'), ('\u{1208}', '\u{1246}'), ('\u{1247}', '\u{1247}'), ('\u{1248}',
        '\u{1248}'), ('\u{124a}', '\u{124d}'), ('\u{1250}', '\u{1256}'), ('\u{1258}', '\u{1258}'),
        ('\u{125a}', '\u{125d}'), ('\u{1260}', '\u{1286}'), ('\u{1287}', '\u{1287}'), ('\u{1288}',
        '\u{1288}'), ('\u{128a}', '\u{128d}'), ('\u{1290}', '\u{12ae}'), ('\u{12af}', '\u{12af}'),
        ('\u{12b0}', '\u{12b0}'), ('\u{12b2}', '\u{12b5}'), ('\u{12b8}', '\u{12be}'), ('\u{12c0}',
        '\u{12c0}'), ('\u{12c2}', '\u{12c5}'), ('\u{12c8}', '\u{12ce}'), ('\u{12cf}', '\u{12cf}'),
        ('\u{12d0}', '\u{12d6}'), ('\u{12d8}', '\u{12ee}'), ('\u{12ef}', '\u{12ef}'), ('\u{12f0}',
        '\u{130e}'), ('\u{130f}', '\u{130f}'), ('\u{1310}', '\u{1310}'), ('\u{1312}', '\u{1315}'),
        ('\u{1318}', '\u{131e}'), ('\u{131f}', '\u{131f}'), ('\u{1320}', '\u{1346}'), ('\u{1347}',
        '\u{1347}'), ('\u{1348}', '\u{135a}'), ('\u{135d}', '\u{135e}'), ('\u{135f}', '\u{135f}'),
        ('\u{1380}', '\u{138f}'), ('\u{1780}', '\u{17a2}'), ('\u{17a5}', '\u{17a7}'), ('\u{17a9}',
        '\u{17b3}'), ('\u{17b6}', '\u{17cd}'), ('\u{17d0}', '\u{17d0}'), ('\u{17d2}', '\u{17d2}'),
        ('\u{17d7}', '\u{17d7}'), ('\u{17dc}', '\u{17dc}'), ('\u{17e0}', '\u{17e9}'), ('\u{1c90}',
        '\u{1cba}'), ('\u{1cbd}', '\u{1cbf}'), ('\u{1e00}', '\u{1e99}'), ('\u{1e9e}', '\u{1e9e}'),
        ('\u{1ea0}', '\u{1ef9}'), ('\u{1f00}', '\u{1f15}'), ('\u{1f18}', '\u{1f1d}'), ('\u{1f20}',
        '\u{1f45}'), ('\u{1f48}', '\u{1f4d}'), ('\u{1f50}', '\u{1f57}'), ('\u{1f59}', '\u{1f59}'),
        ('\u{1f5b}', '\u{1f5b}'), ('\u{1f5d}', '\u{1f5d}'), ('\u{1f5f}', '\u{1f70}'), ('\u{1f72}',
        '\u{1f72}'), ('\u{1f74}', '\u{1f74}'), ('\u{1f76}', '\u{1f76}'), ('\u{1f78}', '\u{1f78}'),
        ('\u{1f7a}', '\u{1f7a}'), ('\u{1f7c}', '\u{1f7c}'), ('\u{1f80}', '\u{1fb4}'), ('\u{1fb6}',
        '\u{1fba}'), ('\u{1fbc}', '\u{1fbc}'), ('\u{1fc2}', '\u{1fc4}'), ('\u{1fc6}', '\u{1fc8}'),
        ('\u{1fca}', '\u{1fca}'), ('\u{1fcc}', '\u{1fcc}'), ('\u{1fd0}', '\u{1fd2}'), ('\u{1fd6}',
        '\u{1fda}'), ('\u{1fe0}', '\u{1fe2}'), ('\u{1fe4}', '\u{1fea}'), ('\u{1fec}', '\u{1fec}'),
        ('\u{1ff2}', '\u{1ff4}'), ('\u{1ff6}', '\u{1ff8}'), ('\u{1ffa}', '\u{1ffa}'), ('\u{1ffc}',
        '\u{1ffc}'), ('\u{2010}', '\u{2010}'), ('\u{2019}', '\u{2019}'), ('\u{2027}', '\u{2027}'),
        ('\u{2d27}', '\u{2d27}'), ('\u{2d2d}', '\u{2d2d}'), ('\u{2d80}', '\u{2d96}'), ('\u{2da0}',
        '\u{2da6}'), ('\u{2da8}', '\u{2dae}'), ('\u{2db0}', '\u{2db6}'), ('\u{2db8}', '\u{2dbe}'),
        ('\u{2dc0}', '\u{2dc6}'), ('\u{2dc8}', '\u{2dce}'), ('\u{2dd0}', '\u{2dd6}'), ('\u{2dd8}',
        '\u{2dde}'), ('\u{3005}', '\u{3007}'), ('\u{3041}', '\u{3094}'), ('\u{3095}', '\u{3096}'),
        ('\u{3099}', '\u{309a}'), ('\u{309d}', '\u{309e}'), ('\u{30a0}', '\u{30a0}'), ('\u{30a1}',
        '\u{30fe}'), ('\u{3105}', '\u{312c}'), ('\u{312d}', '\u{312d}'), ('\u{312f}', '\u{312f}'),
        ('\u{31a0}', '\u{31b7}'), ('\u{31b8}', '\u{31ba}'), ('\u{31bb}', '\u{31bf}'), ('\u{3400}',
        '\u{4db5}'), ('\u{4db6}', '\u{4dbf}'), ('\u{4e00}', '\u{9fa5}'), ('\u{9fa6}', '\u{9fbb}'),
        ('\u{9fbc}', '\u{9fc3}'), ('\u{9fc4}', '\u{9fcb}'), ('\u{9fcc}', '\u{9fcc}'), ('\u{9fcd}',
        '\u{9fd5}'), ('\u{9fd6}', '\u{9fea}'), ('\u{9feb}', '\u{9fef}'), ('\u{9ff0}', '\u{9ffc}'),
        ('\u{9ffd}', '\u{9fff}'), ('\u{a67f}', '\u{a67f}'), ('\u{a717}', '\u{a71a}'), ('\u{a71b}',
        '\u{a71f}'), ('\u{a788}', '\u{a788}'), ('\u{a78d}', '\u{a78d}'), ('\u{a792}', '\u{a793}'),
        ('\u{a7aa}', '\u{a7aa}'), ('\u{a7c0}', '\u{a7c1}'), ('\u{a7c2}', '\u{a7c6}'), ('\u{a7c7}',
        '\u{a7ca}'), ('\u{a7d0}', '\u{a7d1}'), ('\u{a7d3}', '\u{a7d3}'), ('\u{a7d5}', '\u{a7d9}'),
        ('\u{a9e7}', '\u{a9fe}'), ('\u{aa60}', '\u{aa76}'), ('\u{aa7a}', '\u{aa7b}'), ('\u{aa7c}',
        '\u{aa7f}'), ('\u{ab01}', '\u{ab06}'), ('\u{ab09}', '\u{ab0e}'), ('\u{ab11}', '\u{ab16}'),
        ('\u{ab20}', '\u{ab26}'), ('\u{ab28}', '\u{ab2e}'), ('\u{ab66}', '\u{ab67}'), ('\u{ac00}',
        '\u{d7a3}'), ('\u{fa0e}', '\u{fa0f}'), ('\u{fa11}', '\u{fa11}'), ('\u{fa13}', '\u{fa14}'),
        ('\u{fa1f}', '\u{fa1f}'), ('\u{fa21}', '\u{fa21}'), ('\u{fa23}', '\u{fa24}'), ('\u{fa27}',
        '\u{fa29}'), ('\u{11301}', '\u{11301}'), ('\u{11303}', '\u{11303}'), ('\u{1133b}',
        '\u{1133b}'), ('\u{1133c}', '\u{1133c}'), ('\u{16ff0}', '\u{16ff1}'), ('\u{1b11f}',
        '\u{1b122}'), ('\u{1b132}', '\u{1b132}'), ('\u{1b150}', '\u{1b152}'), ('\u{1b155}',
        '\u{1b155}'), ('\u{1b164}', '\u{1b167}'), ('\u{1df00}', '\u{1df1e}'), ('\u{1df25}',
        '\u{1df2a}'), ('\u{1e08f}', '\u{1e08f}'), ('\u{1e7e0}', '\u{1e7e6}'), ('\u{1e7e8}',
        '\u{1e7eb}'), ('\u{1e7ed}', '\u{1e7ee}'), ('\u{1e7f0}', '\u{1e7fe}'), ('\u{20000}',
        '\u{2a6d6}'), ('\u{2a6d7}', '\u{2a6dd}'), ('\u{2a6de}', '\u{2a6df}'), ('\u{2a700}',
        '\u{2b734}'), ('\u{2b735}', '\u{2b738}'), ('\u{2b739}', '\u{2b739}'), ('\u{2b740}',
        '\u{2b81d}'), ('\u{2b820}', '\u{2cea1}'), ('\u{2ceb0}', '\u{2ebe0}'), ('\u{2ebf0}',
        '\u{2ee5d}'), ('\u{30000}', '\u{3134a}'), ('\u{31350}', '\u{323af}')
    ];

    const IDENTIFIER_TYPE: &'static [(char, char, IdentifierType)] = &[
        ('\u{9}', '\u{d}',  IdentifierType::Not_XID), ('\u{20}', '\u{26}',
         IdentifierType::Not_XID), ('\u{27}', '\u{27}',  IdentifierType::Inclusion), ('\u{28}',
        '\u{2c}',  IdentifierType::Not_XID), ('\u{2d}', '\u{2e}',  IdentifierType::Inclusion),
        ('\u{2f}', '\u{2f}',  IdentifierType::Not_XID), ('\u{30}', '\u{39}',
         IdentifierType::Recommended), ('\u{3a}', '\u{3a}',  IdentifierType::Inclusion), ('\u{3b}',
        '\u{40}',  IdentifierType::Not_XID), ('\u{41}', '\u{5a}',  IdentifierType::Recommended),
        ('\u{5b}', '\u{5e}',  IdentifierType::Not_XID), ('\u{5f}', '\u{5f}',
         IdentifierType::Recommended), ('\u{60}', '\u{60}',  IdentifierType::Not_XID), ('\u{61}',
        '\u{7a}',  IdentifierType::Recommended), ('\u{7b}', '\u{7e}',  IdentifierType::Not_XID),
        ('\u{85}', '\u{85}',  IdentifierType::Not_XID), ('\u{a0}', '\u{a0}',
         IdentifierType::Not_NFKC), ('\u{a1}', '\u{a7}',  IdentifierType::Not_XID), ('\u{a8}',
        '\u{a8}',  IdentifierType::Not_NFKC), ('\u{a9}', '\u{a9}',  IdentifierType::Not_XID),
        ('\u{aa}', '\u{aa}',  IdentifierType::Not_NFKC), ('\u{ab}', '\u{ac}',
         IdentifierType::Not_XID), ('\u{ad}', '\u{ad}',  IdentifierType::Default_Ignorable),
        ('\u{ae}', '\u{ae}',  IdentifierType::Not_XID), ('\u{af}', '\u{af}',
         IdentifierType::Not_NFKC), ('\u{b0}', '\u{b1}',  IdentifierType::Not_XID), ('\u{b2}',
        '\u{b5}',  IdentifierType::Not_NFKC), ('\u{b6}', '\u{b6}',  IdentifierType::Not_XID),
        ('\u{b7}', '\u{b7}',  IdentifierType::Inclusion), ('\u{b8}', '\u{ba}',
         IdentifierType::Not_NFKC), ('\u{bb}', '\u{bb}',  IdentifierType::Not_XID), ('\u{bc}',
        '\u{be}',  IdentifierType::Not_NFKC), ('\u{bf}', '\u{bf}',  IdentifierType::Not_XID),
        ('\u{c0}', '\u{d6}',  IdentifierType::Recommended), ('\u{d7}', '\u{d7}',
         IdentifierType::Not_XID), ('\u{d8}', '\u{f6}',  IdentifierType::Recommended), ('\u{f7}',
        '\u{f7}',  IdentifierType::Not_XID), ('\u{f8}', '\u{131}',  IdentifierType::Recommended),
        ('\u{132}', '\u{133}',  IdentifierType::Not_NFKC), ('\u{134}', '\u{13e}',
         IdentifierType::Recommended), ('\u{13f}', '\u{140}',  IdentifierType::Not_NFKC),
        ('\u{141}', '\u{148}',  IdentifierType::Recommended), ('\u{149}', '\u{149}',
         IdentifierType::Deprecated), ('\u{14a}', '\u{17e}',  IdentifierType::Recommended),
        ('\u{17f}', '\u{17f}',  IdentifierType::Not_NFKC), ('\u{180}', '\u{180}',
         IdentifierType::Technical), ('\u{181}', '\u{18c}',  IdentifierType::Uncommon_Use),
        ('\u{18d}', '\u{18d}',  IdentifierType::Technical), ('\u{18e}', '\u{18e}',
         IdentifierType::Uncommon_Use), ('\u{18f}', '\u{18f}',  IdentifierType::Recommended),
        ('\u{190}', '\u{19f}',  IdentifierType::Uncommon_Use), ('\u{1a0}', '\u{1a1}',
         IdentifierType::Recommended), ('\u{1a2}', '\u{1a9}',  IdentifierType::Uncommon_Use),
        ('\u{1aa}', '\u{1ab}',  IdentifierType::Technical), ('\u{1ac}', '\u{1ae}',
         IdentifierType::Uncommon_Use), ('\u{1af}', '\u{1b0}',  IdentifierType::Recommended),
        ('\u{1b1}', '\u{1b8}',  IdentifierType::Uncommon_Use), ('\u{1b9}', '\u{1b9}',
         IdentifierType::Obsolete), ('\u{1ba}', '\u{1bb}',  IdentifierType::Technical), ('\u{1bc}',
        '\u{1bd}',  IdentifierType::Uncommon_Use), ('\u{1be}', '\u{1be}',
         IdentifierType::Technical), ('\u{1bf}', '\u{1bf}',  IdentifierType::Obsolete), ('\u{1c0}',
        '\u{1c3}',  IdentifierType::Technical), ('\u{1c4}', '\u{1cc}',  IdentifierType::Not_NFKC),
        ('\u{1cd}', '\u{1dc}',  IdentifierType::Recommended), ('\u{1dd}', '\u{1dd}',
         IdentifierType::Uncommon_Use), ('\u{1de}', '\u{1e3}',  IdentifierType::Recommended),
        ('\u{1e4}', '\u{1e5}',  IdentifierType::Uncommon_Use), ('\u{1e6}', '\u{1f0}',
         IdentifierType::Recommended), ('\u{1f1}', '\u{1f3}',  IdentifierType::Not_NFKC),
        ('\u{1f4}', '\u{1f5}',  IdentifierType::Recommended), ('\u{1f6}', '\u{1f7}',
         IdentifierType::Obsolete), ('\u{1f8}', '\u{1f9}',  IdentifierType::Recommended),
        ('\u{1fa}', '\u{217}',  IdentifierType::Recommended), ('\u{218}', '\u{21b}',
         IdentifierType::Recommended), ('\u{21c}', '\u{21d}',  IdentifierType::Obsolete),
        ('\u{21e}', '\u{21f}',  IdentifierType::Recommended), ('\u{220}', '\u{220}',
         IdentifierType::Uncommon_Use), ('\u{221}', '\u{221}',  IdentifierType::Uncommon_Use),
        ('\u{222}', '\u{225}',  IdentifierType::Uncommon_Use), ('\u{226}', '\u{233}',
         IdentifierType::Recommended), ('\u{234}', '\u{236}',  IdentifierType::Technical),
        ('\u{237}', '\u{241}',  IdentifierType::Uncommon_Use), ('\u{242}', '\u{24f}',
         IdentifierType::Uncommon_Use), ('\u{250}', '\u{252}',  IdentifierType::Technical),
        ('\u{253}', '\u{254}',  IdentifierType::Uncommon_Use), ('\u{255}', '\u{255}',
         IdentifierType::Technical), ('\u{256}', '\u{257}',  IdentifierType::Uncommon_Use),
        ('\u{258}', '\u{258}',  IdentifierType::Technical), ('\u{259}', '\u{259}',
         IdentifierType::Recommended), ('\u{25a}', '\u{25a}',  IdentifierType::Technical),
        ('\u{25b}', '\u{25b}',  IdentifierType::Uncommon_Use), ('\u{25c}', '\u{262}',
         IdentifierType::Technical), ('\u{263}', '\u{263}',  IdentifierType::Uncommon_Use),
        ('\u{264}', '\u{267}',  IdentifierType::Technical), ('\u{268}', '\u{269}',
         IdentifierType::Uncommon_Use), ('\u{26a}', '\u{271}',  IdentifierType::Technical),
        ('\u{272}', '\u{272}',  IdentifierType::Uncommon_Use), ('\u{273}', '\u{276}',
         IdentifierType::Technical), ('\u{277}', '\u{277}',  IdentifierType::Technical), ('\u{278}',
        '\u{27b}',  IdentifierType::Technical), ('\u{27c}', '\u{27c}',  IdentifierType::Technical),
        ('\u{27d}', '\u{288}',  IdentifierType::Technical), ('\u{289}', '\u{289}',
         IdentifierType::Uncommon_Use), ('\u{28a}', '\u{291}',  IdentifierType::Technical),
        ('\u{292}', '\u{292}',  IdentifierType::Uncommon_Use), ('\u{293}', '\u{29d}',
         IdentifierType::Technical), ('\u{29e}', '\u{29e}',  IdentifierType::Technical), ('\u{29f}',
        '\u{2a8}',  IdentifierType::Technical), ('\u{2a9}', '\u{2ad}',  IdentifierType::Technical),
        ('\u{2ae}', '\u{2af}',  IdentifierType::Technical), ('\u{2b0}', '\u{2b8}',
         IdentifierType::Not_NFKC), ('\u{2b9}', '\u{2ba}',  IdentifierType::Technical), ('\u{2bb}',
        '\u{2bc}',  IdentifierType::Recommended), ('\u{2bd}', '\u{2c1}',
         IdentifierType::Technical), ('\u{2c2}', '\u{2c5}',  IdentifierType::Not_XID), ('\u{2c6}',
        '\u{2d1}',  IdentifierType::Technical), ('\u{2d2}', '\u{2d7}',  IdentifierType::Not_XID),
        ('\u{2d8}', '\u{2dd}',  IdentifierType::Not_NFKC), ('\u{2de}', '\u{2de}',
         IdentifierType::Not_XID), ('\u{2df}', '\u{2df}',  IdentifierType::Not_XID), ('\u{2e0}',
        '\u{2e4}',  IdentifierType::Not_NFKC), ('\u{2e5}', '\u{2e9}',  IdentifierType::Not_XID),
        ('\u{2ea}', '\u{2eb}',  IdentifierType::Not_XID), ('\u{2ec}', '\u{2ec}',
         IdentifierType::Recommended), ('\u{2ed}', '\u{2ed}',  IdentifierType::Not_XID), ('\u{2ee}',
        '\u{2ee}',  IdentifierType::Technical), ('\u{2ef}', '\u{2ff}',  IdentifierType::Not_XID),
        ('\u{300}', '\u{304}',  IdentifierType::Recommended), ('\u{305}', '\u{305}',
         IdentifierType::Uncommon_Use), ('\u{306}', '\u{30c}',  IdentifierType::Recommended),
        ('\u{30d}', '\u{30d}',  IdentifierType::Uncommon_Use), ('\u{30e}', '\u{30e}',
         IdentifierType::Technical), ('\u{30f}', '\u{311}',  IdentifierType::Recommended),
        ('\u{312}', '\u{312}',  IdentifierType::Technical), ('\u{313}', '\u{314}',
         IdentifierType::Recommended), ('\u{315}', '\u{315}',  IdentifierType::Technical),
        ('\u{316}', '\u{316}',  IdentifierType::Uncommon_Use), ('\u{317}', '\u{31a}',
         IdentifierType::Technical), ('\u{31b}', '\u{31b}',  IdentifierType::Recommended),
        ('\u{31c}', '\u{320}',  IdentifierType::Technical), ('\u{321}', '\u{322}',
         IdentifierType::Uncommon_Use), ('\u{323}', '\u{328}',  IdentifierType::Recommended),
        ('\u{329}', '\u{32c}',  IdentifierType::Technical), ('\u{32d}', '\u{32e}',
         IdentifierType::Recommended), ('\u{32f}', '\u{32f}',  IdentifierType::Technical),
        ('\u{330}', '\u{331}',  IdentifierType::Recommended), ('\u{332}', '\u{332}',
         IdentifierType::Uncommon_Use), ('\u{333}', '\u{333}',  IdentifierType::Technical),
        ('\u{334}', '\u{334}',  IdentifierType::Uncommon_Use), ('\u{335}', '\u{335}',
         IdentifierType::Recommended), ('\u{336}', '\u{336}',  IdentifierType::Uncommon_Use),
        ('\u{337}', '\u{337}',  IdentifierType::Technical), ('\u{338}', '\u{339}',
         IdentifierType::Recommended), ('\u{33a}', '\u{33f}',  IdentifierType::Technical),
        ('\u{340}', '\u{341}',  IdentifierType::Not_NFKC), ('\u{342}', '\u{342}',
         IdentifierType::Recommended), ('\u{343}', '\u{344}',  IdentifierType::Not_NFKC),
        ('\u{345}', '\u{345}',  IdentifierType::Recommended), ('\u{346}', '\u{34e}',
         IdentifierType::Technical), ('\u{34f}', '\u{34f}',  IdentifierType::Default_Ignorable),
        ('\u{350}', '\u{357}',  IdentifierType::Technical), ('\u{358}', '\u{358}',
         IdentifierType::Uncommon_Use), ('\u{359}', '\u{35c}',  IdentifierType::Technical),
        ('\u{35d}', '\u{35f}',  IdentifierType::Technical), ('\u{360}', '\u{361}',
         IdentifierType::Technical), ('\u{362}', '\u{362}',  IdentifierType::Technical), ('\u{363}',
        '\u{36f}',  IdentifierType::Obsolete), ('\u{370}', '\u{373}',  IdentifierType::Obsolete),
        ('\u{374}', '\u{374}',  IdentifierType::Not_NFKC), ('\u{375}', '\u{375}',
         IdentifierType::Inclusion), ('\u{376}', '\u{377}',  IdentifierType::Obsolete), ('\u{37a}',
        '\u{37a}',  IdentifierType::Not_NFKC), ('\u{37b}', '\u{37d}',  IdentifierType::Recommended),
        ('\u{37e}', '\u{37e}',  IdentifierType::Not_NFKC), ('\u{37f}', '\u{37f}',
         IdentifierType::Obsolete), ('\u{384}', '\u{385}',  IdentifierType::Not_NFKC), ('\u{386}',
        '\u{386}',  IdentifierType::Recommended), ('\u{387}', '\u{387}',  IdentifierType::Not_NFKC),
        ('\u{388}', '\u{38a}',  IdentifierType::Recommended), ('\u{38c}', '\u{38c}',
         IdentifierType::Recommended), ('\u{38e}', '\u{3a1}',  IdentifierType::Recommended),
        ('\u{3a3}', '\u{3ce}',  IdentifierType::Recommended), ('\u{3cf}', '\u{3cf}',
         IdentifierType::Technical), ('\u{3d0}', '\u{3d6}',  IdentifierType::Not_NFKC), ('\u{3d7}',
        '\u{3d7}',  IdentifierType::Technical), ('\u{3d8}', '\u{3d9}',  IdentifierType::Obsolete),
        ('\u{3da}', '\u{3da}',  IdentifierType::Obsolete), ('\u{3db}', '\u{3db}',
         IdentifierType::Obsolete), ('\u{3dc}', '\u{3dc}',  IdentifierType::Obsolete), ('\u{3dd}',
        '\u{3dd}',  IdentifierType::Obsolete), ('\u{3de}', '\u{3de}',  IdentifierType::Obsolete),
        ('\u{3df}', '\u{3df}',  IdentifierType::Obsolete), ('\u{3e0}', '\u{3e0}',
         IdentifierType::Obsolete), ('\u{3e1}', '\u{3e1}',  IdentifierType::Obsolete), ('\u{3e2}',
        '\u{3ef}',  IdentifierType::Exclusion), ('\u{3f0}', '\u{3f2}',  IdentifierType::Not_NFKC),
        ('\u{3f3}', '\u{3f3}',  IdentifierType::Technical), ('\u{3f4}', '\u{3f5}',
         IdentifierType::Not_NFKC), ('\u{3f6}', '\u{3f6}',  IdentifierType::Not_XID), ('\u{3f7}',
        '\u{3f8}',  IdentifierType::Obsolete), ('\u{3f9}', '\u{3f9}',  IdentifierType::Not_NFKC),
        ('\u{3fa}', '\u{3fb}',  IdentifierType::Obsolete), ('\u{3fc}', '\u{3ff}',
         IdentifierType::Recommended), ('\u{400}', '\u{400}',  IdentifierType::Recommended),
        ('\u{401}', '\u{40c}',  IdentifierType::Recommended), ('\u{40d}', '\u{40d}',
         IdentifierType::Recommended), ('\u{40e}', '\u{44f}',  IdentifierType::Recommended),
        ('\u{450}', '\u{450}',  IdentifierType::Recommended), ('\u{451}', '\u{45c}',
         IdentifierType::Recommended), ('\u{45d}', '\u{45d}',  IdentifierType::Recommended),
        ('\u{45e}', '\u{45f}',  IdentifierType::Recommended), ('\u{460}', '\u{481}',
         IdentifierType::Obsolete), ('\u{482}', '\u{482}',  IdentifierType::Obsolete), ('\u{483}',
        '\u{483}',  IdentifierType::Obsolete), ('\u{484}', '\u{486}',  IdentifierType::Technical),
        ('\u{487}', '\u{487}',  IdentifierType::Technical), ('\u{488}', '\u{489}',
         IdentifierType::Obsolete), ('\u{48a}', '\u{48b}',  IdentifierType::Recommended),
        ('\u{48c}', '\u{48f}',  IdentifierType::Recommended), ('\u{490}', '\u{4c4}',
         IdentifierType::Recommended), ('\u{4c5}', '\u{4c6}',  IdentifierType::Recommended),
        ('\u{4c7}', '\u{4c8}',  IdentifierType::Recommended), ('\u{4c9}', '\u{4ca}',
         IdentifierType::Recommended), ('\u{4cb}', '\u{4cc}',  IdentifierType::Recommended),
        ('\u{4cd}', '\u{4ce}',  IdentifierType::Recommended), ('\u{4cf}', '\u{4cf}',
         IdentifierType::Recommended), ('\u{4d0}', '\u{4eb}',  IdentifierType::Recommended),
        ('\u{4ec}', '\u{4ed}',  IdentifierType::Recommended), ('\u{4ee}', '\u{4f5}',
         IdentifierType::Recommended), ('\u{4f6}', '\u{4f7}',  IdentifierType::Recommended),
        ('\u{4f8}', '\u{4f9}',  IdentifierType::Recommended), ('\u{4fa}', '\u{4ff}',
         IdentifierType::Recommended), ('\u{500}', '\u{50f}',  IdentifierType::Obsolete),
        ('\u{510}', '\u{513}',  IdentifierType::Recommended), ('\u{514}', '\u{523}',
         IdentifierType::Recommended), ('\u{524}', '\u{525}',  IdentifierType::Recommended),
        ('\u{526}', '\u{527}',  IdentifierType::Recommended), ('\u{528}', '\u{529}',
         IdentifierType::Recommended), ('\u{52a}', '\u{52d}',  IdentifierType::Obsolete),
        ('\u{52e}', '\u{52f}',  IdentifierType::Recommended), ('\u{531}', '\u{556}',
         IdentifierType::Recommended), ('\u{559}', '\u{559}',  IdentifierType::Recommended),
        ('\u{55a}', '\u{55f}',  IdentifierType::Not_XID), ('\u{560}', '\u{560}',
         IdentifierType::Technical), ('\u{561}', '\u{586}',  IdentifierType::Recommended),
        ('\u{587}', '\u{587}',  IdentifierType::Not_NFKC), ('\u{588}', '\u{588}',
         IdentifierType::Technical), ('\u{589}', '\u{589}',  IdentifierType::Not_XID), ('\u{58a}',
        '\u{58a}',  IdentifierType::Inclusion), ('\u{58d}', '\u{58e}',  IdentifierType::Not_XID),
        ('\u{58f}', '\u{58f}',  IdentifierType::Not_XID), ('\u{591}', '\u{5a1}',
         IdentifierType::Uncommon_Use), ('\u{5a2}', '\u{5a2}',  IdentifierType::Uncommon_Use),
        ('\u{5a3}', '\u{5af}',  IdentifierType::Uncommon_Use), ('\u{5b0}', '\u{5b3}',
         IdentifierType::Uncommon_Use), ('\u{5b4}', '\u{5b4}',  IdentifierType::Recommended),
        ('\u{5b5}', '\u{5b9}',  IdentifierType::Uncommon_Use), ('\u{5ba}', '\u{5ba}',
         IdentifierType::Uncommon_Use), ('\u{5bb}', '\u{5bd}',  IdentifierType::Uncommon_Use),
        ('\u{5be}', '\u{5be}',  IdentifierType::Not_XID), ('\u{5bf}', '\u{5bf}',
         IdentifierType::Uncommon_Use), ('\u{5c0}', '\u{5c0}',  IdentifierType::Not_XID),
        ('\u{5c1}', '\u{5c2}',  IdentifierType::Uncommon_Use), ('\u{5c3}', '\u{5c3}',
         IdentifierType::Not_XID), ('\u{5c4}', '\u{5c4}',  IdentifierType::Uncommon_Use),
        ('\u{5c5}', '\u{5c5}',  IdentifierType::Uncommon_Use), ('\u{5c6}', '\u{5c6}',
         IdentifierType::Obsolete), ('\u{5c7}', '\u{5c7}',  IdentifierType::Uncommon_Use),
        ('\u{5d0}', '\u{5ea}',  IdentifierType::Recommended), ('\u{5ef}', '\u{5ef}',
         IdentifierType::Recommended), ('\u{5f0}', '\u{5f2}',  IdentifierType::Recommended),
        ('\u{5f3}', '\u{5f4}',  IdentifierType::Inclusion), ('\u{600}', '\u{603}',
         IdentifierType::Not_XID), ('\u{604}', '\u{604}',  IdentifierType::Not_XID), ('\u{605}',
        '\u{605}',  IdentifierType::Not_XID), ('\u{606}', '\u{60a}',  IdentifierType::Not_XID),
        ('\u{60b}', '\u{60b}',  IdentifierType::Not_XID), ('\u{60c}', '\u{60c}',
         IdentifierType::Not_XID), ('\u{60d}', '\u{60f}',  IdentifierType::Not_XID), ('\u{610}',
        '\u{615}',  IdentifierType::Uncommon_Use), ('\u{616}', '\u{61a}',
         IdentifierType::Uncommon_Use), ('\u{61b}', '\u{61b}',  IdentifierType::Not_XID),
        ('\u{61c}', '\u{61c}',  IdentifierType::Default_Ignorable), ('\u{61d}', '\u{61d}',
         IdentifierType::Not_XID), ('\u{61e}', '\u{61e}',  IdentifierType::Not_XID), ('\u{61f}',
        '\u{61f}',  IdentifierType::Not_XID), ('\u{620}', '\u{620}',  IdentifierType::Recommended),
        ('\u{621}', '\u{63a}',  IdentifierType::Recommended), ('\u{63b}', '\u{63f}',
         IdentifierType::Recommended), ('\u{640}', '\u{640}',  IdentifierType::Obsolete),
        ('\u{641}', '\u{652}',  IdentifierType::Recommended), ('\u{653}', '\u{655}',
         IdentifierType::Recommended), ('\u{656}', '\u{658}',  IdentifierType::Uncommon_Use),
        ('\u{659}', '\u{65e}',  IdentifierType::Uncommon_Use), ('\u{65f}', '\u{65f}',
         IdentifierType::Uncommon_Use), ('\u{660}', '\u{669}',  IdentifierType::Recommended),
        ('\u{66a}', '\u{66d}',  IdentifierType::Not_XID), ('\u{66e}', '\u{66f}',
         IdentifierType::Obsolete), ('\u{670}', '\u{672}',  IdentifierType::Recommended),
        ('\u{673}', '\u{673}',  IdentifierType::Deprecated), ('\u{674}', '\u{674}',
         IdentifierType::Recommended), ('\u{675}', '\u{678}',  IdentifierType::Not_NFKC),
        ('\u{679}', '\u{68d}',  IdentifierType::Recommended), ('\u{68e}', '\u{68e}',
         IdentifierType::Obsolete), ('\u{68f}', '\u{6a0}',  IdentifierType::Recommended),
        ('\u{6a1}', '\u{6a1}',  IdentifierType::Obsolete), ('\u{6a2}', '\u{6b7}',
         IdentifierType::Recommended), ('\u{6b8}', '\u{6b9}',  IdentifierType::Recommended),
        ('\u{6ba}', '\u{6be}',  IdentifierType::Recommended), ('\u{6bf}', '\u{6bf}',
         IdentifierType::Recommended), ('\u{6c0}', '\u{6ce}',  IdentifierType::Recommended),
        ('\u{6cf}', '\u{6cf}',  IdentifierType::Recommended), ('\u{6d0}', '\u{6d3}',
         IdentifierType::Recommended), ('\u{6d4}', '\u{6d4}',  IdentifierType::Not_XID), ('\u{6d5}',
        '\u{6d5}',  IdentifierType::Recommended), ('\u{6d6}', '\u{6dc}',
         IdentifierType::Uncommon_Use), ('\u{6dd}', '\u{6dd}',  IdentifierType::Not_XID),
        ('\u{6de}', '\u{6de}',  IdentifierType::Not_XID), ('\u{6df}', '\u{6e4}',
         IdentifierType::Uncommon_Use), ('\u{6e5}', '\u{6e6}',  IdentifierType::Recommended),
        ('\u{6e7}', '\u{6e8}',  IdentifierType::Uncommon_Use), ('\u{6e9}', '\u{6e9}',
         IdentifierType::Not_XID), ('\u{6ea}', '\u{6ed}',  IdentifierType::Uncommon_Use),
        ('\u{6ee}', '\u{6ef}',  IdentifierType::Recommended), ('\u{6f0}', '\u{6f9}',
         IdentifierType::Recommended), ('\u{6fa}', '\u{6fc}',  IdentifierType::Recommended),
        ('\u{6fd}', '\u{6fe}',  IdentifierType::Inclusion), ('\u{6ff}', '\u{6ff}',
         IdentifierType::Recommended), ('\u{700}', '\u{70d}',  IdentifierType::Limited_Use),
        ('\u{70f}', '\u{70f}',  IdentifierType::Limited_Use), ('\u{710}', '\u{72c}',
         IdentifierType::Limited_Use), ('\u{72d}', '\u{72f}',  IdentifierType::Limited_Use),
        ('\u{730}', '\u{73f}',  IdentifierType::Limited_Use), ('\u{740}', '\u{74a}',
         IdentifierType::Limited_Use), ('\u{74d}', '\u{74f}',  IdentifierType::Limited_Use),
        ('\u{750}', '\u{76d}',  IdentifierType::Recommended), ('\u{76e}', '\u{77f}',
         IdentifierType::Recommended), ('\u{780}', '\u{7b0}',  IdentifierType::Recommended),
        ('\u{7b1}', '\u{7b1}',  IdentifierType::Recommended), ('\u{7c0}', '\u{7e7}',
         IdentifierType::Limited_Use), ('\u{7e8}', '\u{7ea}',  IdentifierType::Limited_Use),
        ('\u{7eb}', '\u{7f5}',  IdentifierType::Limited_Use), ('\u{7f6}', '\u{7f9}',
         IdentifierType::Limited_Use), ('\u{7fa}', '\u{7fa}',  IdentifierType::Limited_Use),
        ('\u{7fd}', '\u{7fd}',  IdentifierType::Limited_Use), ('\u{7fe}', '\u{7ff}',
         IdentifierType::Limited_Use), ('\u{800}', '\u{82d}',  IdentifierType::Exclusion),
        ('\u{830}', '\u{83e}',  IdentifierType::Exclusion), ('\u{840}', '\u{85b}',
         IdentifierType::Limited_Use), ('\u{85e}', '\u{85e}',  IdentifierType::Limited_Use),
        ('\u{860}', '\u{86a}',  IdentifierType::Limited_Use), ('\u{870}', '\u{887}',
         IdentifierType::Recommended), ('\u{888}', '\u{888}',  IdentifierType::Not_XID), ('\u{889}',
        '\u{88e}',  IdentifierType::Recommended), ('\u{890}', '\u{891}',  IdentifierType::Not_XID),
        ('\u{897}', '\u{897}',  IdentifierType::Uncommon_Use), ('\u{898}', '\u{89f}',
         IdentifierType::Uncommon_Use), ('\u{8a0}', '\u{8a0}',  IdentifierType::Recommended),
        ('\u{8a1}', '\u{8a1}',  IdentifierType::Recommended), ('\u{8a2}', '\u{8ac}',
         IdentifierType::Recommended), ('\u{8ad}', '\u{8b1}',  IdentifierType::Obsolete),
        ('\u{8b2}', '\u{8b2}',  IdentifierType::Recommended), ('\u{8b3}', '\u{8b4}',
         IdentifierType::Uncommon_Use), ('\u{8b5}', '\u{8b5}',  IdentifierType::Recommended),
        ('\u{8b6}', '\u{8bd}',  IdentifierType::Recommended), ('\u{8be}', '\u{8c7}',
         IdentifierType::Recommended), ('\u{8c8}', '\u{8c9}',  IdentifierType::Recommended),
        ('\u{8ca}', '\u{8d2}',  IdentifierType::Uncommon_Use), ('\u{8d3}', '\u{8d3}',
         IdentifierType::Uncommon_Use), ('\u{8d4}', '\u{8e1}',  IdentifierType::Uncommon_Use),
        ('\u{8e2}', '\u{8e2}',  IdentifierType::Not_XID), ('\u{8e3}', '\u{8e3}',
         IdentifierType::Uncommon_Use), ('\u{8e4}', '\u{8fe}',  IdentifierType::Uncommon_Use),
        ('\u{8ff}', '\u{8ff}',  IdentifierType::Uncommon_Use), ('\u{900}', '\u{900}',
         IdentifierType::Uncommon_Use), ('\u{901}', '\u{903}',  IdentifierType::Recommended),
        ('\u{904}', '\u{904}',  IdentifierType::Recommended), ('\u{905}', '\u{939}',
         IdentifierType::Recommended), ('\u{93a}', '\u{93b}',  IdentifierType::Recommended),
        ('\u{93c}', '\u{94d}',  IdentifierType::Recommended), ('\u{94e}', '\u{94e}',
         IdentifierType::Obsolete), ('\u{94f}', '\u{94f}',  IdentifierType::Recommended),
        ('\u{950}', '\u{950}',  IdentifierType::Recommended), ('\u{951}', '\u{952}',
         IdentifierType::Obsolete), ('\u{953}', '\u{954}',  IdentifierType::Technical), ('\u{955}',
        '\u{955}',  IdentifierType::Uncommon_Use), ('\u{956}', '\u{957}',
         IdentifierType::Recommended), ('\u{958}', '\u{95f}',  IdentifierType::Not_NFKC),
        ('\u{960}', '\u{963}',  IdentifierType::Recommended), ('\u{964}', '\u{965}',
         IdentifierType::Not_XID), ('\u{966}', '\u{96f}',  IdentifierType::Recommended), ('\u{970}',
        '\u{970}',  IdentifierType::Not_XID), ('\u{971}', '\u{972}',  IdentifierType::Recommended),
        ('\u{973}', '\u{977}',  IdentifierType::Recommended), ('\u{978}', '\u{978}',
         IdentifierType::Obsolete), ('\u{979}', '\u{97a}',  IdentifierType::Recommended),
        ('\u{97b}', '\u{97c}',  IdentifierType::Recommended), ('\u{97d}', '\u{97d}',
         IdentifierType::Recommended), ('\u{97e}', '\u{97f}',  IdentifierType::Recommended),
        ('\u{980}', '\u{980}',  IdentifierType::Obsolete), ('\u{981}', '\u{983}',
         IdentifierType::Recommended), ('\u{985}', '\u{98c}',  IdentifierType::Recommended),
        ('\u{98f}', '\u{990}',  IdentifierType::Recommended), ('\u{993}', '\u{9a8}',
         IdentifierType::Recommended), ('\u{9aa}', '\u{9b0}',  IdentifierType::Recommended),
        ('\u{9b2}', '\u{9b2}',  IdentifierType::Recommended), ('\u{9b6}', '\u{9b9}',
         IdentifierType::Recommended), ('\u{9bc}', '\u{9bc}',  IdentifierType::Recommended),
        ('\u{9bd}', '\u{9bd}',  IdentifierType::Recommended), ('\u{9be}', '\u{9c4}',
         IdentifierType::Recommended), ('\u{9c7}', '\u{9c8}',  IdentifierType::Recommended),
        ('\u{9cb}', '\u{9cd}',  IdentifierType::Recommended), ('\u{9ce}', '\u{9ce}',
         IdentifierType::Recommended), ('\u{9d7}', '\u{9d7}',  IdentifierType::Recommended),
        ('\u{9dc}', '\u{9dd}',  IdentifierType::Not_NFKC), ('\u{9df}', '\u{9df}',
         IdentifierType::Not_NFKC), ('\u{9e0}', '\u{9e3}',  IdentifierType::Recommended),
        ('\u{9e6}', '\u{9f1}',  IdentifierType::Recommended), ('\u{9f2}', '\u{9fa}',
         IdentifierType::Not_XID), ('\u{9fb}', '\u{9fb}',  IdentifierType::Not_XID), ('\u{9fc}',
        '\u{9fc}',  IdentifierType::Obsolete), ('\u{9fd}', '\u{9fd}',  IdentifierType::Not_XID),
        ('\u{9fe}', '\u{9fe}',  IdentifierType::Recommended), ('\u{a01}', '\u{a01}',
         IdentifierType::Recommended), ('\u{a02}', '\u{a02}',  IdentifierType::Recommended),
        ('\u{a03}', '\u{a03}',  IdentifierType::Recommended), ('\u{a05}', '\u{a0a}',
         IdentifierType::Recommended), ('\u{a0f}', '\u{a10}',  IdentifierType::Recommended),
        ('\u{a13}', '\u{a28}',  IdentifierType::Recommended), ('\u{a2a}', '\u{a30}',
         IdentifierType::Recommended), ('\u{a32}', '\u{a32}',  IdentifierType::Recommended),
        ('\u{a33}', '\u{a33}',  IdentifierType::Not_NFKC), ('\u{a35}', '\u{a35}',
         IdentifierType::Recommended), ('\u{a36}', '\u{a36}',  IdentifierType::Not_NFKC),
        ('\u{a38}', '\u{a39}',  IdentifierType::Recommended), ('\u{a3c}', '\u{a3c}',
         IdentifierType::Recommended), ('\u{a3e}', '\u{a42}',  IdentifierType::Recommended),
        ('\u{a47}', '\u{a48}',  IdentifierType::Recommended), ('\u{a4b}', '\u{a4d}',
         IdentifierType::Recommended), ('\u{a51}', '\u{a51}',  IdentifierType::Uncommon_Use),
        ('\u{a59}', '\u{a5b}',  IdentifierType::Not_NFKC), ('\u{a5c}', '\u{a5c}',
         IdentifierType::Recommended), ('\u{a5e}', '\u{a5e}',  IdentifierType::Not_NFKC),
        ('\u{a66}', '\u{a74}',  IdentifierType::Recommended), ('\u{a75}', '\u{a75}',
         IdentifierType::Uncommon_Use), ('\u{a76}', '\u{a76}',  IdentifierType::Not_XID),
        ('\u{a81}', '\u{a83}',  IdentifierType::Recommended), ('\u{a85}', '\u{a8b}',
         IdentifierType::Recommended), ('\u{a8c}', '\u{a8c}',  IdentifierType::Recommended),
        ('\u{a8d}', '\u{a8d}',  IdentifierType::Recommended), ('\u{a8f}', '\u{a91}',
         IdentifierType::Recommended), ('\u{a93}', '\u{aa8}',  IdentifierType::Recommended),
        ('\u{aaa}', '\u{ab0}',  IdentifierType::Recommended), ('\u{ab2}', '\u{ab3}',
         IdentifierType::Recommended), ('\u{ab5}', '\u{ab9}',  IdentifierType::Recommended),
        ('\u{abc}', '\u{ac5}',  IdentifierType::Recommended), ('\u{ac7}', '\u{ac9}',
         IdentifierType::Recommended), ('\u{acb}', '\u{acd}',  IdentifierType::Recommended),
        ('\u{ad0}', '\u{ad0}',  IdentifierType::Recommended), ('\u{ae0}', '\u{ae0}',
         IdentifierType::Recommended), ('\u{ae1}', '\u{ae3}',  IdentifierType::Recommended),
        ('\u{ae6}', '\u{aef}',  IdentifierType::Recommended), ('\u{af0}', '\u{af0}',
         IdentifierType::Not_XID), ('\u{af1}', '\u{af1}',  IdentifierType::Not_XID), ('\u{af9}',
        '\u{af9}',  IdentifierType::Uncommon_Use), ('\u{afa}', '\u{aff}',
         IdentifierType::Recommended), ('\u{b01}', '\u{b03}',  IdentifierType::Recommended),
        ('\u{b05}', '\u{b0c}',  IdentifierType::Recommended), ('\u{b0f}', '\u{b10}',
         IdentifierType::Recommended), ('\u{b13}', '\u{b28}',  IdentifierType::Recommended),
        ('\u{b2a}', '\u{b30}',  IdentifierType::Recommended), ('\u{b32}', '\u{b33}',
         IdentifierType::Recommended), ('\u{b35}', '\u{b35}',  IdentifierType::Recommended),
        ('\u{b36}', '\u{b39}',  IdentifierType::Recommended), ('\u{b3c}', '\u{b43}',
         IdentifierType::Recommended), ('\u{b44}', '\u{b44}',  IdentifierType::Uncommon_Use),
        ('\u{b47}', '\u{b48}',  IdentifierType::Recommended), ('\u{b4b}', '\u{b4d}',
         IdentifierType::Recommended), ('\u{b55}', '\u{b55}',  IdentifierType::Recommended),
        ('\u{b56}', '\u{b57}',  IdentifierType::Recommended), ('\u{b5c}', '\u{b5d}',
         IdentifierType::Not_NFKC), ('\u{b5f}', '\u{b61}',  IdentifierType::Recommended),
        ('\u{b62}', '\u{b63}',  IdentifierType::Uncommon_Use), ('\u{b66}', '\u{b6f}',
         IdentifierType::Recommended), ('\u{b70}', '\u{b70}',  IdentifierType::Not_XID), ('\u{b71}',
        '\u{b71}',  IdentifierType::Recommended), ('\u{b72}', '\u{b77}',  IdentifierType::Not_XID),
        ('\u{b82}', '\u{b83}',  IdentifierType::Recommended), ('\u{b85}', '\u{b8a}',
         IdentifierType::Recommended), ('\u{b8e}', '\u{b90}',  IdentifierType::Recommended),
        ('\u{b92}', '\u{b95}',  IdentifierType::Recommended), ('\u{b99}', '\u{b9a}',
         IdentifierType::Recommended), ('\u{b9c}', '\u{b9c}',  IdentifierType::Recommended),
        ('\u{b9e}', '\u{b9f}',  IdentifierType::Recommended), ('\u{ba3}', '\u{ba4}',
         IdentifierType::Recommended), ('\u{ba8}', '\u{baa}',  IdentifierType::Recommended),
        ('\u{bae}', '\u{bb5}',  IdentifierType::Recommended), ('\u{bb6}', '\u{bb6}',
         IdentifierType::Recommended), ('\u{bb7}', '\u{bb9}',  IdentifierType::Recommended),
        ('\u{bbe}', '\u{bc2}',  IdentifierType::Recommended), ('\u{bc6}', '\u{bc8}',
         IdentifierType::Recommended), ('\u{bca}', '\u{bcd}',  IdentifierType::Recommended),
        ('\u{bd0}', '\u{bd0}',  IdentifierType::Recommended), ('\u{bd7}', '\u{bd7}',
         IdentifierType::Recommended), ('\u{be6}', '\u{be6}',  IdentifierType::Recommended),
        ('\u{be7}', '\u{bef}',  IdentifierType::Recommended), ('\u{bf0}', '\u{bf2}',
         IdentifierType::Not_XID), ('\u{bf3}', '\u{bfa}',  IdentifierType::Not_XID), ('\u{c00}',
        '\u{c00}',  IdentifierType::Obsolete), ('\u{c01}', '\u{c03}',  IdentifierType::Recommended),
        ('\u{c04}', '\u{c04}',  IdentifierType::Recommended), ('\u{c05}', '\u{c0c}',
         IdentifierType::Recommended), ('\u{c0e}', '\u{c10}',  IdentifierType::Recommended),
        ('\u{c12}', '\u{c28}',  IdentifierType::Recommended), ('\u{c2a}', '\u{c33}',
         IdentifierType::Recommended), ('\u{c34}', '\u{c34}',  IdentifierType::Obsolete),
        ('\u{c35}', '\u{c39}',  IdentifierType::Recommended), ('\u{c3c}', '\u{c3c}',
         IdentifierType::Recommended), ('\u{c3d}', '\u{c3d}',  IdentifierType::Recommended),
        ('\u{c3e}', '\u{c44}',  IdentifierType::Recommended), ('\u{c46}', '\u{c48}',
         IdentifierType::Recommended), ('\u{c4a}', '\u{c4d}',  IdentifierType::Recommended),
        ('\u{c55}', '\u{c56}',  IdentifierType::Recommended), ('\u{c58}', '\u{c59}',
         IdentifierType::Obsolete), ('\u{c5a}', '\u{c5a}',  IdentifierType::Uncommon_Use),
        ('\u{c5d}', '\u{c5d}',  IdentifierType::Recommended), ('\u{c60}', '\u{c61}',
         IdentifierType::Recommended), ('\u{c62}', '\u{c63}',  IdentifierType::Uncommon_Use),
        ('\u{c66}', '\u{c6f}',  IdentifierType::Recommended), ('\u{c77}', '\u{c77}',
         IdentifierType::Not_XID), ('\u{c78}', '\u{c7f}',  IdentifierType::Not_XID), ('\u{c80}',
        '\u{c80}',  IdentifierType::Recommended), ('\u{c81}', '\u{c81}',  IdentifierType::Obsolete),
        ('\u{c82}', '\u{c83}',  IdentifierType::Recommended), ('\u{c84}', '\u{c84}',
         IdentifierType::Not_XID), ('\u{c85}', '\u{c8c}',  IdentifierType::Recommended), ('\u{c8e}',
        '\u{c90}',  IdentifierType::Recommended), ('\u{c92}', '\u{ca8}',
         IdentifierType::Recommended), ('\u{caa}', '\u{cb3}',  IdentifierType::Recommended),
        ('\u{cb5}', '\u{cb9}',  IdentifierType::Recommended), ('\u{cbc}', '\u{cbd}',
         IdentifierType::Recommended), ('\u{cbe}', '\u{cc4}',  IdentifierType::Recommended),
        ('\u{cc6}', '\u{cc8}',  IdentifierType::Recommended), ('\u{cca}', '\u{ccd}',
         IdentifierType::Recommended), ('\u{cd5}', '\u{cd6}',  IdentifierType::Recommended),
        ('\u{cdd}', '\u{cdd}',  IdentifierType::Recommended), ('\u{cde}', '\u{cde}',
         IdentifierType::Obsolete), ('\u{ce0}', '\u{ce1}',  IdentifierType::Recommended),
        ('\u{ce2}', '\u{ce3}',  IdentifierType::Recommended), ('\u{ce6}', '\u{cef}',
         IdentifierType::Recommended), ('\u{cf1}', '\u{cf2}',  IdentifierType::Recommended),
        ('\u{cf3}', '\u{cf3}',  IdentifierType::Recommended), ('\u{d00}', '\u{d00}',
         IdentifierType::Recommended), ('\u{d01}', '\u{d01}',  IdentifierType::Obsolete),
        ('\u{d02}', '\u{d03}',  IdentifierType::Recommended), ('\u{d04}', '\u{d04}',
         IdentifierType::Technical), ('\u{d05}', '\u{d0c}',  IdentifierType::Recommended),
        ('\u{d0e}', '\u{d10}',  IdentifierType::Recommended), ('\u{d12}', '\u{d28}',
         IdentifierType::Recommended), ('\u{d29}', '\u{d29}',  IdentifierType::Recommended),
        ('\u{d2a}', '\u{d39}',  IdentifierType::Recommended), ('\u{d3a}', '\u{d3a}',
         IdentifierType::Recommended), ('\u{d3b}', '\u{d3c}',  IdentifierType::Obsolete),
        ('\u{d3d}', '\u{d3d}',  IdentifierType::Recommended), ('\u{d3e}', '\u{d43}',
         IdentifierType::Recommended), ('\u{d44}', '\u{d44}',  IdentifierType::Uncommon_Use),
        ('\u{d46}', '\u{d48}',  IdentifierType::Recommended), ('\u{d4a}', '\u{d4d}',
         IdentifierType::Recommended), ('\u{d4e}', '\u{d4e}',  IdentifierType::Recommended),
        ('\u{d4f}', '\u{d4f}',  IdentifierType::Not_XID), ('\u{d54}', '\u{d56}',
         IdentifierType::Recommended), ('\u{d57}', '\u{d57}',  IdentifierType::Recommended),
        ('\u{d58}', '\u{d5e}',  IdentifierType::Not_XID), ('\u{d5f}', '\u{d5f}',
         IdentifierType::Obsolete), ('\u{d60}', '\u{d61}',  IdentifierType::Recommended),
        ('\u{d62}', '\u{d63}',  IdentifierType::Uncommon_Use), ('\u{d66}', '\u{d6f}',
         IdentifierType::Recommended), ('\u{d70}', '\u{d75}',  IdentifierType::Not_XID), ('\u{d76}',
        '\u{d78}',  IdentifierType::Not_XID), ('\u{d79}', '\u{d79}',  IdentifierType::Not_XID),
        ('\u{d7a}', '\u{d7f}',  IdentifierType::Recommended), ('\u{d81}', '\u{d81}',
         IdentifierType::Technical), ('\u{d82}', '\u{d83}',  IdentifierType::Recommended),
        ('\u{d85}', '\u{d8e}',  IdentifierType::Recommended), ('\u{d8f}', '\u{d90}',
         IdentifierType::Uncommon_Use), ('\u{d91}', '\u{d96}',  IdentifierType::Recommended),
        ('\u{d9a}', '\u{da5}',  IdentifierType::Recommended), ('\u{da6}', '\u{da6}',
         IdentifierType::Uncommon_Use), ('\u{da7}', '\u{db1}',  IdentifierType::Recommended),
        ('\u{db3}', '\u{dbb}',  IdentifierType::Recommended), ('\u{dbd}', '\u{dbd}',
         IdentifierType::Recommended), ('\u{dc0}', '\u{dc6}',  IdentifierType::Recommended),
        ('\u{dca}', '\u{dca}',  IdentifierType::Recommended), ('\u{dcf}', '\u{dd4}',
         IdentifierType::Recommended), ('\u{dd6}', '\u{dd6}',  IdentifierType::Recommended),
        ('\u{dd8}', '\u{dde}',  IdentifierType::Recommended), ('\u{ddf}', '\u{ddf}',
         IdentifierType::Uncommon_Use), ('\u{de6}', '\u{def}',  IdentifierType::Obsolete),
        ('\u{df2}', '\u{df2}',  IdentifierType::Recommended), ('\u{df3}', '\u{df3}',
         IdentifierType::Uncommon_Use), ('\u{df4}', '\u{df4}',  IdentifierType::Not_XID),
        ('\u{e01}', '\u{e32}',  IdentifierType::Recommended), ('\u{e33}', '\u{e33}',
         IdentifierType::Not_NFKC), ('\u{e34}', '\u{e3a}',  IdentifierType::Recommended),
        ('\u{e3f}', '\u{e3f}',  IdentifierType::Not_XID), ('\u{e40}', '\u{e4e}',
         IdentifierType::Recommended), ('\u{e4f}', '\u{e4f}',  IdentifierType::Not_XID), ('\u{e50}',
        '\u{e59}',  IdentifierType::Recommended), ('\u{e5a}', '\u{e5b}',  IdentifierType::Not_XID),
        ('\u{e81}', '\u{e82}',  IdentifierType::Recommended), ('\u{e84}', '\u{e84}',
         IdentifierType::Recommended), ('\u{e86}', '\u{e86}',  IdentifierType::Recommended),
        ('\u{e87}', '\u{e88}',  IdentifierType::Recommended), ('\u{e89}', '\u{e89}',
         IdentifierType::Recommended), ('\u{e8a}', '\u{e8a}',  IdentifierType::Recommended),
        ('\u{e8c}', '\u{e8c}',  IdentifierType::Recommended), ('\u{e8d}', '\u{e8d}',
         IdentifierType::Recommended), ('\u{e8e}', '\u{e93}',  IdentifierType::Recommended),
        ('\u{e94}', '\u{e97}',  IdentifierType::Recommended), ('\u{e98}', '\u{e98}',
         IdentifierType::Recommended), ('\u{e99}', '\u{e9f}',  IdentifierType::Recommended),
        ('\u{ea0}', '\u{ea0}',  IdentifierType::Recommended), ('\u{ea1}', '\u{ea3}',
         IdentifierType::Recommended), ('\u{ea5}', '\u{ea5}',  IdentifierType::Recommended),
        ('\u{ea7}', '\u{ea7}',  IdentifierType::Recommended), ('\u{ea8}', '\u{ea9}',
         IdentifierType::Recommended), ('\u{eaa}', '\u{eab}',  IdentifierType::Recommended),
        ('\u{eac}', '\u{eac}',  IdentifierType::Recommended), ('\u{ead}', '\u{eb2}',
         IdentifierType::Recommended), ('\u{eb3}', '\u{eb3}',  IdentifierType::Not_NFKC),
        ('\u{eb4}', '\u{eb9}',  IdentifierType::Recommended), ('\u{eba}', '\u{eba}',
         IdentifierType::Recommended), ('\u{ebb}', '\u{ebd}',  IdentifierType::Recommended),
        ('\u{ec0}', '\u{ec4}',  IdentifierType::Recommended), ('\u{ec6}', '\u{ec6}',
         IdentifierType::Recommended), ('\u{ec8}', '\u{ecd}',  IdentifierType::Recommended),
        ('\u{ece}', '\u{ece}',  IdentifierType::Recommended), ('\u{ed0}', '\u{ed9}',
         IdentifierType::Recommended), ('\u{edc}', '\u{edd}',  IdentifierType::Not_NFKC),
        ('\u{ede}', '\u{edf}',  IdentifierType::Recommended), ('\u{f00}', '\u{f00}',
         IdentifierType::Recommended), ('\u{f01}', '\u{f0a}',  IdentifierType::Not_XID), ('\u{f0b}',
        '\u{f0b}',  IdentifierType::Inclusion), ('\u{f0c}', '\u{f0c}',  IdentifierType::Not_NFKC),
        ('\u{f0d}', '\u{f17}',  IdentifierType::Not_XID), ('\u{f18}', '\u{f19}',
         IdentifierType::Technical), ('\u{f1a}', '\u{f1f}',  IdentifierType::Not_XID), ('\u{f20}',
        '\u{f29}',  IdentifierType::Recommended), ('\u{f2a}', '\u{f34}',  IdentifierType::Not_XID),
        ('\u{f35}', '\u{f35}',  IdentifierType::Recommended), ('\u{f36}', '\u{f36}',
         IdentifierType::Not_XID), ('\u{f37}', '\u{f37}',  IdentifierType::Recommended), ('\u{f38}',
        '\u{f38}',  IdentifierType::Not_XID), ('\u{f39}', '\u{f39}',  IdentifierType::Uncommon_Use),
        ('\u{f3a}', '\u{f3d}',  IdentifierType::Not_XID), ('\u{f3e}', '\u{f42}',
         IdentifierType::Recommended), ('\u{f43}', '\u{f43}',  IdentifierType::Not_NFKC),
        ('\u{f44}', '\u{f47}',  IdentifierType::Recommended), ('\u{f49}', '\u{f4c}',
         IdentifierType::Recommended), ('\u{f4d}', '\u{f4d}',  IdentifierType::Not_NFKC),
        ('\u{f4e}', '\u{f51}',  IdentifierType::Recommended), ('\u{f52}', '\u{f52}',
         IdentifierType::Not_NFKC), ('\u{f53}', '\u{f56}',  IdentifierType::Recommended),
        ('\u{f57}', '\u{f57}',  IdentifierType::Not_NFKC), ('\u{f58}', '\u{f5b}',
         IdentifierType::Recommended), ('\u{f5c}', '\u{f5c}',  IdentifierType::Not_NFKC),
        ('\u{f5d}', '\u{f68}',  IdentifierType::Recommended), ('\u{f69}', '\u{f69}',
         IdentifierType::Not_NFKC), ('\u{f6a}', '\u{f6a}',  IdentifierType::Recommended),
        ('\u{f6b}', '\u{f6c}',  IdentifierType::Recommended), ('\u{f71}', '\u{f72}',
         IdentifierType::Recommended), ('\u{f73}', '\u{f73}',  IdentifierType::Not_NFKC),
        ('\u{f74}', '\u{f74}',  IdentifierType::Recommended), ('\u{f75}', '\u{f76}',
         IdentifierType::Not_NFKC), ('\u{f77}', '\u{f77}',  IdentifierType::Deprecated), ('\u{f78}',
        '\u{f78}',  IdentifierType::Not_NFKC), ('\u{f79}', '\u{f79}',  IdentifierType::Deprecated),
        ('\u{f7a}', '\u{f80}',  IdentifierType::Recommended), ('\u{f81}', '\u{f81}',
         IdentifierType::Not_NFKC), ('\u{f82}', '\u{f84}',  IdentifierType::Recommended),
        ('\u{f85}', '\u{f85}',  IdentifierType::Not_XID), ('\u{f86}', '\u{f8b}',
         IdentifierType::Recommended), ('\u{f8c}', '\u{f8f}',  IdentifierType::Recommended),
        ('\u{f90}', '\u{f92}',  IdentifierType::Recommended), ('\u{f93}', '\u{f93}',
         IdentifierType::Not_NFKC), ('\u{f94}', '\u{f95}',  IdentifierType::Recommended),
        ('\u{f96}', '\u{f96}',  IdentifierType::Recommended), ('\u{f97}', '\u{f97}',
         IdentifierType::Recommended), ('\u{f99}', '\u{f9c}',  IdentifierType::Recommended),
        ('\u{f9d}', '\u{f9d}',  IdentifierType::Not_NFKC), ('\u{f9e}', '\u{fa1}',
         IdentifierType::Recommended), ('\u{fa2}', '\u{fa2}',  IdentifierType::Not_NFKC),
        ('\u{fa3}', '\u{fa6}',  IdentifierType::Recommended), ('\u{fa7}', '\u{fa7}',
         IdentifierType::Not_NFKC), ('\u{fa8}', '\u{fab}',  IdentifierType::Recommended),
        ('\u{fac}', '\u{fac}',  IdentifierType::Not_NFKC), ('\u{fad}', '\u{fad}',
         IdentifierType::Recommended), ('\u{fae}', '\u{fb0}',  IdentifierType::Recommended),
        ('\u{fb1}', '\u{fb7}',  IdentifierType::Recommended), ('\u{fb8}', '\u{fb8}',
         IdentifierType::Recommended), ('\u{fb9}', '\u{fb9}',  IdentifierType::Not_NFKC),
        ('\u{fba}', '\u{fbc}',  IdentifierType::Recommended), ('\u{fbe}', '\u{fc5}',
         IdentifierType::Not_XID), ('\u{fc6}', '\u{fc6}',  IdentifierType::Recommended), ('\u{fc7}',
        '\u{fcc}',  IdentifierType::Not_XID), ('\u{fce}', '\u{fce}',  IdentifierType::Not_XID),
        ('\u{fcf}', '\u{fcf}',  IdentifierType::Not_XID), ('\u{fd0}', '\u{fd1}',
         IdentifierType::Not_XID), ('\u{fd2}', '\u{fd4}',  IdentifierType::Not_XID), ('\u{fd5}',
        '\u{fd8}',  IdentifierType::Not_XID), ('\u{fd9}', '\u{fda}',  IdentifierType::Not_XID),
        ('\u{1000}', '\u{1021}',  IdentifierType::Recommended), ('\u{1022}', '\u{1022}',
         IdentifierType::Recommended), ('\u{1023}', '\u{1027}',  IdentifierType::Recommended),
        ('\u{1028}', '\u{1028}',  IdentifierType::Recommended), ('\u{1029}', '\u{102a}',
         IdentifierType::Recommended), ('\u{102b}', '\u{102b}',  IdentifierType::Recommended),
        ('\u{102c}', '\u{1032}',  IdentifierType::Recommended), ('\u{1033}', '\u{1035}',
         IdentifierType::Recommended), ('\u{1036}', '\u{1039}',  IdentifierType::Recommended),
        ('\u{103a}', '\u{103f}',  IdentifierType::Recommended), ('\u{1040}', '\u{1049}',
         IdentifierType::Recommended), ('\u{104a}', '\u{104f}',  IdentifierType::Not_XID),
        ('\u{1050}', '\u{1059}',  IdentifierType::Recommended), ('\u{105a}', '\u{1099}',
         IdentifierType::Recommended), ('\u{109a}', '\u{109d}',  IdentifierType::Recommended),
        ('\u{109e}', '\u{109f}',  IdentifierType::Not_XID), ('\u{10a0}', '\u{10c5}',
         IdentifierType::Obsolete), ('\u{10c7}', '\u{10c7}',  IdentifierType::Recommended),
        ('\u{10cd}', '\u{10cd}',  IdentifierType::Recommended), ('\u{10d0}', '\u{10f0}',
         IdentifierType::Recommended), ('\u{10f1}', '\u{10f6}',  IdentifierType::Obsolete),
        ('\u{10f7}', '\u{10f8}',  IdentifierType::Recommended), ('\u{10f9}', '\u{10fa}',
         IdentifierType::Recommended), ('\u{10fb}', '\u{10fb}',  IdentifierType::Not_XID),
        ('\u{10fc}', '\u{10fc}',  IdentifierType::Not_NFKC), ('\u{10fd}', '\u{10ff}',
         IdentifierType::Recommended), ('\u{1100}', '\u{1159}',  IdentifierType::Obsolete),
        ('\u{115a}', '\u{115e}',  IdentifierType::Obsolete), ('\u{115f}', '\u{1160}',
         IdentifierType::Default_Ignorable), ('\u{1161}', '\u{11a2}',  IdentifierType::Obsolete),
        ('\u{11a3}', '\u{11a7}',  IdentifierType::Obsolete), ('\u{11a8}', '\u{11f9}',
         IdentifierType::Obsolete), ('\u{11fa}', '\u{11ff}',  IdentifierType::Obsolete),
        ('\u{1200}', '\u{1206}',  IdentifierType::Recommended), ('\u{1207}', '\u{1207}',
         IdentifierType::Recommended), ('\u{1208}', '\u{1246}',  IdentifierType::Recommended),
        ('\u{1247}', '\u{1247}',  IdentifierType::Recommended), ('\u{1248}', '\u{1248}',
         IdentifierType::Recommended), ('\u{124a}', '\u{124d}',  IdentifierType::Recommended),
        ('\u{1250}', '\u{1256}',  IdentifierType::Recommended), ('\u{1258}', '\u{1258}',
         IdentifierType::Recommended), ('\u{125a}', '\u{125d}',  IdentifierType::Recommended),
        ('\u{1260}', '\u{1286}',  IdentifierType::Recommended), ('\u{1287}', '\u{1287}',
         IdentifierType::Recommended), ('\u{1288}', '\u{1288}',  IdentifierType::Recommended),
        ('\u{128a}', '\u{128d}',  IdentifierType::Recommended), ('\u{1290}', '\u{12ae}',
         IdentifierType::Recommended), ('\u{12af}', '\u{12af}',  IdentifierType::Recommended),
        ('\u{12b0}', '\u{12b0}',  IdentifierType::Recommended), ('\u{12b2}', '\u{12b5}',
         IdentifierType::Recommended), ('\u{12b8}', '\u{12be}',  IdentifierType::Recommended),
        ('\u{12c0}', '\u{12c0}',  IdentifierType::Recommended), ('\u{12c2}', '\u{12c5}',
         IdentifierType::Recommended), ('\u{12c8}', '\u{12ce}',  IdentifierType::Recommended),
        ('\u{12cf}', '\u{12cf}',  IdentifierType::Recommended), ('\u{12d0}', '\u{12d6}',
         IdentifierType::Recommended), ('\u{12d8}', '\u{12ee}',  IdentifierType::Recommended),
        ('\u{12ef}', '\u{12ef}',  IdentifierType::Recommended), ('\u{12f0}', '\u{130e}',
         IdentifierType::Recommended), ('\u{130f}', '\u{130f}',  IdentifierType::Recommended),
        ('\u{1310}', '\u{1310}',  IdentifierType::Recommended), ('\u{1312}', '\u{1315}',
         IdentifierType::Recommended), ('\u{1318}', '\u{131e}',  IdentifierType::Recommended),
        ('\u{131f}', '\u{131f}',  IdentifierType::Recommended), ('\u{1320}', '\u{1346}',
         IdentifierType::Recommended), ('\u{1347}', '\u{1347}',  IdentifierType::Recommended),
        ('\u{1348}', '\u{135a}',  IdentifierType::Recommended), ('\u{135d}', '\u{135e}',
         IdentifierType::Recommended), ('\u{135f}', '\u{135f}',  IdentifierType::Recommended),
        ('\u{1360}', '\u{1360}',  IdentifierType::Not_XID), ('\u{1361}', '\u{1368}',
         IdentifierType::Not_XID), ('\u{1369}', '\u{1371}',  IdentifierType::Obsolete), ('\u{1372}',
        '\u{137c}',  IdentifierType::Not_XID), ('\u{1380}', '\u{138f}',
         IdentifierType::Recommended), ('\u{1390}', '\u{1399}',  IdentifierType::Not_XID),
        ('\u{13a0}', '\u{13f4}',  IdentifierType::Limited_Use), ('\u{13f5}', '\u{13f5}',
         IdentifierType::Limited_Use), ('\u{13f8}', '\u{13fd}',  IdentifierType::Limited_Use),
        ('\u{1400}', '\u{1400}',  IdentifierType::Limited_Use), ('\u{1401}', '\u{166c}',
         IdentifierType::Limited_Use), ('\u{166d}', '\u{166e}',  IdentifierType::Limited_Use),
        ('\u{166f}', '\u{1676}',  IdentifierType::Limited_Use), ('\u{1677}', '\u{167f}',
         IdentifierType::Limited_Use), ('\u{1680}', '\u{1680}',  IdentifierType::Exclusion),
        ('\u{1681}', '\u{169a}',  IdentifierType::Exclusion), ('\u{169b}', '\u{169c}',
         IdentifierType::Exclusion), ('\u{16a0}', '\u{16ea}',  IdentifierType::Exclusion),
        ('\u{16eb}', '\u{16ed}',  IdentifierType::Exclusion), ('\u{16ee}', '\u{16f0}',
         IdentifierType::Exclusion), ('\u{16f1}', '\u{16f8}',  IdentifierType::Exclusion),
        ('\u{1700}', '\u{170c}',  IdentifierType::Exclusion), ('\u{170d}', '\u{170d}',
         IdentifierType::Exclusion), ('\u{170e}', '\u{1714}',  IdentifierType::Exclusion),
        ('\u{1715}', '\u{1715}',  IdentifierType::Exclusion), ('\u{171f}', '\u{171f}',
         IdentifierType::Exclusion), ('\u{1720}', '\u{1734}',  IdentifierType::Exclusion),
        ('\u{1735}', '\u{1736}',  IdentifierType::Exclusion), ('\u{1740}', '\u{1753}',
         IdentifierType::Exclusion), ('\u{1760}', '\u{176c}',  IdentifierType::Exclusion),
        ('\u{176e}', '\u{1770}',  IdentifierType::Exclusion), ('\u{1772}', '\u{1773}',
         IdentifierType::Exclusion), ('\u{1780}', '\u{17a2}',  IdentifierType::Recommended),
        ('\u{17a3}', '\u{17a4}',  IdentifierType::Deprecated), ('\u{17a5}', '\u{17a7}',
         IdentifierType::Recommended), ('\u{17a8}', '\u{17a8}',  IdentifierType::Obsolete),
        ('\u{17a9}', '\u{17b3}',  IdentifierType::Recommended), ('\u{17b4}', '\u{17b5}',
         IdentifierType::Default_Ignorable), ('\u{17b6}', '\u{17cd}',  IdentifierType::Recommended),
        ('\u{17ce}', '\u{17cf}',  IdentifierType::Technical), ('\u{17d0}', '\u{17d0}',
         IdentifierType::Recommended), ('\u{17d1}', '\u{17d1}',  IdentifierType::Technical),
        ('\u{17d2}', '\u{17d2}',  IdentifierType::Recommended), ('\u{17d3}', '\u{17d3}',
         IdentifierType::Obsolete), ('\u{17d4}', '\u{17d6}',  IdentifierType::Not_XID), ('\u{17d7}',
        '\u{17d7}',  IdentifierType::Recommended), ('\u{17d8}', '\u{17d8}',
         IdentifierType::Obsolete), ('\u{17d9}', '\u{17db}',  IdentifierType::Not_XID), ('\u{17dc}',
        '\u{17dc}',  IdentifierType::Recommended), ('\u{17dd}', '\u{17dd}',
         IdentifierType::Technical), ('\u{17e0}', '\u{17e9}',  IdentifierType::Recommended),
        ('\u{17f0}', '\u{17f9}',  IdentifierType::Not_XID), ('\u{1800}', '\u{180a}',
         IdentifierType::Exclusion), ('\u{180b}', '\u{180d}',  IdentifierType::Default_Ignorable),
        ('\u{180e}', '\u{180e}',  IdentifierType::Default_Ignorable), ('\u{180f}', '\u{180f}',
         IdentifierType::Default_Ignorable), ('\u{1810}', '\u{1819}',  IdentifierType::Exclusion),
        ('\u{1820}', '\u{1877}',  IdentifierType::Exclusion), ('\u{1878}', '\u{1878}',
         IdentifierType::Exclusion), ('\u{1880}', '\u{18a8}',  IdentifierType::Exclusion),
        ('\u{18a9}', '\u{18a9}',  IdentifierType::Uncommon_Use), ('\u{18aa}', '\u{18aa}',
         IdentifierType::Exclusion), ('\u{18b0}', '\u{18f5}',  IdentifierType::Limited_Use),
        ('\u{1900}', '\u{191c}',  IdentifierType::Limited_Use), ('\u{191d}', '\u{191e}',
         IdentifierType::Limited_Use), ('\u{1920}', '\u{192b}',  IdentifierType::Limited_Use),
        ('\u{1930}', '\u{193b}',  IdentifierType::Limited_Use), ('\u{1940}', '\u{1940}',
         IdentifierType::Limited_Use), ('\u{1944}', '\u{1945}',  IdentifierType::Limited_Use),
        ('\u{1946}', '\u{196d}',  IdentifierType::Limited_Use), ('\u{1970}', '\u{1974}',
         IdentifierType::Limited_Use), ('\u{1980}', '\u{19a9}',  IdentifierType::Limited_Use),
        ('\u{19aa}', '\u{19ab}',  IdentifierType::Limited_Use), ('\u{19b0}', '\u{19c9}',
         IdentifierType::Limited_Use), ('\u{19d0}', '\u{19d9}',  IdentifierType::Limited_Use),
        ('\u{19da}', '\u{19da}',  IdentifierType::Limited_Use), ('\u{19de}', '\u{19df}',
         IdentifierType::Limited_Use), ('\u{19e0}', '\u{19ff}',  IdentifierType::Not_XID),
        ('\u{1a00}', '\u{1a1b}',  IdentifierType::Exclusion), ('\u{1a1e}', '\u{1a1f}',
         IdentifierType::Exclusion), ('\u{1a20}', '\u{1a5e}',  IdentifierType::Limited_Use),
        ('\u{1a60}', '\u{1a7c}',  IdentifierType::Limited_Use), ('\u{1a7f}', '\u{1a89}',
         IdentifierType::Limited_Use), ('\u{1a90}', '\u{1a99}',  IdentifierType::Limited_Use),
        ('\u{1aa0}', '\u{1aa6}',  IdentifierType::Limited_Use), ('\u{1aa7}', '\u{1aa7}',
         IdentifierType::Limited_Use), ('\u{1aa8}', '\u{1aad}',  IdentifierType::Limited_Use),
        ('\u{1ab0}', '\u{1abd}',  IdentifierType::Obsolete), ('\u{1abe}', '\u{1abe}',
         IdentifierType::Not_XID), ('\u{1abf}', '\u{1ac0}',  IdentifierType::Technical),
        ('\u{1ac1}', '\u{1ace}',  IdentifierType::Uncommon_Use), ('\u{1b00}', '\u{1b4b}',
         IdentifierType::Limited_Use), ('\u{1b4c}', '\u{1b4c}',  IdentifierType::Limited_Use),
        ('\u{1b4e}', '\u{1b4f}',  IdentifierType::Limited_Use), ('\u{1b50}', '\u{1b59}',
         IdentifierType::Limited_Use), ('\u{1b5a}', '\u{1b6a}',  IdentifierType::Limited_Use),
        ('\u{1b6b}', '\u{1b73}',  IdentifierType::Limited_Use), ('\u{1b74}', '\u{1b7c}',
         IdentifierType::Limited_Use), ('\u{1b7d}', '\u{1b7e}',  IdentifierType::Limited_Use),
        ('\u{1b7f}', '\u{1b7f}',  IdentifierType::Limited_Use), ('\u{1b80}', '\u{1baa}',
         IdentifierType::Limited_Use), ('\u{1bab}', '\u{1bad}',  IdentifierType::Limited_Use),
        ('\u{1bae}', '\u{1bb9}',  IdentifierType::Limited_Use), ('\u{1bba}', '\u{1bbf}',
         IdentifierType::Limited_Use), ('\u{1bc0}', '\u{1bf3}',  IdentifierType::Limited_Use),
        ('\u{1bfc}', '\u{1bff}',  IdentifierType::Limited_Use), ('\u{1c00}', '\u{1c37}',
         IdentifierType::Limited_Use), ('\u{1c3b}', '\u{1c3f}',  IdentifierType::Limited_Use),
        ('\u{1c40}', '\u{1c49}',  IdentifierType::Limited_Use), ('\u{1c4d}', '\u{1c7d}',
         IdentifierType::Limited_Use), ('\u{1c7e}', '\u{1c7f}',  IdentifierType::Limited_Use),
        ('\u{1c80}', '\u{1c88}',  IdentifierType::Obsolete), ('\u{1c89}', '\u{1c8a}',
         IdentifierType::Uncommon_Use), ('\u{1c90}', '\u{1cba}',  IdentifierType::Recommended),
        ('\u{1cbd}', '\u{1cbf}',  IdentifierType::Recommended), ('\u{1cc0}', '\u{1cc7}',
         IdentifierType::Limited_Use), ('\u{1cd0}', '\u{1cd2}',  IdentifierType::Obsolete),
        ('\u{1cd3}', '\u{1cd3}',  IdentifierType::Obsolete), ('\u{1cd4}', '\u{1cf2}',
         IdentifierType::Obsolete), ('\u{1cf3}', '\u{1cf6}',  IdentifierType::Obsolete),
        ('\u{1cf7}', '\u{1cf7}',  IdentifierType::Obsolete), ('\u{1cf8}', '\u{1cf9}',
         IdentifierType::Obsolete), ('\u{1cfa}', '\u{1cfa}',  IdentifierType::Exclusion),
        ('\u{1d00}', '\u{1d2b}',  IdentifierType::Technical), ('\u{1d2c}', '\u{1d2e}',
         IdentifierType::Not_NFKC), ('\u{1d2f}', '\u{1d2f}',  IdentifierType::Technical),
        ('\u{1d30}', '\u{1d3a}',  IdentifierType::Not_NFKC), ('\u{1d3b}', '\u{1d3b}',
         IdentifierType::Technical), ('\u{1d3c}', '\u{1d4d}',  IdentifierType::Not_NFKC),
        ('\u{1d4e}', '\u{1d4e}',  IdentifierType::Technical), ('\u{1d4f}', '\u{1d6a}',
         IdentifierType::Not_NFKC), ('\u{1d6b}', '\u{1d6b}',  IdentifierType::Technical),
        ('\u{1d6c}', '\u{1d77}',  IdentifierType::Technical), ('\u{1d78}', '\u{1d78}',
         IdentifierType::Not_NFKC), ('\u{1d79}', '\u{1d9a}',  IdentifierType::Technical),
        ('\u{1d9b}', '\u{1dbf}',  IdentifierType::Not_NFKC), ('\u{1dc0}', '\u{1dc3}',
         IdentifierType::Technical), ('\u{1dc4}', '\u{1dca}',  IdentifierType::Technical),
        ('\u{1dcb}', '\u{1dcd}',  IdentifierType::Technical), ('\u{1dce}', '\u{1dce}',
         IdentifierType::Technical), ('\u{1dcf}', '\u{1dd0}',  IdentifierType::Technical),
        ('\u{1dd1}', '\u{1de6}',  IdentifierType::Technical), ('\u{1de7}', '\u{1df5}',
         IdentifierType::Technical), ('\u{1df6}', '\u{1df9}',  IdentifierType::Technical),
        ('\u{1dfa}', '\u{1dfa}',  IdentifierType::Limited_Use), ('\u{1dfb}', '\u{1dfb}',
         IdentifierType::Technical), ('\u{1dfc}', '\u{1dfc}',  IdentifierType::Technical),
        ('\u{1dfd}', '\u{1dfd}',  IdentifierType::Technical), ('\u{1dfe}', '\u{1dff}',
         IdentifierType::Technical), ('\u{1e00}', '\u{1e99}',  IdentifierType::Recommended),
        ('\u{1e9a}', '\u{1e9a}',  IdentifierType::Not_NFKC), ('\u{1e9b}', '\u{1e9b}',
         IdentifierType::Not_NFKC), ('\u{1e9c}', '\u{1e9d}',  IdentifierType::Technical),
        ('\u{1e9e}', '\u{1e9e}',  IdentifierType::Recommended), ('\u{1e9f}', '\u{1e9f}',
         IdentifierType::Technical), ('\u{1ea0}', '\u{1ef9}',  IdentifierType::Recommended),
        ('\u{1efa}', '\u{1eff}',  IdentifierType::Technical), ('\u{1f00}', '\u{1f15}',
         IdentifierType::Recommended), ('\u{1f18}', '\u{1f1d}',  IdentifierType::Recommended),
        ('\u{1f20}', '\u{1f45}',  IdentifierType::Recommended), ('\u{1f48}', '\u{1f4d}',
         IdentifierType::Recommended), ('\u{1f50}', '\u{1f57}',  IdentifierType::Recommended),
        ('\u{1f59}', '\u{1f59}',  IdentifierType::Recommended), ('\u{1f5b}', '\u{1f5b}',
         IdentifierType::Recommended), ('\u{1f5d}', '\u{1f5d}',  IdentifierType::Recommended),
        ('\u{1f5f}', '\u{1f70}',  IdentifierType::Recommended), ('\u{1f71}', '\u{1f71}',
         IdentifierType::Not_NFKC), ('\u{1f72}', '\u{1f72}',  IdentifierType::Recommended),
        ('\u{1f73}', '\u{1f73}',  IdentifierType::Not_NFKC), ('\u{1f74}', '\u{1f74}',
         IdentifierType::Recommended), ('\u{1f75}', '\u{1f75}',  IdentifierType::Not_NFKC),
        ('\u{1f76}', '\u{1f76}',  IdentifierType::Recommended), ('\u{1f77}', '\u{1f77}',
         IdentifierType::Not_NFKC), ('\u{1f78}', '\u{1f78}',  IdentifierType::Recommended),
        ('\u{1f79}', '\u{1f79}',  IdentifierType::Not_NFKC), ('\u{1f7a}', '\u{1f7a}',
         IdentifierType::Recommended), ('\u{1f7b}', '\u{1f7b}',  IdentifierType::Not_NFKC),
        ('\u{1f7c}', '\u{1f7c}',  IdentifierType::Recommended), ('\u{1f7d}', '\u{1f7d}',
         IdentifierType::Not_NFKC), ('\u{1f80}', '\u{1fb4}',  IdentifierType::Recommended),
        ('\u{1fb6}', '\u{1fba}',  IdentifierType::Recommended), ('\u{1fbb}', '\u{1fbb}',
         IdentifierType::Not_NFKC), ('\u{1fbc}', '\u{1fbc}',  IdentifierType::Recommended),
        ('\u{1fbd}', '\u{1fc1}',  IdentifierType::Not_NFKC), ('\u{1fc2}', '\u{1fc4}',
         IdentifierType::Recommended), ('\u{1fc6}', '\u{1fc8}',  IdentifierType::Recommended),
        ('\u{1fc9}', '\u{1fc9}',  IdentifierType::Not_NFKC), ('\u{1fca}', '\u{1fca}',
         IdentifierType::Recommended), ('\u{1fcb}', '\u{1fcb}',  IdentifierType::Not_NFKC),
        ('\u{1fcc}', '\u{1fcc}',  IdentifierType::Recommended), ('\u{1fcd}', '\u{1fcf}',
         IdentifierType::Not_NFKC), ('\u{1fd0}', '\u{1fd2}',  IdentifierType::Recommended),
        ('\u{1fd3}', '\u{1fd3}',  IdentifierType::Not_NFKC), ('\u{1fd6}', '\u{1fda}',
         IdentifierType::Recommended), ('\u{1fdb}', '\u{1fdb}',  IdentifierType::Not_NFKC),
        ('\u{1fdd}', '\u{1fdf}',  IdentifierType::Not_NFKC), ('\u{1fe0}', '\u{1fe2}',
         IdentifierType::Recommended), ('\u{1fe3}', '\u{1fe3}',  IdentifierType::Not_NFKC),
        ('\u{1fe4}', '\u{1fea}',  IdentifierType::Recommended), ('\u{1feb}', '\u{1feb}',
         IdentifierType::Not_NFKC), ('\u{1fec}', '\u{1fec}',  IdentifierType::Recommended),
        ('\u{1fed}', '\u{1fef}',  IdentifierType::Not_NFKC), ('\u{1ff2}', '\u{1ff4}',
         IdentifierType::Recommended), ('\u{1ff6}', '\u{1ff8}',  IdentifierType::Recommended),
        ('\u{1ff9}', '\u{1ff9}',  IdentifierType::Not_NFKC), ('\u{1ffa}', '\u{1ffa}',
         IdentifierType::Recommended), ('\u{1ffb}', '\u{1ffb}',  IdentifierType::Not_NFKC),
        ('\u{1ffc}', '\u{1ffc}',  IdentifierType::Recommended), ('\u{1ffd}', '\u{1ffe}',
         IdentifierType::Not_NFKC), ('\u{2000}', '\u{200a}',  IdentifierType::Not_NFKC),
        ('\u{200b}', '\u{200f}',  IdentifierType::Default_Ignorable), ('\u{2010}', '\u{2010}',
         IdentifierType::Inclusion), ('\u{2011}', '\u{2011}',  IdentifierType::Not_NFKC),
        ('\u{2012}', '\u{2016}',  IdentifierType::Not_XID), ('\u{2017}', '\u{2017}',
         IdentifierType::Not_NFKC), ('\u{2018}', '\u{2018}',  IdentifierType::Not_XID), ('\u{2019}',
        '\u{2019}',  IdentifierType::Inclusion), ('\u{201a}', '\u{2023}',  IdentifierType::Not_XID),
        ('\u{2024}', '\u{2026}',  IdentifierType::Not_NFKC), ('\u{2027}', '\u{2027}',
         IdentifierType::Inclusion), ('\u{2028}', '\u{2029}',  IdentifierType::Not_XID),
        ('\u{202a}', '\u{202e}',  IdentifierType::Default_Ignorable), ('\u{202f}', '\u{202f}',
         IdentifierType::Not_NFKC), ('\u{2030}', '\u{2032}',  IdentifierType::Not_XID), ('\u{2033}',
        '\u{2034}',  IdentifierType::Not_NFKC), ('\u{2035}', '\u{2035}',  IdentifierType::Not_XID),
        ('\u{2036}', '\u{2037}',  IdentifierType::Not_NFKC), ('\u{2038}', '\u{203b}',
         IdentifierType::Not_XID), ('\u{203c}', '\u{203c}',  IdentifierType::Not_NFKC), ('\u{203d}',
        '\u{203d}',  IdentifierType::Not_XID), ('\u{203e}', '\u{203e}',  IdentifierType::Not_NFKC),
        ('\u{203f}', '\u{2040}',  IdentifierType::Technical), ('\u{2041}', '\u{2046}',
         IdentifierType::Not_XID), ('\u{2047}', '\u{2047}',  IdentifierType::Not_NFKC), ('\u{2048}',
        '\u{2049}',  IdentifierType::Not_NFKC), ('\u{204a}', '\u{204d}',  IdentifierType::Not_XID),
        ('\u{204e}', '\u{2052}',  IdentifierType::Not_XID), ('\u{2053}', '\u{2053}',
         IdentifierType::Not_XID), ('\u{2054}', '\u{2054}',  IdentifierType::Uncommon_Use),
        ('\u{2055}', '\u{2055}',  IdentifierType::Not_XID), ('\u{2056}', '\u{2056}',
         IdentifierType::Obsolete), ('\u{2057}', '\u{2057}',  IdentifierType::Not_NFKC),
        ('\u{2058}', '\u{205e}',  IdentifierType::Obsolete), ('\u{205f}', '\u{205f}',
         IdentifierType::Not_NFKC), ('\u{2060}', '\u{2063}',  IdentifierType::Default_Ignorable),
        ('\u{2064}', '\u{2064}',  IdentifierType::Default_Ignorable), ('\u{2066}', '\u{2069}',
         IdentifierType::Default_Ignorable), ('\u{206a}', '\u{206f}',  IdentifierType::Deprecated),
        ('\u{2070}', '\u{2070}',  IdentifierType::Not_NFKC), ('\u{2071}', '\u{2071}',
         IdentifierType::Not_NFKC), ('\u{2074}', '\u{208e}',  IdentifierType::Not_NFKC),
        ('\u{2090}', '\u{2094}',  IdentifierType::Not_NFKC), ('\u{2095}', '\u{209c}',
         IdentifierType::Not_NFKC), ('\u{20a0}', '\u{20a7}',  IdentifierType::Not_XID), ('\u{20a8}',
        '\u{20a8}',  IdentifierType::Not_NFKC), ('\u{20a9}', '\u{20aa}',  IdentifierType::Not_XID),
        ('\u{20ab}', '\u{20ab}',  IdentifierType::Not_XID), ('\u{20ac}', '\u{20ac}',
         IdentifierType::Not_XID), ('\u{20ad}', '\u{20af}',  IdentifierType::Not_XID), ('\u{20b0}',
        '\u{20b1}',  IdentifierType::Not_XID), ('\u{20b2}', '\u{20b5}',  IdentifierType::Not_XID),
        ('\u{20b6}', '\u{20b8}',  IdentifierType::Not_XID), ('\u{20b9}', '\u{20b9}',
         IdentifierType::Not_XID), ('\u{20ba}', '\u{20ba}',  IdentifierType::Not_XID), ('\u{20bb}',
        '\u{20bd}',  IdentifierType::Not_XID), ('\u{20be}', '\u{20be}',  IdentifierType::Not_XID),
        ('\u{20bf}', '\u{20bf}',  IdentifierType::Not_XID), ('\u{20c0}', '\u{20c0}',
         IdentifierType::Not_XID), ('\u{20d0}', '\u{20dc}',  IdentifierType::Technical),
        ('\u{20dd}', '\u{20e0}',  IdentifierType::Technical), ('\u{20e1}', '\u{20e1}',
         IdentifierType::Technical), ('\u{20e2}', '\u{20e3}',  IdentifierType::Technical),
        ('\u{20e4}', '\u{20e4}',  IdentifierType::Technical), ('\u{20e5}', '\u{20ea}',
         IdentifierType::Technical), ('\u{20eb}', '\u{20eb}',  IdentifierType::Technical),
        ('\u{20ec}', '\u{20ef}',  IdentifierType::Technical), ('\u{20f0}', '\u{20f0}',
         IdentifierType::Technical), ('\u{2100}', '\u{2103}',  IdentifierType::Not_NFKC),
        ('\u{2104}', '\u{2104}',  IdentifierType::Not_XID), ('\u{2105}', '\u{2107}',
         IdentifierType::Not_NFKC), ('\u{2108}', '\u{2108}',  IdentifierType::Not_XID), ('\u{2109}',
        '\u{2113}',  IdentifierType::Not_NFKC), ('\u{2114}', '\u{2114}',  IdentifierType::Not_XID),
        ('\u{2115}', '\u{2116}',  IdentifierType::Not_NFKC), ('\u{2117}', '\u{2117}',
         IdentifierType::Not_XID), ('\u{2118}', '\u{2118}',  IdentifierType::Technical),
        ('\u{2119}', '\u{211d}',  IdentifierType::Not_NFKC), ('\u{211e}', '\u{211f}',
         IdentifierType::Not_XID), ('\u{2120}', '\u{2122}',  IdentifierType::Not_NFKC), ('\u{2123}',
        '\u{2123}',  IdentifierType::Not_XID), ('\u{2124}', '\u{2124}',  IdentifierType::Not_NFKC),
        ('\u{2125}', '\u{2125}',  IdentifierType::Not_XID), ('\u{2126}', '\u{2126}',
         IdentifierType::Not_NFKC), ('\u{2127}', '\u{2127}',  IdentifierType::Obsolete),
        ('\u{2128}', '\u{2128}',  IdentifierType::Not_NFKC), ('\u{2129}', '\u{2129}',
         IdentifierType::Not_XID), ('\u{212a}', '\u{212d}',  IdentifierType::Not_NFKC), ('\u{212e}',
        '\u{212e}',  IdentifierType::Technical), ('\u{212f}', '\u{2131}',
         IdentifierType::Not_NFKC), ('\u{2132}', '\u{2132}',  IdentifierType::Obsolete),
        ('\u{2133}', '\u{2138}',  IdentifierType::Not_NFKC), ('\u{2139}', '\u{2139}',
         IdentifierType::Not_NFKC), ('\u{213a}', '\u{213a}',  IdentifierType::Not_XID), ('\u{213b}',
        '\u{213b}',  IdentifierType::Not_NFKC), ('\u{213c}', '\u{213c}',  IdentifierType::Not_NFKC),
        ('\u{213d}', '\u{2140}',  IdentifierType::Not_NFKC), ('\u{2141}', '\u{2144}',
         IdentifierType::Not_XID), ('\u{2145}', '\u{2149}',  IdentifierType::Not_NFKC), ('\u{214a}',
        '\u{214b}',  IdentifierType::Not_XID), ('\u{214c}', '\u{214c}',  IdentifierType::Not_XID),
        ('\u{214d}', '\u{214d}',  IdentifierType::Not_XID), ('\u{214e}', '\u{214e}',
         IdentifierType::Obsolete), ('\u{214f}', '\u{214f}',  IdentifierType::Obsolete),
        ('\u{2150}', '\u{2152}',  IdentifierType::Not_NFKC), ('\u{2153}', '\u{217f}',
         IdentifierType::Not_NFKC), ('\u{2180}', '\u{2182}',  IdentifierType::Technical),
        ('\u{2183}', '\u{2183}',  IdentifierType::Technical), ('\u{2184}', '\u{2184}',
         IdentifierType::Obsolete), ('\u{2185}', '\u{2188}',  IdentifierType::Obsolete),
        ('\u{2189}', '\u{2189}',  IdentifierType::Not_NFKC), ('\u{218a}', '\u{218b}',
         IdentifierType::Uncommon_Use), ('\u{2190}', '\u{21ea}',  IdentifierType::Not_XID),
        ('\u{21eb}', '\u{21f3}',  IdentifierType::Not_XID), ('\u{21f4}', '\u{21ff}',
         IdentifierType::Not_XID), ('\u{2200}', '\u{222b}',  IdentifierType::Not_XID), ('\u{222c}',
        '\u{222d}',  IdentifierType::Not_NFKC), ('\u{222e}', '\u{222e}',  IdentifierType::Not_XID),
        ('\u{222f}', '\u{2230}',  IdentifierType::Not_NFKC), ('\u{2231}', '\u{22f1}',
         IdentifierType::Not_XID), ('\u{22f2}', '\u{22ff}',  IdentifierType::Not_XID), ('\u{2300}',
        '\u{2300}',  IdentifierType::Not_XID), ('\u{2301}', '\u{2301}',  IdentifierType::Not_XID),
        ('\u{2302}', '\u{2328}',  IdentifierType::Not_XID), ('\u{2329}', '\u{232a}',
         IdentifierType::Deprecated), ('\u{232b}', '\u{237a}',  IdentifierType::Not_XID),
        ('\u{237b}', '\u{237b}',  IdentifierType::Not_XID), ('\u{237c}', '\u{237c}',
         IdentifierType::Not_XID), ('\u{237d}', '\u{239a}',  IdentifierType::Not_XID), ('\u{239b}',
        '\u{23ce}',  IdentifierType::Not_XID), ('\u{23cf}', '\u{23d0}',  IdentifierType::Not_XID),
        ('\u{23d1}', '\u{23db}',  IdentifierType::Not_XID), ('\u{23dc}', '\u{23e7}',
         IdentifierType::Not_XID), ('\u{23e8}', '\u{23e8}',  IdentifierType::Not_XID), ('\u{23e9}',
        '\u{23f3}',  IdentifierType::Not_XID), ('\u{23f4}', '\u{23fa}',  IdentifierType::Not_XID),
        ('\u{23fb}', '\u{23fe}',  IdentifierType::Not_XID), ('\u{23ff}', '\u{23ff}',
         IdentifierType::Not_XID), ('\u{2400}', '\u{2424}',  IdentifierType::Not_XID), ('\u{2425}',
        '\u{2426}',  IdentifierType::Not_XID), ('\u{2427}', '\u{2429}',  IdentifierType::Not_XID),
        ('\u{2440}', '\u{244a}',  IdentifierType::Not_XID), ('\u{2460}', '\u{24ea}',
         IdentifierType::Not_NFKC), ('\u{24eb}', '\u{24fe}',  IdentifierType::Technical),
        ('\u{24ff}', '\u{24ff}',  IdentifierType::Technical), ('\u{2500}', '\u{2595}',
         IdentifierType::Not_XID), ('\u{2596}', '\u{259f}',  IdentifierType::Not_XID), ('\u{25a0}',
        '\u{25ef}',  IdentifierType::Not_XID), ('\u{25f0}', '\u{25f7}',  IdentifierType::Not_XID),
        ('\u{25f8}', '\u{25ff}',  IdentifierType::Not_XID), ('\u{2600}', '\u{2613}',
         IdentifierType::Not_XID), ('\u{2614}', '\u{2615}',  IdentifierType::Not_XID), ('\u{2616}',
        '\u{2617}',  IdentifierType::Not_XID), ('\u{2618}', '\u{2618}',  IdentifierType::Not_XID),
        ('\u{2619}', '\u{2619}',  IdentifierType::Not_XID), ('\u{261a}', '\u{266f}',
         IdentifierType::Not_XID), ('\u{2670}', '\u{2671}',  IdentifierType::Not_XID), ('\u{2672}',
        '\u{267d}',  IdentifierType::Not_XID), ('\u{267e}', '\u{267f}',  IdentifierType::Not_XID),
        ('\u{2680}', '\u{2689}',  IdentifierType::Not_XID), ('\u{268a}', '\u{2691}',
         IdentifierType::Not_XID), ('\u{2692}', '\u{269c}',  IdentifierType::Not_XID), ('\u{269d}',
        '\u{269d}',  IdentifierType::Not_XID), ('\u{269e}', '\u{269f}',  IdentifierType::Not_XID),
        ('\u{26a0}', '\u{26a1}',  IdentifierType::Not_XID), ('\u{26a2}', '\u{26b1}',
         IdentifierType::Not_XID), ('\u{26b2}', '\u{26b2}',  IdentifierType::Not_XID), ('\u{26b3}',
        '\u{26bc}',  IdentifierType::Not_XID), ('\u{26bd}', '\u{26bf}',  IdentifierType::Not_XID),
        ('\u{26c0}', '\u{26c3}',  IdentifierType::Not_XID), ('\u{26c4}', '\u{26cd}',
         IdentifierType::Not_XID), ('\u{26ce}', '\u{26ce}',  IdentifierType::Not_XID), ('\u{26cf}',
        '\u{26e1}',  IdentifierType::Not_XID), ('\u{26e2}', '\u{26e2}',  IdentifierType::Not_XID),
        ('\u{26e3}', '\u{26e3}',  IdentifierType::Not_XID), ('\u{26e4}', '\u{26e7}',
         IdentifierType::Not_XID), ('\u{26e8}', '\u{26ff}',  IdentifierType::Not_XID), ('\u{2700}',
        '\u{2700}',  IdentifierType::Not_XID), ('\u{2701}', '\u{2704}',  IdentifierType::Not_XID),
        ('\u{2705}', '\u{2705}',  IdentifierType::Not_XID), ('\u{2706}', '\u{2709}',
         IdentifierType::Not_XID), ('\u{270a}', '\u{270b}',  IdentifierType::Not_XID), ('\u{270c}',
        '\u{2727}',  IdentifierType::Not_XID), ('\u{2728}', '\u{2728}',  IdentifierType::Not_XID),
        ('\u{2729}', '\u{274b}',  IdentifierType::Not_XID), ('\u{274c}', '\u{274c}',
         IdentifierType::Not_XID), ('\u{274d}', '\u{274d}',  IdentifierType::Not_XID), ('\u{274e}',
        '\u{274e}',  IdentifierType::Not_XID), ('\u{274f}', '\u{2752}',  IdentifierType::Not_XID),
        ('\u{2753}', '\u{2755}',  IdentifierType::Not_XID), ('\u{2756}', '\u{2756}',
         IdentifierType::Not_XID), ('\u{2757}', '\u{2757}',  IdentifierType::Not_XID), ('\u{2758}',
        '\u{275e}',  IdentifierType::Not_XID), ('\u{275f}', '\u{2760}',  IdentifierType::Not_XID),
        ('\u{2761}', '\u{2767}',  IdentifierType::Not_XID), ('\u{2768}', '\u{2775}',
         IdentifierType::Not_XID), ('\u{2776}', '\u{2794}',  IdentifierType::Not_XID), ('\u{2795}',
        '\u{2797}',  IdentifierType::Not_XID), ('\u{2798}', '\u{27af}',  IdentifierType::Not_XID),
        ('\u{27b0}', '\u{27b0}',  IdentifierType::Not_XID), ('\u{27b1}', '\u{27be}',
         IdentifierType::Not_XID), ('\u{27bf}', '\u{27bf}',  IdentifierType::Not_XID), ('\u{27c0}',
        '\u{27c6}',  IdentifierType::Not_XID), ('\u{27c7}', '\u{27ca}',  IdentifierType::Not_XID),
        ('\u{27cb}', '\u{27cb}',  IdentifierType::Not_XID), ('\u{27cc}', '\u{27cc}',
         IdentifierType::Not_XID), ('\u{27cd}', '\u{27cd}',  IdentifierType::Not_XID), ('\u{27ce}',
        '\u{27cf}',  IdentifierType::Not_XID), ('\u{27d0}', '\u{27eb}',  IdentifierType::Not_XID),
        ('\u{27ec}', '\u{27ef}',  IdentifierType::Not_XID), ('\u{27f0}', '\u{27ff}',
         IdentifierType::Not_XID), ('\u{2800}', '\u{28ff}',  IdentifierType::Technical),
        ('\u{2900}', '\u{2a0b}',  IdentifierType::Not_XID), ('\u{2a0c}', '\u{2a0c}',
         IdentifierType::Not_NFKC), ('\u{2a0d}', '\u{2a73}',  IdentifierType::Not_XID), ('\u{2a74}',
        '\u{2a76}',  IdentifierType::Not_NFKC), ('\u{2a77}', '\u{2adb}',  IdentifierType::Not_XID),
        ('\u{2adc}', '\u{2adc}',  IdentifierType::Not_NFKC), ('\u{2add}', '\u{2aff}',
         IdentifierType::Not_XID), ('\u{2b00}', '\u{2b0d}',  IdentifierType::Not_XID), ('\u{2b0e}',
        '\u{2b13}',  IdentifierType::Not_XID), ('\u{2b14}', '\u{2b1a}',  IdentifierType::Not_XID),
        ('\u{2b1b}', '\u{2b1f}',  IdentifierType::Not_XID), ('\u{2b20}', '\u{2b23}',
         IdentifierType::Not_XID), ('\u{2b24}', '\u{2b4c}',  IdentifierType::Not_XID), ('\u{2b4d}',
        '\u{2b4f}',  IdentifierType::Not_XID), ('\u{2b50}', '\u{2b54}',  IdentifierType::Not_XID),
        ('\u{2b55}', '\u{2b59}',  IdentifierType::Not_XID), ('\u{2b5a}', '\u{2b73}',
         IdentifierType::Not_XID), ('\u{2b76}', '\u{2b95}',  IdentifierType::Not_XID), ('\u{2b97}',
        '\u{2b97}',  IdentifierType::Not_XID), ('\u{2b98}', '\u{2bb9}',  IdentifierType::Not_XID),
        ('\u{2bba}', '\u{2bbc}',  IdentifierType::Not_XID), ('\u{2bbd}', '\u{2bc8}',
         IdentifierType::Not_XID), ('\u{2bc9}', '\u{2bc9}',  IdentifierType::Not_XID), ('\u{2bca}',
        '\u{2bd1}',  IdentifierType::Not_XID), ('\u{2bd2}', '\u{2bd2}',  IdentifierType::Not_XID),
        ('\u{2bd3}', '\u{2beb}',  IdentifierType::Not_XID), ('\u{2bec}', '\u{2bef}',
         IdentifierType::Uncommon_Use), ('\u{2bf0}', '\u{2bfe}',  IdentifierType::Not_XID),
        ('\u{2bff}', '\u{2bff}',  IdentifierType::Not_XID), ('\u{2c00}', '\u{2c2e}',
         IdentifierType::Exclusion), ('\u{2c2f}', '\u{2c2f}',  IdentifierType::Exclusion),
        ('\u{2c30}', '\u{2c5e}',  IdentifierType::Exclusion), ('\u{2c5f}', '\u{2c5f}',
         IdentifierType::Exclusion), ('\u{2c60}', '\u{2c67}',  IdentifierType::Technical),
        ('\u{2c68}', '\u{2c6c}',  IdentifierType::Uncommon_Use), ('\u{2c6d}', '\u{2c6f}',
         IdentifierType::Obsolete), ('\u{2c70}', '\u{2c70}',  IdentifierType::Obsolete),
        ('\u{2c71}', '\u{2c73}',  IdentifierType::Obsolete), ('\u{2c74}', '\u{2c76}',
         IdentifierType::Obsolete), ('\u{2c77}', '\u{2c77}',  IdentifierType::Technical),
        ('\u{2c78}', '\u{2c7b}',  IdentifierType::Technical), ('\u{2c7c}', '\u{2c7d}',
         IdentifierType::Not_NFKC), ('\u{2c7e}', '\u{2c7f}',  IdentifierType::Obsolete),
        ('\u{2c80}', '\u{2ce4}',  IdentifierType::Exclusion), ('\u{2ce5}', '\u{2cea}',
         IdentifierType::Exclusion), ('\u{2ceb}', '\u{2cef}',  IdentifierType::Exclusion),
        ('\u{2cf0}', '\u{2cf1}',  IdentifierType::Technical), ('\u{2cf2}', '\u{2cf3}',
         IdentifierType::Exclusion), ('\u{2cf9}', '\u{2cff}',  IdentifierType::Exclusion),
        ('\u{2d00}', '\u{2d25}',  IdentifierType::Obsolete), ('\u{2d27}', '\u{2d27}',
         IdentifierType::Recommended), ('\u{2d2d}', '\u{2d2d}',  IdentifierType::Recommended),
        ('\u{2d30}', '\u{2d65}',  IdentifierType::Limited_Use), ('\u{2d66}', '\u{2d67}',
         IdentifierType::Limited_Use), ('\u{2d6f}', '\u{2d6f}',  IdentifierType::Not_NFKC),
        ('\u{2d70}', '\u{2d70}',  IdentifierType::Limited_Use), ('\u{2d7f}', '\u{2d7f}',
         IdentifierType::Limited_Use), ('\u{2d80}', '\u{2d96}',  IdentifierType::Recommended),
        ('\u{2da0}', '\u{2da6}',  IdentifierType::Recommended), ('\u{2da8}', '\u{2dae}',
         IdentifierType::Recommended), ('\u{2db0}', '\u{2db6}',  IdentifierType::Recommended),
        ('\u{2db8}', '\u{2dbe}',  IdentifierType::Recommended), ('\u{2dc0}', '\u{2dc6}',
         IdentifierType::Recommended), ('\u{2dc8}', '\u{2dce}',  IdentifierType::Recommended),
        ('\u{2dd0}', '\u{2dd6}',  IdentifierType::Recommended), ('\u{2dd8}', '\u{2dde}',
         IdentifierType::Recommended), ('\u{2de0}', '\u{2dff}',  IdentifierType::Obsolete),
        ('\u{2e00}', '\u{2e0d}',  IdentifierType::Technical), ('\u{2e0e}', '\u{2e16}',
         IdentifierType::Obsolete), ('\u{2e17}', '\u{2e17}',  IdentifierType::Not_XID), ('\u{2e18}',
        '\u{2e1b}',  IdentifierType::Not_XID), ('\u{2e1c}', '\u{2e1d}',  IdentifierType::Not_XID),
        ('\u{2e1e}', '\u{2e29}',  IdentifierType::Not_XID), ('\u{2e2a}', '\u{2e2f}',
         IdentifierType::Obsolete), ('\u{2e30}', '\u{2e30}',  IdentifierType::Exclusion),
        ('\u{2e31}', '\u{2e31}',  IdentifierType::Obsolete), ('\u{2e32}', '\u{2e32}',
         IdentifierType::Obsolete), ('\u{2e33}', '\u{2e34}',  IdentifierType::Not_XID), ('\u{2e35}',
        '\u{2e35}',  IdentifierType::Obsolete), ('\u{2e36}', '\u{2e38}',  IdentifierType::Not_XID),
        ('\u{2e39}', '\u{2e39}',  IdentifierType::Obsolete), ('\u{2e3a}', '\u{2e3b}',
         IdentifierType::Not_XID), ('\u{2e3c}', '\u{2e3c}',  IdentifierType::Exclusion),
        ('\u{2e3d}', '\u{2e42}',  IdentifierType::Not_XID), ('\u{2e43}', '\u{2e44}',
         IdentifierType::Not_XID), ('\u{2e45}', '\u{2e49}',  IdentifierType::Not_XID), ('\u{2e4a}',
        '\u{2e4e}',  IdentifierType::Not_XID), ('\u{2e4f}', '\u{2e4f}',  IdentifierType::Not_XID),
        ('\u{2e50}', '\u{2e52}',  IdentifierType::Not_XID), ('\u{2e53}', '\u{2e5d}',
         IdentifierType::Not_XID), ('\u{2e80}', '\u{2e99}',  IdentifierType::Not_XID), ('\u{2e9b}',
        '\u{2e9e}',  IdentifierType::Not_XID), ('\u{2e9f}', '\u{2e9f}',  IdentifierType::Not_NFKC),
        ('\u{2ea0}', '\u{2ef2}',  IdentifierType::Not_XID), ('\u{2ef3}', '\u{2ef3}',
         IdentifierType::Not_NFKC), ('\u{2f00}', '\u{2fd5}',  IdentifierType::Not_NFKC),
        ('\u{2ff0}', '\u{2ffb}',  IdentifierType::Not_XID), ('\u{2ffc}', '\u{2fff}',
         IdentifierType::Not_XID), ('\u{3000}', '\u{3000}',  IdentifierType::Not_NFKC), ('\u{3001}',
        '\u{3004}',  IdentifierType::Not_XID), ('\u{3005}', '\u{3007}',
         IdentifierType::Recommended), ('\u{3008}', '\u{301d}',  IdentifierType::Not_XID),
        ('\u{301e}', '\u{301e}',  IdentifierType::Obsolete), ('\u{301f}', '\u{3020}',
         IdentifierType::Not_XID), ('\u{3021}', '\u{302d}',  IdentifierType::Technical),
        ('\u{302e}', '\u{302f}',  IdentifierType::Technical), ('\u{3030}', '\u{3030}',
         IdentifierType::Not_XID), ('\u{3031}', '\u{3035}',  IdentifierType::Technical),
        ('\u{3036}', '\u{3036}',  IdentifierType::Not_NFKC), ('\u{3037}', '\u{3037}',
         IdentifierType::Not_XID), ('\u{3038}', '\u{303a}',  IdentifierType::Not_NFKC), ('\u{303b}',
        '\u{303c}',  IdentifierType::Technical), ('\u{303d}', '\u{303d}',  IdentifierType::Not_XID),
        ('\u{303e}', '\u{303e}',  IdentifierType::Not_XID), ('\u{303f}', '\u{303f}',
         IdentifierType::Not_XID), ('\u{3041}', '\u{3094}',  IdentifierType::Recommended),
        ('\u{3095}', '\u{3096}',  IdentifierType::Recommended), ('\u{3099}', '\u{309a}',
         IdentifierType::Recommended), ('\u{309b}', '\u{309c}',  IdentifierType::Not_NFKC),
        ('\u{309d}', '\u{309e}',  IdentifierType::Recommended), ('\u{309f}', '\u{309f}',
         IdentifierType::Not_NFKC), ('\u{30a0}', '\u{30a0}',  IdentifierType::Inclusion),
        ('\u{30a1}', '\u{30fa}',  IdentifierType::Recommended), ('\u{30fb}', '\u{30fb}',
         IdentifierType::Inclusion), ('\u{30fc}', '\u{30fe}',  IdentifierType::Recommended),
        ('\u{30ff}', '\u{30ff}',  IdentifierType::Not_NFKC), ('\u{3105}', '\u{312c}',
         IdentifierType::Recommended), ('\u{312d}', '\u{312d}',  IdentifierType::Recommended),
        ('\u{312e}', '\u{312e}',  IdentifierType::Obsolete), ('\u{312f}', '\u{312f}',
         IdentifierType::Recommended), ('\u{3131}', '\u{3163}',  IdentifierType::Not_NFKC),
        ('\u{3164}', '\u{3164}',  IdentifierType::Default_Ignorable), ('\u{3165}', '\u{318e}',
         IdentifierType::Not_NFKC), ('\u{3190}', '\u{3191}',  IdentifierType::Not_XID), ('\u{3192}',
        '\u{319f}',  IdentifierType::Not_NFKC), ('\u{31a0}', '\u{31b7}',
         IdentifierType::Recommended), ('\u{31b8}', '\u{31ba}',  IdentifierType::Recommended),
        ('\u{31bb}', '\u{31bf}',  IdentifierType::Recommended), ('\u{31c0}', '\u{31cf}',
         IdentifierType::Not_XID), ('\u{31d0}', '\u{31e3}',  IdentifierType::Not_XID), ('\u{31e4}',
        '\u{31e5}',  IdentifierType::Not_XID), ('\u{31ef}', '\u{31ef}',  IdentifierType::Not_XID),
        ('\u{31f0}', '\u{31ff}',  IdentifierType::Obsolete), ('\u{3200}', '\u{321c}',
         IdentifierType::Not_NFKC), ('\u{321d}', '\u{321e}',  IdentifierType::Not_NFKC),
        ('\u{3220}', '\u{3243}',  IdentifierType::Not_NFKC), ('\u{3244}', '\u{3247}',
         IdentifierType::Not_NFKC), ('\u{3248}', '\u{324f}',  IdentifierType::Not_XID), ('\u{3250}',
        '\u{3250}',  IdentifierType::Not_NFKC), ('\u{3251}', '\u{325f}',  IdentifierType::Not_NFKC),
        ('\u{3260}', '\u{327b}',  IdentifierType::Not_NFKC), ('\u{327c}', '\u{327d}',
         IdentifierType::Not_NFKC), ('\u{327e}', '\u{327e}',  IdentifierType::Not_NFKC),
        ('\u{327f}', '\u{327f}',  IdentifierType::Technical), ('\u{3280}', '\u{32b0}',
         IdentifierType::Not_NFKC), ('\u{32b1}', '\u{32bf}',  IdentifierType::Not_NFKC),
        ('\u{32c0}', '\u{32cb}',  IdentifierType::Not_NFKC), ('\u{32cc}', '\u{32cf}',
         IdentifierType::Not_NFKC), ('\u{32d0}', '\u{32fe}',  IdentifierType::Not_NFKC),
        ('\u{32ff}', '\u{32ff}',  IdentifierType::Not_NFKC), ('\u{3300}', '\u{3376}',
         IdentifierType::Not_NFKC), ('\u{3377}', '\u{337a}',  IdentifierType::Not_NFKC),
        ('\u{337b}', '\u{33dd}',  IdentifierType::Not_NFKC), ('\u{33de}', '\u{33df}',
         IdentifierType::Not_NFKC), ('\u{33e0}', '\u{33fe}',  IdentifierType::Not_NFKC),
        ('\u{33ff}', '\u{33ff}',  IdentifierType::Not_NFKC), ('\u{3400}', '\u{4db5}',
         IdentifierType::Recommended), ('\u{4db6}', '\u{4dbf}',  IdentifierType::Recommended),
        ('\u{4dc0}', '\u{4dff}',  IdentifierType::Technical), ('\u{4e00}', '\u{9fa5}',
         IdentifierType::Recommended), ('\u{9fa6}', '\u{9fbb}',  IdentifierType::Recommended),
        ('\u{9fbc}', '\u{9fc3}',  IdentifierType::Recommended), ('\u{9fc4}', '\u{9fcb}',
         IdentifierType::Recommended), ('\u{9fcc}', '\u{9fcc}',  IdentifierType::Recommended),
        ('\u{9fcd}', '\u{9fd5}',  IdentifierType::Recommended), ('\u{9fd6}', '\u{9fea}',
         IdentifierType::Recommended), ('\u{9feb}', '\u{9fef}',  IdentifierType::Recommended),
        ('\u{9ff0}', '\u{9ffc}',  IdentifierType::Recommended), ('\u{9ffd}', '\u{9fff}',
         IdentifierType::Recommended), ('\u{a000}', '\u{a48c}',  IdentifierType::Limited_Use),
        ('\u{a490}', '\u{a4a1}',  IdentifierType::Limited_Use), ('\u{a4a2}', '\u{a4a3}',
         IdentifierType::Limited_Use), ('\u{a4a4}', '\u{a4b3}',  IdentifierType::Limited_Use),
        ('\u{a4b4}', '\u{a4b4}',  IdentifierType::Limited_Use), ('\u{a4b5}', '\u{a4c0}',
         IdentifierType::Limited_Use), ('\u{a4c1}', '\u{a4c1}',  IdentifierType::Limited_Use),
        ('\u{a4c2}', '\u{a4c4}',  IdentifierType::Limited_Use), ('\u{a4c5}', '\u{a4c5}',
         IdentifierType::Limited_Use), ('\u{a4c6}', '\u{a4c6}',  IdentifierType::Limited_Use),
        ('\u{a4d0}', '\u{a4fd}',  IdentifierType::Limited_Use), ('\u{a4fe}', '\u{a4ff}',
         IdentifierType::Limited_Use), ('\u{a500}', '\u{a60c}',  IdentifierType::Limited_Use),
        ('\u{a60d}', '\u{a60f}',  IdentifierType::Limited_Use), ('\u{a610}', '\u{a612}',
         IdentifierType::Limited_Use), ('\u{a613}', '\u{a629}',  IdentifierType::Limited_Use),
        ('\u{a62a}', '\u{a62b}',  IdentifierType::Limited_Use), ('\u{a640}', '\u{a65f}',
         IdentifierType::Obsolete), ('\u{a660}', '\u{a661}',  IdentifierType::Obsolete),
        ('\u{a662}', '\u{a66e}',  IdentifierType::Obsolete), ('\u{a66f}', '\u{a66f}',
         IdentifierType::Uncommon_Use), ('\u{a670}', '\u{a673}',  IdentifierType::Obsolete),
        ('\u{a674}', '\u{a67b}',  IdentifierType::Obsolete), ('\u{a67c}', '\u{a67d}',
         IdentifierType::Uncommon_Use), ('\u{a67e}', '\u{a67e}',  IdentifierType::Not_XID),
        ('\u{a67f}', '\u{a67f}',  IdentifierType::Recommended), ('\u{a680}', '\u{a697}',
         IdentifierType::Obsolete), ('\u{a698}', '\u{a69b}',  IdentifierType::Obsolete),
        ('\u{a69c}', '\u{a69d}',  IdentifierType::Not_NFKC), ('\u{a69e}', '\u{a69e}',
         IdentifierType::Uncommon_Use), ('\u{a69f}', '\u{a69f}',  IdentifierType::Obsolete),
        ('\u{a6a0}', '\u{a6f1}',  IdentifierType::Limited_Use), ('\u{a6f2}', '\u{a6f7}',
         IdentifierType::Limited_Use), ('\u{a700}', '\u{a707}',  IdentifierType::Obsolete),
        ('\u{a708}', '\u{a716}',  IdentifierType::Technical), ('\u{a717}', '\u{a71a}',
         IdentifierType::Recommended), ('\u{a71b}', '\u{a71f}',  IdentifierType::Recommended),
        ('\u{a720}', '\u{a721}',  IdentifierType::Not_XID), ('\u{a722}', '\u{a72f}',
         IdentifierType::Technical), ('\u{a730}', '\u{a76f}',  IdentifierType::Obsolete),
        ('\u{a770}', '\u{a770}',  IdentifierType::Not_NFKC), ('\u{a771}', '\u{a787}',
         IdentifierType::Obsolete), ('\u{a788}', '\u{a788}',  IdentifierType::Recommended),
        ('\u{a789}', '\u{a78a}',  IdentifierType::Not_XID), ('\u{a78b}', '\u{a78c}',
         IdentifierType::Uncommon_Use), ('\u{a78d}', '\u{a78d}',  IdentifierType::Recommended),
        ('\u{a78e}', '\u{a78e}',  IdentifierType::Technical), ('\u{a78f}', '\u{a78f}',
         IdentifierType::Uncommon_Use), ('\u{a790}', '\u{a791}',  IdentifierType::Obsolete),
        ('\u{a792}', '\u{a793}',  IdentifierType::Recommended), ('\u{a794}', '\u{a79f}',
         IdentifierType::Obsolete), ('\u{a7a0}', '\u{a7a9}',  IdentifierType::Obsolete),
        ('\u{a7aa}', '\u{a7aa}',  IdentifierType::Recommended), ('\u{a7ab}', '\u{a7ad}',
         IdentifierType::Obsolete), ('\u{a7ae}', '\u{a7ae}',  IdentifierType::Technical),
        ('\u{a7af}', '\u{a7af}',  IdentifierType::Technical), ('\u{a7b0}', '\u{a7b1}',
         IdentifierType::Obsolete), ('\u{a7b2}', '\u{a7b7}',  IdentifierType::Uncommon_Use),
        ('\u{a7b8}', '\u{a7b9}',  IdentifierType::Uncommon_Use), ('\u{a7ba}', '\u{a7bf}',
         IdentifierType::Technical), ('\u{a7c0}', '\u{a7c1}',  IdentifierType::Recommended),
        ('\u{a7c2}', '\u{a7c6}',  IdentifierType::Recommended), ('\u{a7c7}', '\u{a7ca}',
         IdentifierType::Recommended), ('\u{a7cb}', '\u{a7cd}',  IdentifierType::Uncommon_Use),
        ('\u{a7d0}', '\u{a7d1}',  IdentifierType::Recommended), ('\u{a7d3}', '\u{a7d3}',
         IdentifierType::Recommended), ('\u{a7d5}', '\u{a7d9}',  IdentifierType::Recommended),
        ('\u{a7da}', '\u{a7dc}',  IdentifierType::Uncommon_Use), ('\u{a7f2}', '\u{a7f4}',
         IdentifierType::Not_NFKC), ('\u{a7f5}', '\u{a7f6}',  IdentifierType::Obsolete),
        ('\u{a7f7}', '\u{a7f7}',  IdentifierType::Obsolete), ('\u{a7f8}', '\u{a7f9}',
         IdentifierType::Not_NFKC), ('\u{a7fa}', '\u{a7fa}',  IdentifierType::Technical),
        ('\u{a7fb}', '\u{a7ff}',  IdentifierType::Obsolete), ('\u{a800}', '\u{a827}',
         IdentifierType::Limited_Use), ('\u{a828}', '\u{a82b}',  IdentifierType::Limited_Use),
        ('\u{a82c}', '\u{a82c}',  IdentifierType::Limited_Use), ('\u{a830}', '\u{a839}',
         IdentifierType::Not_XID), ('\u{a840}', '\u{a873}',  IdentifierType::Exclusion),
        ('\u{a874}', '\u{a877}',  IdentifierType::Exclusion), ('\u{a880}', '\u{a8c4}',
         IdentifierType::Limited_Use), ('\u{a8c5}', '\u{a8c5}',  IdentifierType::Limited_Use),
        ('\u{a8ce}', '\u{a8cf}',  IdentifierType::Limited_Use), ('\u{a8d0}', '\u{a8d9}',
         IdentifierType::Limited_Use), ('\u{a8e0}', '\u{a8f7}',  IdentifierType::Obsolete),
        ('\u{a8f8}', '\u{a8fa}',  IdentifierType::Obsolete), ('\u{a8fb}', '\u{a8fb}',
         IdentifierType::Obsolete), ('\u{a8fc}', '\u{a8fc}',  IdentifierType::Uncommon_Use),
        ('\u{a8fd}', '\u{a8fd}',  IdentifierType::Uncommon_Use), ('\u{a8fe}', '\u{a8ff}',
         IdentifierType::Obsolete), ('\u{a900}', '\u{a92d}',  IdentifierType::Limited_Use),
        ('\u{a92e}', '\u{a92e}',  IdentifierType::Not_XID), ('\u{a92f}', '\u{a92f}',
         IdentifierType::Limited_Use), ('\u{a930}', '\u{a953}',  IdentifierType::Exclusion),
        ('\u{a95f}', '\u{a95f}',  IdentifierType::Exclusion), ('\u{a960}', '\u{a97c}',
         IdentifierType::Obsolete), ('\u{a980}', '\u{a9c0}',  IdentifierType::Limited_Use),
        ('\u{a9c1}', '\u{a9cd}',  IdentifierType::Limited_Use), ('\u{a9cf}', '\u{a9cf}',
         IdentifierType::Limited_Use), ('\u{a9d0}', '\u{a9d9}',  IdentifierType::Limited_Use),
        ('\u{a9de}', '\u{a9df}',  IdentifierType::Limited_Use), ('\u{a9e0}', '\u{a9e6}',
         IdentifierType::Obsolete), ('\u{a9e7}', '\u{a9fe}',  IdentifierType::Recommended),
        ('\u{aa00}', '\u{aa36}',  IdentifierType::Limited_Use), ('\u{aa40}', '\u{aa4d}',
         IdentifierType::Limited_Use), ('\u{aa50}', '\u{aa59}',  IdentifierType::Limited_Use),
        ('\u{aa5c}', '\u{aa5f}',  IdentifierType::Limited_Use), ('\u{aa60}', '\u{aa76}',
         IdentifierType::Recommended), ('\u{aa77}', '\u{aa79}',  IdentifierType::Not_XID),
        ('\u{aa7a}', '\u{aa7b}',  IdentifierType::Recommended), ('\u{aa7c}', '\u{aa7f}',
         IdentifierType::Recommended), ('\u{aa80}', '\u{aac2}',  IdentifierType::Limited_Use),
        ('\u{aadb}', '\u{aadd}',  IdentifierType::Limited_Use), ('\u{aade}', '\u{aadf}',
         IdentifierType::Limited_Use), ('\u{aae0}', '\u{aaef}',  IdentifierType::Limited_Use),
        ('\u{aaf0}', '\u{aaf1}',  IdentifierType::Limited_Use), ('\u{aaf2}', '\u{aaf6}',
         IdentifierType::Limited_Use), ('\u{ab01}', '\u{ab06}',  IdentifierType::Recommended),
        ('\u{ab09}', '\u{ab0e}',  IdentifierType::Recommended), ('\u{ab11}', '\u{ab16}',
         IdentifierType::Recommended), ('\u{ab20}', '\u{ab26}',  IdentifierType::Recommended),
        ('\u{ab28}', '\u{ab2e}',  IdentifierType::Recommended), ('\u{ab30}', '\u{ab5a}',
         IdentifierType::Obsolete), ('\u{ab5b}', '\u{ab5b}',  IdentifierType::Not_XID), ('\u{ab5c}',
        '\u{ab5f}',  IdentifierType::Not_NFKC), ('\u{ab60}', '\u{ab63}',
         IdentifierType::Uncommon_Use), ('\u{ab64}', '\u{ab65}',  IdentifierType::Obsolete),
        ('\u{ab66}', '\u{ab67}',  IdentifierType::Recommended), ('\u{ab68}', '\u{ab68}',
         IdentifierType::Technical), ('\u{ab69}', '\u{ab69}',  IdentifierType::Not_NFKC),
        ('\u{ab6a}', '\u{ab6b}',  IdentifierType::Not_XID), ('\u{ab70}', '\u{abbf}',
         IdentifierType::Limited_Use), ('\u{abc0}', '\u{abea}',  IdentifierType::Limited_Use),
        ('\u{abeb}', '\u{abeb}',  IdentifierType::Limited_Use), ('\u{abec}', '\u{abed}',
         IdentifierType::Limited_Use), ('\u{abf0}', '\u{abf9}',  IdentifierType::Limited_Use),
        ('\u{ac00}', '\u{d7a3}',  IdentifierType::Recommended), ('\u{d7b0}', '\u{d7c6}',
         IdentifierType::Obsolete), ('\u{d7cb}', '\u{d7fb}',  IdentifierType::Obsolete),
        ('\u{f900}', '\u{fa0d}',  IdentifierType::Not_NFKC), ('\u{fa0e}', '\u{fa0f}',
         IdentifierType::Recommended), ('\u{fa10}', '\u{fa10}',  IdentifierType::Not_NFKC),
        ('\u{fa11}', '\u{fa11}',  IdentifierType::Recommended), ('\u{fa12}', '\u{fa12}',
         IdentifierType::Not_NFKC), ('\u{fa13}', '\u{fa14}',  IdentifierType::Recommended),
        ('\u{fa15}', '\u{fa1e}',  IdentifierType::Not_NFKC), ('\u{fa1f}', '\u{fa1f}',
         IdentifierType::Recommended), ('\u{fa20}', '\u{fa20}',  IdentifierType::Not_NFKC),
        ('\u{fa21}', '\u{fa21}',  IdentifierType::Recommended), ('\u{fa22}', '\u{fa22}',
         IdentifierType::Not_NFKC), ('\u{fa23}', '\u{fa24}',  IdentifierType::Recommended),
        ('\u{fa25}', '\u{fa26}',  IdentifierType::Not_NFKC), ('\u{fa27}', '\u{fa29}',
         IdentifierType::Recommended), ('\u{fa2a}', '\u{fa2d}',  IdentifierType::Not_NFKC),
        ('\u{fa2e}', '\u{fa2f}',  IdentifierType::Not_NFKC), ('\u{fa30}', '\u{fa6a}',
         IdentifierType::Not_NFKC), ('\u{fa6b}', '\u{fa6d}',  IdentifierType::Not_NFKC),
        ('\u{fa70}', '\u{fad9}',  IdentifierType::Not_NFKC), ('\u{fb00}', '\u{fb06}',
         IdentifierType::Not_NFKC), ('\u{fb13}', '\u{fb17}',  IdentifierType::Not_NFKC),
        ('\u{fb1d}', '\u{fb1d}',  IdentifierType::Not_NFKC), ('\u{fb1e}', '\u{fb1e}',
         IdentifierType::Uncommon_Use), ('\u{fb1f}', '\u{fb36}',  IdentifierType::Not_NFKC),
        ('\u{fb38}', '\u{fb3c}',  IdentifierType::Not_NFKC), ('\u{fb3e}', '\u{fb3e}',
         IdentifierType::Not_NFKC), ('\u{fb40}', '\u{fb41}',  IdentifierType::Not_NFKC),
        ('\u{fb43}', '\u{fb44}',  IdentifierType::Not_NFKC), ('\u{fb46}', '\u{fbb1}',
         IdentifierType::Not_NFKC), ('\u{fbb2}', '\u{fbc1}',  IdentifierType::Technical),
        ('\u{fbc2}', '\u{fbc2}',  IdentifierType::Technical), ('\u{fbd3}', '\u{fd3d}',
         IdentifierType::Not_NFKC), ('\u{fd3e}', '\u{fd3f}',  IdentifierType::Technical),
        ('\u{fd40}', '\u{fd4f}',  IdentifierType::Technical), ('\u{fd50}', '\u{fd8f}',
         IdentifierType::Not_NFKC), ('\u{fd92}', '\u{fdc7}',  IdentifierType::Not_NFKC),
        ('\u{fdcf}', '\u{fdcf}',  IdentifierType::Technical), ('\u{fdf0}', '\u{fdfb}',
         IdentifierType::Not_NFKC), ('\u{fdfc}', '\u{fdfc}',  IdentifierType::Not_NFKC),
        ('\u{fdfd}', '\u{fdfd}',  IdentifierType::Technical), ('\u{fdfe}', '\u{fdff}',
         IdentifierType::Technical), ('\u{fe00}', '\u{fe0f}',  IdentifierType::Default_Ignorable),
        ('\u{fe10}', '\u{fe19}',  IdentifierType::Not_NFKC), ('\u{fe20}', '\u{fe23}',
         IdentifierType::Technical), ('\u{fe24}', '\u{fe26}',  IdentifierType::Technical),
        ('\u{fe27}', '\u{fe2d}',  IdentifierType::Technical), ('\u{fe2e}', '\u{fe2f}',
         IdentifierType::Uncommon_Use), ('\u{fe30}', '\u{fe44}',  IdentifierType::Not_NFKC),
        ('\u{fe45}', '\u{fe46}',  IdentifierType::Technical), ('\u{fe47}', '\u{fe48}',
         IdentifierType::Not_NFKC), ('\u{fe49}', '\u{fe52}',  IdentifierType::Not_NFKC),
        ('\u{fe54}', '\u{fe66}',  IdentifierType::Not_NFKC), ('\u{fe68}', '\u{fe6b}',
         IdentifierType::Not_NFKC), ('\u{fe70}', '\u{fe72}',  IdentifierType::Not_NFKC),
        ('\u{fe73}', '\u{fe73}',  IdentifierType::Technical), ('\u{fe74}', '\u{fe74}',
         IdentifierType::Not_NFKC), ('\u{fe76}', '\u{fefc}',  IdentifierType::Not_NFKC),
        ('\u{feff}', '\u{feff}',  IdentifierType::Default_Ignorable), ('\u{ff01}', '\u{ff5e}',
         IdentifierType::Not_NFKC), ('\u{ff5f}', '\u{ff60}',  IdentifierType::Not_NFKC),
        ('\u{ff61}', '\u{ff9f}',  IdentifierType::Not_NFKC), ('\u{ffa0}', '\u{ffa0}',
         IdentifierType::Default_Ignorable), ('\u{ffa1}', '\u{ffbe}',  IdentifierType::Not_NFKC),
        ('\u{ffc2}', '\u{ffc7}',  IdentifierType::Not_NFKC), ('\u{ffca}', '\u{ffcf}',
         IdentifierType::Not_NFKC), ('\u{ffd2}', '\u{ffd7}',  IdentifierType::Not_NFKC),
        ('\u{ffda}', '\u{ffdc}',  IdentifierType::Not_NFKC), ('\u{ffe0}', '\u{ffe6}',
         IdentifierType::Not_NFKC), ('\u{ffe8}', '\u{ffee}',  IdentifierType::Not_NFKC),
        ('\u{fff9}', '\u{fffb}',  IdentifierType::Not_XID), ('\u{fffc}', '\u{fffc}',
         IdentifierType::Not_XID), ('\u{fffd}', '\u{fffd}',  IdentifierType::Not_XID), ('\u{10000}',
        '\u{1000b}',  IdentifierType::Exclusion), ('\u{1000d}', '\u{10026}',
         IdentifierType::Exclusion), ('\u{10028}', '\u{1003a}',  IdentifierType::Exclusion),
        ('\u{1003c}', '\u{1003d}',  IdentifierType::Exclusion), ('\u{1003f}', '\u{1004d}',
         IdentifierType::Exclusion), ('\u{10050}', '\u{1005d}',  IdentifierType::Exclusion),
        ('\u{10080}', '\u{100fa}',  IdentifierType::Exclusion), ('\u{10100}', '\u{10102}',
         IdentifierType::Exclusion), ('\u{10107}', '\u{10133}',  IdentifierType::Exclusion),
        ('\u{10137}', '\u{1013f}',  IdentifierType::Exclusion), ('\u{10140}', '\u{10174}',
         IdentifierType::Obsolete), ('\u{10175}', '\u{1018a}',  IdentifierType::Not_XID),
        ('\u{1018b}', '\u{1018c}',  IdentifierType::Not_XID), ('\u{1018d}', '\u{1018e}',
         IdentifierType::Not_XID), ('\u{10190}', '\u{1019b}',  IdentifierType::Not_XID),
        ('\u{1019c}', '\u{1019c}',  IdentifierType::Not_XID), ('\u{101a0}', '\u{101a0}',
         IdentifierType::Not_XID), ('\u{101d0}', '\u{101fc}',  IdentifierType::Obsolete),
        ('\u{101fd}', '\u{101fd}',  IdentifierType::Obsolete), ('\u{10280}', '\u{1029c}',
         IdentifierType::Exclusion), ('\u{102a0}', '\u{102d0}',  IdentifierType::Exclusion),
        ('\u{102e0}', '\u{102e0}',  IdentifierType::Obsolete), ('\u{102e1}', '\u{102fb}',
         IdentifierType::Obsolete), ('\u{10300}', '\u{1031e}',  IdentifierType::Exclusion),
        ('\u{1031f}', '\u{1031f}',  IdentifierType::Exclusion), ('\u{10320}', '\u{10323}',
         IdentifierType::Exclusion), ('\u{1032d}', '\u{1032f}',  IdentifierType::Exclusion),
        ('\u{10330}', '\u{1034a}',  IdentifierType::Exclusion), ('\u{10350}', '\u{1037a}',
         IdentifierType::Exclusion), ('\u{10380}', '\u{1039d}',  IdentifierType::Exclusion),
        ('\u{1039f}', '\u{1039f}',  IdentifierType::Exclusion), ('\u{103a0}', '\u{103c3}',
         IdentifierType::Exclusion), ('\u{103c8}', '\u{103cf}',  IdentifierType::Exclusion),
        ('\u{103d0}', '\u{103d0}',  IdentifierType::Exclusion), ('\u{103d1}', '\u{103d5}',
         IdentifierType::Exclusion), ('\u{10400}', '\u{10425}',  IdentifierType::Exclusion),
        ('\u{10426}', '\u{10427}',  IdentifierType::Exclusion), ('\u{10428}', '\u{1044d}',
         IdentifierType::Exclusion), ('\u{1044e}', '\u{1049d}',  IdentifierType::Exclusion),
        ('\u{104a0}', '\u{104a9}',  IdentifierType::Exclusion), ('\u{104b0}', '\u{104d3}',
         IdentifierType::Limited_Use), ('\u{104d8}', '\u{104fb}',  IdentifierType::Limited_Use),
        ('\u{10500}', '\u{10527}',  IdentifierType::Exclusion), ('\u{10530}', '\u{10563}',
         IdentifierType::Exclusion), ('\u{1056f}', '\u{1056f}',  IdentifierType::Exclusion),
        ('\u{10570}', '\u{1057a}',  IdentifierType::Exclusion), ('\u{1057c}', '\u{1058a}',
         IdentifierType::Exclusion), ('\u{1058c}', '\u{10592}',  IdentifierType::Exclusion),
        ('\u{10594}', '\u{10595}',  IdentifierType::Exclusion), ('\u{10597}', '\u{105a1}',
         IdentifierType::Exclusion), ('\u{105a3}', '\u{105b1}',  IdentifierType::Exclusion),
        ('\u{105b3}', '\u{105b9}',  IdentifierType::Exclusion), ('\u{105bb}', '\u{105bc}',
         IdentifierType::Exclusion), ('\u{105c0}', '\u{105f3}',  IdentifierType::Exclusion),
        ('\u{10600}', '\u{10736}',  IdentifierType::Exclusion), ('\u{10740}', '\u{10755}',
         IdentifierType::Exclusion), ('\u{10760}', '\u{10767}',  IdentifierType::Exclusion),
        ('\u{10780}', '\u{10780}',  IdentifierType::Uncommon_Use), ('\u{10781}', '\u{10785}',
         IdentifierType::Not_NFKC), ('\u{10787}', '\u{107b0}',  IdentifierType::Not_NFKC),
        ('\u{107b2}', '\u{107ba}',  IdentifierType::Not_NFKC), ('\u{10800}', '\u{10805}',
         IdentifierType::Exclusion), ('\u{10808}', '\u{10808}',  IdentifierType::Exclusion),
        ('\u{1080a}', '\u{10835}',  IdentifierType::Exclusion), ('\u{10837}', '\u{10838}',
         IdentifierType::Exclusion), ('\u{1083c}', '\u{1083c}',  IdentifierType::Exclusion),
        ('\u{1083f}', '\u{1083f}',  IdentifierType::Exclusion), ('\u{10840}', '\u{10855}',
         IdentifierType::Exclusion), ('\u{10857}', '\u{1085f}',  IdentifierType::Exclusion),
        ('\u{10860}', '\u{10876}',  IdentifierType::Exclusion), ('\u{10877}', '\u{1087f}',
         IdentifierType::Exclusion), ('\u{10880}', '\u{1089e}',  IdentifierType::Exclusion),
        ('\u{108a7}', '\u{108af}',  IdentifierType::Exclusion), ('\u{108e0}', '\u{108f2}',
         IdentifierType::Exclusion), ('\u{108f4}', '\u{108f5}',  IdentifierType::Exclusion),
        ('\u{108fb}', '\u{108ff}',  IdentifierType::Exclusion), ('\u{10900}', '\u{10915}',
         IdentifierType::Exclusion), ('\u{10916}', '\u{10919}',  IdentifierType::Exclusion),
        ('\u{1091a}', '\u{1091b}',  IdentifierType::Exclusion), ('\u{1091f}', '\u{1091f}',
         IdentifierType::Exclusion), ('\u{10920}', '\u{10939}',  IdentifierType::Exclusion),
        ('\u{1093f}', '\u{1093f}',  IdentifierType::Exclusion), ('\u{10980}', '\u{109b7}',
         IdentifierType::Exclusion), ('\u{109bc}', '\u{109bd}',  IdentifierType::Exclusion),
        ('\u{109be}', '\u{109bf}',  IdentifierType::Exclusion), ('\u{109c0}', '\u{109cf}',
         IdentifierType::Exclusion), ('\u{109d2}', '\u{109ff}',  IdentifierType::Exclusion),
        ('\u{10a00}', '\u{10a03}',  IdentifierType::Exclusion), ('\u{10a05}', '\u{10a06}',
         IdentifierType::Exclusion), ('\u{10a0c}', '\u{10a13}',  IdentifierType::Exclusion),
        ('\u{10a15}', '\u{10a17}',  IdentifierType::Exclusion), ('\u{10a19}', '\u{10a33}',
         IdentifierType::Exclusion), ('\u{10a34}', '\u{10a35}',  IdentifierType::Exclusion),
        ('\u{10a38}', '\u{10a3a}',  IdentifierType::Exclusion), ('\u{10a3f}', '\u{10a3f}',
         IdentifierType::Exclusion), ('\u{10a40}', '\u{10a47}',  IdentifierType::Exclusion),
        ('\u{10a48}', '\u{10a48}',  IdentifierType::Exclusion), ('\u{10a50}', '\u{10a58}',
         IdentifierType::Exclusion), ('\u{10a60}', '\u{10a7c}',  IdentifierType::Exclusion),
        ('\u{10a7d}', '\u{10a7f}',  IdentifierType::Exclusion), ('\u{10a80}', '\u{10a9c}',
         IdentifierType::Exclusion), ('\u{10a9d}', '\u{10a9f}',  IdentifierType::Exclusion),
        ('\u{10ac0}', '\u{10ac7}',  IdentifierType::Exclusion), ('\u{10ac8}', '\u{10ac8}',
         IdentifierType::Exclusion), ('\u{10ac9}', '\u{10ae6}',  IdentifierType::Exclusion),
        ('\u{10aeb}', '\u{10af6}',  IdentifierType::Exclusion), ('\u{10b00}', '\u{10b35}',
         IdentifierType::Exclusion), ('\u{10b39}', '\u{10b3f}',  IdentifierType::Exclusion),
        ('\u{10b40}', '\u{10b55}',  IdentifierType::Exclusion), ('\u{10b58}', '\u{10b5f}',
         IdentifierType::Exclusion), ('\u{10b60}', '\u{10b72}',  IdentifierType::Exclusion),
        ('\u{10b78}', '\u{10b7f}',  IdentifierType::Exclusion), ('\u{10b80}', '\u{10b91}',
         IdentifierType::Exclusion), ('\u{10b99}', '\u{10b9c}',  IdentifierType::Exclusion),
        ('\u{10ba9}', '\u{10baf}',  IdentifierType::Exclusion), ('\u{10c00}', '\u{10c48}',
         IdentifierType::Exclusion), ('\u{10c80}', '\u{10cb2}',  IdentifierType::Exclusion),
        ('\u{10cc0}', '\u{10cf2}',  IdentifierType::Exclusion), ('\u{10cfa}', '\u{10cff}',
         IdentifierType::Exclusion), ('\u{10d00}', '\u{10d27}',  IdentifierType::Limited_Use),
        ('\u{10d30}', '\u{10d39}',  IdentifierType::Limited_Use), ('\u{10d40}', '\u{10d65}',
         IdentifierType::Exclusion), ('\u{10d69}', '\u{10d6d}',  IdentifierType::Exclusion),
        ('\u{10d6e}', '\u{10d6e}',  IdentifierType::Exclusion), ('\u{10d6f}', '\u{10d85}',
         IdentifierType::Exclusion), ('\u{10d8e}', '\u{10d8f}',  IdentifierType::Exclusion),
        ('\u{10e60}', '\u{10e7e}',  IdentifierType::Not_XID), ('\u{10e80}', '\u{10ea9}',
         IdentifierType::Exclusion), ('\u{10eab}', '\u{10eac}',  IdentifierType::Exclusion),
        ('\u{10ead}', '\u{10ead}',  IdentifierType::Exclusion), ('\u{10eb0}', '\u{10eb1}',
         IdentifierType::Exclusion), ('\u{10ec2}', '\u{10ec4}',  IdentifierType::Uncommon_Use),
        ('\u{10efc}', '\u{10efc}',  IdentifierType::Uncommon_Use), ('\u{10efd}', '\u{10eff}',
         IdentifierType::Uncommon_Use), ('\u{10f00}', '\u{10f1c}',  IdentifierType::Exclusion),
        ('\u{10f1d}', '\u{10f26}',  IdentifierType::Exclusion), ('\u{10f27}', '\u{10f27}',
         IdentifierType::Exclusion), ('\u{10f30}', '\u{10f50}',  IdentifierType::Exclusion),
        ('\u{10f51}', '\u{10f59}',  IdentifierType::Exclusion), ('\u{10f70}', '\u{10f85}',
         IdentifierType::Exclusion), ('\u{10f86}', '\u{10f89}',  IdentifierType::Exclusion),
        ('\u{10fb0}', '\u{10fc4}',  IdentifierType::Exclusion), ('\u{10fc5}', '\u{10fcb}',
         IdentifierType::Exclusion), ('\u{10fe0}', '\u{10ff6}',  IdentifierType::Exclusion),
        ('\u{11000}', '\u{11046}',  IdentifierType::Exclusion), ('\u{11047}', '\u{1104d}',
         IdentifierType::Exclusion), ('\u{11052}', '\u{11065}',  IdentifierType::Exclusion),
        ('\u{11066}', '\u{1106f}',  IdentifierType::Exclusion), ('\u{11070}', '\u{11075}',
         IdentifierType::Exclusion), ('\u{1107f}', '\u{1107f}',  IdentifierType::Exclusion),
        ('\u{11080}', '\u{110ba}',  IdentifierType::Exclusion), ('\u{110bb}', '\u{110bc}',
         IdentifierType::Exclusion), ('\u{110bd}', '\u{110bd}',  IdentifierType::Exclusion),
        ('\u{110be}', '\u{110c1}',  IdentifierType::Exclusion), ('\u{110c2}', '\u{110c2}',
         IdentifierType::Exclusion), ('\u{110cd}', '\u{110cd}',  IdentifierType::Exclusion),
        ('\u{110d0}', '\u{110e8}',  IdentifierType::Exclusion), ('\u{110f0}', '\u{110f9}',
         IdentifierType::Exclusion), ('\u{11100}', '\u{11134}',  IdentifierType::Limited_Use),
        ('\u{11136}', '\u{1113f}',  IdentifierType::Limited_Use), ('\u{11140}', '\u{11143}',
         IdentifierType::Limited_Use), ('\u{11144}', '\u{11146}',  IdentifierType::Limited_Use),
        ('\u{11147}', '\u{11147}',  IdentifierType::Limited_Use), ('\u{11150}', '\u{11173}',
         IdentifierType::Exclusion), ('\u{11174}', '\u{11175}',  IdentifierType::Exclusion),
        ('\u{11176}', '\u{11176}',  IdentifierType::Exclusion), ('\u{11180}', '\u{111c4}',
         IdentifierType::Exclusion), ('\u{111c5}', '\u{111c8}',  IdentifierType::Exclusion),
        ('\u{111c9}', '\u{111cc}',  IdentifierType::Exclusion), ('\u{111cd}', '\u{111cd}',
         IdentifierType::Exclusion), ('\u{111ce}', '\u{111cf}',  IdentifierType::Exclusion),
        ('\u{111d0}', '\u{111d9}',  IdentifierType::Exclusion), ('\u{111da}', '\u{111da}',
         IdentifierType::Exclusion), ('\u{111db}', '\u{111db}',  IdentifierType::Exclusion),
        ('\u{111dc}', '\u{111dc}',  IdentifierType::Exclusion), ('\u{111dd}', '\u{111df}',
         IdentifierType::Exclusion), ('\u{111e1}', '\u{111f4}',  IdentifierType::Not_XID),
        ('\u{11200}', '\u{11211}',  IdentifierType::Exclusion), ('\u{11213}', '\u{11237}',
         IdentifierType::Exclusion), ('\u{11238}', '\u{1123d}',  IdentifierType::Exclusion),
        ('\u{1123e}', '\u{1123e}',  IdentifierType::Exclusion), ('\u{1123f}', '\u{11241}',
         IdentifierType::Exclusion), ('\u{11280}', '\u{11286}',  IdentifierType::Exclusion),
        ('\u{11288}', '\u{11288}',  IdentifierType::Exclusion), ('\u{1128a}', '\u{1128d}',
         IdentifierType::Exclusion), ('\u{1128f}', '\u{1129d}',  IdentifierType::Exclusion),
        ('\u{1129f}', '\u{112a8}',  IdentifierType::Exclusion), ('\u{112a9}', '\u{112a9}',
         IdentifierType::Exclusion), ('\u{112b0}', '\u{112ea}',  IdentifierType::Exclusion),
        ('\u{112f0}', '\u{112f9}',  IdentifierType::Exclusion), ('\u{11300}', '\u{11300}',
         IdentifierType::Exclusion), ('\u{11301}', '\u{11301}',  IdentifierType::Recommended),
        ('\u{11302}', '\u{11302}',  IdentifierType::Exclusion), ('\u{11303}', '\u{11303}',
         IdentifierType::Recommended), ('\u{11305}', '\u{1130c}',  IdentifierType::Exclusion),
        ('\u{1130f}', '\u{11310}',  IdentifierType::Exclusion), ('\u{11313}', '\u{11328}',
         IdentifierType::Exclusion), ('\u{1132a}', '\u{11330}',  IdentifierType::Exclusion),
        ('\u{11332}', '\u{11333}',  IdentifierType::Exclusion), ('\u{11335}', '\u{11339}',
         IdentifierType::Exclusion), ('\u{1133b}', '\u{1133b}',  IdentifierType::Recommended),
        ('\u{1133c}', '\u{1133c}',  IdentifierType::Recommended), ('\u{1133d}', '\u{11344}',
         IdentifierType::Exclusion), ('\u{11347}', '\u{11348}',  IdentifierType::Exclusion),
        ('\u{1134b}', '\u{1134d}',  IdentifierType::Exclusion), ('\u{11350}', '\u{11350}',
         IdentifierType::Exclusion), ('\u{11357}', '\u{11357}',  IdentifierType::Exclusion),
        ('\u{1135d}', '\u{11363}',  IdentifierType::Exclusion), ('\u{11366}', '\u{1136c}',
         IdentifierType::Exclusion), ('\u{11370}', '\u{11374}',  IdentifierType::Exclusion),
        ('\u{11380}', '\u{11389}',  IdentifierType::Exclusion), ('\u{1138b}', '\u{1138b}',
         IdentifierType::Exclusion), ('\u{1138e}', '\u{1138e}',  IdentifierType::Exclusion),
        ('\u{11390}', '\u{113b5}',  IdentifierType::Exclusion), ('\u{113b7}', '\u{113c0}',
         IdentifierType::Exclusion), ('\u{113c2}', '\u{113c2}',  IdentifierType::Exclusion),
        ('\u{113c5}', '\u{113c5}',  IdentifierType::Exclusion), ('\u{113c7}', '\u{113ca}',
         IdentifierType::Exclusion), ('\u{113cc}', '\u{113d3}',  IdentifierType::Exclusion),
        ('\u{113d4}', '\u{113d5}',  IdentifierType::Exclusion), ('\u{113d7}', '\u{113d8}',
         IdentifierType::Exclusion), ('\u{113e1}', '\u{113e2}',  IdentifierType::Exclusion),
        ('\u{11400}', '\u{1144a}',  IdentifierType::Limited_Use), ('\u{1144b}', '\u{1144f}',
         IdentifierType::Limited_Use), ('\u{11450}', '\u{11459}',  IdentifierType::Limited_Use),
        ('\u{1145a}', '\u{1145a}',  IdentifierType::Limited_Use), ('\u{1145b}', '\u{1145b}',
         IdentifierType::Limited_Use), ('\u{1145d}', '\u{1145d}',  IdentifierType::Limited_Use),
        ('\u{1145e}', '\u{1145e}',  IdentifierType::Limited_Use), ('\u{1145f}', '\u{1145f}',
         IdentifierType::Limited_Use), ('\u{11460}', '\u{11461}',  IdentifierType::Limited_Use),
        ('\u{11480}', '\u{114c5}',  IdentifierType::Exclusion), ('\u{114c6}', '\u{114c6}',
         IdentifierType::Exclusion), ('\u{114c7}', '\u{114c7}',  IdentifierType::Exclusion),
        ('\u{114d0}', '\u{114d9}',  IdentifierType::Exclusion), ('\u{11580}', '\u{115b5}',
         IdentifierType::Exclusion), ('\u{115b8}', '\u{115c0}',  IdentifierType::Exclusion),
        ('\u{115c1}', '\u{115c9}',  IdentifierType::Exclusion), ('\u{115ca}', '\u{115d7}',
         IdentifierType::Exclusion), ('\u{115d8}', '\u{115dd}',  IdentifierType::Exclusion),
        ('\u{11600}', '\u{11640}',  IdentifierType::Exclusion), ('\u{11641}', '\u{11643}',
         IdentifierType::Exclusion), ('\u{11644}', '\u{11644}',  IdentifierType::Exclusion),
        ('\u{11650}', '\u{11659}',  IdentifierType::Exclusion), ('\u{11660}', '\u{1166c}',
         IdentifierType::Exclusion), ('\u{11680}', '\u{116b7}',  IdentifierType::Exclusion),
        ('\u{116b8}', '\u{116b8}',  IdentifierType::Exclusion), ('\u{116b9}', '\u{116b9}',
         IdentifierType::Exclusion), ('\u{116c0}', '\u{116c9}',  IdentifierType::Exclusion),
        ('\u{116d0}', '\u{116e3}',  IdentifierType::Uncommon_Use), ('\u{11700}', '\u{11719}',
         IdentifierType::Exclusion), ('\u{1171a}', '\u{1171a}',  IdentifierType::Exclusion),
        ('\u{1171d}', '\u{1172b}',  IdentifierType::Exclusion), ('\u{11730}', '\u{11739}',
         IdentifierType::Exclusion), ('\u{1173a}', '\u{1173f}',  IdentifierType::Exclusion),
        ('\u{11740}', '\u{11746}',  IdentifierType::Exclusion), ('\u{11800}', '\u{1183a}',
         IdentifierType::Exclusion), ('\u{1183b}', '\u{1183b}',  IdentifierType::Exclusion),
        ('\u{118a0}', '\u{118e9}',  IdentifierType::Exclusion), ('\u{118ea}', '\u{118f2}',
         IdentifierType::Exclusion), ('\u{118ff}', '\u{118ff}',  IdentifierType::Exclusion),
        ('\u{11900}', '\u{11906}',  IdentifierType::Exclusion), ('\u{11909}', '\u{11909}',
         IdentifierType::Exclusion), ('\u{1190c}', '\u{11913}',  IdentifierType::Exclusion),
        ('\u{11915}', '\u{11916}',  IdentifierType::Exclusion), ('\u{11918}', '\u{11935}',
         IdentifierType::Exclusion), ('\u{11937}', '\u{11938}',  IdentifierType::Exclusion),
        ('\u{1193b}', '\u{11943}',  IdentifierType::Exclusion), ('\u{11944}', '\u{11946}',
         IdentifierType::Exclusion), ('\u{11950}', '\u{11959}',  IdentifierType::Exclusion),
        ('\u{119a0}', '\u{119a7}',  IdentifierType::Exclusion), ('\u{119aa}', '\u{119d7}',
         IdentifierType::Exclusion), ('\u{119da}', '\u{119e1}',  IdentifierType::Exclusion),
        ('\u{119e2}', '\u{119e2}',  IdentifierType::Exclusion), ('\u{119e3}', '\u{119e4}',
         IdentifierType::Exclusion), ('\u{11a00}', '\u{11a3e}',  IdentifierType::Exclusion),
        ('\u{11a3f}', '\u{11a46}',  IdentifierType::Exclusion), ('\u{11a47}', '\u{11a47}',
         IdentifierType::Exclusion), ('\u{11a50}', '\u{11a83}',  IdentifierType::Exclusion),
        ('\u{11a84}', '\u{11a85}',  IdentifierType::Exclusion), ('\u{11a86}', '\u{11a99}',
         IdentifierType::Exclusion), ('\u{11a9a}', '\u{11a9c}',  IdentifierType::Exclusion),
        ('\u{11a9d}', '\u{11a9d}',  IdentifierType::Exclusion), ('\u{11a9e}', '\u{11aa2}',
         IdentifierType::Exclusion), ('\u{11ab0}', '\u{11abf}',  IdentifierType::Limited_Use),
        ('\u{11ac0}', '\u{11af8}',  IdentifierType::Exclusion), ('\u{11b00}', '\u{11b09}',
         IdentifierType::Not_XID), ('\u{11bc0}', '\u{11be0}',  IdentifierType::Exclusion),
        ('\u{11be1}', '\u{11be1}',  IdentifierType::Exclusion), ('\u{11bf0}', '\u{11bf9}',
         IdentifierType::Exclusion), ('\u{11c00}', '\u{11c08}',  IdentifierType::Exclusion),
        ('\u{11c0a}', '\u{11c36}',  IdentifierType::Exclusion), ('\u{11c38}', '\u{11c40}',
         IdentifierType::Exclusion), ('\u{11c41}', '\u{11c45}',  IdentifierType::Exclusion),
        ('\u{11c50}', '\u{11c59}',  IdentifierType::Exclusion), ('\u{11c5a}', '\u{11c6c}',
         IdentifierType::Exclusion), ('\u{11c70}', '\u{11c71}',  IdentifierType::Exclusion),
        ('\u{11c72}', '\u{11c8f}',  IdentifierType::Exclusion), ('\u{11c92}', '\u{11ca7}',
         IdentifierType::Exclusion), ('\u{11ca9}', '\u{11cb6}',  IdentifierType::Exclusion),
        ('\u{11d00}', '\u{11d06}',  IdentifierType::Exclusion), ('\u{11d08}', '\u{11d09}',
         IdentifierType::Exclusion), ('\u{11d0b}', '\u{11d36}',  IdentifierType::Exclusion),
        ('\u{11d3a}', '\u{11d3a}',  IdentifierType::Exclusion), ('\u{11d3c}', '\u{11d3d}',
         IdentifierType::Exclusion), ('\u{11d3f}', '\u{11d47}',  IdentifierType::Exclusion),
        ('\u{11d50}', '\u{11d59}',  IdentifierType::Exclusion), ('\u{11d60}', '\u{11d65}',
         IdentifierType::Limited_Use), ('\u{11d67}', '\u{11d68}',  IdentifierType::Limited_Use),
        ('\u{11d6a}', '\u{11d8e}',  IdentifierType::Limited_Use), ('\u{11d90}', '\u{11d91}',
         IdentifierType::Limited_Use), ('\u{11d93}', '\u{11d98}',  IdentifierType::Limited_Use),
        ('\u{11da0}', '\u{11da9}',  IdentifierType::Limited_Use), ('\u{11ee0}', '\u{11ef6}',
         IdentifierType::Exclusion), ('\u{11ef7}', '\u{11ef8}',  IdentifierType::Exclusion),
        ('\u{11f00}', '\u{11f10}',  IdentifierType::Exclusion), ('\u{11f12}', '\u{11f3a}',
         IdentifierType::Exclusion), ('\u{11f3e}', '\u{11f42}',  IdentifierType::Exclusion),
        ('\u{11f43}', '\u{11f4f}',  IdentifierType::Exclusion), ('\u{11f50}', '\u{11f59}',
         IdentifierType::Exclusion), ('\u{11f5a}', '\u{11f5a}',  IdentifierType::Exclusion),
        ('\u{11fb0}', '\u{11fb0}',  IdentifierType::Limited_Use), ('\u{11fc0}', '\u{11ff1}',
         IdentifierType::Not_XID), ('\u{11fff}', '\u{11fff}',  IdentifierType::Not_XID),
        ('\u{12000}', '\u{1236e}',  IdentifierType::Exclusion), ('\u{1236f}', '\u{12398}',
         IdentifierType::Exclusion), ('\u{12399}', '\u{12399}',  IdentifierType::Exclusion),
        ('\u{12400}', '\u{12462}',  IdentifierType::Exclusion), ('\u{12463}', '\u{1246e}',
         IdentifierType::Exclusion), ('\u{12470}', '\u{12473}',  IdentifierType::Exclusion),
        ('\u{12474}', '\u{12474}',  IdentifierType::Exclusion), ('\u{12480}', '\u{12543}',
         IdentifierType::Exclusion), ('\u{12f90}', '\u{12ff0}',  IdentifierType::Exclusion),
        ('\u{12ff1}', '\u{12ff2}',  IdentifierType::Exclusion), ('\u{13000}', '\u{1342e}',
         IdentifierType::Exclusion), ('\u{1342f}', '\u{1342f}',  IdentifierType::Exclusion),
        ('\u{13430}', '\u{13438}',  IdentifierType::Exclusion), ('\u{13439}', '\u{1343f}',
         IdentifierType::Exclusion), ('\u{13440}', '\u{13455}',  IdentifierType::Exclusion),
        ('\u{13460}', '\u{143fa}',  IdentifierType::Exclusion), ('\u{14400}', '\u{14646}',
         IdentifierType::Exclusion), ('\u{16100}', '\u{16139}',  IdentifierType::Exclusion),
        ('\u{16800}', '\u{16a38}',  IdentifierType::Limited_Use), ('\u{16a40}', '\u{16a5e}',
         IdentifierType::Uncommon_Use), ('\u{16a60}', '\u{16a69}',  IdentifierType::Uncommon_Use),
        ('\u{16a6e}', '\u{16a6f}',  IdentifierType::Exclusion), ('\u{16a70}', '\u{16abe}',
         IdentifierType::Exclusion), ('\u{16ac0}', '\u{16ac9}',  IdentifierType::Exclusion),
        ('\u{16ad0}', '\u{16aed}',  IdentifierType::Exclusion), ('\u{16af0}', '\u{16af4}',
         IdentifierType::Exclusion), ('\u{16af5}', '\u{16af5}',  IdentifierType::Exclusion),
        ('\u{16b00}', '\u{16b36}',  IdentifierType::Exclusion), ('\u{16b37}', '\u{16b3f}',
         IdentifierType::Exclusion), ('\u{16b40}', '\u{16b43}',  IdentifierType::Exclusion),
        ('\u{16b44}', '\u{16b45}',  IdentifierType::Exclusion), ('\u{16b50}', '\u{16b59}',
         IdentifierType::Exclusion), ('\u{16b5b}', '\u{16b61}',  IdentifierType::Exclusion),
        ('\u{16b63}', '\u{16b77}',  IdentifierType::Exclusion), ('\u{16b7d}', '\u{16b8f}',
         IdentifierType::Exclusion), ('\u{16d40}', '\u{16d6c}',  IdentifierType::Exclusion),
        ('\u{16d6d}', '\u{16d6f}',  IdentifierType::Exclusion), ('\u{16d70}', '\u{16d79}',
         IdentifierType::Exclusion), ('\u{16e40}', '\u{16e7f}',  IdentifierType::Exclusion),
        ('\u{16e80}', '\u{16e9a}',  IdentifierType::Exclusion), ('\u{16f00}', '\u{16f44}',
         IdentifierType::Limited_Use), ('\u{16f45}', '\u{16f4a}',  IdentifierType::Limited_Use),
        ('\u{16f4f}', '\u{16f4f}',  IdentifierType::Limited_Use), ('\u{16f50}', '\u{16f7e}',
         IdentifierType::Limited_Use), ('\u{16f7f}', '\u{16f87}',  IdentifierType::Limited_Use),
        ('\u{16f8f}', '\u{16f9f}',  IdentifierType::Limited_Use), ('\u{16fe0}', '\u{16fe0}',
         IdentifierType::Exclusion), ('\u{16fe1}', '\u{16fe1}',  IdentifierType::Exclusion),
        ('\u{16fe2}', '\u{16fe2}',  IdentifierType::Not_XID), ('\u{16fe3}', '\u{16fe3}',
         IdentifierType::Obsolete), ('\u{16fe4}', '\u{16fe4}',  IdentifierType::Exclusion),
        ('\u{16ff0}', '\u{16ff1}',  IdentifierType::Recommended), ('\u{17000}', '\u{187ec}',
         IdentifierType::Exclusion), ('\u{187ed}', '\u{187f1}',  IdentifierType::Exclusion),
        ('\u{187f2}', '\u{187f7}',  IdentifierType::Exclusion), ('\u{18800}', '\u{18af2}',
         IdentifierType::Exclusion), ('\u{18af3}', '\u{18cd5}',  IdentifierType::Exclusion),
        ('\u{18cff}', '\u{18cff}',  IdentifierType::Exclusion), ('\u{18d00}', '\u{18d08}',
         IdentifierType::Exclusion), ('\u{1aff0}', '\u{1aff3}',  IdentifierType::Uncommon_Use),
        ('\u{1aff5}', '\u{1affb}',  IdentifierType::Uncommon_Use), ('\u{1affd}', '\u{1affe}',
         IdentifierType::Uncommon_Use), ('\u{1b000}', '\u{1b001}',  IdentifierType::Obsolete),
        ('\u{1b002}', '\u{1b11e}',  IdentifierType::Obsolete), ('\u{1b11f}', '\u{1b122}',
         IdentifierType::Recommended), ('\u{1b132}', '\u{1b132}',  IdentifierType::Recommended),
        ('\u{1b150}', '\u{1b152}',  IdentifierType::Recommended), ('\u{1b155}', '\u{1b155}',
         IdentifierType::Recommended), ('\u{1b164}', '\u{1b167}',  IdentifierType::Recommended),
        ('\u{1b170}', '\u{1b2fb}',  IdentifierType::Exclusion), ('\u{1bc00}', '\u{1bc6a}',
         IdentifierType::Exclusion), ('\u{1bc70}', '\u{1bc7c}',  IdentifierType::Exclusion),
        ('\u{1bc80}', '\u{1bc88}',  IdentifierType::Exclusion), ('\u{1bc90}', '\u{1bc99}',
         IdentifierType::Exclusion), ('\u{1bc9c}', '\u{1bc9c}',  IdentifierType::Exclusion),
        ('\u{1bc9d}', '\u{1bc9e}',  IdentifierType::Exclusion), ('\u{1bc9f}', '\u{1bc9f}',
         IdentifierType::Exclusion), ('\u{1bca0}', '\u{1bca3}',  IdentifierType::Default_Ignorable),
        ('\u{1cc00}', '\u{1ccd5}',  IdentifierType::Not_XID), ('\u{1ccd6}', '\u{1ccf9}',
         IdentifierType::Not_NFKC), ('\u{1cd00}', '\u{1ceb3}',  IdentifierType::Not_XID),
        ('\u{1cf00}', '\u{1cf2d}',  IdentifierType::Technical), ('\u{1cf30}', '\u{1cf46}',
         IdentifierType::Technical), ('\u{1cf50}', '\u{1cfc3}',  IdentifierType::Technical),
        ('\u{1d000}', '\u{1d0f5}',  IdentifierType::Technical), ('\u{1d100}', '\u{1d126}',
         IdentifierType::Technical), ('\u{1d129}', '\u{1d129}',  IdentifierType::Technical),
        ('\u{1d12a}', '\u{1d15d}',  IdentifierType::Technical), ('\u{1d15e}', '\u{1d164}',
         IdentifierType::Not_NFKC), ('\u{1d165}', '\u{1d169}',  IdentifierType::Technical),
        ('\u{1d16a}', '\u{1d16c}',  IdentifierType::Technical), ('\u{1d16d}', '\u{1d172}',
         IdentifierType::Technical), ('\u{1d173}', '\u{1d17a}',  IdentifierType::Default_Ignorable),
        ('\u{1d17b}', '\u{1d182}',  IdentifierType::Technical), ('\u{1d183}', '\u{1d184}',
         IdentifierType::Technical), ('\u{1d185}', '\u{1d18b}',  IdentifierType::Technical),
        ('\u{1d18c}', '\u{1d1a9}',  IdentifierType::Technical), ('\u{1d1aa}', '\u{1d1ad}',
         IdentifierType::Technical), ('\u{1d1ae}', '\u{1d1ba}',  IdentifierType::Technical),
        ('\u{1d1bb}', '\u{1d1c0}',  IdentifierType::Not_NFKC), ('\u{1d1c1}', '\u{1d1dd}',
         IdentifierType::Technical), ('\u{1d1de}', '\u{1d1e8}',  IdentifierType::Uncommon_Use),
        ('\u{1d1e9}', '\u{1d1ea}',  IdentifierType::Technical), ('\u{1d200}', '\u{1d241}',
         IdentifierType::Obsolete), ('\u{1d242}', '\u{1d244}',  IdentifierType::Technical),
        ('\u{1d245}', '\u{1d245}',  IdentifierType::Obsolete), ('\u{1d2c0}', '\u{1d2d3}',
         IdentifierType::Not_XID), ('\u{1d2e0}', '\u{1d2f3}',  IdentifierType::Not_XID),
        ('\u{1d300}', '\u{1d356}',  IdentifierType::Technical), ('\u{1d360}', '\u{1d371}',
         IdentifierType::Not_XID), ('\u{1d372}', '\u{1d378}',  IdentifierType::Not_XID),
        ('\u{1d400}', '\u{1d454}',  IdentifierType::Not_NFKC), ('\u{1d456}', '\u{1d49c}',
         IdentifierType::Not_NFKC), ('\u{1d49e}', '\u{1d49f}',  IdentifierType::Not_NFKC),
        ('\u{1d4a2}', '\u{1d4a2}',  IdentifierType::Not_NFKC), ('\u{1d4a5}', '\u{1d4a6}',
         IdentifierType::Not_NFKC), ('\u{1d4a9}', '\u{1d4ac}',  IdentifierType::Not_NFKC),
        ('\u{1d4ae}', '\u{1d4b9}',  IdentifierType::Not_NFKC), ('\u{1d4bb}', '\u{1d4bb}',
         IdentifierType::Not_NFKC), ('\u{1d4bd}', '\u{1d4c0}',  IdentifierType::Not_NFKC),
        ('\u{1d4c1}', '\u{1d4c1}',  IdentifierType::Not_NFKC), ('\u{1d4c2}', '\u{1d4c3}',
         IdentifierType::Not_NFKC), ('\u{1d4c5}', '\u{1d505}',  IdentifierType::Not_NFKC),
        ('\u{1d507}', '\u{1d50a}',  IdentifierType::Not_NFKC), ('\u{1d50d}', '\u{1d514}',
         IdentifierType::Not_NFKC), ('\u{1d516}', '\u{1d51c}',  IdentifierType::Not_NFKC),
        ('\u{1d51e}', '\u{1d539}',  IdentifierType::Not_NFKC), ('\u{1d53b}', '\u{1d53e}',
         IdentifierType::Not_NFKC), ('\u{1d540}', '\u{1d544}',  IdentifierType::Not_NFKC),
        ('\u{1d546}', '\u{1d546}',  IdentifierType::Not_NFKC), ('\u{1d54a}', '\u{1d550}',
         IdentifierType::Not_NFKC), ('\u{1d552}', '\u{1d6a3}',  IdentifierType::Not_NFKC),
        ('\u{1d6a4}', '\u{1d6a5}',  IdentifierType::Not_NFKC), ('\u{1d6a8}', '\u{1d7c9}',
         IdentifierType::Not_NFKC), ('\u{1d7ca}', '\u{1d7cb}',  IdentifierType::Not_NFKC),
        ('\u{1d7ce}', '\u{1d7ff}',  IdentifierType::Not_NFKC), ('\u{1d800}', '\u{1d9ff}',
         IdentifierType::Exclusion), ('\u{1da00}', '\u{1da36}',  IdentifierType::Exclusion),
        ('\u{1da37}', '\u{1da3a}',  IdentifierType::Exclusion), ('\u{1da3b}', '\u{1da6c}',
         IdentifierType::Exclusion), ('\u{1da6d}', '\u{1da74}',  IdentifierType::Exclusion),
        ('\u{1da75}', '\u{1da75}',  IdentifierType::Exclusion), ('\u{1da76}', '\u{1da83}',
         IdentifierType::Exclusion), ('\u{1da84}', '\u{1da84}',  IdentifierType::Exclusion),
        ('\u{1da85}', '\u{1da8b}',  IdentifierType::Exclusion), ('\u{1da9b}', '\u{1da9f}',
         IdentifierType::Exclusion), ('\u{1daa1}', '\u{1daaf}',  IdentifierType::Exclusion),
        ('\u{1df00}', '\u{1df1e}',  IdentifierType::Recommended), ('\u{1df25}', '\u{1df2a}',
         IdentifierType::Recommended), ('\u{1e000}', '\u{1e006}',  IdentifierType::Exclusion),
        ('\u{1e008}', '\u{1e018}',  IdentifierType::Exclusion), ('\u{1e01b}', '\u{1e021}',
         IdentifierType::Exclusion), ('\u{1e023}', '\u{1e024}',  IdentifierType::Exclusion),
        ('\u{1e026}', '\u{1e02a}',  IdentifierType::Exclusion), ('\u{1e030}', '\u{1e06d}',
         IdentifierType::Not_NFKC), ('\u{1e08f}', '\u{1e08f}',  IdentifierType::Recommended),
        ('\u{1e100}', '\u{1e12c}',  IdentifierType::Limited_Use), ('\u{1e130}', '\u{1e13d}',
         IdentifierType::Limited_Use), ('\u{1e140}', '\u{1e149}',  IdentifierType::Limited_Use),
        ('\u{1e14e}', '\u{1e14e}',  IdentifierType::Limited_Use), ('\u{1e14f}', '\u{1e14f}',
         IdentifierType::Limited_Use), ('\u{1e290}', '\u{1e2ae}',  IdentifierType::Exclusion),
        ('\u{1e2c0}', '\u{1e2f9}',  IdentifierType::Limited_Use), ('\u{1e2ff}', '\u{1e2ff}',
         IdentifierType::Limited_Use), ('\u{1e4d0}', '\u{1e4f9}',  IdentifierType::Exclusion),
        ('\u{1e5d0}', '\u{1e5fa}',  IdentifierType::Exclusion), ('\u{1e5ff}', '\u{1e5ff}',
         IdentifierType::Exclusion), ('\u{1e7e0}', '\u{1e7e6}',  IdentifierType::Recommended),
        ('\u{1e7e8}', '\u{1e7eb}',  IdentifierType::Recommended), ('\u{1e7ed}', '\u{1e7ee}',
         IdentifierType::Recommended), ('\u{1e7f0}', '\u{1e7fe}',  IdentifierType::Recommended),
        ('\u{1e800}', '\u{1e8c4}',  IdentifierType::Exclusion), ('\u{1e8c7}', '\u{1e8cf}',
         IdentifierType::Exclusion), ('\u{1e8d0}', '\u{1e8d6}',  IdentifierType::Exclusion),
        ('\u{1e900}', '\u{1e94a}',  IdentifierType::Limited_Use), ('\u{1e94b}', '\u{1e94b}',
         IdentifierType::Limited_Use), ('\u{1e950}', '\u{1e959}',  IdentifierType::Limited_Use),
        ('\u{1e95e}', '\u{1e95f}',  IdentifierType::Limited_Use), ('\u{1ec71}', '\u{1ecb4}',
         IdentifierType::Not_XID), ('\u{1ed01}', '\u{1ed3d}',  IdentifierType::Not_XID),
        ('\u{1ee00}', '\u{1ee03}',  IdentifierType::Not_NFKC), ('\u{1ee05}', '\u{1ee1f}',
         IdentifierType::Not_NFKC), ('\u{1ee21}', '\u{1ee22}',  IdentifierType::Not_NFKC),
        ('\u{1ee24}', '\u{1ee24}',  IdentifierType::Not_NFKC), ('\u{1ee27}', '\u{1ee27}',
         IdentifierType::Not_NFKC), ('\u{1ee29}', '\u{1ee32}',  IdentifierType::Not_NFKC),
        ('\u{1ee34}', '\u{1ee37}',  IdentifierType::Not_NFKC), ('\u{1ee39}', '\u{1ee39}',
         IdentifierType::Not_NFKC), ('\u{1ee3b}', '\u{1ee3b}',  IdentifierType::Not_NFKC),
        ('\u{1ee42}', '\u{1ee42}',  IdentifierType::Not_NFKC), ('\u{1ee47}', '\u{1ee47}',
         IdentifierType::Not_NFKC), ('\u{1ee49}', '\u{1ee49}',  IdentifierType::Not_NFKC),
        ('\u{1ee4b}', '\u{1ee4b}',  IdentifierType::Not_NFKC), ('\u{1ee4d}', '\u{1ee4f}',
         IdentifierType::Not_NFKC), ('\u{1ee51}', '\u{1ee52}',  IdentifierType::Not_NFKC),
        ('\u{1ee54}', '\u{1ee54}',  IdentifierType::Not_NFKC), ('\u{1ee57}', '\u{1ee57}',
         IdentifierType::Not_NFKC), ('\u{1ee59}', '\u{1ee59}',  IdentifierType::Not_NFKC),
        ('\u{1ee5b}', '\u{1ee5b}',  IdentifierType::Not_NFKC), ('\u{1ee5d}', '\u{1ee5d}',
         IdentifierType::Not_NFKC), ('\u{1ee5f}', '\u{1ee5f}',  IdentifierType::Not_NFKC),
        ('\u{1ee61}', '\u{1ee62}',  IdentifierType::Not_NFKC), ('\u{1ee64}', '\u{1ee64}',
         IdentifierType::Not_NFKC), ('\u{1ee67}', '\u{1ee6a}',  IdentifierType::Not_NFKC),
        ('\u{1ee6c}', '\u{1ee72}',  IdentifierType::Not_NFKC), ('\u{1ee74}', '\u{1ee77}',
         IdentifierType::Not_NFKC), ('\u{1ee79}', '\u{1ee7c}',  IdentifierType::Not_NFKC),
        ('\u{1ee7e}', '\u{1ee7e}',  IdentifierType::Not_NFKC), ('\u{1ee80}', '\u{1ee89}',
         IdentifierType::Not_NFKC), ('\u{1ee8b}', '\u{1ee9b}',  IdentifierType::Not_NFKC),
        ('\u{1eea1}', '\u{1eea3}',  IdentifierType::Not_NFKC), ('\u{1eea5}', '\u{1eea9}',
         IdentifierType::Not_NFKC), ('\u{1eeab}', '\u{1eebb}',  IdentifierType::Not_NFKC),
        ('\u{1eef0}', '\u{1eef1}',  IdentifierType::Not_XID), ('\u{1f000}', '\u{1f02b}',
         IdentifierType::Not_XID), ('\u{1f030}', '\u{1f093}',  IdentifierType::Not_XID),
        ('\u{1f0a0}', '\u{1f0ae}',  IdentifierType::Not_XID), ('\u{1f0b1}', '\u{1f0be}',
         IdentifierType::Not_XID), ('\u{1f0bf}', '\u{1f0bf}',  IdentifierType::Not_XID),
        ('\u{1f0c1}', '\u{1f0cf}',  IdentifierType::Not_XID), ('\u{1f0d1}', '\u{1f0df}',
         IdentifierType::Not_XID), ('\u{1f0e0}', '\u{1f0f5}',  IdentifierType::Not_XID),
        ('\u{1f100}', '\u{1f10a}',  IdentifierType::Not_NFKC), ('\u{1f10b}', '\u{1f10c}',
         IdentifierType::Not_XID), ('\u{1f10d}', '\u{1f10f}',  IdentifierType::Not_XID),
        ('\u{1f110}', '\u{1f12e}',  IdentifierType::Not_NFKC), ('\u{1f12f}', '\u{1f12f}',
         IdentifierType::Not_XID), ('\u{1f130}', '\u{1f130}',  IdentifierType::Not_NFKC),
        ('\u{1f131}', '\u{1f131}',  IdentifierType::Not_NFKC), ('\u{1f132}', '\u{1f13c}',
         IdentifierType::Not_NFKC), ('\u{1f13d}', '\u{1f13d}',  IdentifierType::Not_NFKC),
        ('\u{1f13e}', '\u{1f13e}',  IdentifierType::Not_NFKC), ('\u{1f13f}', '\u{1f13f}',
         IdentifierType::Not_NFKC), ('\u{1f140}', '\u{1f141}',  IdentifierType::Not_NFKC),
        ('\u{1f142}', '\u{1f142}',  IdentifierType::Not_NFKC), ('\u{1f143}', '\u{1f145}',
         IdentifierType::Not_NFKC), ('\u{1f146}', '\u{1f146}',  IdentifierType::Not_NFKC),
        ('\u{1f147}', '\u{1f149}',  IdentifierType::Not_NFKC), ('\u{1f14a}', '\u{1f14e}',
         IdentifierType::Not_NFKC), ('\u{1f14f}', '\u{1f14f}',  IdentifierType::Not_NFKC),
        ('\u{1f150}', '\u{1f156}',  IdentifierType::Not_XID), ('\u{1f157}', '\u{1f157}',
         IdentifierType::Not_XID), ('\u{1f158}', '\u{1f15e}',  IdentifierType::Not_XID),
        ('\u{1f15f}', '\u{1f15f}',  IdentifierType::Not_XID), ('\u{1f160}', '\u{1f169}',
         IdentifierType::Not_XID), ('\u{1f16a}', '\u{1f16b}',  IdentifierType::Not_NFKC),
        ('\u{1f16c}', '\u{1f16c}',  IdentifierType::Not_NFKC), ('\u{1f16d}', '\u{1f16f}',
         IdentifierType::Not_XID), ('\u{1f170}', '\u{1f178}',  IdentifierType::Not_XID),
        ('\u{1f179}', '\u{1f179}',  IdentifierType::Not_XID), ('\u{1f17a}', '\u{1f17a}',
         IdentifierType::Not_XID), ('\u{1f17b}', '\u{1f17c}',  IdentifierType::Not_XID),
        ('\u{1f17d}', '\u{1f17e}',  IdentifierType::Not_XID), ('\u{1f17f}', '\u{1f17f}',
         IdentifierType::Not_XID), ('\u{1f180}', '\u{1f189}',  IdentifierType::Not_XID),
        ('\u{1f18a}', '\u{1f18d}',  IdentifierType::Not_XID), ('\u{1f18e}', '\u{1f18f}',
         IdentifierType::Not_XID), ('\u{1f190}', '\u{1f190}',  IdentifierType::Not_NFKC),
        ('\u{1f191}', '\u{1f19a}',  IdentifierType::Not_XID), ('\u{1f19b}', '\u{1f1ac}',
         IdentifierType::Not_XID), ('\u{1f1ad}', '\u{1f1ad}',  IdentifierType::Not_XID),
        ('\u{1f1e6}', '\u{1f1ff}',  IdentifierType::Not_XID), ('\u{1f200}', '\u{1f200}',
         IdentifierType::Not_NFKC), ('\u{1f201}', '\u{1f202}',  IdentifierType::Not_NFKC),
        ('\u{1f210}', '\u{1f231}',  IdentifierType::Not_NFKC), ('\u{1f232}', '\u{1f23a}',
         IdentifierType::Not_NFKC), ('\u{1f23b}', '\u{1f23b}',  IdentifierType::Not_NFKC),
        ('\u{1f240}', '\u{1f248}',  IdentifierType::Not_NFKC), ('\u{1f250}', '\u{1f251}',
         IdentifierType::Not_NFKC), ('\u{1f260}', '\u{1f265}',  IdentifierType::Not_XID),
        ('\u{1f300}', '\u{1f320}',  IdentifierType::Not_XID), ('\u{1f321}', '\u{1f32c}',
         IdentifierType::Not_XID), ('\u{1f32d}', '\u{1f32f}',  IdentifierType::Not_XID),
        ('\u{1f330}', '\u{1f335}',  IdentifierType::Not_XID), ('\u{1f336}', '\u{1f336}',
         IdentifierType::Not_XID), ('\u{1f337}', '\u{1f37c}',  IdentifierType::Not_XID),
        ('\u{1f37d}', '\u{1f37d}',  IdentifierType::Not_XID), ('\u{1f37e}', '\u{1f37f}',
         IdentifierType::Not_XID), ('\u{1f380}', '\u{1f393}',  IdentifierType::Not_XID),
        ('\u{1f394}', '\u{1f39f}',  IdentifierType::Not_XID), ('\u{1f3a0}', '\u{1f3c4}',
         IdentifierType::Not_XID), ('\u{1f3c5}', '\u{1f3c5}',  IdentifierType::Not_XID),
        ('\u{1f3c6}', '\u{1f3ca}',  IdentifierType::Not_XID), ('\u{1f3cb}', '\u{1f3ce}',
         IdentifierType::Not_XID), ('\u{1f3cf}', '\u{1f3d3}',  IdentifierType::Not_XID),
        ('\u{1f3d4}', '\u{1f3df}',  IdentifierType::Not_XID), ('\u{1f3e0}', '\u{1f3f0}',
         IdentifierType::Not_XID), ('\u{1f3f1}', '\u{1f3f7}',  IdentifierType::Not_XID),
        ('\u{1f3f8}', '\u{1f3ff}',  IdentifierType::Not_XID), ('\u{1f400}', '\u{1f43e}',
         IdentifierType::Not_XID), ('\u{1f43f}', '\u{1f43f}',  IdentifierType::Not_XID),
        ('\u{1f440}', '\u{1f440}',  IdentifierType::Not_XID), ('\u{1f441}', '\u{1f441}',
         IdentifierType::Not_XID), ('\u{1f442}', '\u{1f4f7}',  IdentifierType::Not_XID),
        ('\u{1f4f8}', '\u{1f4f8}',  IdentifierType::Not_XID), ('\u{1f4f9}', '\u{1f4fc}',
         IdentifierType::Not_XID), ('\u{1f4fd}', '\u{1f4fe}',  IdentifierType::Not_XID),
        ('\u{1f4ff}', '\u{1f4ff}',  IdentifierType::Not_XID), ('\u{1f500}', '\u{1f53d}',
         IdentifierType::Not_XID), ('\u{1f53e}', '\u{1f53f}',  IdentifierType::Not_XID),
        ('\u{1f540}', '\u{1f543}',  IdentifierType::Not_XID), ('\u{1f544}', '\u{1f54a}',
         IdentifierType::Not_XID), ('\u{1f54b}', '\u{1f54e}',  IdentifierType::Not_XID),
        ('\u{1f54f}', '\u{1f54f}',  IdentifierType::Uncommon_Use), ('\u{1f550}', '\u{1f567}',
         IdentifierType::Not_XID), ('\u{1f568}', '\u{1f579}',  IdentifierType::Not_XID),
        ('\u{1f57a}', '\u{1f57a}',  IdentifierType::Not_XID), ('\u{1f57b}', '\u{1f5a3}',
         IdentifierType::Not_XID), ('\u{1f5a4}', '\u{1f5a4}',  IdentifierType::Not_XID),
        ('\u{1f5a5}', '\u{1f5fa}',  IdentifierType::Not_XID), ('\u{1f5fb}', '\u{1f5ff}',
         IdentifierType::Not_XID), ('\u{1f600}', '\u{1f600}',  IdentifierType::Not_XID),
        ('\u{1f601}', '\u{1f610}',  IdentifierType::Not_XID), ('\u{1f611}', '\u{1f611}',
         IdentifierType::Not_XID), ('\u{1f612}', '\u{1f614}',  IdentifierType::Not_XID),
        ('\u{1f615}', '\u{1f615}',  IdentifierType::Not_XID), ('\u{1f616}', '\u{1f616}',
         IdentifierType::Not_XID), ('\u{1f617}', '\u{1f617}',  IdentifierType::Not_XID),
        ('\u{1f618}', '\u{1f618}',  IdentifierType::Not_XID), ('\u{1f619}', '\u{1f619}',
         IdentifierType::Not_XID), ('\u{1f61a}', '\u{1f61a}',  IdentifierType::Not_XID),
        ('\u{1f61b}', '\u{1f61b}',  IdentifierType::Not_XID), ('\u{1f61c}', '\u{1f61e}',
         IdentifierType::Not_XID), ('\u{1f61f}', '\u{1f61f}',  IdentifierType::Not_XID),
        ('\u{1f620}', '\u{1f625}',  IdentifierType::Not_XID), ('\u{1f626}', '\u{1f627}',
         IdentifierType::Not_XID), ('\u{1f628}', '\u{1f62b}',  IdentifierType::Not_XID),
        ('\u{1f62c}', '\u{1f62c}',  IdentifierType::Not_XID), ('\u{1f62d}', '\u{1f62d}',
         IdentifierType::Not_XID), ('\u{1f62e}', '\u{1f62f}',  IdentifierType::Not_XID),
        ('\u{1f630}', '\u{1f633}',  IdentifierType::Not_XID), ('\u{1f634}', '\u{1f634}',
         IdentifierType::Not_XID), ('\u{1f635}', '\u{1f640}',  IdentifierType::Not_XID),
        ('\u{1f641}', '\u{1f642}',  IdentifierType::Not_XID), ('\u{1f643}', '\u{1f644}',
         IdentifierType::Not_XID), ('\u{1f645}', '\u{1f64f}',  IdentifierType::Not_XID),
        ('\u{1f650}', '\u{1f67f}',  IdentifierType::Not_XID), ('\u{1f680}', '\u{1f6c5}',
         IdentifierType::Not_XID), ('\u{1f6c6}', '\u{1f6cf}',  IdentifierType::Not_XID),
        ('\u{1f6d0}', '\u{1f6d0}',  IdentifierType::Not_XID), ('\u{1f6d1}', '\u{1f6d2}',
         IdentifierType::Not_XID), ('\u{1f6d3}', '\u{1f6d4}',  IdentifierType::Not_XID),
        ('\u{1f6d5}', '\u{1f6d5}',  IdentifierType::Not_XID), ('\u{1f6d6}', '\u{1f6d7}',
         IdentifierType::Not_XID), ('\u{1f6dc}', '\u{1f6dc}',  IdentifierType::Not_XID),
        ('\u{1f6dd}', '\u{1f6df}',  IdentifierType::Not_XID), ('\u{1f6e0}', '\u{1f6ec}',
         IdentifierType::Not_XID), ('\u{1f6f0}', '\u{1f6f3}',  IdentifierType::Not_XID),
        ('\u{1f6f4}', '\u{1f6f6}',  IdentifierType::Not_XID), ('\u{1f6f7}', '\u{1f6f8}',
         IdentifierType::Not_XID), ('\u{1f6f9}', '\u{1f6f9}',  IdentifierType::Not_XID),
        ('\u{1f6fa}', '\u{1f6fa}',  IdentifierType::Not_XID), ('\u{1f6fb}', '\u{1f6fc}',
         IdentifierType::Not_XID), ('\u{1f700}', '\u{1f773}',  IdentifierType::Not_XID),
        ('\u{1f774}', '\u{1f776}',  IdentifierType::Not_XID), ('\u{1f77b}', '\u{1f77f}',
         IdentifierType::Not_XID), ('\u{1f780}', '\u{1f7d4}',  IdentifierType::Not_XID),
        ('\u{1f7d5}', '\u{1f7d8}',  IdentifierType::Not_XID), ('\u{1f7d9}', '\u{1f7d9}',
         IdentifierType::Not_XID), ('\u{1f7e0}', '\u{1f7eb}',  IdentifierType::Not_XID),
        ('\u{1f7f0}', '\u{1f7f0}',  IdentifierType::Not_XID), ('\u{1f800}', '\u{1f80b}',
         IdentifierType::Not_XID), ('\u{1f810}', '\u{1f847}',  IdentifierType::Not_XID),
        ('\u{1f850}', '\u{1f859}',  IdentifierType::Not_XID), ('\u{1f860}', '\u{1f887}',
         IdentifierType::Not_XID), ('\u{1f890}', '\u{1f8ad}',  IdentifierType::Not_XID),
        ('\u{1f8b0}', '\u{1f8b1}',  IdentifierType::Not_XID), ('\u{1f8b2}', '\u{1f8bb}',
         IdentifierType::Not_XID), ('\u{1f8c0}', '\u{1f8c1}',  IdentifierType::Not_XID),
        ('\u{1f900}', '\u{1f90b}',  IdentifierType::Not_XID), ('\u{1f90c}', '\u{1f90c}',
         IdentifierType::Not_XID), ('\u{1f90d}', '\u{1f90f}',  IdentifierType::Not_XID),
        ('\u{1f910}', '\u{1f918}',  IdentifierType::Not_XID), ('\u{1f919}', '\u{1f91e}',
         IdentifierType::Not_XID), ('\u{1f91f}', '\u{1f91f}',  IdentifierType::Not_XID),
        ('\u{1f920}', '\u{1f927}',  IdentifierType::Not_XID), ('\u{1f928}', '\u{1f92f}',
         IdentifierType::Not_XID), ('\u{1f930}', '\u{1f930}',  IdentifierType::Not_XID),
        ('\u{1f931}', '\u{1f932}',  IdentifierType::Not_XID), ('\u{1f933}', '\u{1f93e}',
         IdentifierType::Not_XID), ('\u{1f93f}', '\u{1f93f}',  IdentifierType::Not_XID),
        ('\u{1f940}', '\u{1f94b}',  IdentifierType::Not_XID), ('\u{1f94c}', '\u{1f94c}',
         IdentifierType::Not_XID), ('\u{1f94d}', '\u{1f94f}',  IdentifierType::Not_XID),
        ('\u{1f950}', '\u{1f95e}',  IdentifierType::Not_XID), ('\u{1f95f}', '\u{1f96b}',
         IdentifierType::Not_XID), ('\u{1f96c}', '\u{1f970}',  IdentifierType::Not_XID),
        ('\u{1f971}', '\u{1f971}',  IdentifierType::Not_XID), ('\u{1f972}', '\u{1f972}',
         IdentifierType::Not_XID), ('\u{1f973}', '\u{1f976}',  IdentifierType::Not_XID),
        ('\u{1f977}', '\u{1f978}',  IdentifierType::Not_XID), ('\u{1f979}', '\u{1f979}',
         IdentifierType::Not_XID), ('\u{1f97a}', '\u{1f97a}',  IdentifierType::Not_XID),
        ('\u{1f97b}', '\u{1f97b}',  IdentifierType::Not_XID), ('\u{1f97c}', '\u{1f97f}',
         IdentifierType::Not_XID), ('\u{1f980}', '\u{1f984}',  IdentifierType::Not_XID),
        ('\u{1f985}', '\u{1f991}',  IdentifierType::Not_XID), ('\u{1f992}', '\u{1f997}',
         IdentifierType::Not_XID), ('\u{1f998}', '\u{1f9a2}',  IdentifierType::Not_XID),
        ('\u{1f9a3}', '\u{1f9a4}',  IdentifierType::Not_XID), ('\u{1f9a5}', '\u{1f9aa}',
         IdentifierType::Not_XID), ('\u{1f9ab}', '\u{1f9ad}',  IdentifierType::Not_XID),
        ('\u{1f9ae}', '\u{1f9af}',  IdentifierType::Not_XID), ('\u{1f9b0}', '\u{1f9b9}',
         IdentifierType::Not_XID), ('\u{1f9ba}', '\u{1f9bf}',  IdentifierType::Not_XID),
        ('\u{1f9c0}', '\u{1f9c0}',  IdentifierType::Not_XID), ('\u{1f9c1}', '\u{1f9c2}',
         IdentifierType::Not_XID), ('\u{1f9c3}', '\u{1f9ca}',  IdentifierType::Not_XID),
        ('\u{1f9cb}', '\u{1f9cb}',  IdentifierType::Not_XID), ('\u{1f9cc}', '\u{1f9cc}',
         IdentifierType::Not_XID), ('\u{1f9cd}', '\u{1f9cf}',  IdentifierType::Not_XID),
        ('\u{1f9d0}', '\u{1f9e6}',  IdentifierType::Not_XID), ('\u{1f9e7}', '\u{1f9ff}',
         IdentifierType::Not_XID), ('\u{1fa00}', '\u{1fa53}',  IdentifierType::Not_XID),
        ('\u{1fa60}', '\u{1fa6d}',  IdentifierType::Not_XID), ('\u{1fa70}', '\u{1fa73}',
         IdentifierType::Not_XID), ('\u{1fa74}', '\u{1fa74}',  IdentifierType::Not_XID),
        ('\u{1fa75}', '\u{1fa77}',  IdentifierType::Not_XID), ('\u{1fa78}', '\u{1fa7a}',
         IdentifierType::Not_XID), ('\u{1fa7b}', '\u{1fa7c}',  IdentifierType::Not_XID),
        ('\u{1fa80}', '\u{1fa82}',  IdentifierType::Not_XID), ('\u{1fa83}', '\u{1fa86}',
         IdentifierType::Not_XID), ('\u{1fa87}', '\u{1fa88}',  IdentifierType::Not_XID),
        ('\u{1fa89}', '\u{1fa89}',  IdentifierType::Not_XID), ('\u{1fa8f}', '\u{1fa8f}',
         IdentifierType::Not_XID), ('\u{1fa90}', '\u{1fa95}',  IdentifierType::Not_XID),
        ('\u{1fa96}', '\u{1faa8}',  IdentifierType::Not_XID), ('\u{1faa9}', '\u{1faac}',
         IdentifierType::Not_XID), ('\u{1faad}', '\u{1faaf}',  IdentifierType::Not_XID),
        ('\u{1fab0}', '\u{1fab6}',  IdentifierType::Not_XID), ('\u{1fab7}', '\u{1faba}',
         IdentifierType::Not_XID), ('\u{1fabb}', '\u{1fabd}',  IdentifierType::Not_XID),
        ('\u{1fabe}', '\u{1fabe}',  IdentifierType::Not_XID), ('\u{1fabf}', '\u{1fabf}',
         IdentifierType::Not_XID), ('\u{1fac0}', '\u{1fac2}',  IdentifierType::Not_XID),
        ('\u{1fac3}', '\u{1fac5}',  IdentifierType::Not_XID), ('\u{1fac6}', '\u{1fac6}',
         IdentifierType::Not_XID), ('\u{1face}', '\u{1facf}',  IdentifierType::Not_XID),
        ('\u{1fad0}', '\u{1fad6}',  IdentifierType::Not_XID), ('\u{1fad7}', '\u{1fad9}',
         IdentifierType::Not_XID), ('\u{1fada}', '\u{1fadb}',  IdentifierType::Not_XID),
        ('\u{1fadc}', '\u{1fadc}',  IdentifierType::Not_XID), ('\u{1fadf}', '\u{1fadf}',
         IdentifierType::Not_XID), ('\u{1fae0}', '\u{1fae7}',  IdentifierType::Not_XID),
        ('\u{1fae8}', '\u{1fae8}',  IdentifierType::Not_XID), ('\u{1fae9}', '\u{1fae9}',
         IdentifierType::Not_XID), ('\u{1faf0}', '\u{1faf6}',  IdentifierType::Not_XID),
        ('\u{1faf7}', '\u{1faf8}',  IdentifierType::Not_XID), ('\u{1fb00}', '\u{1fb92}',
         IdentifierType::Not_XID), ('\u{1fb94}', '\u{1fbca}',  IdentifierType::Not_XID),
        ('\u{1fbcb}', '\u{1fbef}',  IdentifierType::Not_XID), ('\u{1fbf0}', '\u{1fbf9}',
         IdentifierType::Not_NFKC), ('\u{20000}', '\u{2a6d6}',  IdentifierType::Recommended),
        ('\u{2a6d7}', '\u{2a6dd}',  IdentifierType::Recommended), ('\u{2a6de}', '\u{2a6df}',
         IdentifierType::Recommended), ('\u{2a700}', '\u{2b734}',  IdentifierType::Recommended),
        ('\u{2b735}', '\u{2b738}',  IdentifierType::Recommended), ('\u{2b739}', '\u{2b739}',
         IdentifierType::Recommended), ('\u{2b740}', '\u{2b81d}',  IdentifierType::Recommended),
        ('\u{2b820}', '\u{2cea1}',  IdentifierType::Recommended), ('\u{2ceb0}', '\u{2ebe0}',
         IdentifierType::Recommended), ('\u{2ebf0}', '\u{2ee5d}',  IdentifierType::Recommended),
        ('\u{2f800}', '\u{2fa1d}',  IdentifierType::Not_NFKC), ('\u{30000}', '\u{3134a}',
         IdentifierType::Recommended), ('\u{31350}', '\u{323af}',  IdentifierType::Recommended),
        ('\u{e0001}', '\u{e0001}',  IdentifierType::Deprecated), ('\u{e0020}', '\u{e007f}',
         IdentifierType::Default_Ignorable), ('\u{e0100}', '\u{e01ef}',
         IdentifierType::Default_Ignorable)
    ];

}

pub mod confusable_detection {

    #[inline]
    pub fn char_confusable_prototype(c: char) -> Option<&'static [char]> {
        // FIXME: do we want to special case ASCII here?
        match c as usize {
            _ => super::util::bsearch_value_table(c, CONFUSABLES)
        }
    }

    // Confusable table:
    const CONFUSABLES: &'static [(char, &'static [char])] = &[
        ('\u{22}',  &['\u{27}',  '\u{27}']), ('\u{25}',  &['\u{ba}',  '\u{2f}',  '\u{2080}']),
        ('\u{30}',  &['\u{4f}']), ('\u{31}',  &['\u{6c}']), ('\u{49}',  &['\u{6c}']), ('\u{60}',
         &['\u{27}']), ('\u{6d}',  &['\u{72}',  '\u{6e}']), ('\u{7c}',  &['\u{6c}']), ('\u{a0}',
         &['\u{20}']), ('\u{a2}',  &['\u{63}',  '\u{338}']), ('\u{a5}',  &['\u{59}',  '\u{335}']),
        ('\u{af}',  &['\u{2c9}']), ('\u{b4}',  &['\u{27}']), ('\u{b5}',  &['\u{3bc}']), ('\u{b8}',
         &['\u{2c}']), ('\u{c6}',  &['\u{41}',  '\u{45}']), ('\u{c7}',  &['\u{43}',  '\u{326}']),
        ('\u{d0}',  &['\u{44}',  '\u{335}']), ('\u{d7}',  &['\u{78}']), ('\u{d8}',  &['\u{4f}',
         '\u{338}']), ('\u{e6}',  &['\u{61}',  '\u{65}']), ('\u{e7}',  &['\u{63}',  '\u{326}']),
        ('\u{f0}',  &['\u{2202}',  '\u{335}']), ('\u{f6}',  &['\u{629}']), ('\u{f8}',  &['\u{6f}',
         '\u{338}']), ('\u{110}',  &['\u{44}',  '\u{335}']), ('\u{111}',  &['\u{64}',  '\u{335}']),
        ('\u{11a}',  &['\u{114}']), ('\u{11b}',  &['\u{115}']), ('\u{126}',  &['\u{48}',
         '\u{335}']), ('\u{127}',  &['\u{68}',  '\u{335}']), ('\u{131}',  &['\u{69}']), ('\u{132}',
         &['\u{6c}',  '\u{4a}']), ('\u{133}',  &['\u{69}',  '\u{6a}']), ('\u{13f}',  &['\u{6c}',
         '\u{b7}']), ('\u{140}',  &['\u{6c}',  '\u{b7}']), ('\u{141}',  &['\u{4c}',  '\u{338}']),
        ('\u{142}',  &['\u{6c}',  '\u{338}']), ('\u{146}',  &['\u{272}']), ('\u{149}',  &['\u{27}',
         '\u{6e}']), ('\u{150}',  &['\u{d6}']), ('\u{152}',  &['\u{4f}',  '\u{45}']), ('\u{153}',
         &['\u{6f}',  '\u{65}']), ('\u{163}',  &['\u{1ab}']), ('\u{166}',  &['\u{54}',  '\u{335}']),
        ('\u{167}',  &['\u{74}',  '\u{335}']), ('\u{17f}',  &['\u{66}']), ('\u{180}',  &['\u{62}',
         '\u{335}']), ('\u{181}',  &['\u{27}',  '\u{42}']), ('\u{182}',  &['\u{62}',  '\u{304}']),
        ('\u{183}',  &['\u{62}',  '\u{304}']), ('\u{184}',  &['\u{62}']), ('\u{187}',  &['\u{43}',
         '\u{27}']), ('\u{189}',  &['\u{44}',  '\u{335}']), ('\u{18a}',  &['\u{27}',  '\u{44}']),
        ('\u{18c}',  &['\u{64}',  '\u{304}']), ('\u{18d}',  &['\u{67}']), ('\u{191}',  &['\u{46}',
         '\u{326}']), ('\u{192}',  &['\u{66}',  '\u{326}']), ('\u{193}',  &['\u{47}',  '\u{27}']),
        ('\u{196}',  &['\u{6c}']), ('\u{197}',  &['\u{6c}',  '\u{335}']), ('\u{198}',  &['\u{4b}',
         '\u{27}']), ('\u{199}',  &['\u{6b}',  '\u{314}']), ('\u{19a}',  &['\u{6c}',  '\u{335}']),
        ('\u{19b}',  &['\u{3bb}',  '\u{338}']), ('\u{19d}',  &['\u{4e}',  '\u{326}']), ('\u{19e}',
         &['\u{6e}',  '\u{329}']), ('\u{19f}',  &['\u{4f}',  '\u{335}']), ('\u{1a0}',  &['\u{4f}',
         '\u{27}']), ('\u{1a1}',  &['\u{6f}',  '\u{27}']), ('\u{1a4}',  &['\u{27}',  '\u{50}']),
        ('\u{1a5}',  &['\u{70}',  '\u{314}']), ('\u{1a6}',  &['\u{52}']), ('\u{1a7}',  &['\u{32}']),
        ('\u{1ac}',  &['\u{27}',  '\u{54}']), ('\u{1ad}',  &['\u{74}',  '\u{314}']), ('\u{1ae}',
         &['\u{54}',  '\u{328}']), ('\u{1b3}',  &['\u{27}',  '\u{59}']), ('\u{1b4}',  &['\u{79}',
         '\u{314}']), ('\u{1b5}',  &['\u{5a}',  '\u{335}']), ('\u{1b6}',  &['\u{7a}',  '\u{335}']),
        ('\u{1b7}',  &['\u{33}']), ('\u{1bb}',  &['\u{32}',  '\u{335}']), ('\u{1bc}',  &['\u{35}']),
        ('\u{1bd}',  &['\u{73}']), ('\u{1bf}',  &['\u{fe}']), ('\u{1c0}',  &['\u{6c}']), ('\u{1c1}',
         &['\u{6c}',  '\u{6c}']), ('\u{1c3}',  &['\u{21}']), ('\u{1c4}',  &['\u{44}',  '\u{17d}']),
        ('\u{1c5}',  &['\u{44}',  '\u{17e}']), ('\u{1c6}',  &['\u{64}',  '\u{17e}']), ('\u{1c7}',
         &['\u{4c}',  '\u{4a}']), ('\u{1c8}',  &['\u{4c}',  '\u{6a}']), ('\u{1c9}',  &['\u{6c}',
         '\u{6a}']), ('\u{1ca}',  &['\u{4e}',  '\u{4a}']), ('\u{1cb}',  &['\u{4e}',  '\u{6a}']),
        ('\u{1cc}',  &['\u{6e}',  '\u{6a}']), ('\u{1cd}',  &['\u{102}']), ('\u{1ce}',
         &['\u{103}']), ('\u{1cf}',  &['\u{12c}']), ('\u{1d0}',  &['\u{12d}']), ('\u{1d1}',
         &['\u{14e}']), ('\u{1d2}',  &['\u{14f}']), ('\u{1d3}',  &['\u{16c}']), ('\u{1d4}',
         &['\u{16d}']), ('\u{1e4}',  &['\u{47}',  '\u{335}']), ('\u{1e5}',  &['\u{67}',
         '\u{335}']), ('\u{1e6}',  &['\u{11e}']), ('\u{1e7}',  &['\u{11f}']), ('\u{1f1}',
         &['\u{44}',  '\u{5a}']), ('\u{1f2}',  &['\u{44}',  '\u{7a}']), ('\u{1f3}',  &['\u{64}',
         '\u{7a}']), ('\u{1f5}',  &['\u{123}']), ('\u{1fe}',  &['\u{4f}',  '\u{338}',  '\u{301}']),
        ('\u{21a}',  &['\u{162}']), ('\u{21b}',  &['\u{1ab}']), ('\u{21c}',  &['\u{33}']),
        ('\u{222}',  &['\u{38}']), ('\u{223}',  &['\u{38}']), ('\u{224}',  &['\u{5a}',  '\u{326}']),
        ('\u{225}',  &['\u{7a}',  '\u{326}']), ('\u{226}',  &['\u{c5}']), ('\u{227}',  &['\u{e5}']),
        ('\u{23c}',  &['\u{63}',  '\u{338}']), ('\u{23e}',  &['\u{54}',  '\u{338}']), ('\u{241}',
         &['\u{3f}']), ('\u{244}',  &['\u{55}',  '\u{335}']), ('\u{246}',  &['\u{45}',  '\u{338}']),
        ('\u{247}',  &['\u{65}',  '\u{338}']), ('\u{248}',  &['\u{4a}',  '\u{335}']), ('\u{249}',
         &['\u{6a}',  '\u{335}']), ('\u{24d}',  &['\u{72}',  '\u{335}']), ('\u{24e}',  &['\u{59}',
         '\u{335}']), ('\u{24f}',  &['\u{79}',  '\u{335}']), ('\u{251}',  &['\u{61}']), ('\u{253}',
         &['\u{62}',  '\u{314}']), ('\u{256}',  &['\u{64}',  '\u{328}']), ('\u{257}',  &['\u{64}',
         '\u{314}']), ('\u{259}',  &['\u{1dd}']), ('\u{25a}',  &['\u{1dd}',  '\u{2de}']),
        ('\u{25b}',  &['\u{a793}']), ('\u{260}',  &['\u{67}',  '\u{314}']), ('\u{261}',
         &['\u{67}']), ('\u{263}',  &['\u{79}']), ('\u{266}',  &['\u{68}',  '\u{314}']), ('\u{268}',
         &['\u{69}',  '\u{335}']), ('\u{269}',  &['\u{69}']), ('\u{26a}',  &['\u{69}']), ('\u{26b}',
         &['\u{6c}',  '\u{334}']), ('\u{26d}',  &['\u{6c}',  '\u{328}']), ('\u{26e}',  &['\u{6c}',
         '\u{21d}']), ('\u{26f}',  &['\u{77}']), ('\u{271}',  &['\u{72}',  '\u{6e}',  '\u{326}']),
        ('\u{273}',  &['\u{6e}',  '\u{328}']), ('\u{275}',  &['\u{6f}',  '\u{335}']), ('\u{276}',
         &['\u{6f}',  '\u{1d07}']), ('\u{27c}',  &['\u{72}',  '\u{329}']), ('\u{27d}',  &['\u{72}',
         '\u{328}']), ('\u{282}',  &['\u{73}',  '\u{328}']), ('\u{28b}',  &['\u{75}']), ('\u{28f}',
         &['\u{79}']), ('\u{290}',  &['\u{7a}',  '\u{328}']), ('\u{292}',  &['\u{21d}']),
        ('\u{294}',  &['\u{3f}']), ('\u{2a0}',  &['\u{71}',  '\u{314}']), ('\u{2a3}',  &['\u{64}',
         '\u{7a}']), ('\u{2a4}',  &['\u{64}',  '\u{21d}']), ('\u{2a5}',  &['\u{64}',  '\u{291}']),
        ('\u{2a6}',  &['\u{74}',  '\u{73}']), ('\u{2a7}',  &['\u{74}',  '\u{283}']), ('\u{2a8}',
         &['\u{74}',  '\u{255}']), ('\u{2a9}',  &['\u{66}',  '\u{14b}']), ('\u{2aa}',  &['\u{6c}',
         '\u{73}']), ('\u{2ab}',  &['\u{6c}',  '\u{7a}']), ('\u{2b3}',  &['\u{18f4}']), ('\u{2b9}',
         &['\u{27}']), ('\u{2ba}',  &['\u{27}',  '\u{27}']), ('\u{2bb}',  &['\u{27}']), ('\u{2bc}',
         &['\u{27}']), ('\u{2bd}',  &['\u{27}']), ('\u{2be}',  &['\u{27}']), ('\u{2bf}',
         &['\u{559}']), ('\u{2c2}',  &['\u{3c}']), ('\u{2c3}',  &['\u{3e}']), ('\u{2c4}',
         &['\u{5e}']), ('\u{2c6}',  &['\u{5e}']), ('\u{2c8}',  &['\u{27}']), ('\u{2ca}',
         &['\u{27}']), ('\u{2cb}',  &['\u{27}']), ('\u{2d0}',  &['\u{3a}']), ('\u{2d3}',
         &['\u{559}']), ('\u{2d7}',  &['\u{2d}']), ('\u{2d8}',  &['\u{2c7}']), ('\u{2d9}',
         &['\u{971}']), ('\u{2da}',  &['\u{b0}']), ('\u{2db}',  &['\u{69}']), ('\u{2dc}',
         &['\u{7e}']), ('\u{2dd}',  &['\u{27}',  '\u{27}']), ('\u{2e1}',  &['\u{18f3}']),
        ('\u{2e2}',  &['\u{18f5}']), ('\u{2e4}',  &['\u{2c1}']), ('\u{2ee}',  &['\u{27}',
         '\u{27}']), ('\u{2f4}',  &['\u{27}']), ('\u{2f6}',  &['\u{27}',  '\u{27}']), ('\u{2f8}',
         &['\u{3a}']), ('\u{2fb}',  &['\u{2ea}']), ('\u{305}',  &['\u{304}']), ('\u{30c}',
         &['\u{306}']), ('\u{30d}',  &['\u{670}']), ('\u{310}',  &['\u{306}',  '\u{307}']),
        ('\u{311}',  &['\u{302}']), ('\u{315}',  &['\u{313}']), ('\u{317}',  &['\u{650}']),
        ('\u{320}',  &['\u{331}']), ('\u{321}',  &['\u{326}']), ('\u{322}',  &['\u{328}']),
        ('\u{327}',  &['\u{326}']), ('\u{336}',  &['\u{335}']), ('\u{337}',  &['\u{338}']),
        ('\u{339}',  &['\u{326}']), ('\u{340}',  &['\u{300}']), ('\u{341}',  &['\u{301}']),
        ('\u{342}',  &['\u{303}']), ('\u{343}',  &['\u{313}']), ('\u{345}',  &['\u{328}']),
        ('\u{347}',  &['\u{333}']), ('\u{357}',  &['\u{350}']), ('\u{358}',  &['\u{307}']),
        ('\u{366}',  &['\u{30a}']), ('\u{36e}',  &['\u{306}']), ('\u{370}',  &['\u{2c75}']),
        ('\u{374}',  &['\u{27}']), ('\u{375}',  &['\u{2cf}']), ('\u{376}',  &['\u{418}']),
        ('\u{377}',  &['\u{1d0e}']), ('\u{37a}',  &['\u{69}']), ('\u{37b}',  &['\u{254}']),
        ('\u{37d}',  &['\u{a73f}']), ('\u{37e}',  &['\u{3b}']), ('\u{37f}',  &['\u{4a}']),
        ('\u{384}',  &['\u{27}']), ('\u{387}',  &['\u{b7}']), ('\u{391}',  &['\u{41}']), ('\u{392}',
         &['\u{42}']), ('\u{395}',  &['\u{45}']), ('\u{396}',  &['\u{5a}']), ('\u{397}',
         &['\u{48}']), ('\u{398}',  &['\u{4f}',  '\u{335}']), ('\u{399}',  &['\u{6c}']), ('\u{39a}',
         &['\u{4b}']), ('\u{39b}',  &['\u{245}']), ('\u{39c}',  &['\u{4d}']), ('\u{39d}',
         &['\u{4e}']), ('\u{39f}',  &['\u{4f}']), ('\u{3a1}',  &['\u{50}']), ('\u{3a3}',
         &['\u{1a9}']), ('\u{3a4}',  &['\u{54}']), ('\u{3a5}',  &['\u{59}']), ('\u{3a7}',
         &['\u{58}']), ('\u{3b1}',  &['\u{61}']), ('\u{3b2}',  &['\u{df}']), ('\u{3b3}',
         &['\u{79}']), ('\u{3b4}',  &['\u{1e9f}']), ('\u{3b5}',  &['\u{a793}']), ('\u{3b7}',
         &['\u{6e}',  '\u{329}']), ('\u{3b8}',  &['\u{4f}',  '\u{335}']), ('\u{3b9}',  &['\u{69}']),
        ('\u{3ba}',  &['\u{138}']), ('\u{3bd}',  &['\u{76}']), ('\u{3bf}',  &['\u{6f}']),
        ('\u{3c1}',  &['\u{70}']), ('\u{3c3}',  &['\u{6f}']), ('\u{3c4}',  &['\u{1d1b}']),
        ('\u{3c5}',  &['\u{75}']), ('\u{3c6}',  &['\u{278}']), ('\u{3d0}',  &['\u{df}']),
        ('\u{3d1}',  &['\u{4f}',  '\u{335}']), ('\u{3d2}',  &['\u{59}']), ('\u{3d5}',
         &['\u{278}']), ('\u{3d6}',  &['\u{3c0}']), ('\u{3db}',  &['\u{3c2}']), ('\u{3dc}',
         &['\u{46}']), ('\u{3e8}',  &['\u{32}']), ('\u{3e9}',  &['\u{1a8}']), ('\u{3f0}',
         &['\u{138}']), ('\u{3f1}',  &['\u{70}']), ('\u{3f2}',  &['\u{63}']), ('\u{3f3}',
         &['\u{6a}']), ('\u{3f4}',  &['\u{4f}',  '\u{335}']), ('\u{3f5}',  &['\u{a793}']),
        ('\u{3f7}',  &['\u{de}']), ('\u{3f8}',  &['\u{fe}']), ('\u{3f9}',  &['\u{43}']), ('\u{3fa}',
         &['\u{4d}']), ('\u{3fd}',  &['\u{186}']), ('\u{3ff}',  &['\u{a73e}']), ('\u{404}',
         &['\u{a792}']), ('\u{405}',  &['\u{53}']), ('\u{406}',  &['\u{6c}']), ('\u{408}',
         &['\u{4a}']), ('\u{410}',  &['\u{41}']), ('\u{411}',  &['\u{62}',  '\u{304}']), ('\u{412}',
         &['\u{42}']), ('\u{413}',  &['\u{393}']), ('\u{415}',  &['\u{45}']), ('\u{417}',
         &['\u{33}']), ('\u{419}',  &['\u{40d}']), ('\u{41a}',  &['\u{4b}']), ('\u{41b}',
         &['\u{245}']), ('\u{41c}',  &['\u{4d}']), ('\u{41d}',  &['\u{48}']), ('\u{41e}',
         &['\u{4f}']), ('\u{41f}',  &['\u{3a0}']), ('\u{420}',  &['\u{50}']), ('\u{421}',
         &['\u{43}']), ('\u{422}',  &['\u{54}']), ('\u{423}',  &['\u{59}']), ('\u{424}',
         &['\u{3a6}']), ('\u{425}',  &['\u{58}']), ('\u{42b}',  &['\u{62}',  '\u{6c}']), ('\u{42c}',
         &['\u{62}']), ('\u{42e}',  &['\u{6c}',  '\u{4f}']), ('\u{430}',  &['\u{61}']), ('\u{431}',
         &['\u{36}']), ('\u{432}',  &['\u{299}']), ('\u{433}',  &['\u{72}']), ('\u{435}',
         &['\u{65}']), ('\u{437}',  &['\u{25c}']), ('\u{438}',  &['\u{1d0e}']), ('\u{43a}',
         &['\u{138}']), ('\u{43c}',  &['\u{28d}']), ('\u{43d}',  &['\u{29c}']), ('\u{43e}',
         &['\u{6f}']), ('\u{43f}',  &['\u{3c0}']), ('\u{440}',  &['\u{70}']), ('\u{441}',
         &['\u{63}']), ('\u{442}',  &['\u{1d1b}']), ('\u{443}',  &['\u{79}']), ('\u{444}',
         &['\u{278}']), ('\u{445}',  &['\u{78}']), ('\u{44a}',  &['\u{2c9}',  '\u{62}']),
        ('\u{44b}',  &['\u{185}',  '\u{69}']), ('\u{44c}',  &['\u{185}']), ('\u{44f}',
         &['\u{1d19}']), ('\u{454}',  &['\u{a793}']), ('\u{455}',  &['\u{73}']), ('\u{456}',
         &['\u{69}']), ('\u{458}',  &['\u{6a}']), ('\u{45b}',  &['\u{68}',  '\u{335}']), ('\u{45d}',
         &['\u{439}']), ('\u{461}',  &['\u{77}']), ('\u{462}',  &['\u{62}',  '\u{335}']),
        ('\u{463}',  &['\u{62}',  '\u{335}']), ('\u{470}',  &['\u{3a8}']), ('\u{471}',
         &['\u{3c8}']), ('\u{472}',  &['\u{4f}',  '\u{335}']), ('\u{473}',  &['\u{6f}',
         '\u{335}']), ('\u{474}',  &['\u{56}']), ('\u{475}',  &['\u{76}']), ('\u{47c}',
         &['\u{460}',  '\u{486}',  '\u{487}']), ('\u{47d}',  &['\u{77}',  '\u{486}',  '\u{487}']),
        ('\u{48a}',  &['\u{40d}',  '\u{326}']), ('\u{48b}',  &['\u{439}',  '\u{326}']), ('\u{48c}',
         &['\u{62}',  '\u{335}']), ('\u{48d}',  &['\u{62}',  '\u{335}']), ('\u{490}',  &['\u{393}',
         '\u{27}']), ('\u{491}',  &['\u{72}',  '\u{27}']), ('\u{492}',  &['\u{393}',  '\u{335}']),
        ('\u{493}',  &['\u{72}',  '\u{335}']), ('\u{496}',  &['\u{416}',  '\u{329}']), ('\u{497}',
         &['\u{436}',  '\u{329}']), ('\u{498}',  &['\u{33}',  '\u{326}']), ('\u{499}',  &['\u{25c}',
         '\u{326}']), ('\u{49a}',  &['\u{4b}',  '\u{329}']), ('\u{49b}',  &['\u{138}',  '\u{329}']),
        ('\u{49e}',  &['\u{4b}',  '\u{335}']), ('\u{49f}',  &['\u{138}',  '\u{335}']), ('\u{4a2}',
         &['\u{48}',  '\u{329}']), ('\u{4a3}',  &['\u{29c}',  '\u{329}']), ('\u{4aa}',  &['\u{43}',
         '\u{326}']), ('\u{4ab}',  &['\u{63}',  '\u{326}']), ('\u{4ac}',  &['\u{54}',  '\u{329}']),
        ('\u{4ad}',  &['\u{1d1b}',  '\u{329}']), ('\u{4ae}',  &['\u{59}']), ('\u{4af}',
         &['\u{79}']), ('\u{4b0}',  &['\u{59}',  '\u{335}']), ('\u{4b1}',  &['\u{79}',  '\u{335}']),
        ('\u{4b2}',  &['\u{58}',  '\u{329}']), ('\u{4bb}',  &['\u{68}']), ('\u{4bd}',  &['\u{65}']),
        ('\u{4be}',  &['\u{4bc}',  '\u{328}']), ('\u{4bf}',  &['\u{65}',  '\u{328}']), ('\u{4c0}',
         &['\u{6c}']), ('\u{4c5}',  &['\u{245}',  '\u{326}']), ('\u{4c6}',  &['\u{43b}',
         '\u{326}']), ('\u{4c7}',  &['\u{48}',  '\u{326}']), ('\u{4c8}',  &['\u{29c}',  '\u{326}']),
        ('\u{4c9}',  &['\u{48}',  '\u{326}']), ('\u{4ca}',  &['\u{29c}',  '\u{326}']), ('\u{4cb}',
         &['\u{4b6}']), ('\u{4cc}',  &['\u{4b7}']), ('\u{4cd}',  &['\u{4d}',  '\u{326}']),
        ('\u{4ce}',  &['\u{28d}',  '\u{326}']), ('\u{4cf}',  &['\u{69}']), ('\u{4d4}',  &['\u{41}',
         '\u{45}']), ('\u{4d5}',  &['\u{61}',  '\u{65}']), ('\u{4d8}',  &['\u{18f}']), ('\u{4d9}',
         &['\u{1dd}']), ('\u{4e0}',  &['\u{33}']), ('\u{4e1}',  &['\u{21d}']), ('\u{4e8}',
         &['\u{4f}',  '\u{335}']), ('\u{4e9}',  &['\u{6f}',  '\u{335}']), ('\u{501}',  &['\u{64}']),
        ('\u{50a}',  &['\u{1f6}']), ('\u{50c}',  &['\u{47}']), ('\u{50d}',  &['\u{262}']),
        ('\u{510}',  &['\u{190}']), ('\u{511}',  &['\u{a793}']), ('\u{51b}',  &['\u{71}']),
        ('\u{51c}',  &['\u{57}']), ('\u{51d}',  &['\u{77}']), ('\u{53b}',  &['\u{12ae}']),
        ('\u{544}',  &['\u{1206}']), ('\u{54a}',  &['\u{1323}']), ('\u{54c}',  &['\u{1261}']),
        ('\u{54d}',  &['\u{55}']), ('\u{54f}',  &['\u{53}']), ('\u{553}',  &['\u{3a6}']),
        ('\u{555}',  &['\u{4f}']), ('\u{55a}',  &['\u{27}']), ('\u{55d}',  &['\u{27}']), ('\u{561}',
         &['\u{77}']), ('\u{563}',  &['\u{71}']), ('\u{566}',  &['\u{71}']), ('\u{56e}',
         &['\u{1e9f}']), ('\u{570}',  &['\u{68}']), ('\u{575}',  &['\u{237}']), ('\u{578}',
         &['\u{6e}']), ('\u{57a}',  &['\u{270}']), ('\u{57c}',  &['\u{6e}']), ('\u{57d}',
         &['\u{75}']), ('\u{581}',  &['\u{67}']), ('\u{584}',  &['\u{66}']), ('\u{585}',
         &['\u{6f}']), ('\u{587}',  &['\u{565}',  '\u{582}']), ('\u{589}',  &['\u{3a}']),
        ('\u{59c}',  &['\u{301}']), ('\u{59d}',  &['\u{301}']), ('\u{5a4}',  &['\u{59a}']),
        ('\u{5a8}',  &['\u{599}']), ('\u{5ad}',  &['\u{596}']), ('\u{5ae}',  &['\u{598}']),
        ('\u{5af}',  &['\u{30a}']), ('\u{5b4}',  &['\u{323}']), ('\u{5b9}',  &['\u{307}']),
        ('\u{5ba}',  &['\u{307}']), ('\u{5c0}',  &['\u{6c}']), ('\u{5c1}',  &['\u{307}']),
        ('\u{5c2}',  &['\u{307}']), ('\u{5c3}',  &['\u{3a}']), ('\u{5c4}',  &['\u{307}']),
        ('\u{5c5}',  &['\u{323}']), ('\u{5d5}',  &['\u{6c}']), ('\u{5d8}',  &['\u{76}']),
        ('\u{5d9}',  &['\u{27}']), ('\u{5df}',  &['\u{6c}']), ('\u{5e1}',  &['\u{6f}']), ('\u{5f0}',
         &['\u{6c}',  '\u{6c}']), ('\u{5f1}',  &['\u{6c}',  '\u{27}']), ('\u{5f2}',  &['\u{27}',
         '\u{27}']), ('\u{5f3}',  &['\u{27}']), ('\u{5f4}',  &['\u{27}',  '\u{27}']), ('\u{609}',
         &['\u{ba}',  '\u{2f}',  '\u{2080}',  '\u{2080}']), ('\u{60a}',  &['\u{ba}',  '\u{2f}',
         '\u{2080}',  '\u{2080}',  '\u{2080}']), ('\u{60d}',  &['\u{2c}']), ('\u{60f}',
         &['\u{639}']), ('\u{618}',  &['\u{301}']), ('\u{619}',  &['\u{313}']), ('\u{61a}',
         &['\u{650}']), ('\u{623}',  &['\u{6c}',  '\u{674}']), ('\u{624}',  &['\u{648}',
         '\u{674}']), ('\u{625}',  &['\u{6c}',  '\u{655}']), ('\u{626}',  &['\u{649}',  '\u{674}']),
        ('\u{627}',  &['\u{6c}']), ('\u{62b}',  &['\u{649}',  '\u{6db}']), ('\u{634}',  &['\u{633}',
         '\u{6db}']), ('\u{63d}',  &['\u{649}',  '\u{302}']), ('\u{63f}',  &['\u{649}',
         '\u{6db}']), ('\u{647}',  &['\u{6f}']), ('\u{64a}',  &['\u{649}']), ('\u{64b}',
         &['\u{30b}']), ('\u{64e}',  &['\u{301}']), ('\u{64f}',  &['\u{313}']), ('\u{652}',
         &['\u{30a}']), ('\u{653}',  &['\u{303}']), ('\u{656}',  &['\u{329}']), ('\u{657}',
         &['\u{312}']), ('\u{658}',  &['\u{306}']), ('\u{659}',  &['\u{304}']), ('\u{65a}',
         &['\u{306}']), ('\u{65b}',  &['\u{302}']), ('\u{65c}',  &['\u{323}']), ('\u{65d}',
         &['\u{314}']), ('\u{65f}',  &['\u{655}']), ('\u{660}',  &['\u{2e}']), ('\u{661}',
         &['\u{6c}']), ('\u{665}',  &['\u{6f}']), ('\u{667}',  &['\u{56}']), ('\u{668}',
         &['\u{245}']), ('\u{66a}',  &['\u{ba}',  '\u{2f}',  '\u{2080}']), ('\u{66b}',
         &['\u{2c}']), ('\u{66c}',  &['\u{60c}']), ('\u{66d}',  &['\u{2a}']), ('\u{66e}',
         &['\u{649}']), ('\u{66f}',  &['\u{6a1}']), ('\u{672}',  &['\u{6c}',  '\u{674}']),
        ('\u{673}',  &['\u{6c}',  '\u{655}']), ('\u{675}',  &['\u{6c}',  '\u{674}']), ('\u{676}',
         &['\u{648}',  '\u{674}']), ('\u{677}',  &['\u{648}',  '\u{313}',  '\u{674}']), ('\u{678}',
         &['\u{649}',  '\u{674}']), ('\u{679}',  &['\u{649}',  '\u{615}']), ('\u{67e}',
         &['\u{649}',  '\u{6db}']), ('\u{681}',  &['\u{62d}',  '\u{654}']), ('\u{685}',
         &['\u{62d}',  '\u{6db}']), ('\u{688}',  &['\u{62f}',  '\u{615}']), ('\u{68b}',
         &['\u{68a}',  '\u{615}']), ('\u{68e}',  &['\u{62f}',  '\u{6db}']), ('\u{691}',
         &['\u{631}',  '\u{615}']), ('\u{692}',  &['\u{631}',  '\u{306}']), ('\u{698}',
         &['\u{631}',  '\u{6db}']), ('\u{69e}',  &['\u{635}',  '\u{6db}']), ('\u{69f}',
         &['\u{637}',  '\u{6db}']), ('\u{6a4}',  &['\u{6a1}',  '\u{6db}']), ('\u{6a7}',
         &['\u{641}']), ('\u{6a8}',  &['\u{6a1}',  '\u{6db}']), ('\u{6a9}',  &['\u{643}']),
        ('\u{6aa}',  &['\u{643}']), ('\u{6ad}',  &['\u{643}',  '\u{6db}']), ('\u{6b4}',
         &['\u{6af}',  '\u{6db}']), ('\u{6b5}',  &['\u{644}',  '\u{306}']), ('\u{6b7}',
         &['\u{644}',  '\u{6db}']), ('\u{6ba}',  &['\u{649}']), ('\u{6bb}',  &['\u{649}',
         '\u{615}']), ('\u{6bd}',  &['\u{649}',  '\u{6db}']), ('\u{6be}',  &['\u{6f}']), ('\u{6c1}',
         &['\u{6f}']), ('\u{6c2}',  &['\u{6c0}']), ('\u{6c3}',  &['\u{629}']), ('\u{6c6}',
         &['\u{648}',  '\u{306}']), ('\u{6c7}',  &['\u{648}',  '\u{313}']), ('\u{6c8}',
         &['\u{648}',  '\u{670}']), ('\u{6c9}',  &['\u{648}',  '\u{302}']), ('\u{6cb}',
         &['\u{648}',  '\u{6db}']), ('\u{6cc}',  &['\u{649}']), ('\u{6ce}',  &['\u{649}',
         '\u{306}']), ('\u{6d0}',  &['\u{67b}']), ('\u{6d1}',  &['\u{649}',  '\u{6db}']),
        ('\u{6d2}',  &['\u{649}']), ('\u{6d4}',  &['\u{2d}']), ('\u{6d5}',  &['\u{6f}']),
        ('\u{6df}',  &['\u{30a}']), ('\u{6e8}',  &['\u{306}',  '\u{307}']), ('\u{6ec}',
         &['\u{307}']), ('\u{6ee}',  &['\u{62f}',  '\u{302}']), ('\u{6ef}',  &['\u{631}',
         '\u{302}']), ('\u{6f0}',  &['\u{2e}']), ('\u{6f1}',  &['\u{6c}']), ('\u{6f2}',
         &['\u{662}']), ('\u{6f3}',  &['\u{663}']), ('\u{6f4}',  &['\u{664}']), ('\u{6f5}',
         &['\u{6f}']), ('\u{6f6}',  &['\u{666}']), ('\u{6f7}',  &['\u{56}']), ('\u{6f8}',
         &['\u{245}']), ('\u{6f9}',  &['\u{669}']), ('\u{6fd}',  &['\u{621}',  '\u{348}']),
        ('\u{6fe}',  &['\u{645}',  '\u{348}']), ('\u{6ff}',  &['\u{6f}',  '\u{302}']), ('\u{701}',
         &['\u{2e}']), ('\u{702}',  &['\u{2e}']), ('\u{703}',  &['\u{3a}']), ('\u{704}',
         &['\u{3a}']), ('\u{740}',  &['\u{307}']), ('\u{741}',  &['\u{307}']), ('\u{742}',
         &['\u{73c}']), ('\u{747}',  &['\u{301}']), ('\u{751}',  &['\u{628}',  '\u{6db}']),
        ('\u{756}',  &['\u{649}',  '\u{306}']), ('\u{762}',  &['\u{6ac}']), ('\u{763}',
         &['\u{643}',  '\u{6db}']), ('\u{767}',  &['\u{754}']), ('\u{768}',  &['\u{646}',
         '\u{615}']), ('\u{769}',  &['\u{646}',  '\u{306}']), ('\u{76c}',  &['\u{631}',
         '\u{654}']), ('\u{771}',  &['\u{697}',  '\u{615}']), ('\u{772}',  &['\u{62d}',
         '\u{654}']), ('\u{77e}',  &['\u{633}',  '\u{302}']), ('\u{7c0}',  &['\u{4f}']), ('\u{7ca}',
         &['\u{6c}']), ('\u{7eb}',  &['\u{304}']), ('\u{7ed}',  &['\u{307}']), ('\u{7ee}',
         &['\u{302}']), ('\u{7f3}',  &['\u{308}']), ('\u{7f4}',  &['\u{27}']), ('\u{7f5}',
         &['\u{27}']), ('\u{7fa}',  &['\u{5f}']), ('\u{8a1}',  &['\u{628}',  '\u{654}']),
        ('\u{8a4}',  &['\u{6a2}',  '\u{6db}']), ('\u{8a7}',  &['\u{645}',  '\u{6db}']), ('\u{8a8}',
         &['\u{649}',  '\u{654}']), ('\u{8a9}',  &['\u{754}']), ('\u{8ae}',  &['\u{62f}',
         '\u{324}',  '\u{323}']), ('\u{8af}',  &['\u{635}',  '\u{324}',  '\u{323}']), ('\u{8b0}',
         &['\u{6af}']), ('\u{8b1}',  &['\u{648}']), ('\u{8b2}',  &['\u{632}',  '\u{302}']),
        ('\u{8b6}',  &['\u{628}',  '\u{6e2}']), ('\u{8b7}',  &['\u{649}',  '\u{6db}',  '\u{6e2}']),
        ('\u{8b9}',  &['\u{631}',  '\u{306}',  '\u{307}']), ('\u{8ba}',  &['\u{649}',  '\u{306}',
         '\u{307}']), ('\u{8bb}',  &['\u{6a1}']), ('\u{8bc}',  &['\u{6a1}']), ('\u{8bd}',
         &['\u{649}']), ('\u{8e5}',  &['\u{64c}']), ('\u{8e8}',  &['\u{64c}']), ('\u{8ea}',
         &['\u{307}']), ('\u{8eb}',  &['\u{308}']), ('\u{8ed}',  &['\u{323}']), ('\u{8ee}',
         &['\u{324}']), ('\u{8f0}',  &['\u{30b}']), ('\u{8f1}',  &['\u{64c}']), ('\u{8f2}',
         &['\u{64d}']), ('\u{8f3}',  &['\u{313}']), ('\u{8f8}',  &['\u{350}']), ('\u{8f9}',
         &['\u{354}']), ('\u{8fa}',  &['\u{355}']), ('\u{8ff}',  &['\u{350}']), ('\u{900}',
         &['\u{352}']), ('\u{901}',  &['\u{306}',  '\u{307}']), ('\u{902}',  &['\u{307}']),
        ('\u{903}',  &['\u{3a}']), ('\u{904}',  &['\u{905}',  '\u{946}']), ('\u{906}',  &['\u{905}',
         '\u{93e}']), ('\u{908}',  &['\u{930}',  '\u{94d}',  '\u{907}']), ('\u{90d}',  &['\u{90f}',
         '\u{945}']), ('\u{90e}',  &['\u{90f}',  '\u{946}']), ('\u{910}',  &['\u{90f}',
         '\u{947}']), ('\u{911}',  &['\u{905}',  '\u{949}']), ('\u{912}',  &['\u{905}',  '\u{93e}',
         '\u{946}']), ('\u{913}',  &['\u{905}',  '\u{93e}',  '\u{947}']), ('\u{914}',  &['\u{905}',
         '\u{93e}',  '\u{948}']), ('\u{93c}',  &['\u{323}']), ('\u{952}',  &['\u{331}']),
        ('\u{953}',  &['\u{300}']), ('\u{954}',  &['\u{301}']), ('\u{965}',  &['\u{964}',
         '\u{964}']), ('\u{966}',  &['\u{6f}']), ('\u{967}',  &['\u{669}']), ('\u{97d}',
         &['\u{3f}']), ('\u{981}',  &['\u{306}',  '\u{307}']), ('\u{986}',  &['\u{985}',
         '\u{9be}']), ('\u{9bc}',  &['\u{323}']), ('\u{9e0}',  &['\u{98b}',  '\u{9c3}']),
        ('\u{9e1}',  &['\u{98b}',  '\u{9c3}']), ('\u{9e6}',  &['\u{4f}']), ('\u{9ea}',
         &['\u{38}']), ('\u{9ed}',  &['\u{39}']), ('\u{a02}',  &['\u{307}']), ('\u{a03}',
         &['\u{983}']), ('\u{a06}',  &['\u{a05}',  '\u{a3e}']), ('\u{a07}',  &['\u{a72}',
         '\u{a3f}']), ('\u{a08}',  &['\u{a72}',  '\u{a40}']), ('\u{a09}',  &['\u{a73}',
         '\u{a41}']), ('\u{a0a}',  &['\u{a73}',  '\u{a42}']), ('\u{a0f}',  &['\u{a72}',
         '\u{a47}']), ('\u{a10}',  &['\u{a05}',  '\u{a48}']), ('\u{a14}',  &['\u{a05}',
         '\u{a4c}']), ('\u{a3c}',  &['\u{323}']), ('\u{a4b}',  &['\u{946}']), ('\u{a4d}',
         &['\u{94d}']), ('\u{a66}',  &['\u{6f}']), ('\u{a67}',  &['\u{39}']), ('\u{a6a}',
         &['\u{38}']), ('\u{a81}',  &['\u{306}',  '\u{307}']), ('\u{a82}',  &['\u{307}']),
        ('\u{a83}',  &['\u{3a}']), ('\u{a86}',  &['\u{a85}',  '\u{abe}']), ('\u{a8d}',  &['\u{a85}',
         '\u{ac5}']), ('\u{a8f}',  &['\u{a85}',  '\u{ac7}']), ('\u{a90}',  &['\u{a85}',
         '\u{ac8}']), ('\u{a91}',  &['\u{a85}',  '\u{abe}',  '\u{ac5}']), ('\u{a93}',  &['\u{a85}',
         '\u{abe}',  '\u{ac7}']), ('\u{a94}',  &['\u{a85}',  '\u{abe}',  '\u{ac8}']), ('\u{abc}',
         &['\u{323}']), ('\u{abd}',  &['\u{93d}']), ('\u{ac1}',  &['\u{941}']), ('\u{ac2}',
         &['\u{942}']), ('\u{acd}',  &['\u{94d}']), ('\u{ae6}',  &['\u{6f}']), ('\u{ae8}',
         &['\u{968}']), ('\u{ae9}',  &['\u{969}']), ('\u{aea}',  &['\u{96a}']), ('\u{aee}',
         &['\u{96e}']), ('\u{af0}',  &['\u{970}']), ('\u{b01}',  &['\u{306}',  '\u{307}']),
        ('\u{b03}',  &['\u{38}']), ('\u{b06}',  &['\u{b05}',  '\u{b3e}']), ('\u{b20}',
         &['\u{4f}']), ('\u{b3c}',  &['\u{323}']), ('\u{b66}',  &['\u{4f}']), ('\u{b68}',
         &['\u{39}']), ('\u{b82}',  &['\u{30a}']), ('\u{b8a}',  &['\u{b89}',  '\u{bb3}']),
        ('\u{b9c}',  &['\u{b90}']), ('\u{bb0}',  &['\u{b88}']), ('\u{bbe}',  &['\u{b88}']),
        ('\u{bc8}',  &['\u{ba9}']), ('\u{bca}',  &['\u{bc6}',  '\u{b88}']), ('\u{bcb}',
         &['\u{bc7}',  '\u{b88}']), ('\u{bcc}',  &['\u{bc6}',  '\u{bb3}']), ('\u{bcd}',
         &['\u{307}']), ('\u{bd7}',  &['\u{bb3}']), ('\u{be6}',  &['\u{6f}']), ('\u{be7}',
         &['\u{b95}']), ('\u{be8}',  &['\u{b89}']), ('\u{bea}',  &['\u{b9a}']), ('\u{beb}',
         &['\u{b88}',  '\u{bc1}']), ('\u{bec}',  &['\u{b9a}',  '\u{bc1}']), ('\u{bed}',
         &['\u{b8e}']), ('\u{bee}',  &['\u{b85}']), ('\u{bf0}',  &['\u{baf}']), ('\u{bf2}',
         &['\u{b9a}',  '\u{bc2}']), ('\u{bf4}',  &['\u{bae}',  '\u{bc0}']), ('\u{bf5}',
         &['\u{bf3}']), ('\u{bf7}',  &['\u{b8e}',  '\u{bb5}']), ('\u{bf8}',  &['\u{bb7}']),
        ('\u{bfa}',  &['\u{ba8}',  '\u{bc0}']), ('\u{c00}',  &['\u{306}',  '\u{307}']), ('\u{c02}',
         &['\u{6f}']), ('\u{c03}',  &['\u{983}']), ('\u{c13}',  &['\u{c12}',  '\u{c55}']),
        ('\u{c14}',  &['\u{c12}',  '\u{c4c}']), ('\u{c20}',  &['\u{c30}',  '\u{5bc}']), ('\u{c22}',
         &['\u{c21}',  '\u{323}']), ('\u{c25}',  &['\u{c27}',  '\u{5bc}']), ('\u{c2d}',
         &['\u{c2c}',  '\u{323}']), ('\u{c2e}',  &['\u{c35}',  '\u{c41}']), ('\u{c37}',
         &['\u{c35}',  '\u{323}']), ('\u{c39}',  &['\u{c35}',  '\u{c3e}']), ('\u{c42}',
         &['\u{c41}',  '\u{c3e}']), ('\u{c44}',  &['\u{c43}',  '\u{c3e}']), ('\u{c60}',
         &['\u{c0b}',  '\u{c3e}']), ('\u{c61}',  &['\u{c0c}',  '\u{c3e}']), ('\u{c66}',
         &['\u{6f}']), ('\u{c81}',  &['\u{306}',  '\u{307}']), ('\u{c82}',  &['\u{6f}']),
        ('\u{c83}',  &['\u{983}']), ('\u{c85}',  &['\u{c05}']), ('\u{c86}',  &['\u{c06}']),
        ('\u{c87}',  &['\u{c07}']), ('\u{c92}',  &['\u{c12}']), ('\u{c93}',  &['\u{c12}',
         '\u{c55}']), ('\u{c94}',  &['\u{c12}',  '\u{c4c}']), ('\u{c9c}',  &['\u{c1c}']),
        ('\u{c9e}',  &['\u{c1e}']), ('\u{ca3}',  &['\u{c23}']), ('\u{caf}',  &['\u{c2f}']),
        ('\u{cb1}',  &['\u{c31}']), ('\u{cb2}',  &['\u{c32}']), ('\u{ce1}',  &['\u{c8c}',
         '\u{cbe}']), ('\u{ce6}',  &['\u{6f}']), ('\u{ce7}',  &['\u{c67}']), ('\u{ce8}',
         &['\u{c68}']), ('\u{cef}',  &['\u{c6f}']), ('\u{d01}',  &['\u{306}',  '\u{307}']),
        ('\u{d02}',  &['\u{6f}']), ('\u{d03}',  &['\u{983}']), ('\u{d08}',  &['\u{d07}',
         '\u{d57}']), ('\u{d09}',  &['\u{b89}']), ('\u{d0a}',  &['\u{b89}',  '\u{d57}']),
        ('\u{d0c}',  &['\u{d28}',  '\u{d41}']), ('\u{d10}',  &['\u{d0e}',  '\u{d46}']), ('\u{d13}',
         &['\u{d12}',  '\u{d3e}']), ('\u{d14}',  &['\u{d12}',  '\u{d57}']), ('\u{d19}',
         &['\u{d28}',  '\u{d41}']), ('\u{d1c}',  &['\u{b90}']), ('\u{d20}',  &['\u{6f}']),
        ('\u{d23}',  &['\u{ba3}']), ('\u{d31}',  &['\u{d30}']), ('\u{d34}',  &['\u{bb4}']),
        ('\u{d36}',  &['\u{bb6}']), ('\u{d3a}',  &['\u{b9f}',  '\u{bbf}']), ('\u{d3f}',
         &['\u{bbf}']), ('\u{d40}',  &['\u{bbf}']), ('\u{d42}',  &['\u{d41}']), ('\u{d43}',
         &['\u{d41}']), ('\u{d48}',  &['\u{d46}',  '\u{d46}']), ('\u{d4e}',  &['\u{971}']),
        ('\u{d5a}',  &['\u{d28}',  '\u{d4d}',  '\u{d2e}']), ('\u{d5f}',  &['\u{6f}',  '\u{d30}',
         '\u{6f}']), ('\u{d61}',  &['\u{d1e}']), ('\u{d66}',  &['\u{6f}']), ('\u{d6a}',
         &['\u{d30}',  '\u{d4d}']), ('\u{d6b}',  &['\u{d26}',  '\u{d4d}',  '\u{d30}']), ('\u{d6c}',
         &['\u{d28}',  '\u{d4d}',  '\u{d28}']), ('\u{d6d}',  &['\u{39}']), ('\u{d6e}',  &['\u{d35}',
         '\u{d4d}',  '\u{d30}']), ('\u{d6f}',  &['\u{d28}',  '\u{d4d}']), ('\u{d76}',  &['\u{d39}',
         '\u{d4d}',  '\u{d2e}']), ('\u{d79}',  &['\u{d28}',  '\u{d41}']), ('\u{d7b}',  &['\u{d28}',
         '\u{d4d}']), ('\u{d7c}',  &['\u{d30}',  '\u{d4d}']), ('\u{d82}',  &['\u{6f}']), ('\u{d83}',
         &['\u{983}']), ('\u{de9}',  &['\u{de8}',  '\u{dcf}']), ('\u{dea}',  &['\u{da2}']),
        ('\u{deb}',  &['\u{daf}']), ('\u{def}',  &['\u{de8}',  '\u{dd3}']), ('\u{e03}',
         &['\u{e02}']), ('\u{e0b}',  &['\u{e0a}']), ('\u{e0f}',  &['\u{e0e}']), ('\u{e14}',
         &['\u{e04}']), ('\u{e15}',  &['\u{e04}']), ('\u{e17}',  &['\u{e11}']), ('\u{e21}',
         &['\u{e06}']), ('\u{e26}',  &['\u{e20}']), ('\u{e33}',  &['\u{30a}',  '\u{e32}']),
        ('\u{e41}',  &['\u{e40}',  '\u{e40}']), ('\u{e45}',  &['\u{e32}']), ('\u{e4d}',
         &['\u{30a}']), ('\u{e50}',  &['\u{6f}']), ('\u{e88}',  &['\u{e08}']), ('\u{e8d}',
         &['\u{e22}']), ('\u{e9a}',  &['\u{e1a}']), ('\u{e9b}',  &['\u{e1b}']), ('\u{e9d}',
         &['\u{e1d}']), ('\u{e9e}',  &['\u{e1e}']), ('\u{e9f}',  &['\u{e1f}']), ('\u{eb3}',
         &['\u{30a}',  '\u{eb2}']), ('\u{eb8}',  &['\u{e38}']), ('\u{eb9}',  &['\u{e39}']),
        ('\u{ec8}',  &['\u{e48}']), ('\u{ec9}',  &['\u{e49}']), ('\u{eca}',  &['\u{e4a}']),
        ('\u{ecb}',  &['\u{e4b}']), ('\u{ecd}',  &['\u{30a}']), ('\u{ed0}',  &['\u{6f}']),
        ('\u{edc}',  &['\u{eab}',  '\u{e99}']), ('\u{edd}',  &['\u{eab}',  '\u{ea1}']), ('\u{f00}',
         &['\u{f68}',  '\u{f7c}',  '\u{f7e}']), ('\u{f02}',  &['\u{f60}',  '\u{f74}',  '\u{f82}',
         '\u{f7f}']), ('\u{f03}',  &['\u{f60}',  '\u{f74}',  '\u{f82}',  '\u{f14}']), ('\u{f0c}',
         &['\u{f0b}']), ('\u{f0e}',  &['\u{f0d}',  '\u{f0d}']), ('\u{f1b}',  &['\u{f1a}',
         '\u{f1a}']), ('\u{f1e}',  &['\u{f1d}',  '\u{f1d}']), ('\u{f1f}',  &['\u{f1a}',
         '\u{f1d}']), ('\u{f37}',  &['\u{325}']), ('\u{f6a}',  &['\u{f62}']), ('\u{f77}',
         &['\u{fb2}',  '\u{f71}',  '\u{f80}']), ('\u{f79}',  &['\u{fb3}',  '\u{f71}',  '\u{f80}']),
        ('\u{fce}',  &['\u{f1d}',  '\u{f1a}']), ('\u{fd5}',  &['\u{5350}']), ('\u{fd6}',
         &['\u{534d}']), ('\u{1000}',  &['\u{1002}',  '\u{102c}']), ('\u{1010}',  &['\u{6f}',
         '\u{102c}']), ('\u{101d}',  &['\u{6f}']), ('\u{101f}',  &['\u{1015}',  '\u{102c}']),
        ('\u{1029}',  &['\u{101e}',  '\u{103c}']), ('\u{102a}',  &['\u{101e}',  '\u{103c}',
         '\u{1031}',  '\u{102c}',  '\u{103a}']), ('\u{1036}',  &['\u{30a}']), ('\u{1038}',
         &['\u{983}']), ('\u{1040}',  &['\u{6f}']), ('\u{104b}',  &['\u{104a}',  '\u{104a}']),
        ('\u{1065}',  &['\u{1041}']), ('\u{1066}',  &['\u{1015}',  '\u{103e}']), ('\u{106f}',
         &['\u{1015}',  '\u{102c}',  '\u{103e}']), ('\u{1070}',  &['\u{1003}',  '\u{103e}']),
        ('\u{107e}',  &['\u{107d}',  '\u{103e}']), ('\u{1081}',  &['\u{1002}',  '\u{103e}']),
        ('\u{109e}',  &['\u{1083}',  '\u{30a}']), ('\u{10a0}',  &['\u{a786}']), ('\u{10e7}',
         &['\u{79}']), ('\u{10f3}',  &['\u{21d}']), ('\u{10ff}',  &['\u{6f}']), ('\u{1101}',
         &['\u{1100}',  '\u{1100}']), ('\u{1104}',  &['\u{1103}',  '\u{1103}']), ('\u{1108}',
         &['\u{1107}',  '\u{1107}']), ('\u{110a}',  &['\u{1109}',  '\u{1109}']), ('\u{110d}',
         &['\u{110c}',  '\u{110c}']), ('\u{1113}',  &['\u{1102}',  '\u{1100}']), ('\u{1114}',
         &['\u{1102}',  '\u{1102}']), ('\u{1115}',  &['\u{1102}',  '\u{1103}']), ('\u{1116}',
         &['\u{1102}',  '\u{1107}']), ('\u{1117}',  &['\u{1103}',  '\u{1100}']), ('\u{1118}',
         &['\u{1105}',  '\u{1102}']), ('\u{1119}',  &['\u{1105}',  '\u{1105}']), ('\u{111a}',
         &['\u{1105}',  '\u{1112}']), ('\u{111b}',  &['\u{1105}',  '\u{110b}']), ('\u{111c}',
         &['\u{1106}',  '\u{1107}']), ('\u{111d}',  &['\u{1106}',  '\u{110b}']), ('\u{111e}',
         &['\u{1107}',  '\u{1100}']), ('\u{111f}',  &['\u{1107}',  '\u{1102}']), ('\u{1120}',
         &['\u{1107}',  '\u{1103}']), ('\u{1121}',  &['\u{1107}',  '\u{1109}']), ('\u{1122}',
         &['\u{1107}',  '\u{1109}',  '\u{1100}']), ('\u{1123}',  &['\u{1107}',  '\u{1109}',
         '\u{1103}']), ('\u{1124}',  &['\u{1107}',  '\u{1109}',  '\u{1107}']), ('\u{1125}',
         &['\u{1107}',  '\u{1109}',  '\u{1109}']), ('\u{1126}',  &['\u{1107}',  '\u{1109}',
         '\u{110c}']), ('\u{1127}',  &['\u{1107}',  '\u{110c}']), ('\u{1128}',  &['\u{1107}',
         '\u{110e}']), ('\u{1129}',  &['\u{1107}',  '\u{1110}']), ('\u{112a}',  &['\u{1107}',
         '\u{1111}']), ('\u{112b}',  &['\u{1107}',  '\u{110b}']), ('\u{112c}',  &['\u{1107}',
         '\u{1107}',  '\u{110b}']), ('\u{112d}',  &['\u{1109}',  '\u{1100}']), ('\u{112e}',
         &['\u{1109}',  '\u{1102}']), ('\u{112f}',  &['\u{1109}',  '\u{1103}']), ('\u{1130}',
         &['\u{1109}',  '\u{1105}']), ('\u{1131}',  &['\u{1109}',  '\u{1106}']), ('\u{1132}',
         &['\u{1109}',  '\u{1107}']), ('\u{1133}',  &['\u{1109}',  '\u{1107}',  '\u{1100}']),
        ('\u{1134}',  &['\u{1109}',  '\u{1109}',  '\u{1109}']), ('\u{1135}',  &['\u{1109}',
         '\u{110b}']), ('\u{1136}',  &['\u{1109}',  '\u{110c}']), ('\u{1137}',  &['\u{1109}',
         '\u{110e}']), ('\u{1138}',  &['\u{1109}',  '\u{110f}']), ('\u{1139}',  &['\u{1109}',
         '\u{1110}']), ('\u{113a}',  &['\u{1109}',  '\u{1111}']), ('\u{113b}',  &['\u{1105}',
         '\u{1112}']), ('\u{113d}',  &['\u{113c}',  '\u{113c}']), ('\u{113f}',  &['\u{113e}',
         '\u{113e}']), ('\u{1141}',  &['\u{110b}',  '\u{1100}']), ('\u{1142}',  &['\u{110b}',
         '\u{1103}']), ('\u{1143}',  &['\u{110b}',  '\u{1106}']), ('\u{1144}',  &['\u{110b}',
         '\u{1107}']), ('\u{1145}',  &['\u{110b}',  '\u{1109}']), ('\u{1146}',  &['\u{110b}',
         '\u{1140}']), ('\u{1147}',  &['\u{110b}',  '\u{110b}']), ('\u{1148}',  &['\u{110b}',
         '\u{110c}']), ('\u{1149}',  &['\u{110b}',  '\u{110e}']), ('\u{114a}',  &['\u{110b}',
         '\u{1110}']), ('\u{114b}',  &['\u{110b}',  '\u{1111}']), ('\u{114d}',  &['\u{110c}',
         '\u{110b}']), ('\u{114f}',  &['\u{114e}',  '\u{114e}']), ('\u{1151}',  &['\u{1150}',
         '\u{1150}']), ('\u{1152}',  &['\u{110e}',  '\u{110f}']), ('\u{1153}',  &['\u{110e}',
         '\u{1112}']), ('\u{1156}',  &['\u{1111}',  '\u{1107}']), ('\u{1157}',  &['\u{1111}',
         '\u{110b}']), ('\u{1158}',  &['\u{1112}',  '\u{1112}']), ('\u{115a}',  &['\u{1100}',
         '\u{1103}']), ('\u{115b}',  &['\u{1102}',  '\u{1109}']), ('\u{115c}',  &['\u{1102}',
         '\u{110c}']), ('\u{115d}',  &['\u{1102}',  '\u{1112}']), ('\u{115e}',  &['\u{1103}',
         '\u{1105}']), ('\u{1162}',  &['\u{1161}',  '\u{4e28}']), ('\u{1164}',  &['\u{1163}',
         '\u{4e28}']), ('\u{1166}',  &['\u{1165}',  '\u{4e28}']), ('\u{1168}',  &['\u{1167}',
         '\u{4e28}']), ('\u{116a}',  &['\u{1169}',  '\u{1161}']), ('\u{116b}',  &['\u{1169}',
         '\u{1161}',  '\u{4e28}']), ('\u{116c}',  &['\u{1169}',  '\u{4e28}']), ('\u{116f}',
         &['\u{116e}',  '\u{1165}']), ('\u{1170}',  &['\u{116e}',  '\u{1165}',  '\u{4e28}']),
        ('\u{1171}',  &['\u{116e}',  '\u{4e28}']), ('\u{1173}',  &['\u{30fc}']), ('\u{1174}',
         &['\u{30fc}',  '\u{4e28}']), ('\u{1175}',  &['\u{4e28}']), ('\u{1176}',  &['\u{1161}',
         '\u{1169}']), ('\u{1177}',  &['\u{1161}',  '\u{116e}']), ('\u{1178}',  &['\u{1163}',
         '\u{1169}']), ('\u{1179}',  &['\u{1163}',  '\u{116d}']), ('\u{117a}',  &['\u{1165}',
         '\u{1169}']), ('\u{117b}',  &['\u{1165}',  '\u{116e}']), ('\u{117c}',  &['\u{1165}',
         '\u{30fc}']), ('\u{117d}',  &['\u{1167}',  '\u{1169}']), ('\u{117e}',  &['\u{1167}',
         '\u{116e}']), ('\u{117f}',  &['\u{1169}',  '\u{1165}']), ('\u{1180}',  &['\u{1169}',
         '\u{1165}',  '\u{4e28}']), ('\u{1181}',  &['\u{1169}',  '\u{1167}',  '\u{4e28}']),
        ('\u{1182}',  &['\u{1169}',  '\u{1169}']), ('\u{1183}',  &['\u{1169}',  '\u{116e}']),
        ('\u{1184}',  &['\u{116d}',  '\u{1163}']), ('\u{1185}',  &['\u{116d}',  '\u{1163}',
         '\u{4e28}']), ('\u{1186}',  &['\u{116d}',  '\u{1163}']), ('\u{1187}',  &['\u{116d}',
         '\u{1169}']), ('\u{1188}',  &['\u{116d}',  '\u{4e28}']), ('\u{1189}',  &['\u{116e}',
         '\u{1161}']), ('\u{118a}',  &['\u{116e}',  '\u{1161}',  '\u{4e28}']), ('\u{118b}',
         &['\u{116e}',  '\u{1165}',  '\u{30fc}']), ('\u{118c}',  &['\u{116e}',  '\u{1167}',
         '\u{4e28}']), ('\u{118d}',  &['\u{116e}',  '\u{116e}']), ('\u{118e}',  &['\u{1172}',
         '\u{1161}']), ('\u{118f}',  &['\u{1172}',  '\u{1165}']), ('\u{1190}',  &['\u{1172}',
         '\u{1165}',  '\u{4e28}']), ('\u{1191}',  &['\u{1172}',  '\u{1167}']), ('\u{1192}',
         &['\u{1172}',  '\u{1167}',  '\u{4e28}']), ('\u{1193}',  &['\u{1172}',  '\u{116e}']),
        ('\u{1194}',  &['\u{1172}',  '\u{4e28}']), ('\u{1195}',  &['\u{30fc}',  '\u{116e}']),
        ('\u{1196}',  &['\u{30fc}',  '\u{30fc}']), ('\u{1197}',  &['\u{30fc}',  '\u{4e28}',
         '\u{116e}']), ('\u{1198}',  &['\u{4e28}',  '\u{1161}']), ('\u{1199}',  &['\u{4e28}',
         '\u{1163}']), ('\u{119a}',  &['\u{4e28}',  '\u{1169}']), ('\u{119b}',  &['\u{4e28}',
         '\u{116e}']), ('\u{119c}',  &['\u{4e28}',  '\u{30fc}']), ('\u{119d}',  &['\u{4e28}',
         '\u{119e}']), ('\u{119f}',  &['\u{119e}',  '\u{1165}']), ('\u{11a0}',  &['\u{119e}',
         '\u{116e}']), ('\u{11a1}',  &['\u{119e}',  '\u{4e28}']), ('\u{11a2}',  &['\u{119e}',
         '\u{119e}']), ('\u{11a3}',  &['\u{1161}',  '\u{30fc}']), ('\u{11a4}',  &['\u{1163}',
         '\u{116e}']), ('\u{11a5}',  &['\u{1167}',  '\u{1163}']), ('\u{11a6}',  &['\u{1169}',
         '\u{1163}']), ('\u{11a7}',  &['\u{1169}',  '\u{1163}',  '\u{4e28}']), ('\u{11a8}',
         &['\u{1100}']), ('\u{11a9}',  &['\u{1100}',  '\u{1100}']), ('\u{11aa}',  &['\u{1100}',
         '\u{1109}']), ('\u{11ab}',  &['\u{1102}']), ('\u{11ac}',  &['\u{1102}',  '\u{110c}']),
        ('\u{11ad}',  &['\u{1102}',  '\u{1112}']), ('\u{11ae}',  &['\u{1103}']), ('\u{11af}',
         &['\u{1105}']), ('\u{11b0}',  &['\u{1105}',  '\u{1100}']), ('\u{11b1}',  &['\u{1105}',
         '\u{1106}']), ('\u{11b2}',  &['\u{1105}',  '\u{1107}']), ('\u{11b3}',  &['\u{1105}',
         '\u{1109}']), ('\u{11b4}',  &['\u{1105}',  '\u{1110}']), ('\u{11b5}',  &['\u{1105}',
         '\u{1111}']), ('\u{11b6}',  &['\u{1105}',  '\u{1112}']), ('\u{11b7}',  &['\u{1106}']),
        ('\u{11b8}',  &['\u{1107}']), ('\u{11b9}',  &['\u{1107}',  '\u{1109}']), ('\u{11ba}',
         &['\u{1109}']), ('\u{11bb}',  &['\u{1109}',  '\u{1109}']), ('\u{11bc}',  &['\u{110b}']),
        ('\u{11bd}',  &['\u{110c}']), ('\u{11be}',  &['\u{110e}']), ('\u{11bf}',  &['\u{110f}']),
        ('\u{11c0}',  &['\u{1110}']), ('\u{11c1}',  &['\u{1111}']), ('\u{11c2}',  &['\u{1112}']),
        ('\u{11c3}',  &['\u{1100}',  '\u{1105}']), ('\u{11c4}',  &['\u{1100}',  '\u{1109}',
         '\u{1100}']), ('\u{11c5}',  &['\u{1102}',  '\u{1100}']), ('\u{11c6}',  &['\u{1102}',
         '\u{1103}']), ('\u{11c7}',  &['\u{1102}',  '\u{1109}']), ('\u{11c8}',  &['\u{1102}',
         '\u{1140}']), ('\u{11c9}',  &['\u{1102}',  '\u{1110}']), ('\u{11ca}',  &['\u{1103}',
         '\u{1100}']), ('\u{11cb}',  &['\u{1103}',  '\u{1105}']), ('\u{11cc}',  &['\u{1105}',
         '\u{1100}',  '\u{1109}']), ('\u{11cd}',  &['\u{1105}',  '\u{1102}']), ('\u{11ce}',
         &['\u{1105}',  '\u{1103}']), ('\u{11cf}',  &['\u{1105}',  '\u{1103}',  '\u{1112}']),
        ('\u{11d0}',  &['\u{1105}',  '\u{1105}']), ('\u{11d1}',  &['\u{1105}',  '\u{1106}',
         '\u{1100}']), ('\u{11d2}',  &['\u{1105}',  '\u{1106}',  '\u{1109}']), ('\u{11d3}',
         &['\u{1105}',  '\u{1107}',  '\u{1109}']), ('\u{11d4}',  &['\u{1105}',  '\u{1107}',
         '\u{1112}']), ('\u{11d5}',  &['\u{1105}',  '\u{1107}',  '\u{110b}']), ('\u{11d6}',
         &['\u{1105}',  '\u{1109}',  '\u{1109}']), ('\u{11d7}',  &['\u{1105}',  '\u{1140}']),
        ('\u{11d8}',  &['\u{1105}',  '\u{110f}']), ('\u{11d9}',  &['\u{1105}',  '\u{1159}']),
        ('\u{11da}',  &['\u{1106}',  '\u{1100}']), ('\u{11db}',  &['\u{1106}',  '\u{1105}']),
        ('\u{11dc}',  &['\u{1106}',  '\u{1107}']), ('\u{11dd}',  &['\u{1106}',  '\u{1109}']),
        ('\u{11de}',  &['\u{1106}',  '\u{1109}',  '\u{1109}']), ('\u{11df}',  &['\u{1106}',
         '\u{1140}']), ('\u{11e0}',  &['\u{1106}',  '\u{110e}']), ('\u{11e1}',  &['\u{1106}',
         '\u{1112}']), ('\u{11e2}',  &['\u{1106}',  '\u{110b}']), ('\u{11e3}',  &['\u{1107}',
         '\u{1105}']), ('\u{11e4}',  &['\u{1107}',  '\u{1111}']), ('\u{11e5}',  &['\u{1107}',
         '\u{1112}']), ('\u{11e6}',  &['\u{1107}',  '\u{110b}']), ('\u{11e7}',  &['\u{1109}',
         '\u{1100}']), ('\u{11e8}',  &['\u{1109}',  '\u{1103}']), ('\u{11e9}',  &['\u{1109}',
         '\u{1105}']), ('\u{11ea}',  &['\u{1109}',  '\u{1107}']), ('\u{11eb}',  &['\u{1140}']),
        ('\u{11ec}',  &['\u{110b}',  '\u{1100}']), ('\u{11ed}',  &['\u{110b}',  '\u{1100}',
         '\u{1100}']), ('\u{11ee}',  &['\u{110b}',  '\u{110b}']), ('\u{11ef}',  &['\u{110b}',
         '\u{110f}']), ('\u{11f0}',  &['\u{114c}']), ('\u{11f1}',  &['\u{110b}',  '\u{1109}']),
        ('\u{11f2}',  &['\u{110b}',  '\u{1140}']), ('\u{11f3}',  &['\u{1111}',  '\u{1107}']),
        ('\u{11f4}',  &['\u{1111}',  '\u{110b}']), ('\u{11f5}',  &['\u{1112}',  '\u{1102}']),
        ('\u{11f6}',  &['\u{1112}',  '\u{1105}']), ('\u{11f7}',  &['\u{1112}',  '\u{1106}']),
        ('\u{11f8}',  &['\u{1112}',  '\u{1107}']), ('\u{11f9}',  &['\u{1159}']), ('\u{11fa}',
         &['\u{1100}',  '\u{1102}']), ('\u{11fb}',  &['\u{1100}',  '\u{1107}']), ('\u{11fc}',
         &['\u{1100}',  '\u{110e}']), ('\u{11fd}',  &['\u{1100}',  '\u{110f}']), ('\u{11fe}',
         &['\u{1100}',  '\u{1112}']), ('\u{11ff}',  &['\u{1102}',  '\u{1102}']), ('\u{1200}',
         &['\u{55}']), ('\u{1223}',  &['\u{270}']), ('\u{1240}',  &['\u{3a6}']), ('\u{1260}',
         &['\u{548}']), ('\u{1294}',  &['\u{571}']), ('\u{12d0}',  &['\u{4f}']), ('\u{13a0}',
         &['\u{44}']), ('\u{13a1}',  &['\u{52}']), ('\u{13a2}',  &['\u{54}']), ('\u{13a4}',
         &['\u{4f}',  '\u{27}']), ('\u{13a5}',  &['\u{69}']), ('\u{13a8}',  &['\u{2c75}']),
        ('\u{13a9}',  &['\u{59}']), ('\u{13aa}',  &['\u{41}']), ('\u{13ab}',  &['\u{4a}']),
        ('\u{13ac}',  &['\u{45}']), ('\u{13ae}',  &['\u{3f}']), ('\u{13b0}',  &['\u{2c75}']),
        ('\u{13b1}',  &['\u{393}']), ('\u{13b3}',  &['\u{57}']), ('\u{13b7}',  &['\u{4d}']),
        ('\u{13bb}',  &['\u{48}']), ('\u{13bd}',  &['\u{59}']), ('\u{13be}',  &['\u{4f}',
         '\u{335}']), ('\u{13bf}',  &['\u{1ab}']), ('\u{13c0}',  &['\u{47}']), ('\u{13c2}',
         &['\u{68}']), ('\u{13c3}',  &['\u{5a}']), ('\u{13c7}',  &['\u{460}']), ('\u{13cb}',
         &['\u{190}']), ('\u{13cc}',  &['\u{55}',  '\u{335}']), ('\u{13ce}',  &['\u{34}']),
        ('\u{13cf}',  &['\u{62}']), ('\u{13d2}',  &['\u{52}']), ('\u{13d4}',  &['\u{57}']),
        ('\u{13d5}',  &['\u{53}']), ('\u{13d9}',  &['\u{56}']), ('\u{13da}',  &['\u{53}']),
        ('\u{13de}',  &['\u{4c}']), ('\u{13df}',  &['\u{43}']), ('\u{13e2}',  &['\u{50}']),
        ('\u{13e6}',  &['\u{4b}']), ('\u{13e7}',  &['\u{64}']), ('\u{13eb}',  &['\u{4f}',
         '\u{335}']), ('\u{13ee}',  &['\u{36}']), ('\u{13f0}',  &['\u{df}']), ('\u{13f2}',
         &['\u{68}',  '\u{314}']), ('\u{13f3}',  &['\u{47}']), ('\u{13f4}',  &['\u{42}']),
        ('\u{13fb}',  &['\u{262}']), ('\u{13fc}',  &['\u{299}']), ('\u{1400}',  &['\u{3d}']),
        ('\u{1403}',  &['\u{394}']), ('\u{140c}',  &['\u{b7}',  '\u{1401}']), ('\u{140d}',
         &['\u{1401}',  '\u{b7}']), ('\u{140e}',  &['\u{b7}',  '\u{394}']), ('\u{140f}',
         &['\u{394}',  '\u{b7}']), ('\u{1410}',  &['\u{b7}',  '\u{1404}']), ('\u{1411}',
         &['\u{1404}',  '\u{b7}']), ('\u{1412}',  &['\u{b7}',  '\u{1405}']), ('\u{1413}',
         &['\u{1405}',  '\u{b7}']), ('\u{1414}',  &['\u{b7}',  '\u{1406}']), ('\u{1415}',
         &['\u{1406}',  '\u{b7}']), ('\u{1417}',  &['\u{b7}',  '\u{140a}']), ('\u{1418}',
         &['\u{140a}',  '\u{b7}']), ('\u{1419}',  &['\u{b7}',  '\u{140b}']), ('\u{141a}',
         &['\u{140b}',  '\u{b7}']), ('\u{1427}',  &['\u{b7}']), ('\u{142b}',  &['\u{1401}',
         '\u{1420}']), ('\u{142c}',  &['\u{394}',  '\u{1420}']), ('\u{142d}',  &['\u{1405}',
         '\u{1420}']), ('\u{142e}',  &['\u{140a}',  '\u{1420}']), ('\u{142f}',  &['\u{56}']),
        ('\u{1431}',  &['\u{245}']), ('\u{1433}',  &['\u{3e}']), ('\u{1437}',  &['\u{b7}',
         '\u{3e}']), ('\u{1438}',  &['\u{3c}']), ('\u{143a}',  &['\u{b7}',  '\u{56}']), ('\u{143b}',
         &['\u{56}',  '\u{b7}']), ('\u{143c}',  &['\u{b7}',  '\u{245}']), ('\u{143d}',  &['\u{245}',
         '\u{b7}']), ('\u{143e}',  &['\u{b7}',  '\u{1432}']), ('\u{143f}',  &['\u{1432}',
         '\u{b7}']), ('\u{1440}',  &['\u{b7}',  '\u{3e}']), ('\u{1441}',  &['\u{3e}',  '\u{b7}']),
        ('\u{1442}',  &['\u{b7}',  '\u{1434}']), ('\u{1443}',  &['\u{1434}',  '\u{b7}']),
        ('\u{1444}',  &['\u{b7}',  '\u{3c}']), ('\u{1445}',  &['\u{3c}',  '\u{b7}']), ('\u{1446}',
         &['\u{b7}',  '\u{1439}']), ('\u{1447}',  &['\u{1439}',  '\u{b7}']), ('\u{144a}',
         &['\u{27}']), ('\u{144c}',  &['\u{55}']), ('\u{144e}',  &['\u{548}']), ('\u{1454}',
         &['\u{b7}',  '\u{1450}']), ('\u{1457}',  &['\u{b7}',  '\u{55}']), ('\u{1458}',  &['\u{55}',
         '\u{b7}']), ('\u{1459}',  &['\u{b7}',  '\u{548}']), ('\u{145a}',  &['\u{548}',  '\u{b7}']),
        ('\u{145b}',  &['\u{b7}',  '\u{144f}']), ('\u{145c}',  &['\u{144f}',  '\u{b7}']),
        ('\u{145d}',  &['\u{b7}',  '\u{1450}']), ('\u{145e}',  &['\u{1450}',  '\u{b7}']),
        ('\u{145f}',  &['\u{b7}',  '\u{1451}']), ('\u{1460}',  &['\u{1451}',  '\u{b7}']),
        ('\u{1461}',  &['\u{b7}',  '\u{1455}']), ('\u{1462}',  &['\u{1455}',  '\u{b7}']),
        ('\u{1463}',  &['\u{b7}',  '\u{1456}']), ('\u{1464}',  &['\u{1456}',  '\u{b7}']),
        ('\u{1467}',  &['\u{55}',  '\u{27}']), ('\u{1468}',  &['\u{548}',  '\u{27}']), ('\u{1469}',
         &['\u{1450}',  '\u{27}']), ('\u{146a}',  &['\u{1455}',  '\u{27}']), ('\u{146d}',
         &['\u{50}']), ('\u{146f}',  &['\u{64}']), ('\u{1472}',  &['\u{62}']), ('\u{1473}',
         &['\u{62}',  '\u{307}']), ('\u{1474}',  &['\u{b7}',  '\u{146b}']), ('\u{1475}',
         &['\u{146b}',  '\u{b7}']), ('\u{1476}',  &['\u{b7}',  '\u{50}']), ('\u{1477}',  &['\u{70}',
         '\u{b7}']), ('\u{1478}',  &['\u{b7}',  '\u{146e}']), ('\u{1479}',  &['\u{146e}',
         '\u{b7}']), ('\u{147a}',  &['\u{b7}',  '\u{64}']), ('\u{147b}',  &['\u{64}',  '\u{b7}']),
        ('\u{147c}',  &['\u{b7}',  '\u{1470}']), ('\u{147d}',  &['\u{1470}',  '\u{b7}']),
        ('\u{147e}',  &['\u{b7}',  '\u{62}']), ('\u{147f}',  &['\u{62}',  '\u{b7}']), ('\u{1480}',
         &['\u{b7}',  '\u{62}',  '\u{307}']), ('\u{1481}',  &['\u{62}',  '\u{307}',  '\u{b7}']),
        ('\u{1485}',  &['\u{146b}',  '\u{27}']), ('\u{1486}',  &['\u{50}',  '\u{27}']), ('\u{1487}',
         &['\u{64}',  '\u{27}']), ('\u{1488}',  &['\u{62}',  '\u{27}']), ('\u{148d}',  &['\u{4a}']),
        ('\u{1492}',  &['\u{b7}',  '\u{1489}']), ('\u{1493}',  &['\u{1489}',  '\u{b7}']),
        ('\u{1494}',  &['\u{b7}',  '\u{148b}']), ('\u{1495}',  &['\u{148b}',  '\u{b7}']),
        ('\u{1496}',  &['\u{b7}',  '\u{148c}']), ('\u{1497}',  &['\u{148c}',  '\u{b7}']),
        ('\u{1498}',  &['\u{b7}',  '\u{4a}']), ('\u{1499}',  &['\u{4a}',  '\u{b7}']), ('\u{149a}',
         &['\u{b7}',  '\u{148e}']), ('\u{149b}',  &['\u{148e}',  '\u{b7}']), ('\u{149c}',
         &['\u{b7}',  '\u{1490}']), ('\u{149d}',  &['\u{1490}',  '\u{b7}']), ('\u{149e}',
         &['\u{b7}',  '\u{1491}']), ('\u{149f}',  &['\u{1491}',  '\u{b7}']), ('\u{14a5}',
         &['\u{393}']), ('\u{14aa}',  &['\u{4c}']), ('\u{14ac}',  &['\u{b7}',  '\u{14a3}']),
        ('\u{14ad}',  &['\u{14a3}',  '\u{b7}']), ('\u{14ae}',  &['\u{b7}',  '\u{393}']),
        ('\u{14af}',  &['\u{393}',  '\u{b7}']), ('\u{14b0}',  &['\u{b7}',  '\u{14a6}']),
        ('\u{14b1}',  &['\u{14a6}',  '\u{b7}']), ('\u{14b2}',  &['\u{b7}',  '\u{14a7}']),
        ('\u{14b3}',  &['\u{14a7}',  '\u{b7}']), ('\u{14b4}',  &['\u{b7}',  '\u{14a8}']),
        ('\u{14b5}',  &['\u{14a8}',  '\u{b7}']), ('\u{14b6}',  &['\u{b7}',  '\u{4c}']), ('\u{14b7}',
         &['\u{6c}',  '\u{b7}']), ('\u{14b8}',  &['\u{b7}',  '\u{14ab}']), ('\u{14b9}',
         &['\u{14ab}',  '\u{b7}']), ('\u{14bf}',  &['\u{32}']), ('\u{14c9}',  &['\u{b7}',
         '\u{14c0}']), ('\u{14ca}',  &['\u{14c0}',  '\u{b7}']), ('\u{14cb}',  &['\u{b7}',
         '\u{14c7}']), ('\u{14cc}',  &['\u{14c7}',  '\u{b7}']), ('\u{14cd}',  &['\u{b7}',
         '\u{14c8}']), ('\u{14ce}',  &['\u{14c8}',  '\u{b7}']), ('\u{14d1}',  &['\u{1421}']),
        ('\u{14dc}',  &['\u{b7}',  '\u{14d3}']), ('\u{14dd}',  &['\u{14d3}',  '\u{b7}']),
        ('\u{14de}',  &['\u{b7}',  '\u{14d5}']), ('\u{14df}',  &['\u{14d5}',  '\u{b7}']),
        ('\u{14e0}',  &['\u{b7}',  '\u{14d6}']), ('\u{14e1}',  &['\u{14d6}',  '\u{b7}']),
        ('\u{14e2}',  &['\u{b7}',  '\u{14d7}']), ('\u{14e3}',  &['\u{14d7}',  '\u{b7}']),
        ('\u{14e4}',  &['\u{b7}',  '\u{14d8}']), ('\u{14e5}',  &['\u{14d8}',  '\u{b7}']),
        ('\u{14e6}',  &['\u{b7}',  '\u{14da}']), ('\u{14e7}',  &['\u{14da}',  '\u{b7}']),
        ('\u{14e8}',  &['\u{b7}',  '\u{14db}']), ('\u{14e9}',  &['\u{14db}',  '\u{b7}']),
        ('\u{14f6}',  &['\u{b7}',  '\u{14ed}']), ('\u{14f7}',  &['\u{14ed}',  '\u{b7}']),
        ('\u{14f8}',  &['\u{b7}',  '\u{14ef}']), ('\u{14f9}',  &['\u{14ef}',  '\u{b7}']),
        ('\u{14fa}',  &['\u{b7}',  '\u{14f0}']), ('\u{14fb}',  &['\u{14f0}',  '\u{b7}']),
        ('\u{14fc}',  &['\u{b7}',  '\u{14f1}']), ('\u{14fd}',  &['\u{14f1}',  '\u{b7}']),
        ('\u{14fe}',  &['\u{b7}',  '\u{14f2}']), ('\u{14ff}',  &['\u{14f2}',  '\u{b7}']),
        ('\u{1500}',  &['\u{b7}',  '\u{14f4}']), ('\u{1501}',  &['\u{14f4}',  '\u{b7}']),
        ('\u{1502}',  &['\u{b7}',  '\u{14f5}']), ('\u{1503}',  &['\u{14f5}',  '\u{b7}']),
        ('\u{150c}',  &['\u{150b}',  '\u{3c}']), ('\u{150d}',  &['\u{150b}',  '\u{1455}']),
        ('\u{150e}',  &['\u{150b}',  '\u{62}']), ('\u{150f}',  &['\u{150b}',  '\u{1490}']),
        ('\u{1517}',  &['\u{b7}',  '\u{1510}']), ('\u{1518}',  &['\u{1510}',  '\u{b7}']),
        ('\u{1519}',  &['\u{b7}',  '\u{1511}']), ('\u{151a}',  &['\u{1511}',  '\u{b7}']),
        ('\u{151b}',  &['\u{b7}',  '\u{1512}']), ('\u{151c}',  &['\u{1512}',  '\u{b7}']),
        ('\u{151d}',  &['\u{b7}',  '\u{1513}']), ('\u{151e}',  &['\u{1513}',  '\u{b7}']),
        ('\u{151f}',  &['\u{b7}',  '\u{1514}']), ('\u{1520}',  &['\u{1514}',  '\u{b7}']),
        ('\u{1521}',  &['\u{b7}',  '\u{1515}']), ('\u{1522}',  &['\u{1515}',  '\u{b7}']),
        ('\u{1523}',  &['\u{b7}',  '\u{1516}']), ('\u{1524}',  &['\u{1516}',  '\u{b7}']),
        ('\u{152f}',  &['\u{b7}',  '\u{34}']), ('\u{1530}',  &['\u{34}',  '\u{b7}']), ('\u{1531}',
         &['\u{b7}',  '\u{1528}']), ('\u{1532}',  &['\u{1528}',  '\u{b7}']), ('\u{1533}',
         &['\u{b7}',  '\u{1529}']), ('\u{1534}',  &['\u{1529}',  '\u{b7}']), ('\u{1535}',
         &['\u{b7}',  '\u{152a}']), ('\u{1536}',  &['\u{152a}',  '\u{b7}']), ('\u{1537}',
         &['\u{b7}',  '\u{152b}']), ('\u{1538}',  &['\u{152b}',  '\u{b7}']), ('\u{1539}',
         &['\u{b7}',  '\u{152d}']), ('\u{153a}',  &['\u{152d}',  '\u{b7}']), ('\u{153b}',
         &['\u{b7}',  '\u{152e}']), ('\u{153c}',  &['\u{152e}',  '\u{b7}']), ('\u{1540}',
         &['\u{1429}']), ('\u{1541}',  &['\u{78}']), ('\u{154e}',  &['\u{b7}',  '\u{154c}']),
        ('\u{154f}',  &['\u{154c}',  '\u{b7}']), ('\u{155b}',  &['\u{b7}',  '\u{155a}']),
        ('\u{155c}',  &['\u{155a}',  '\u{b7}']), ('\u{1568}',  &['\u{b7}',  '\u{1567}']),
        ('\u{1569}',  &['\u{1567}',  '\u{b7}']), ('\u{1577}',  &['\u{1e9f}']), ('\u{157c}',
         &['\u{48}']), ('\u{157d}',  &['\u{78}']), ('\u{157e}',  &['\u{1550}',  '\u{146c}']),
        ('\u{157f}',  &['\u{1550}',  '\u{50}']), ('\u{1580}',  &['\u{1550}',  '\u{146e}']),
        ('\u{1581}',  &['\u{1550}',  '\u{64}']), ('\u{1582}',  &['\u{1550}',  '\u{1470}']),
        ('\u{1583}',  &['\u{1550}',  '\u{62}']), ('\u{1584}',  &['\u{1550}',  '\u{62}',
         '\u{307}']), ('\u{1585}',  &['\u{1550}',  '\u{1483}']), ('\u{1587}',  &['\u{52}']),
        ('\u{158e}',  &['\u{1595}',  '\u{148a}']), ('\u{158f}',  &['\u{1595}',  '\u{148b}']),
        ('\u{1590}',  &['\u{1595}',  '\u{148c}']), ('\u{1591}',  &['\u{1595}',  '\u{4a}']),
        ('\u{1592}',  &['\u{1595}',  '\u{148e}']), ('\u{1593}',  &['\u{1595}',  '\u{1490}']),
        ('\u{1594}',  &['\u{1595}',  '\u{1491}']), ('\u{15af}',  &['\u{62}']), ('\u{15b4}',
         &['\u{46}']), ('\u{15b5}',  &['\u{2132}']), ('\u{15b7}',  &['\u{a7fb}']), ('\u{15c4}',
         &['\u{2c6f}']), ('\u{15c5}',  &['\u{41}']), ('\u{15de}',  &['\u{44}']), ('\u{15ea}',
         &['\u{44}']), ('\u{15ef}',  &['\u{460}']), ('\u{15f0}',  &['\u{4d}']), ('\u{15f7}',
         &['\u{42}']), ('\u{1602}',  &['\u{1490}']), ('\u{1603}',  &['\u{1489}']), ('\u{1604}',
         &['\u{14d3}']), ('\u{1607}',  &['\u{14da}']), ('\u{1622}',  &['\u{1543}']), ('\u{1623}',
         &['\u{1546}']), ('\u{1624}',  &['\u{154a}']), ('\u{162e}',  &['\u{1b1}']), ('\u{162f}',
         &['\u{3a9}']), ('\u{1634}',  &['\u{1b1}']), ('\u{1635}',  &['\u{3a9}']), ('\u{166d}',
         &['\u{58}']), ('\u{166e}',  &['\u{78}']), ('\u{166f}',  &['\u{1550}',  '\u{146b}']),
        ('\u{1670}',  &['\u{1595}',  '\u{1489}']), ('\u{1671}',  &['\u{1596}',  '\u{148b}']),
        ('\u{1672}',  &['\u{1596}',  '\u{148c}']), ('\u{1673}',  &['\u{1596}',  '\u{4a}']),
        ('\u{1674}',  &['\u{1596}',  '\u{148e}']), ('\u{1675}',  &['\u{1596}',  '\u{1490}']),
        ('\u{1676}',  &['\u{1596}',  '\u{1491}']), ('\u{1677}',  &['\u{15a7}',  '\u{b7}']),
        ('\u{1678}',  &['\u{15a8}',  '\u{b7}']), ('\u{1679}',  &['\u{15a9}',  '\u{b7}']),
        ('\u{167a}',  &['\u{15aa}',  '\u{b7}']), ('\u{167b}',  &['\u{15ab}',  '\u{b7}']),
        ('\u{167c}',  &['\u{15ac}',  '\u{b7}']), ('\u{167d}',  &['\u{15ad}',  '\u{b7}']),
        ('\u{1680}',  &['\u{20}']), ('\u{16b2}',  &['\u{3c}']), ('\u{16b7}',  &['\u{58}']),
        ('\u{16c1}',  &['\u{6c}']), ('\u{16c2}',  &['\u{16bd}']), ('\u{16cc}',  &['\u{27}']),
        ('\u{16d5}',  &['\u{4b}']), ('\u{16d6}',  &['\u{4d}']), ('\u{16d8}',  &['\u{3a8}']),
        ('\u{16e1}',  &['\u{16bc}']), ('\u{16eb}',  &['\u{b7}']), ('\u{16ec}',  &['\u{3a}']),
        ('\u{16ed}',  &['\u{2b}']), ('\u{16f0}',  &['\u{3a6}']), ('\u{1734}',  &['\u{1715}']),
        ('\u{1735}',  &['\u{2f}']), ('\u{17a3}',  &['\u{17a2}']), ('\u{17b7}',  &['\u{e34}']),
        ('\u{17b8}',  &['\u{e35}']), ('\u{17b9}',  &['\u{e36}']), ('\u{17ba}',  &['\u{e37}']),
        ('\u{17c6}',  &['\u{30a}']), ('\u{17cb}',  &['\u{e48}']), ('\u{17d3}',  &['\u{30a}']),
        ('\u{17d4}',  &['\u{e2f}']), ('\u{17d5}',  &['\u{e5a}']), ('\u{17d9}',  &['\u{e4f}']),
        ('\u{17da}',  &['\u{e5b}']), ('\u{1803}',  &['\u{3a}']), ('\u{1809}',  &['\u{3a}']),
        ('\u{1855}',  &['\u{1835}']), ('\u{1896}',  &['\u{185c}']), ('\u{18b3}',  &['\u{b7}',
         '\u{18b1}']), ('\u{18b6}',  &['\u{b7}',  '\u{18b4}']), ('\u{18b9}',  &['\u{b7}',
         '\u{18b8}']), ('\u{18c2}',  &['\u{b7}',  '\u{18c0}']), ('\u{18c6}',  &['\u{b7}',
         '\u{14c2}']), ('\u{18c7}',  &['\u{14c2}',  '\u{b7}']), ('\u{18c8}',  &['\u{b7}',
         '\u{14c3}']), ('\u{18c9}',  &['\u{14c3}',  '\u{b7}']), ('\u{18ca}',  &['\u{b7}',
         '\u{14c4}']), ('\u{18cb}',  &['\u{14c4}',  '\u{b7}']), ('\u{18cc}',  &['\u{b7}',
         '\u{14c5}']), ('\u{18cd}',  &['\u{14c5}',  '\u{b7}']), ('\u{18ce}',  &['\u{b7}',
         '\u{1543}']), ('\u{18cf}',  &['\u{b7}',  '\u{1546}']), ('\u{18d0}',  &['\u{b7}',
         '\u{1547}']), ('\u{18d1}',  &['\u{b7}',  '\u{1548}']), ('\u{18d2}',  &['\u{b7}',
         '\u{1549}']), ('\u{18d3}',  &['\u{b7}',  '\u{154b}']), ('\u{18db}',  &['\u{18f5}']),
        ('\u{18dc}',  &['\u{18df}',  '\u{141e}']), ('\u{18dd}',  &['\u{141e}',  '\u{18df}']),
        ('\u{18e0}',  &['\u{1543}',  '\u{b7}']), ('\u{18e3}',  &['\u{155e}',  '\u{b7}']),
        ('\u{18e4}',  &['\u{1566}',  '\u{b7}']), ('\u{18e5}',  &['\u{156b}',  '\u{b7}']),
        ('\u{18e8}',  &['\u{1586}',  '\u{b7}']), ('\u{18ea}',  &['\u{1597}',  '\u{b7}']),
        ('\u{18ed}',  &['\u{460}',  '\u{b7}']), ('\u{18f0}',  &['\u{15f4}',  '\u{b7}']),
        ('\u{18f2}',  &['\u{161b}',  '\u{b7}']), ('\u{19d0}',  &['\u{199e}']), ('\u{19d1}',
         &['\u{19b1}']), ('\u{1a80}',  &['\u{1a45}']), ('\u{1a90}',  &['\u{1a45}']), ('\u{1aa9}',
         &['\u{1aa8}',  '\u{1aa8}']), ('\u{1aab}',  &['\u{1aaa}',  '\u{1aa8}']), ('\u{1ab4}',
         &['\u{6db}']), ('\u{1ab7}',  &['\u{328}']), ('\u{1b52}',  &['\u{1b0d}']), ('\u{1b53}',
         &['\u{1b11}']), ('\u{1b58}',  &['\u{1b28}']), ('\u{1b5c}',  &['\u{1b50}']), ('\u{1b5f}',
         &['\u{1b5e}',  '\u{1b5e}']), ('\u{1c3c}',  &['\u{1c3b}',  '\u{1c3b}']), ('\u{1c7f}',
         &['\u{1c7e}',  '\u{1c7e}']), ('\u{1cd0}',  &['\u{302}']), ('\u{1cd2}',  &['\u{304}']),
        ('\u{1cd3}',  &['\u{27}',  '\u{27}']), ('\u{1cd5}',  &['\u{32b}']), ('\u{1cd8}',
         &['\u{32e}']), ('\u{1cd9}',  &['\u{32d}']), ('\u{1cda}',  &['\u{30e}']), ('\u{1cdc}',
         &['\u{329}']), ('\u{1cdd}',  &['\u{323}']), ('\u{1cde}',  &['\u{324}']), ('\u{1ced}',
         &['\u{316}']), ('\u{1d04}',  &['\u{63}']), ('\u{1d08}',  &['\u{25c}']), ('\u{1d0b}',
         &['\u{138}']), ('\u{1d0d}',  &['\u{28d}']), ('\u{1d0f}',  &['\u{6f}']), ('\u{1d10}',
         &['\u{254}']), ('\u{1d11}',  &['\u{6f}']), ('\u{1d14}',  &['\u{1dd}',  '\u{6f}']),
        ('\u{1d1c}',  &['\u{75}']), ('\u{1d20}',  &['\u{76}']), ('\u{1d21}',  &['\u{77}']),
        ('\u{1d22}',  &['\u{7a}']), ('\u{1d24}',  &['\u{1a8}']), ('\u{1d26}',  &['\u{72}']),
        ('\u{1d27}',  &['\u{28c}']), ('\u{1d28}',  &['\u{3c0}']), ('\u{1d29}',  &['\u{1d18}']),
        ('\u{1d2b}',  &['\u{43b}']), ('\u{1d3e}',  &['\u{18d6}']), ('\u{1d52}',  &['\u{ba}']),
        ('\u{1d6b}',  &['\u{75}',  '\u{65}']), ('\u{1d6e}',  &['\u{66}',  '\u{334}']), ('\u{1d6f}',
         &['\u{72}',  '\u{6e}',  '\u{334}']), ('\u{1d70}',  &['\u{6e}',  '\u{334}']), ('\u{1d72}',
         &['\u{72}',  '\u{334}']), ('\u{1d73}',  &['\u{27e}',  '\u{334}']), ('\u{1d74}',
         &['\u{73}',  '\u{334}']), ('\u{1d75}',  &['\u{74}',  '\u{334}']), ('\u{1d76}',  &['\u{7a}',
         '\u{334}']), ('\u{1d78}',  &['\u{1d34}']), ('\u{1d7b}',  &['\u{69}',  '\u{335}']),
        ('\u{1d7c}',  &['\u{69}',  '\u{335}']), ('\u{1d7d}',  &['\u{70}',  '\u{335}']), ('\u{1d7e}',
         &['\u{75}',  '\u{335}']), ('\u{1d7f}',  &['\u{28a}',  '\u{335}']), ('\u{1d83}',
         &['\u{67}']), ('\u{1d8c}',  &['\u{79}']), ('\u{1d90}',  &['\u{24b}']), ('\u{1d9f}',
         &['\u{1d4b}']), ('\u{1da2}',  &['\u{1d4d}']), ('\u{1dba}',  &['\u{18d4}']), ('\u{1dbb}',
         &['\u{1646}']), ('\u{1dee}',  &['\u{2dec}']), ('\u{1e43}',  &['\u{ab51}']), ('\u{1e9a}',
         &['\u{1ea3}']), ('\u{1e9d}',  &['\u{66}']), ('\u{1e9e}',  &['\u{df}']), ('\u{1eff}',
         &['\u{79}']), ('\u{1f7d}',  &['\u{1ff4}']), ('\u{1fbd}',  &['\u{27}']), ('\u{1fbe}',
         &['\u{69}']), ('\u{1fbf}',  &['\u{27}']), ('\u{1fc0}',  &['\u{7e}']), ('\u{1fef}',
         &['\u{27}']), ('\u{1ff6}',  &['\u{13ef}']), ('\u{1ffd}',  &['\u{27}']), ('\u{1ffe}',
         &['\u{27}']), ('\u{2000}',  &['\u{20}']), ('\u{2001}',  &['\u{20}']), ('\u{2002}',
         &['\u{20}']), ('\u{2003}',  &['\u{20}']), ('\u{2004}',  &['\u{20}']), ('\u{2005}',
         &['\u{20}']), ('\u{2006}',  &['\u{20}']), ('\u{2007}',  &['\u{20}']), ('\u{2008}',
         &['\u{20}']), ('\u{2009}',  &['\u{20}']), ('\u{200a}',  &['\u{20}']), ('\u{2010}',
         &['\u{2d}']), ('\u{2011}',  &['\u{2d}']), ('\u{2012}',  &['\u{2d}']), ('\u{2013}',
         &['\u{2d}']), ('\u{2014}',  &['\u{30fc}']), ('\u{2015}',  &['\u{30fc}']), ('\u{2016}',
         &['\u{6c}',  '\u{6c}']), ('\u{2018}',  &['\u{27}']), ('\u{2019}',  &['\u{27}']),
        ('\u{201a}',  &['\u{2c}']), ('\u{201b}',  &['\u{27}']), ('\u{201c}',  &['\u{27}',
         '\u{27}']), ('\u{201d}',  &['\u{27}',  '\u{27}']), ('\u{201f}',  &['\u{27}',  '\u{27}']),
        ('\u{2022}',  &['\u{b7}']), ('\u{2024}',  &['\u{2e}']), ('\u{2025}',  &['\u{2e}',
         '\u{2e}']), ('\u{2026}',  &['\u{2e}',  '\u{2e}',  '\u{2e}']), ('\u{2027}',  &['\u{b7}']),
        ('\u{2028}',  &['\u{20}']), ('\u{2029}',  &['\u{20}']), ('\u{202f}',  &['\u{20}']),
        ('\u{2030}',  &['\u{ba}',  '\u{2f}',  '\u{2080}',  '\u{2080}']), ('\u{2031}',  &['\u{ba}',
         '\u{2f}',  '\u{2080}',  '\u{2080}',  '\u{2080}']), ('\u{2032}',  &['\u{27}']), ('\u{2033}',
         &['\u{27}',  '\u{27}']), ('\u{2034}',  &['\u{27}',  '\u{27}',  '\u{27}']), ('\u{2035}',
         &['\u{27}']), ('\u{2036}',  &['\u{27}',  '\u{27}']), ('\u{2037}',  &['\u{27}',  '\u{27}',
         '\u{27}']), ('\u{2039}',  &['\u{3c}']), ('\u{203a}',  &['\u{3e}']), ('\u{203c}',
         &['\u{21}',  '\u{21}']), ('\u{203e}',  &['\u{2c9}']), ('\u{2041}',  &['\u{2f}']),
        ('\u{2043}',  &['\u{2d}']), ('\u{2044}',  &['\u{2f}']), ('\u{2047}',  &['\u{3f}',
         '\u{3f}']), ('\u{2048}',  &['\u{3f}',  '\u{21}']), ('\u{2049}',  &['\u{21}',  '\u{3f}']),
        ('\u{204e}',  &['\u{2a}']), ('\u{2052}',  &['\u{ba}',  '\u{2f}',  '\u{2080}']), ('\u{2053}',
         &['\u{7e}']), ('\u{2057}',  &['\u{27}',  '\u{27}',  '\u{27}',  '\u{27}']), ('\u{205a}',
         &['\u{3a}']), ('\u{205d}',  &['\u{2d57}']), ('\u{205e}',  &['\u{2d42}']), ('\u{205f}',
         &['\u{20}']), ('\u{2070}',  &['\u{ba}']), ('\u{2079}',  &['\u{a770}']), ('\u{20a1}',
         &['\u{43}',  '\u{20eb}']), ('\u{20a4}',  &['\u{a3}']), ('\u{20a5}',  &['\u{72}',  '\u{6e}',
         '\u{338}']), ('\u{20a8}',  &['\u{52}',  '\u{73}']), ('\u{20a9}',  &['\u{57}',  '\u{335}']),
        ('\u{20ab}',  &['\u{64}',  '\u{335}',  '\u{331}']), ('\u{20ac}',  &['\u{a792}']),
        ('\u{20ad}',  &['\u{4b}',  '\u{335}']), ('\u{20ae}',  &['\u{54}',  '\u{20eb}']),
        ('\u{20b6}',  &['\u{6c}',  '\u{74}']), ('\u{20bd}',  &['\u{554}']), ('\u{20db}',
         &['\u{6db}']), ('\u{2100}',  &['\u{61}',  '\u{2f}',  '\u{63}']), ('\u{2101}',  &['\u{61}',
         '\u{2f}',  '\u{73}']), ('\u{2102}',  &['\u{43}']), ('\u{2103}',  &['\u{b0}',  '\u{43}']),
        ('\u{2105}',  &['\u{63}',  '\u{2f}',  '\u{6f}']), ('\u{2106}',  &['\u{63}',  '\u{2f}',
         '\u{75}']), ('\u{2107}',  &['\u{190}']), ('\u{2108}',  &['\u{42d}']), ('\u{2109}',
         &['\u{b0}',  '\u{46}']), ('\u{210a}',  &['\u{67}']), ('\u{210b}',  &['\u{48}']),
        ('\u{210c}',  &['\u{48}']), ('\u{210d}',  &['\u{48}']), ('\u{210e}',  &['\u{68}']),
        ('\u{210f}',  &['\u{68}',  '\u{335}']), ('\u{2110}',  &['\u{6c}']), ('\u{2111}',
         &['\u{6c}']), ('\u{2112}',  &['\u{4c}']), ('\u{2113}',  &['\u{6c}']), ('\u{2115}',
         &['\u{4e}']), ('\u{2116}',  &['\u{4e}',  '\u{6f}']), ('\u{2119}',  &['\u{50}']),
        ('\u{211a}',  &['\u{51}']), ('\u{211b}',  &['\u{52}']), ('\u{211c}',  &['\u{52}']),
        ('\u{211d}',  &['\u{52}']), ('\u{2121}',  &['\u{54}',  '\u{45}',  '\u{4c}']), ('\u{2124}',
         &['\u{5a}']), ('\u{2126}',  &['\u{3a9}']), ('\u{2127}',  &['\u{1b1}']), ('\u{2128}',
         &['\u{5a}']), ('\u{2129}',  &['\u{27f}']), ('\u{212a}',  &['\u{4b}']), ('\u{212c}',
         &['\u{42}']), ('\u{212d}',  &['\u{43}']), ('\u{212e}',  &['\u{65}']), ('\u{212f}',
         &['\u{65}']), ('\u{2130}',  &['\u{45}']), ('\u{2131}',  &['\u{46}']), ('\u{2133}',
         &['\u{4d}']), ('\u{2134}',  &['\u{6f}']), ('\u{2135}',  &['\u{5d0}']), ('\u{2136}',
         &['\u{5d1}']), ('\u{2137}',  &['\u{5d2}']), ('\u{2138}',  &['\u{5d3}']), ('\u{2139}',
         &['\u{69}']), ('\u{213b}',  &['\u{46}',  '\u{41}',  '\u{58}']), ('\u{213c}',
         &['\u{3c0}']), ('\u{213d}',  &['\u{79}']), ('\u{213e}',  &['\u{393}']), ('\u{213f}',
         &['\u{3a0}']), ('\u{2140}',  &['\u{1a9}']), ('\u{2141}',  &['\u{a4e8}']), ('\u{2142}',
         &['\u{a4f6}']), ('\u{2143}',  &['\u{16f00}']), ('\u{2145}',  &['\u{44}']), ('\u{2146}',
         &['\u{64}']), ('\u{2147}',  &['\u{65}']), ('\u{2148}',  &['\u{69}']), ('\u{2149}',
         &['\u{6a}']), ('\u{2160}',  &['\u{6c}']), ('\u{2161}',  &['\u{6c}',  '\u{6c}']),
        ('\u{2162}',  &['\u{6c}',  '\u{6c}',  '\u{6c}']), ('\u{2163}',  &['\u{6c}',  '\u{56}']),
        ('\u{2164}',  &['\u{56}']), ('\u{2165}',  &['\u{56}',  '\u{6c}']), ('\u{2166}',  &['\u{56}',
         '\u{6c}',  '\u{6c}']), ('\u{2167}',  &['\u{56}',  '\u{6c}',  '\u{6c}',  '\u{6c}']),
        ('\u{2168}',  &['\u{6c}',  '\u{58}']), ('\u{2169}',  &['\u{58}']), ('\u{216a}',  &['\u{58}',
         '\u{6c}']), ('\u{216b}',  &['\u{58}',  '\u{6c}',  '\u{6c}']), ('\u{216c}',  &['\u{4c}']),
        ('\u{216d}',  &['\u{43}']), ('\u{216e}',  &['\u{44}']), ('\u{216f}',  &['\u{4d}']),
        ('\u{2170}',  &['\u{69}']), ('\u{2171}',  &['\u{69}',  '\u{69}']), ('\u{2172}',  &['\u{69}',
         '\u{69}',  '\u{69}']), ('\u{2173}',  &['\u{69}',  '\u{76}']), ('\u{2174}',  &['\u{76}']),
        ('\u{2175}',  &['\u{76}',  '\u{69}']), ('\u{2176}',  &['\u{76}',  '\u{69}',  '\u{69}']),
        ('\u{2177}',  &['\u{76}',  '\u{69}',  '\u{69}',  '\u{69}']), ('\u{2178}',  &['\u{69}',
         '\u{78}']), ('\u{2179}',  &['\u{78}']), ('\u{217a}',  &['\u{78}',  '\u{69}']), ('\u{217b}',
         &['\u{78}',  '\u{69}',  '\u{69}']), ('\u{217c}',  &['\u{6c}']), ('\u{217d}',  &['\u{63}']),
        ('\u{217e}',  &['\u{64}']), ('\u{217f}',  &['\u{72}',  '\u{6e}']), ('\u{2183}',
         &['\u{186}']), ('\u{2184}',  &['\u{254}']), ('\u{2191}',  &['\u{16cf}']), ('\u{2195}',
         &['\u{16e8}']), ('\u{21b5}',  &['\u{21b2}']), ('\u{21ba}',  &['\u{1f10e}']), ('\u{21be}',
         &['\u{16da}']), ('\u{21bf}',  &['\u{16d0}']), ('\u{2200}',  &['\u{2c6f}']), ('\u{2203}',
         &['\u{18e}']), ('\u{2206}',  &['\u{394}']), ('\u{220f}',  &['\u{3a0}']), ('\u{2211}',
         &['\u{1a9}']), ('\u{2212}',  &['\u{2d}']), ('\u{2214}',  &['\u{2b}',  '\u{307}']),
        ('\u{2215}',  &['\u{2f}']), ('\u{2216}',  &['\u{5c}']), ('\u{2217}',  &['\u{2a}']),
        ('\u{2218}',  &['\u{b0}']), ('\u{2219}',  &['\u{b7}']), ('\u{221e}',  &['\u{6f}',
         '\u{6f}']), ('\u{2223}',  &['\u{6c}']), ('\u{2225}',  &['\u{6c}',  '\u{6c}']), ('\u{2228}',
         &['\u{76}']), ('\u{2229}',  &['\u{548}']), ('\u{222a}',  &['\u{55}']), ('\u{222b}',
         &['\u{283}']), ('\u{222c}',  &['\u{283}',  '\u{283}']), ('\u{222d}',  &['\u{283}',
         '\u{283}',  '\u{283}']), ('\u{222f}',  &['\u{222e}',  '\u{222e}']), ('\u{2230}',
         &['\u{222e}',  '\u{222e}',  '\u{222e}']), ('\u{2236}',  &['\u{3a}']), ('\u{2238}',
         &['\u{2d}',  '\u{307}']), ('\u{223c}',  &['\u{7e}']), ('\u{2250}',  &['\u{3d}',
         '\u{307}']), ('\u{2251}',  &['\u{3d}',  '\u{307}',  '\u{323}']), ('\u{2257}',  &['\u{3d}',
         '\u{30a}']), ('\u{2259}',  &['\u{3d}',  '\u{302}']), ('\u{225a}',  &['\u{3d}',
         '\u{306}']), ('\u{225e}',  &['\u{3d}',  '\u{36b}']), ('\u{2263}',  &['\u{2261}']),
        ('\u{226a}',  &['\u{3c}',  '\u{3c}']), ('\u{226b}',  &['\u{3e}',  '\u{3e}']), ('\u{2282}',
         &['\u{1455}']), ('\u{2283}',  &['\u{1450}']), ('\u{2295}',  &['\u{102a8}']), ('\u{2296}',
         &['\u{4f}',  '\u{335}']), ('\u{2299}',  &['\u{298}']), ('\u{229d}',  &['\u{4f}',
         '\u{335}']), ('\u{22a4}',  &['\u{54}']), ('\u{22a5}',  &['\u{a4d5}']), ('\u{22c0}',
         &['\u{2227}']), ('\u{22c1}',  &['\u{76}']), ('\u{22c2}',  &['\u{548}']), ('\u{22c3}',
         &['\u{55}']), ('\u{22c4}',  &['\u{16dc}']), ('\u{22c5}',  &['\u{b7}']), ('\u{22c8}',
         &['\u{16de}']), ('\u{22d6}',  &['\u{3c}',  '\u{b7}']), ('\u{22d7}',  &['\u{b7}',
         '\u{3e}']), ('\u{22d8}',  &['\u{3c}',  '\u{3c}',  '\u{3c}']), ('\u{22d9}',  &['\u{3e}',
         '\u{3e}',  '\u{3e}']), ('\u{22ee}',  &['\u{2d57}']), ('\u{22ef}',  &['\u{b7}',  '\u{b7}',
         '\u{b7}']), ('\u{22f4}',  &['\u{a793}']), ('\u{22ff}',  &['\u{45}']), ('\u{2300}',
         &['\u{2205}']), ('\u{2325}',  &['\u{2324}']), ('\u{2329}',  &['\u{276c}']), ('\u{232a}',
         &['\u{276d}']), ('\u{2341}',  &['\u{303c}']), ('\u{2359}',  &['\u{394}',  '\u{332}']),
        ('\u{235a}',  &['\u{16dc}',  '\u{332}']), ('\u{235c}',  &['\u{b0}',  '\u{332}']),
        ('\u{235f}',  &['\u{229b}']), ('\u{2361}',  &['\u{54}',  '\u{308}']), ('\u{2362}',
         &['\u{2207}',  '\u{308}']), ('\u{2363}',  &['\u{22c6}',  '\u{308}']), ('\u{2364}',
         &['\u{b0}',  '\u{308}']), ('\u{2365}',  &['\u{629}']), ('\u{2368}',  &['\u{7e}',
         '\u{308}']), ('\u{2369}',  &['\u{1435}']), ('\u{236b}',  &['\u{2207}',  '\u{334}']),
        ('\u{236c}',  &['\u{4f}',  '\u{335}']), ('\u{2373}',  &['\u{69}']), ('\u{2374}',
         &['\u{70}']), ('\u{2375}',  &['\u{3c9}']), ('\u{2376}',  &['\u{61}',  '\u{332}']),
        ('\u{2377}',  &['\u{a793}',  '\u{332}']), ('\u{2378}',  &['\u{69}',  '\u{332}']),
        ('\u{2379}',  &['\u{3c9}',  '\u{332}']), ('\u{237a}',  &['\u{61}']), ('\u{237f}',
         &['\u{16bd}']), ('\u{239c}',  &['\u{4e28}']), ('\u{239f}',  &['\u{4e28}']), ('\u{23a2}',
         &['\u{4e28}']), ('\u{23a5}',  &['\u{4e28}']), ('\u{23aa}',  &['\u{4e28}']), ('\u{23ae}',
         &['\u{4e28}']), ('\u{23c1}',  &['\u{2355}']), ('\u{23c2}',  &['\u{234e}']), ('\u{23c3}',
         &['\u{234b}']), ('\u{23c6}',  &['\u{236d}']), ('\u{23e8}',  &['\u{2081}',  '\u{2080}']),
        ('\u{23fc}',  &['\u{23fb}']), ('\u{23fd}',  &['\u{6c}']), ('\u{23fe}',  &['\u{263e}']),
        ('\u{244a}',  &['\u{5c}',  '\u{5c}']), ('\u{2460}',  &['\u{2780}']), ('\u{2461}',
         &['\u{2781}']), ('\u{2462}',  &['\u{2782}']), ('\u{2463}',  &['\u{2783}']), ('\u{2464}',
         &['\u{2784}']), ('\u{2465}',  &['\u{2785}']), ('\u{2466}',  &['\u{2786}']), ('\u{2467}',
         &['\u{2787}']), ('\u{2468}',  &['\u{2788}']), ('\u{2469}',  &['\u{2789}']), ('\u{2474}',
         &['\u{28}',  '\u{6c}',  '\u{29}']), ('\u{2475}',  &['\u{28}',  '\u{32}',  '\u{29}']),
        ('\u{2476}',  &['\u{28}',  '\u{33}',  '\u{29}']), ('\u{2477}',  &['\u{28}',  '\u{34}',
         '\u{29}']), ('\u{2478}',  &['\u{28}',  '\u{35}',  '\u{29}']), ('\u{2479}',  &['\u{28}',
         '\u{36}',  '\u{29}']), ('\u{247a}',  &['\u{28}',  '\u{37}',  '\u{29}']), ('\u{247b}',
         &['\u{28}',  '\u{38}',  '\u{29}']), ('\u{247c}',  &['\u{28}',  '\u{39}',  '\u{29}']),
        ('\u{247d}',  &['\u{28}',  '\u{6c}',  '\u{4f}',  '\u{29}']), ('\u{247e}',  &['\u{28}',
         '\u{6c}',  '\u{6c}',  '\u{29}']), ('\u{247f}',  &['\u{28}',  '\u{6c}',  '\u{32}',
         '\u{29}']), ('\u{2480}',  &['\u{28}',  '\u{6c}',  '\u{33}',  '\u{29}']), ('\u{2481}',
         &['\u{28}',  '\u{6c}',  '\u{34}',  '\u{29}']), ('\u{2482}',  &['\u{28}',  '\u{6c}',
         '\u{35}',  '\u{29}']), ('\u{2483}',  &['\u{28}',  '\u{6c}',  '\u{36}',  '\u{29}']),
        ('\u{2484}',  &['\u{28}',  '\u{6c}',  '\u{37}',  '\u{29}']), ('\u{2485}',  &['\u{28}',
         '\u{6c}',  '\u{38}',  '\u{29}']), ('\u{2486}',  &['\u{28}',  '\u{6c}',  '\u{39}',
         '\u{29}']), ('\u{2487}',  &['\u{28}',  '\u{32}',  '\u{4f}',  '\u{29}']), ('\u{2488}',
         &['\u{6c}',  '\u{2e}']), ('\u{2489}',  &['\u{32}',  '\u{2e}']), ('\u{248a}',  &['\u{33}',
         '\u{2e}']), ('\u{248b}',  &['\u{34}',  '\u{2e}']), ('\u{248c}',  &['\u{35}',  '\u{2e}']),
        ('\u{248d}',  &['\u{36}',  '\u{2e}']), ('\u{248e}',  &['\u{37}',  '\u{2e}']), ('\u{248f}',
         &['\u{38}',  '\u{2e}']), ('\u{2490}',  &['\u{39}',  '\u{2e}']), ('\u{2491}',  &['\u{6c}',
         '\u{4f}',  '\u{2e}']), ('\u{2492}',  &['\u{6c}',  '\u{6c}',  '\u{2e}']), ('\u{2493}',
         &['\u{6c}',  '\u{32}',  '\u{2e}']), ('\u{2494}',  &['\u{6c}',  '\u{33}',  '\u{2e}']),
        ('\u{2495}',  &['\u{6c}',  '\u{34}',  '\u{2e}']), ('\u{2496}',  &['\u{6c}',  '\u{35}',
         '\u{2e}']), ('\u{2497}',  &['\u{6c}',  '\u{36}',  '\u{2e}']), ('\u{2498}',  &['\u{6c}',
         '\u{37}',  '\u{2e}']), ('\u{2499}',  &['\u{6c}',  '\u{38}',  '\u{2e}']), ('\u{249a}',
         &['\u{6c}',  '\u{39}',  '\u{2e}']), ('\u{249b}',  &['\u{32}',  '\u{4f}',  '\u{2e}']),
        ('\u{249c}',  &['\u{28}',  '\u{61}',  '\u{29}']), ('\u{249d}',  &['\u{28}',  '\u{62}',
         '\u{29}']), ('\u{249e}',  &['\u{28}',  '\u{63}',  '\u{29}']), ('\u{249f}',  &['\u{28}',
         '\u{64}',  '\u{29}']), ('\u{24a0}',  &['\u{28}',  '\u{65}',  '\u{29}']), ('\u{24a1}',
         &['\u{28}',  '\u{66}',  '\u{29}']), ('\u{24a2}',  &['\u{28}',  '\u{67}',  '\u{29}']),
        ('\u{24a3}',  &['\u{28}',  '\u{68}',  '\u{29}']), ('\u{24a4}',  &['\u{28}',  '\u{69}',
         '\u{29}']), ('\u{24a5}',  &['\u{28}',  '\u{6a}',  '\u{29}']), ('\u{24a6}',  &['\u{28}',
         '\u{6b}',  '\u{29}']), ('\u{24a7}',  &['\u{28}',  '\u{6c}',  '\u{29}']), ('\u{24a8}',
         &['\u{28}',  '\u{72}',  '\u{6e}',  '\u{29}']), ('\u{24a9}',  &['\u{28}',  '\u{6e}',
         '\u{29}']), ('\u{24aa}',  &['\u{28}',  '\u{6f}',  '\u{29}']), ('\u{24ab}',  &['\u{28}',
         '\u{70}',  '\u{29}']), ('\u{24ac}',  &['\u{28}',  '\u{71}',  '\u{29}']), ('\u{24ad}',
         &['\u{28}',  '\u{72}',  '\u{29}']), ('\u{24ae}',  &['\u{28}',  '\u{73}',  '\u{29}']),
        ('\u{24af}',  &['\u{28}',  '\u{74}',  '\u{29}']), ('\u{24b0}',  &['\u{28}',  '\u{75}',
         '\u{29}']), ('\u{24b1}',  &['\u{28}',  '\u{76}',  '\u{29}']), ('\u{24b2}',  &['\u{28}',
         '\u{77}',  '\u{29}']), ('\u{24b3}',  &['\u{28}',  '\u{78}',  '\u{29}']), ('\u{24b4}',
         &['\u{28}',  '\u{79}',  '\u{29}']), ('\u{24b5}',  &['\u{28}',  '\u{7a}',  '\u{29}']),
        ('\u{24b8}',  &['\u{a9}']), ('\u{24c5}',  &['\u{2117}']), ('\u{24c7}',  &['\u{ae}']),
        ('\u{24db}',  &['\u{24be}']), ('\u{24ea}',  &['\u{1f10d}']), ('\u{2500}',  &['\u{30fc}']),
        ('\u{2501}',  &['\u{30fc}']), ('\u{2503}',  &['\u{2502}']), ('\u{250f}',  &['\u{250c}']),
        ('\u{2523}',  &['\u{251c}']), ('\u{2571}',  &['\u{2f}']), ('\u{2573}',  &['\u{58}']),
        ('\u{2588}',  &['\u{220e}']), ('\u{2590}',  &['\u{258c}']), ('\u{2594}',  &['\u{2c9}']),
        ('\u{2597}',  &['\u{2596}']), ('\u{259d}',  &['\u{2598}']), ('\u{25a0}',  &['\u{220e}']),
        ('\u{25b1}',  &['\u{23e5}']), ('\u{25b3}',  &['\u{394}']), ('\u{25b7}',  &['\u{22b3}']),
        ('\u{25b8}',  &['\u{25b6}']), ('\u{25ba}',  &['\u{25b6}']), ('\u{25bd}',  &['\u{102bc}']),
        ('\u{25c1}',  &['\u{22b2}']), ('\u{25c7}',  &['\u{16dc}']), ('\u{25ca}',  &['\u{16dc}']),
        ('\u{25cb}',  &['\u{b0}']), ('\u{25ce}',  &['\u{233e}']), ('\u{25e0}',  &['\u{2312}']),
        ('\u{25e6}',  &['\u{b0}']), ('\u{2609}',  &['\u{298}']), ('\u{2610}',  &['\u{25a1}']),
        ('\u{2625}',  &['\u{1099e}']), ('\u{2630}',  &['\u{2cb6}']), ('\u{2638}',  &['\u{2388}']),
        ('\u{264e}',  &['\u{224f}']), ('\u{2662}',  &['\u{16dc}']), ('\u{2669}',  &['\u{1d158}',
         '\u{1d165}']), ('\u{266a}',  &['\u{1d158}',  '\u{1d165}',  '\u{1d16e}']), ('\u{26ac}',
         &['\u{970}']), ('\u{2768}',  &['\u{28}']), ('\u{2769}',  &['\u{29}']), ('\u{276e}',
         &['\u{3c}']), ('\u{276f}',  &['\u{3e}']), ('\u{2772}',  &['\u{28}']), ('\u{2773}',
         &['\u{29}']), ('\u{2774}',  &['\u{7b}']), ('\u{2775}',  &['\u{7d}']), ('\u{2795}',
         &['\u{2b}']), ('\u{2796}',  &['\u{2d}']), ('\u{2797}',  &['\u{f7}']), ('\u{27c2}',
         &['\u{a4d5}']), ('\u{27c8}',  &['\u{5c}',  '\u{1455}']), ('\u{27c9}',  &['\u{1450}',
         '\u{2f}']), ('\u{27cb}',  &['\u{2f}']), ('\u{27cd}',  &['\u{5c}']), ('\u{27d9}',
         &['\u{54}']), ('\u{27e8}',  &['\u{276c}']), ('\u{27e9}',  &['\u{276d}']), ('\u{292b}',
         &['\u{78}']), ('\u{292c}',  &['\u{78}']), ('\u{2963}',  &['\u{16d0}',  '\u{16da}']),
        ('\u{2965}',  &['\u{21c3}',  '\u{21c2}']), ('\u{296e}',  &['\u{16d0}',  '\u{21c2}']),
        ('\u{296f}',  &['\u{21c3}',  '\u{16da}']), ('\u{2999}',  &['\u{2d42}']), ('\u{29b0}',
         &['\u{2349}']), ('\u{29be}',  &['\u{233e}']), ('\u{29c4}',  &['\u{303c}']), ('\u{29c5}',
         &['\u{2342}']), ('\u{29c7}',  &['\u{233b}']), ('\u{29d6}',  &['\u{102c0}']), ('\u{29d9}',
         &['\u{299a}']), ('\u{29f4}',  &['\u{3a}',  '\u{2192}']), ('\u{29f5}',  &['\u{5c}']),
        ('\u{29f6}',  &['\u{2f}',  '\u{304}']), ('\u{29f8}',  &['\u{2f}']), ('\u{29f9}',
         &['\u{5c}']), ('\u{2a00}',  &['\u{298}']), ('\u{2a01}',  &['\u{102a8}']), ('\u{2a02}',
         &['\u{2297}']), ('\u{2a03}',  &['\u{228d}']), ('\u{2a04}',  &['\u{228e}']), ('\u{2a05}',
         &['\u{2293}']), ('\u{2a06}',  &['\u{2294}']), ('\u{2a0c}',  &['\u{283}',  '\u{283}',
         '\u{283}',  '\u{283}']), ('\u{2a1d}',  &['\u{16de}']), ('\u{2a20}',  &['\u{3e}',
         '\u{3e}']), ('\u{2a21}',  &['\u{16da}']), ('\u{2a22}',  &['\u{2b}',  '\u{30a}']),
        ('\u{2a23}',  &['\u{2b}',  '\u{302}']), ('\u{2a24}',  &['\u{2b}',  '\u{303}']), ('\u{2a25}',
         &['\u{2b}',  '\u{323}']), ('\u{2a26}',  &['\u{2b}',  '\u{330}']), ('\u{2a27}',  &['\u{2b}',
         '\u{2082}']), ('\u{2a29}',  &['\u{2d}',  '\u{313}']), ('\u{2a2a}',  &['\u{2d}',
         '\u{323}']), ('\u{2a2f}',  &['\u{78}']), ('\u{2a30}',  &['\u{78}',  '\u{307}']),
        ('\u{2a3d}',  &['\u{2319}']), ('\u{2a3e}',  &['\u{2a1f}']), ('\u{2a3f}',  &['\u{2210}']),
        ('\u{2a6a}',  &['\u{7e}',  '\u{307}']), ('\u{2a6e}',  &['\u{3d}',  '\u{20f0}']),
        ('\u{2a74}',  &['\u{3a}',  '\u{3a}',  '\u{3d}']), ('\u{2a75}',  &['\u{3d}',  '\u{3d}']),
        ('\u{2a76}',  &['\u{3d}',  '\u{3d}',  '\u{3d}']), ('\u{2aa5}',  &['\u{3e}',  '\u{3c}']),
        ('\u{2aaa}',  &['\u{15d5}']), ('\u{2aab}',  &['\u{15d2}']), ('\u{2ad7}',  &['\u{1450}',
         '\u{1455}']), ('\u{2afb}',  &['\u{2f}',  '\u{2f}',  '\u{2f}']), ('\u{2afd}',  &['\u{2f}',
         '\u{2f}']), ('\u{2bec}',  &['\u{219e}']), ('\u{2bed}',  &['\u{219f}']), ('\u{2bee}',
         &['\u{21a0}']), ('\u{2bef}',  &['\u{21a1}']), ('\u{2c67}',  &['\u{48}',  '\u{329}']),
        ('\u{2c69}',  &['\u{4b}',  '\u{329}']), ('\u{2c84}',  &['\u{393}']), ('\u{2c85}',
         &['\u{72}']), ('\u{2c86}',  &['\u{394}']), ('\u{2c88}',  &['\u{a792}']), ('\u{2c89}',
         &['\u{a793}']), ('\u{2c8e}',  &['\u{48}']), ('\u{2c92}',  &['\u{6c}']), ('\u{2c94}',
         &['\u{4b}']), ('\u{2c95}',  &['\u{138}']), ('\u{2c96}',  &['\u{3bb}']), ('\u{2c98}',
         &['\u{4d}']), ('\u{2c9a}',  &['\u{4e}']), ('\u{2c9e}',  &['\u{4f}']), ('\u{2c9f}',
         &['\u{6f}']), ('\u{2ca0}',  &['\u{3a0}']), ('\u{2ca2}',  &['\u{50}']), ('\u{2ca3}',
         &['\u{70}']), ('\u{2ca4}',  &['\u{43}']), ('\u{2ca5}',  &['\u{63}']), ('\u{2ca6}',
         &['\u{54}']), ('\u{2ca8}',  &['\u{59}']), ('\u{2caa}',  &['\u{3a6}']), ('\u{2cab}',
         &['\u{278}']), ('\u{2cac}',  &['\u{58}']), ('\u{2cad}',  &['\u{3c7}']), ('\u{2cae}',
         &['\u{3a8}']), ('\u{2cb1}',  &['\u{3c9}']), ('\u{2cb4}',  &['\u{3c}',  '\u{b7}']),
        ('\u{2cba}',  &['\u{2d}']), ('\u{2cbc}',  &['\u{428}']), ('\u{2cbd}',  &['\u{448}']),
        ('\u{2cc6}',  &['\u{2f}']), ('\u{2cca}',  &['\u{39}']), ('\u{2ccc}',  &['\u{33}']),
        ('\u{2ccd}',  &['\u{21d}']), ('\u{2cd0}',  &['\u{4c}']), ('\u{2cd1}',  &['\u{29f}']),
        ('\u{2cd2}',  &['\u{36}']), ('\u{2cdc}',  &['\u{3ec}']), ('\u{2ce4}',  &['\u{3d7}']),
        ('\u{2ce9}',  &['\u{2627}']), ('\u{2cf9}',  &['\u{5c}',  '\u{5c}']), ('\u{2d31}',
         &['\u{4f}',  '\u{335}']), ('\u{2d37}',  &['\u{245}']), ('\u{2d38}',  &['\u{56}']),
        ('\u{2d39}',  &['\u{45}']), ('\u{2d3a}',  &['\u{18e}']), ('\u{2d41}',  &['\u{4f}',
         '\u{338}']), ('\u{2d48}',  &['\u{b7}',  '\u{b7}',  '\u{b7}']), ('\u{2d49}',  &['\u{1a9}']),
        ('\u{2d4f}',  &['\u{6c}']), ('\u{2d51}',  &['\u{21}']), ('\u{2d54}',  &['\u{4f}']),
        ('\u{2d55}',  &['\u{51}']), ('\u{2d59}',  &['\u{298}']), ('\u{2d5d}',  &['\u{58}']),
        ('\u{2d60}',  &['\u{394}']), ('\u{2d63}',  &['\u{16ef}']), ('\u{2de8}',  &['\u{1ddf}']),
        ('\u{2dea}',  &['\u{30a}']), ('\u{2ded}',  &['\u{368}']), ('\u{2def}',  &['\u{36f}']),
        ('\u{2df6}',  &['\u{363}']), ('\u{2df7}',  &['\u{364}']), ('\u{2e1a}',  &['\u{2d}',
         '\u{308}']), ('\u{2e1e}',  &['\u{7e}',  '\u{307}']), ('\u{2e1f}',  &['\u{7e}',
         '\u{323}']), ('\u{2e26}',  &['\u{1455}']), ('\u{2e27}',  &['\u{1450}']), ('\u{2e28}',
         &['\u{28}',  '\u{28}']), ('\u{2e29}',  &['\u{29}',  '\u{29}']), ('\u{2e2a}',
         &['\u{2235}']), ('\u{2e2b}',  &['\u{2234}']), ('\u{2e2c}',  &['\u{2237}']), ('\u{2e2e}',
         &['\u{61f}']), ('\u{2e30}',  &['\u{b0}']), ('\u{2e31}',  &['\u{b7}']), ('\u{2e32}',
         &['\u{60c}']), ('\u{2e35}',  &['\u{61b}']), ('\u{2e39}',  &['\u{1e9f}']), ('\u{2e3d}',
         &['\u{2d42}']), ('\u{2e3f}',  &['\u{b6}']), ('\u{2e40}',  &['\u{3d}']), ('\u{2e82}',
         &['\u{4e5b}']), ('\u{2e83}',  &['\u{4e5a}']), ('\u{2e85}',  &['\u{4ebb}']), ('\u{2e89}',
         &['\u{5202}']), ('\u{2e8b}',  &['\u{353e}']), ('\u{2e8e}',  &['\u{5140}']), ('\u{2e8f}',
         &['\u{5c23}']), ('\u{2e90}',  &['\u{5c22}']), ('\u{2e92}',  &['\u{5df3}']), ('\u{2e93}',
         &['\u{5e7a}']), ('\u{2e94}',  &['\u{5f51}']), ('\u{2e96}',  &['\u{5fc4}']), ('\u{2e97}',
         &['\u{38fa}']), ('\u{2e98}',  &['\u{624c}']), ('\u{2e99}',  &['\u{6535}']), ('\u{2e9b}',
         &['\u{65e1}']), ('\u{2e9e}',  &['\u{6b7a}']), ('\u{2e9f}',  &['\u{6bcd}']), ('\u{2ea0}',
         &['\u{6c11}']), ('\u{2ea1}',  &['\u{6c35}']), ('\u{2ea2}',  &['\u{6c3a}']), ('\u{2ea3}',
         &['\u{706c}']), ('\u{2ea4}',  &['\u{722b}']), ('\u{2ea6}',  &['\u{4e2c}']), ('\u{2ea8}',
         &['\u{72ad}']), ('\u{2eab}',  &['\u{7f52}']), ('\u{2ead}',  &['\u{793b}']), ('\u{2eaf}',
         &['\u{7cf9}']), ('\u{2eb1}',  &['\u{7f53}']), ('\u{2eb2}',  &['\u{7f52}']), ('\u{2eb9}',
         &['\u{8002}']), ('\u{2eba}',  &['\u{8080}']), ('\u{2ebe}',  &['\u{8279}']), ('\u{2ebf}',
         &['\u{8279}']), ('\u{2ec0}',  &['\u{8279}']), ('\u{2ec1}',  &['\u{864e}']), ('\u{2ec2}',
         &['\u{8864}']), ('\u{2ec3}',  &['\u{8980}']), ('\u{2ec4}',  &['\u{897f}']), ('\u{2ec5}',
         &['\u{89c1}']), ('\u{2ec8}',  &['\u{8ba0}']), ('\u{2ec9}',  &['\u{8d1d}']), ('\u{2ecb}',
         &['\u{8f66}']), ('\u{2ecc}',  &['\u{8fb6}']), ('\u{2ecd}',  &['\u{8fb6}']), ('\u{2ecf}',
         &['\u{961d}']), ('\u{2ed0}',  &['\u{9485}']), ('\u{2ed1}',  &['\u{9577}']), ('\u{2ed2}',
         &['\u{9578}']), ('\u{2ed3}',  &['\u{957f}']), ('\u{2ed4}',  &['\u{95e8}']), ('\u{2ed6}',
         &['\u{961d}']), ('\u{2ed8}',  &['\u{9752}']), ('\u{2ed9}',  &['\u{97e6}']), ('\u{2eda}',
         &['\u{9875}']), ('\u{2edb}',  &['\u{98ce}']), ('\u{2edc}',  &['\u{98de}']), ('\u{2edd}',
         &['\u{98df}']), ('\u{2edf}',  &['\u{98e0}']), ('\u{2ee0}',  &['\u{9963}']), ('\u{2ee2}',
         &['\u{9a6c}']), ('\u{2ee4}',  &['\u{9b3c}']), ('\u{2ee5}',  &['\u{9c7c}']), ('\u{2ee8}',
         &['\u{9ea6}']), ('\u{2ee9}',  &['\u{9ec4}']), ('\u{2eeb}',  &['\u{6589}']), ('\u{2eec}',
         &['\u{9f50}']), ('\u{2eed}',  &['\u{6b6f}']), ('\u{2eee}',  &['\u{9f7f}']), ('\u{2eef}',
         &['\u{7adc}']), ('\u{2ef0}',  &['\u{9f99}']), ('\u{2ef2}',  &['\u{4e80}']), ('\u{2ef3}',
         &['\u{9f9f}']), ('\u{2f00}',  &['\u{30fc}']), ('\u{2f01}',  &['\u{4e28}']), ('\u{2f02}',
         &['\u{5c}']), ('\u{2f03}',  &['\u{2f}']), ('\u{2f04}',  &['\u{4e59}']), ('\u{2f05}',
         &['\u{4e85}']), ('\u{2f06}',  &['\u{4e8c}']), ('\u{2f07}',  &['\u{4ea0}']), ('\u{2f08}',
         &['\u{4eba}']), ('\u{2f09}',  &['\u{513f}']), ('\u{2f0a}',  &['\u{5165}']), ('\u{2f0b}',
         &['\u{516b}']), ('\u{2f0c}',  &['\u{5182}']), ('\u{2f0d}',  &['\u{5196}']), ('\u{2f0e}',
         &['\u{51ab}']), ('\u{2f0f}',  &['\u{51e0}']), ('\u{2f10}',  &['\u{51f5}']), ('\u{2f11}',
         &['\u{5200}']), ('\u{2f12}',  &['\u{529b}']), ('\u{2f13}',  &['\u{52f9}']), ('\u{2f14}',
         &['\u{5315}']), ('\u{2f15}',  &['\u{531a}']), ('\u{2f16}',  &['\u{5338}']), ('\u{2f17}',
         &['\u{5341}']), ('\u{2f18}',  &['\u{535c}']), ('\u{2f19}',  &['\u{5369}']), ('\u{2f1a}',
         &['\u{5382}']), ('\u{2f1b}',  &['\u{53b6}']), ('\u{2f1c}',  &['\u{53c8}']), ('\u{2f1d}',
         &['\u{53e3}']), ('\u{2f1e}',  &['\u{53e3}']), ('\u{2f1f}',  &['\u{571f}']), ('\u{2f20}',
         &['\u{571f}']), ('\u{2f21}',  &['\u{5902}']), ('\u{2f22}',  &['\u{590a}']), ('\u{2f23}',
         &['\u{5915}']), ('\u{2f24}',  &['\u{5927}']), ('\u{2f25}',  &['\u{5973}']), ('\u{2f26}',
         &['\u{5b50}']), ('\u{2f27}',  &['\u{5b80}']), ('\u{2f28}',  &['\u{5bf8}']), ('\u{2f29}',
         &['\u{5c0f}']), ('\u{2f2a}',  &['\u{5c22}']), ('\u{2f2b}',  &['\u{5c38}']), ('\u{2f2c}',
         &['\u{5c6e}']), ('\u{2f2d}',  &['\u{5c71}']), ('\u{2f2e}',  &['\u{5ddb}']), ('\u{2f2f}',
         &['\u{5de5}']), ('\u{2f30}',  &['\u{5df1}']), ('\u{2f31}',  &['\u{5dfe}']), ('\u{2f32}',
         &['\u{5e72}']), ('\u{2f33}',  &['\u{5e7a}']), ('\u{2f34}',  &['\u{5e7f}']), ('\u{2f35}',
         &['\u{5ef4}']), ('\u{2f36}',  &['\u{5efe}']), ('\u{2f37}',  &['\u{5f0b}']), ('\u{2f38}',
         &['\u{5f13}']), ('\u{2f39}',  &['\u{5f50}']), ('\u{2f3a}',  &['\u{5f61}']), ('\u{2f3b}',
         &['\u{5f73}']), ('\u{2f3c}',  &['\u{5fc3}']), ('\u{2f3d}',  &['\u{6208}']), ('\u{2f3e}',
         &['\u{6236}']), ('\u{2f3f}',  &['\u{624b}']), ('\u{2f40}',  &['\u{652f}']), ('\u{2f41}',
         &['\u{6534}']), ('\u{2f42}',  &['\u{6587}']), ('\u{2f43}',  &['\u{6597}']), ('\u{2f44}',
         &['\u{65a4}']), ('\u{2f45}',  &['\u{65b9}']), ('\u{2f46}',  &['\u{65e0}']), ('\u{2f47}',
         &['\u{65e5}']), ('\u{2f48}',  &['\u{66f0}']), ('\u{2f49}',  &['\u{6708}']), ('\u{2f4a}',
         &['\u{6728}']), ('\u{2f4b}',  &['\u{6b20}']), ('\u{2f4c}',  &['\u{6b62}']), ('\u{2f4d}',
         &['\u{6b79}']), ('\u{2f4e}',  &['\u{6bb3}']), ('\u{2f4f}',  &['\u{6bcb}']), ('\u{2f50}',
         &['\u{6bd4}']), ('\u{2f51}',  &['\u{6bdb}']), ('\u{2f52}',  &['\u{6c0f}']), ('\u{2f53}',
         &['\u{6c14}']), ('\u{2f54}',  &['\u{6c34}']), ('\u{2f55}',  &['\u{706b}']), ('\u{2f56}',
         &['\u{722a}']), ('\u{2f57}',  &['\u{7236}']), ('\u{2f58}',  &['\u{723b}']), ('\u{2f59}',
         &['\u{723f}']), ('\u{2f5a}',  &['\u{7247}']), ('\u{2f5b}',  &['\u{7259}']), ('\u{2f5c}',
         &['\u{725b}']), ('\u{2f5d}',  &['\u{72ac}']), ('\u{2f5e}',  &['\u{7384}']), ('\u{2f5f}',
         &['\u{7389}']), ('\u{2f60}',  &['\u{74dc}']), ('\u{2f61}',  &['\u{74e6}']), ('\u{2f62}',
         &['\u{7518}']), ('\u{2f63}',  &['\u{751f}']), ('\u{2f64}',  &['\u{7528}']), ('\u{2f65}',
         &['\u{7530}']), ('\u{2f66}',  &['\u{758b}']), ('\u{2f67}',  &['\u{7592}']), ('\u{2f68}',
         &['\u{7676}']), ('\u{2f69}',  &['\u{767d}']), ('\u{2f6a}',  &['\u{76ae}']), ('\u{2f6b}',
         &['\u{76bf}']), ('\u{2f6c}',  &['\u{76ee}']), ('\u{2f6d}',  &['\u{77db}']), ('\u{2f6e}',
         &['\u{77e2}']), ('\u{2f6f}',  &['\u{77f3}']), ('\u{2f70}',  &['\u{793a}']), ('\u{2f71}',
         &['\u{79b8}']), ('\u{2f72}',  &['\u{79be}']), ('\u{2f73}',  &['\u{7a74}']), ('\u{2f74}',
         &['\u{7acb}']), ('\u{2f75}',  &['\u{7af9}']), ('\u{2f76}',  &['\u{7c73}']), ('\u{2f77}',
         &['\u{7cf8}']), ('\u{2f78}',  &['\u{7f36}']), ('\u{2f79}',  &['\u{7f51}']), ('\u{2f7a}',
         &['\u{7f8a}']), ('\u{2f7b}',  &['\u{7fbd}']), ('\u{2f7c}',  &['\u{8001}']), ('\u{2f7d}',
         &['\u{800c}']), ('\u{2f7e}',  &['\u{8012}']), ('\u{2f7f}',  &['\u{8033}']), ('\u{2f80}',
         &['\u{807f}']), ('\u{2f81}',  &['\u{8089}']), ('\u{2f82}',  &['\u{81e3}']), ('\u{2f83}',
         &['\u{81ea}']), ('\u{2f84}',  &['\u{81f3}']), ('\u{2f85}',  &['\u{81fc}']), ('\u{2f86}',
         &['\u{820c}']), ('\u{2f87}',  &['\u{821b}']), ('\u{2f88}',  &['\u{821f}']), ('\u{2f89}',
         &['\u{826e}']), ('\u{2f8a}',  &['\u{8272}']), ('\u{2f8b}',  &['\u{8278}']), ('\u{2f8c}',
         &['\u{864d}']), ('\u{2f8d}',  &['\u{866b}']), ('\u{2f8e}',  &['\u{8840}']), ('\u{2f8f}',
         &['\u{884c}']), ('\u{2f90}',  &['\u{8863}']), ('\u{2f91}',  &['\u{897e}']), ('\u{2f92}',
         &['\u{898b}']), ('\u{2f93}',  &['\u{89d2}']), ('\u{2f94}',  &['\u{8a00}']), ('\u{2f95}',
         &['\u{8c37}']), ('\u{2f96}',  &['\u{8c46}']), ('\u{2f97}',  &['\u{8c55}']), ('\u{2f98}',
         &['\u{8c78}']), ('\u{2f99}',  &['\u{8c9d}']), ('\u{2f9a}',  &['\u{8d64}']), ('\u{2f9b}',
         &['\u{8d70}']), ('\u{2f9c}',  &['\u{8db3}']), ('\u{2f9d}',  &['\u{8eab}']), ('\u{2f9e}',
         &['\u{8eca}']), ('\u{2f9f}',  &['\u{8f9b}']), ('\u{2fa0}',  &['\u{8fb0}']), ('\u{2fa1}',
         &['\u{8fb5}']), ('\u{2fa2}',  &['\u{9091}']), ('\u{2fa3}',  &['\u{9149}']), ('\u{2fa4}',
         &['\u{91c6}']), ('\u{2fa5}',  &['\u{91cc}']), ('\u{2fa6}',  &['\u{91d1}']), ('\u{2fa7}',
         &['\u{9577}']), ('\u{2fa8}',  &['\u{9580}']), ('\u{2fa9}',  &['\u{961c}']), ('\u{2faa}',
         &['\u{96b6}']), ('\u{2fab}',  &['\u{96b9}']), ('\u{2fac}',  &['\u{96e8}']), ('\u{2fad}',
         &['\u{9751}']), ('\u{2fae}',  &['\u{975e}']), ('\u{2faf}',  &['\u{9762}']), ('\u{2fb0}',
         &['\u{9769}']), ('\u{2fb1}',  &['\u{97cb}']), ('\u{2fb2}',  &['\u{97ed}']), ('\u{2fb3}',
         &['\u{97f3}']), ('\u{2fb4}',  &['\u{9801}']), ('\u{2fb5}',  &['\u{98a8}']), ('\u{2fb6}',
         &['\u{98db}']), ('\u{2fb7}',  &['\u{98df}']), ('\u{2fb8}',  &['\u{9996}']), ('\u{2fb9}',
         &['\u{9999}']), ('\u{2fba}',  &['\u{99ac}']), ('\u{2fbb}',  &['\u{9aa8}']), ('\u{2fbc}',
         &['\u{9ad8}']), ('\u{2fbd}',  &['\u{9adf}']), ('\u{2fbe}',  &['\u{9b25}']), ('\u{2fbf}',
         &['\u{9b2f}']), ('\u{2fc0}',  &['\u{9b32}']), ('\u{2fc1}',  &['\u{9b3c}']), ('\u{2fc2}',
         &['\u{9b5a}']), ('\u{2fc3}',  &['\u{9ce5}']), ('\u{2fc4}',  &['\u{9e75}']), ('\u{2fc5}',
         &['\u{9e7f}']), ('\u{2fc6}',  &['\u{9ea5}']), ('\u{2fc7}',  &['\u{9ebb}']), ('\u{2fc8}',
         &['\u{9ec3}']), ('\u{2fc9}',  &['\u{9ecd}']), ('\u{2fca}',  &['\u{9ed1}']), ('\u{2fcb}',
         &['\u{9ef9}']), ('\u{2fcc}',  &['\u{9efd}']), ('\u{2fcd}',  &['\u{9f0e}']), ('\u{2fce}',
         &['\u{9f13}']), ('\u{2fcf}',  &['\u{9f20}']), ('\u{2fd0}',  &['\u{9f3b}']), ('\u{2fd1}',
         &['\u{9f4a}']), ('\u{2fd2}',  &['\u{9f52}']), ('\u{2fd3}',  &['\u{9f8d}']), ('\u{2fd4}',
         &['\u{9f9c}']), ('\u{2fd5}',  &['\u{9fa0}']), ('\u{3002}',  &['\u{2f3}']), ('\u{3003}',
         &['\u{27}',  '\u{27}']), ('\u{3007}',  &['\u{4f}']), ('\u{3008}',  &['\u{276c}']),
        ('\u{3009}',  &['\u{276d}']), ('\u{3012}',  &['\u{20b8}']), ('\u{3014}',  &['\u{28}']),
        ('\u{3015}',  &['\u{29}']), ('\u{301a}',  &['\u{27e6}']), ('\u{301b}',  &['\u{27e7}']),
        ('\u{302c}',  &['\u{309}']), ('\u{302d}',  &['\u{325}']), ('\u{3033}',  &['\u{2f}']),
        ('\u{3036}',  &['\u{20b8}']), ('\u{3038}',  &['\u{5341}']), ('\u{3039}',  &['\u{5344}']),
        ('\u{303a}',  &['\u{5345}']), ('\u{304f}',  &['\u{276c}']), ('\u{309a}',  &['\u{30a}']),
        ('\u{309b}',  &['\u{ff9e}']), ('\u{309c}',  &['\u{ff9f}']), ('\u{30a0}',  &['\u{3d}']),
        ('\u{30a4}',  &['\u{4ebb}']), ('\u{30a8}',  &['\u{5de5}']), ('\u{30ab}',  &['\u{529b}']),
        ('\u{30bf}',  &['\u{5915}']), ('\u{30c8}',  &['\u{535c}']), ('\u{30cb}',  &['\u{4e8c}']),
        ('\u{30ce}',  &['\u{2f}']), ('\u{30cf}',  &['\u{516b}']), ('\u{30d8}',  &['\u{3078}']),
        ('\u{30ed}',  &['\u{53e3}']), ('\u{30fb}',  &['\u{b7}']), ('\u{3126}',  &['\u{513f}']),
        ('\u{3131}',  &['\u{1100}']), ('\u{3132}',  &['\u{1100}',  '\u{1100}']), ('\u{3133}',
         &['\u{1100}',  '\u{1109}']), ('\u{3134}',  &['\u{1102}']), ('\u{3135}',  &['\u{1102}',
         '\u{110c}']), ('\u{3136}',  &['\u{1102}',  '\u{1112}']), ('\u{3137}',  &['\u{1103}']),
        ('\u{3138}',  &['\u{1103}',  '\u{1103}']), ('\u{3139}',  &['\u{1105}']), ('\u{313a}',
         &['\u{1105}',  '\u{1100}']), ('\u{313b}',  &['\u{1105}',  '\u{1106}']), ('\u{313c}',
         &['\u{1105}',  '\u{1107}']), ('\u{313d}',  &['\u{1105}',  '\u{1109}']), ('\u{313e}',
         &['\u{1105}',  '\u{1110}']), ('\u{313f}',  &['\u{1105}',  '\u{1111}']), ('\u{3140}',
         &['\u{1105}',  '\u{1112}']), ('\u{3141}',  &['\u{1106}']), ('\u{3142}',  &['\u{1107}']),
        ('\u{3143}',  &['\u{1107}',  '\u{1107}']), ('\u{3144}',  &['\u{1107}',  '\u{1109}']),
        ('\u{3145}',  &['\u{1109}']), ('\u{3146}',  &['\u{1109}',  '\u{1109}']), ('\u{3147}',
         &['\u{110b}']), ('\u{3148}',  &['\u{110c}']), ('\u{3149}',  &['\u{110c}',  '\u{110c}']),
        ('\u{314a}',  &['\u{110e}']), ('\u{314b}',  &['\u{110f}']), ('\u{314c}',  &['\u{1110}']),
        ('\u{314d}',  &['\u{1111}']), ('\u{314e}',  &['\u{1112}']), ('\u{314f}',  &['\u{1161}']),
        ('\u{3150}',  &['\u{1161}',  '\u{4e28}']), ('\u{3151}',  &['\u{1163}']), ('\u{3152}',
         &['\u{1163}',  '\u{4e28}']), ('\u{3153}',  &['\u{1165}']), ('\u{3154}',  &['\u{1165}',
         '\u{4e28}']), ('\u{3155}',  &['\u{1167}']), ('\u{3156}',  &['\u{1167}',  '\u{4e28}']),
        ('\u{3157}',  &['\u{1169}']), ('\u{3158}',  &['\u{1169}',  '\u{1161}']), ('\u{3159}',
         &['\u{1169}',  '\u{1161}',  '\u{4e28}']), ('\u{315a}',  &['\u{1169}',  '\u{4e28}']),
        ('\u{315b}',  &['\u{116d}']), ('\u{315c}',  &['\u{116e}']), ('\u{315d}',  &['\u{116e}',
         '\u{1165}']), ('\u{315e}',  &['\u{116e}',  '\u{1165}',  '\u{4e28}']), ('\u{315f}',
         &['\u{116e}',  '\u{4e28}']), ('\u{3160}',  &['\u{1172}']), ('\u{3161}',  &['\u{30fc}']),
        ('\u{3162}',  &['\u{30fc}',  '\u{4e28}']), ('\u{3163}',  &['\u{4e28}']), ('\u{3164}',
         &['\u{1160}']), ('\u{3165}',  &['\u{1102}',  '\u{1102}']), ('\u{3166}',  &['\u{1102}',
         '\u{1103}']), ('\u{3167}',  &['\u{1102}',  '\u{1109}']), ('\u{3168}',  &['\u{1102}',
         '\u{1140}']), ('\u{3169}',  &['\u{1105}',  '\u{1100}',  '\u{1109}']), ('\u{316a}',
         &['\u{1105}',  '\u{1103}']), ('\u{316b}',  &['\u{1105}',  '\u{1107}',  '\u{1109}']),
        ('\u{316c}',  &['\u{1105}',  '\u{1140}']), ('\u{316d}',  &['\u{1105}',  '\u{1159}']),
        ('\u{316e}',  &['\u{1106}',  '\u{1107}']), ('\u{316f}',  &['\u{1106}',  '\u{1109}']),
        ('\u{3170}',  &['\u{1106}',  '\u{1140}']), ('\u{3171}',  &['\u{1106}',  '\u{110b}']),
        ('\u{3172}',  &['\u{1107}',  '\u{1100}']), ('\u{3173}',  &['\u{1107}',  '\u{1103}']),
        ('\u{3174}',  &['\u{1107}',  '\u{1109}',  '\u{1100}']), ('\u{3175}',  &['\u{1107}',
         '\u{1109}',  '\u{1103}']), ('\u{3176}',  &['\u{1107}',  '\u{110c}']), ('\u{3177}',
         &['\u{1107}',  '\u{1110}']), ('\u{3178}',  &['\u{1107}',  '\u{110b}']), ('\u{3179}',
         &['\u{1107}',  '\u{1107}',  '\u{110b}']), ('\u{317a}',  &['\u{1109}',  '\u{1100}']),
        ('\u{317b}',  &['\u{1109}',  '\u{1102}']), ('\u{317c}',  &['\u{1109}',  '\u{1103}']),
        ('\u{317d}',  &['\u{1109}',  '\u{1107}']), ('\u{317e}',  &['\u{1109}',  '\u{110c}']),
        ('\u{317f}',  &['\u{1140}']), ('\u{3180}',  &['\u{110b}',  '\u{110b}']), ('\u{3181}',
         &['\u{114c}']), ('\u{3182}',  &['\u{110b}',  '\u{1109}']), ('\u{3183}',  &['\u{110b}',
         '\u{1140}']), ('\u{3184}',  &['\u{1111}',  '\u{110b}']), ('\u{3185}',  &['\u{1112}',
         '\u{1112}']), ('\u{3186}',  &['\u{1159}']), ('\u{3187}',  &['\u{116d}',  '\u{1163}']),
        ('\u{3188}',  &['\u{116d}',  '\u{1163}',  '\u{4e28}']), ('\u{3189}',  &['\u{116d}',
         '\u{4e28}']), ('\u{318a}',  &['\u{1172}',  '\u{1167}']), ('\u{318b}',  &['\u{1172}',
         '\u{1167}',  '\u{4e28}']), ('\u{318c}',  &['\u{1172}',  '\u{4e28}']), ('\u{318d}',
         &['\u{119e}']), ('\u{318e}',  &['\u{119e}',  '\u{4e28}']), ('\u{31d0}',  &['\u{30fc}']),
        ('\u{31d1}',  &['\u{4e28}']), ('\u{31d3}',  &['\u{2f}']), ('\u{31d4}',  &['\u{5c}']),
        ('\u{31d6}',  &['\u{4e5b}']), ('\u{31da}',  &['\u{4e85}']), ('\u{31db}',  &['\u{276c}']),
        ('\u{31df}',  &['\u{4e5a}']), ('\u{31e0}',  &['\u{4e59}']), ('\u{3200}',  &['\u{28}',
         '\u{1100}',  '\u{29}']), ('\u{3201}',  &['\u{28}',  '\u{1102}',  '\u{29}']), ('\u{3202}',
         &['\u{28}',  '\u{1103}',  '\u{29}']), ('\u{3203}',  &['\u{28}',  '\u{1105}',  '\u{29}']),
        ('\u{3204}',  &['\u{28}',  '\u{1106}',  '\u{29}']), ('\u{3205}',  &['\u{28}',  '\u{1107}',
         '\u{29}']), ('\u{3206}',  &['\u{28}',  '\u{1109}',  '\u{29}']), ('\u{3207}',  &['\u{28}',
         '\u{110b}',  '\u{29}']), ('\u{3208}',  &['\u{28}',  '\u{110c}',  '\u{29}']), ('\u{3209}',
         &['\u{28}',  '\u{110e}',  '\u{29}']), ('\u{320a}',  &['\u{28}',  '\u{110f}',  '\u{29}']),
        ('\u{320b}',  &['\u{28}',  '\u{1110}',  '\u{29}']), ('\u{320c}',  &['\u{28}',  '\u{1111}',
         '\u{29}']), ('\u{320d}',  &['\u{28}',  '\u{1112}',  '\u{29}']), ('\u{320e}',  &['\u{28}',
         '\u{ac00}',  '\u{29}']), ('\u{320f}',  &['\u{28}',  '\u{b098}',  '\u{29}']), ('\u{3210}',
         &['\u{28}',  '\u{b2e4}',  '\u{29}']), ('\u{3211}',  &['\u{28}',  '\u{b77c}',  '\u{29}']),
        ('\u{3212}',  &['\u{28}',  '\u{b9c8}',  '\u{29}']), ('\u{3213}',  &['\u{28}',  '\u{bc14}',
         '\u{29}']), ('\u{3214}',  &['\u{28}',  '\u{c0ac}',  '\u{29}']), ('\u{3215}',  &['\u{28}',
         '\u{c544}',  '\u{29}']), ('\u{3216}',  &['\u{28}',  '\u{c790}',  '\u{29}']), ('\u{3217}',
         &['\u{28}',  '\u{cc28}',  '\u{29}']), ('\u{3218}',  &['\u{28}',  '\u{ce74}',  '\u{29}']),
        ('\u{3219}',  &['\u{28}',  '\u{d0c0}',  '\u{29}']), ('\u{321a}',  &['\u{28}',  '\u{d30c}',
         '\u{29}']), ('\u{321b}',  &['\u{28}',  '\u{d558}',  '\u{29}']), ('\u{321c}',  &['\u{28}',
         '\u{c8fc}',  '\u{29}']), ('\u{321d}',  &['\u{28}',  '\u{c624}',  '\u{c804}',  '\u{29}']),
        ('\u{321e}',  &['\u{28}',  '\u{c624}',  '\u{d6c4}',  '\u{29}']), ('\u{3220}',  &['\u{28}',
         '\u{30fc}',  '\u{29}']), ('\u{3221}',  &['\u{28}',  '\u{4e8c}',  '\u{29}']), ('\u{3222}',
         &['\u{28}',  '\u{4e09}',  '\u{29}']), ('\u{3223}',  &['\u{28}',  '\u{56db}',  '\u{29}']),
        ('\u{3224}',  &['\u{28}',  '\u{4e94}',  '\u{29}']), ('\u{3225}',  &['\u{28}',  '\u{516d}',
         '\u{29}']), ('\u{3226}',  &['\u{28}',  '\u{4e03}',  '\u{29}']), ('\u{3227}',  &['\u{28}',
         '\u{516b}',  '\u{29}']), ('\u{3228}',  &['\u{28}',  '\u{4e5d}',  '\u{29}']), ('\u{3229}',
         &['\u{28}',  '\u{5341}',  '\u{29}']), ('\u{322a}',  &['\u{28}',  '\u{6708}',  '\u{29}']),
        ('\u{322b}',  &['\u{28}',  '\u{706b}',  '\u{29}']), ('\u{322c}',  &['\u{28}',  '\u{6c34}',
         '\u{29}']), ('\u{322d}',  &['\u{28}',  '\u{6728}',  '\u{29}']), ('\u{322e}',  &['\u{28}',
         '\u{91d1}',  '\u{29}']), ('\u{322f}',  &['\u{28}',  '\u{571f}',  '\u{29}']), ('\u{3230}',
         &['\u{28}',  '\u{65e5}',  '\u{29}']), ('\u{3231}',  &['\u{28}',  '\u{682a}',  '\u{29}']),
        ('\u{3232}',  &['\u{28}',  '\u{6709}',  '\u{29}']), ('\u{3233}',  &['\u{28}',  '\u{793e}',
         '\u{29}']), ('\u{3234}',  &['\u{28}',  '\u{540d}',  '\u{29}']), ('\u{3235}',  &['\u{28}',
         '\u{7279}',  '\u{29}']), ('\u{3236}',  &['\u{28}',  '\u{8ca1}',  '\u{29}']), ('\u{3237}',
         &['\u{28}',  '\u{795d}',  '\u{29}']), ('\u{3238}',  &['\u{28}',  '\u{52b4}',  '\u{29}']),
        ('\u{3239}',  &['\u{28}',  '\u{4ee3}',  '\u{29}']), ('\u{323a}',  &['\u{28}',  '\u{547c}',
         '\u{29}']), ('\u{323b}',  &['\u{28}',  '\u{5b66}',  '\u{29}']), ('\u{323c}',  &['\u{28}',
         '\u{76e3}',  '\u{29}']), ('\u{323d}',  &['\u{28}',  '\u{4f01}',  '\u{29}']), ('\u{323e}',
         &['\u{28}',  '\u{8cc7}',  '\u{29}']), ('\u{323f}',  &['\u{28}',  '\u{5354}',  '\u{29}']),
        ('\u{3240}',  &['\u{28}',  '\u{796d}',  '\u{29}']), ('\u{3241}',  &['\u{28}',  '\u{4f11}',
         '\u{29}']), ('\u{3242}',  &['\u{28}',  '\u{81ea}',  '\u{29}']), ('\u{3243}',  &['\u{28}',
         '\u{81f3}',  '\u{29}']), ('\u{32c0}',  &['\u{6c}',  '\u{6708}']), ('\u{32c1}',  &['\u{32}',
         '\u{6708}']), ('\u{32c2}',  &['\u{33}',  '\u{6708}']), ('\u{32c3}',  &['\u{34}',
         '\u{6708}']), ('\u{32c4}',  &['\u{35}',  '\u{6708}']), ('\u{32c5}',  &['\u{36}',
         '\u{6708}']), ('\u{32c6}',  &['\u{37}',  '\u{6708}']), ('\u{32c7}',  &['\u{38}',
         '\u{6708}']), ('\u{32c8}',  &['\u{39}',  '\u{6708}']), ('\u{32c9}',  &['\u{6c}',  '\u{4f}',
         '\u{6708}']), ('\u{32ca}',  &['\u{6c}',  '\u{6c}',  '\u{6708}']), ('\u{32cb}',  &['\u{6c}',
         '\u{32}',  '\u{6708}']), ('\u{3358}',  &['\u{4f}',  '\u{70b9}']), ('\u{3359}',  &['\u{6c}',
         '\u{70b9}']), ('\u{335a}',  &['\u{32}',  '\u{70b9}']), ('\u{335b}',  &['\u{33}',
         '\u{70b9}']), ('\u{335c}',  &['\u{34}',  '\u{70b9}']), ('\u{335d}',  &['\u{35}',
         '\u{70b9}']), ('\u{335e}',  &['\u{36}',  '\u{70b9}']), ('\u{335f}',  &['\u{37}',
         '\u{70b9}']), ('\u{3360}',  &['\u{38}',  '\u{70b9}']), ('\u{3361}',  &['\u{39}',
         '\u{70b9}']), ('\u{3362}',  &['\u{6c}',  '\u{4f}',  '\u{70b9}']), ('\u{3363}',  &['\u{6c}',
         '\u{6c}',  '\u{70b9}']), ('\u{3364}',  &['\u{6c}',  '\u{32}',  '\u{70b9}']), ('\u{3365}',
         &['\u{6c}',  '\u{33}',  '\u{70b9}']), ('\u{3366}',  &['\u{6c}',  '\u{34}',  '\u{70b9}']),
        ('\u{3367}',  &['\u{6c}',  '\u{35}',  '\u{70b9}']), ('\u{3368}',  &['\u{6c}',  '\u{36}',
         '\u{70b9}']), ('\u{3369}',  &['\u{6c}',  '\u{37}',  '\u{70b9}']), ('\u{336a}',  &['\u{6c}',
         '\u{38}',  '\u{70b9}']), ('\u{336b}',  &['\u{6c}',  '\u{39}',  '\u{70b9}']), ('\u{336c}',
         &['\u{32}',  '\u{4f}',  '\u{70b9}']), ('\u{336d}',  &['\u{32}',  '\u{6c}',  '\u{70b9}']),
        ('\u{336e}',  &['\u{32}',  '\u{32}',  '\u{70b9}']), ('\u{336f}',  &['\u{32}',  '\u{33}',
         '\u{70b9}']), ('\u{3370}',  &['\u{32}',  '\u{34}',  '\u{70b9}']), ('\u{33e0}',  &['\u{6c}',
         '\u{65e5}']), ('\u{33e1}',  &['\u{32}',  '\u{65e5}']), ('\u{33e2}',  &['\u{33}',
         '\u{65e5}']), ('\u{33e3}',  &['\u{34}',  '\u{65e5}']), ('\u{33e4}',  &['\u{35}',
         '\u{65e5}']), ('\u{33e5}',  &['\u{36}',  '\u{65e5}']), ('\u{33e6}',  &['\u{37}',
         '\u{65e5}']), ('\u{33e7}',  &['\u{38}',  '\u{65e5}']), ('\u{33e8}',  &['\u{39}',
         '\u{65e5}']), ('\u{33e9}',  &['\u{6c}',  '\u{4f}',  '\u{65e5}']), ('\u{33ea}',  &['\u{6c}',
         '\u{6c}',  '\u{65e5}']), ('\u{33eb}',  &['\u{6c}',  '\u{32}',  '\u{65e5}']), ('\u{33ec}',
         &['\u{6c}',  '\u{33}',  '\u{65e5}']), ('\u{33ed}',  &['\u{6c}',  '\u{34}',  '\u{65e5}']),
        ('\u{33ee}',  &['\u{6c}',  '\u{35}',  '\u{65e5}']), ('\u{33ef}',  &['\u{6c}',  '\u{36}',
         '\u{65e5}']), ('\u{33f0}',  &['\u{6c}',  '\u{37}',  '\u{65e5}']), ('\u{33f1}',  &['\u{6c}',
         '\u{38}',  '\u{65e5}']), ('\u{33f2}',  &['\u{6c}',  '\u{39}',  '\u{65e5}']), ('\u{33f3}',
         &['\u{32}',  '\u{4f}',  '\u{65e5}']), ('\u{33f4}',  &['\u{32}',  '\u{6c}',  '\u{65e5}']),
        ('\u{33f5}',  &['\u{32}',  '\u{32}',  '\u{65e5}']), ('\u{33f6}',  &['\u{32}',  '\u{33}',
         '\u{65e5}']), ('\u{33f7}',  &['\u{32}',  '\u{34}',  '\u{65e5}']), ('\u{33f8}',  &['\u{32}',
         '\u{35}',  '\u{65e5}']), ('\u{33f9}',  &['\u{32}',  '\u{36}',  '\u{65e5}']), ('\u{33fa}',
         &['\u{32}',  '\u{37}',  '\u{65e5}']), ('\u{33fb}',  &['\u{32}',  '\u{38}',  '\u{65e5}']),
        ('\u{33fc}',  &['\u{32}',  '\u{39}',  '\u{65e5}']), ('\u{33fd}',  &['\u{33}',  '\u{4f}',
         '\u{65e5}']), ('\u{33fe}',  &['\u{33}',  '\u{6c}',  '\u{65e5}']), ('\u{39b3}',
         &['\u{363d}']), ('\u{439b}',  &['\u{3588}']), ('\u{4420}',  &['\u{3b3b}']), ('\u{4e00}',
         &['\u{30fc}']), ('\u{4e36}',  &['\u{5c}']), ('\u{4e3f}',  &['\u{2f}']), ('\u{5002}',
         &['\u{4f75}']), ('\u{503c}',  &['\u{5024}']), ('\u{555f}',  &['\u{5553}']), ('\u{56d7}',
         &['\u{53e3}']), ('\u{586b}',  &['\u{5861}']), ('\u{58eb}',  &['\u{571f}']), ('\u{58ff}',
         &['\u{58ab}']), ('\u{5b00}',  &['\u{5aaf}']), ('\u{5e32}',  &['\u{5e21}']), ('\u{5e50}',
         &['\u{3b3a}']), ('\u{6238}',  &['\u{6236}']), ('\u{6409}',  &['\u{3a41}']), ('\u{6663}',
         &['\u{403f}']), ('\u{6669}',  &['\u{665a}']), ('\u{66f6}',  &['\u{3ada}']), ('\u{6726}',
         &['\u{4443}']), ('\u{67ff}',  &['\u{676e}']), ('\u{69e9}',  &['\u{3ba3}']), ('\u{6a27}',
         &['\u{699d}']), ('\u{6f59}',  &['\u{6e88}']), ('\u{784f}',  &['\u{7814}']), ('\u{7d76}',
         &['\u{7d55}']), ('\u{80a6}',  &['\u{670c}']), ('\u{80ca}',  &['\u{6710}']), ('\u{80d0}',
         &['\u{670f}']), ('\u{80f6}',  &['\u{3b35}']), ('\u{8101}',  &['\u{6713}']), ('\u{8127}',
         &['\u{6718}']), ('\u{8141}',  &['\u{80fc}']), ('\u{81a7}',  &['\u{6723}']), ('\u{853f}',
         &['\u{848d}']), ('\u{8641}',  &['\u{8637}']), ('\u{8a1e}',  &['\u{46b6}']), ('\u{8a7d}',
         &['\u{8a2e}']), ('\u{8b8f}',  &['\u{8b86}']), ('\u{8c63}',  &['\u{8c5c}']), ('\u{8d86}',
         &['\u{8d7f}']), ('\u{8dfa}',  &['\u{8de5}']), ('\u{8e9b}',  &['\u{8e97}']), ('\u{8f27}',
         &['\u{8eff}']), ('\u{90de}',  &['\u{90ce}']), ('\u{93ae}',  &['\u{93ad}']), ('\u{96b8}',
         &['\u{96b7}']), ('\u{9e43}',  &['\u{9e42}']), ('\u{9ed2}',  &['\u{9ed1}']), ('\u{9fc3}',
         &['\u{4039}']), ('\u{a494}',  &['\u{a2cd}']), ('\u{a49c}',  &['\u{a0c0}']), ('\u{a49e}',
         &['\u{a04a}']), ('\u{a4a7}',  &['\u{a458}']), ('\u{a4a8}',  &['\u{a132}']), ('\u{a4ac}',
         &['\u{a050}']), ('\u{a4b0}',  &['\u{a3c2}']), ('\u{a4ba}',  &['\u{a3bf}']), ('\u{a4be}',
         &['\u{a2b1}']), ('\u{a4bf}',  &['\u{a259}']), ('\u{a4c0}',  &['\u{a3ab}']), ('\u{a4c2}',
         &['\u{a3b5}']), ('\u{a4d0}',  &['\u{42}']), ('\u{a4d1}',  &['\u{50}']), ('\u{a4d2}',
         &['\u{64}']), ('\u{a4d3}',  &['\u{44}']), ('\u{a4d4}',  &['\u{54}']), ('\u{a4d6}',
         &['\u{47}']), ('\u{a4d7}',  &['\u{4b}']), ('\u{a4d9}',  &['\u{4a}']), ('\u{a4da}',
         &['\u{43}']), ('\u{a4db}',  &['\u{186}']), ('\u{a4dc}',  &['\u{5a}']), ('\u{a4dd}',
         &['\u{46}']), ('\u{a4de}',  &['\u{2132}']), ('\u{a4df}',  &['\u{4d}']), ('\u{a4e0}',
         &['\u{4e}']), ('\u{a4e1}',  &['\u{4c}']), ('\u{a4e2}',  &['\u{53}']), ('\u{a4e3}',
         &['\u{52}']), ('\u{a4e5}',  &['\u{245}']), ('\u{a4e6}',  &['\u{56}']), ('\u{a4e7}',
         &['\u{48}']), ('\u{a4ea}',  &['\u{57}']), ('\u{a4eb}',  &['\u{58}']), ('\u{a4ec}',
         &['\u{59}']), ('\u{a4ed}',  &['\u{1660}']), ('\u{a4ee}',  &['\u{41}']), ('\u{a4ef}',
         &['\u{2c6f}']), ('\u{a4f0}',  &['\u{45}']), ('\u{a4f1}',  &['\u{18e}']), ('\u{a4f2}',
         &['\u{6c}']), ('\u{a4f3}',  &['\u{4f}']), ('\u{a4f4}',  &['\u{55}']), ('\u{a4f5}',
         &['\u{548}']), ('\u{a4f7}',  &['\u{15e1}']), ('\u{a4f8}',  &['\u{2e}']), ('\u{a4f9}',
         &['\u{2c}']), ('\u{a4fa}',  &['\u{2e}',  '\u{2e}']), ('\u{a4fb}',  &['\u{2e}',  '\u{2c}']),
        ('\u{a4fd}',  &['\u{3a}']), ('\u{a4fe}',  &['\u{2d}',  '\u{2e}']), ('\u{a4ff}',
         &['\u{3d}']), ('\u{a60e}',  &['\u{2e}']), ('\u{a644}',  &['\u{32}']), ('\u{a645}',
         &['\u{1a8}']), ('\u{a647}',  &['\u{69}']), ('\u{a64d}',  &['\u{3c9}']), ('\u{a650}',
         &['\u{42a}',  '\u{6c}']), ('\u{a651}',  &['\u{2c9}',  '\u{62}',  '\u{69}']), ('\u{a668}',
         &['\u{298}']), ('\u{a66f}',  &['\u{20e9}']), ('\u{a67c}',  &['\u{306}']), ('\u{a67e}',
         &['\u{2c7}']), ('\u{a695}',  &['\u{68}',  '\u{314}']), ('\u{a698}',  &['\u{4f}',
         '\u{4f}']), ('\u{a699}',  &['\u{6f}',  '\u{6f}']), ('\u{a69a}',  &['\u{102a8}']),
        ('\u{a6a1}',  &['\u{418}']), ('\u{a6b0}',  &['\u{16b9}']), ('\u{a6b1}',  &['\u{2c75}']),
        ('\u{a6cd}',  &['\u{2a1}']), ('\u{a6ce}',  &['\u{245}']), ('\u{a6db}',  &['\u{3a0}']),
        ('\u{a6df}',  &['\u{56}']), ('\u{a6eb}',  &['\u{3f}']), ('\u{a6ef}',  &['\u{32}']),
        ('\u{a6f0}',  &['\u{302}']), ('\u{a6f1}',  &['\u{304}']), ('\u{a6f4}',  &['\u{a6f3}',
         '\u{a6f3}']), ('\u{a714}',  &['\u{2eb}']), ('\u{a716}',  &['\u{2ea}']), ('\u{a728}',
         &['\u{54}',  '\u{33}']), ('\u{a729}',  &['\u{74}',  '\u{21d}']), ('\u{a731}',
         &['\u{73}']), ('\u{a732}',  &['\u{41}',  '\u{41}']), ('\u{a733}',  &['\u{61}',  '\u{61}']),
        ('\u{a734}',  &['\u{41}',  '\u{4f}']), ('\u{a735}',  &['\u{61}',  '\u{6f}']), ('\u{a736}',
         &['\u{41}',  '\u{55}']), ('\u{a737}',  &['\u{61}',  '\u{75}']), ('\u{a738}',  &['\u{41}',
         '\u{56}']), ('\u{a739}',  &['\u{61}',  '\u{76}']), ('\u{a73a}',  &['\u{41}',  '\u{56}']),
        ('\u{a73b}',  &['\u{61}',  '\u{76}']), ('\u{a73c}',  &['\u{41}',  '\u{59}']), ('\u{a73d}',
         &['\u{61}',  '\u{79}']), ('\u{a740}',  &['\u{4b}',  '\u{335}']), ('\u{a74a}',  &['\u{4f}',
         '\u{335}']), ('\u{a74b}',  &['\u{6f}',  '\u{335}']), ('\u{a74e}',  &['\u{4f}',  '\u{4f}']),
        ('\u{a74f}',  &['\u{6f}',  '\u{6f}']), ('\u{a75a}',  &['\u{32}']), ('\u{a761}',  &['\u{77}',
         '\u{326}']), ('\u{a76a}',  &['\u{33}']), ('\u{a76b}',  &['\u{21d}']), ('\u{a76e}',
         &['\u{39}']), ('\u{a777}',  &['\u{74}',  '\u{66}']), ('\u{a778}',  &['\u{26}']),
        ('\u{a77a}',  &['\u{a779}']), ('\u{a789}',  &['\u{3a}']), ('\u{a78c}',  &['\u{27}']),
        ('\u{a78f}',  &['\u{b7}']), ('\u{a795}',  &['\u{a727}']), ('\u{a798}',  &['\u{46}']),
        ('\u{a799}',  &['\u{66}']), ('\u{a79a}',  &['\u{10412}']), ('\u{a79b}',  &['\u{1043a}']),
        ('\u{a79d}',  &['\u{29a}']), ('\u{a79e}',  &['\u{a4e4}']), ('\u{a79f}',  &['\u{75}']),
        ('\u{a7ab}',  &['\u{33}']), ('\u{a7b1}',  &['\u{a4d5}']), ('\u{a7b2}',  &['\u{4a}']),
        ('\u{a7b3}',  &['\u{58}']), ('\u{a7b4}',  &['\u{42}']), ('\u{a7b5}',  &['\u{df}']),
        ('\u{a7b6}',  &['\u{a64c}']), ('\u{a7b7}',  &['\u{3c9}']), ('\u{a7d6}',  &['\u{df}']),
        ('\u{a7da}',  &['\u{245}']), ('\u{a7db}',  &['\u{3bb}']), ('\u{a7dc}',  &['\u{245}',
         '\u{338}']), ('\u{a7f7}',  &['\u{30fc}']), ('\u{a830}',  &['\u{964}']), ('\u{a960}',
         &['\u{1103}',  '\u{1106}']), ('\u{a961}',  &['\u{1103}',  '\u{1107}']), ('\u{a962}',
         &['\u{1103}',  '\u{1109}']), ('\u{a963}',  &['\u{1103}',  '\u{110c}']), ('\u{a964}',
         &['\u{1105}',  '\u{1100}']), ('\u{a965}',  &['\u{1105}',  '\u{1100}',  '\u{1100}']),
        ('\u{a966}',  &['\u{1105}',  '\u{1103}']), ('\u{a967}',  &['\u{1105}',  '\u{1103}',
         '\u{1103}']), ('\u{a968}',  &['\u{1105}',  '\u{1106}']), ('\u{a969}',  &['\u{1105}',
         '\u{1107}']), ('\u{a96a}',  &['\u{1105}',  '\u{1107}',  '\u{1107}']), ('\u{a96b}',
         &['\u{1105}',  '\u{1107}',  '\u{110b}']), ('\u{a96c}',  &['\u{1105}',  '\u{1109}']),
        ('\u{a96d}',  &['\u{1105}',  '\u{110c}']), ('\u{a96e}',  &['\u{1105}',  '\u{110f}']),
        ('\u{a96f}',  &['\u{1106}',  '\u{1100}']), ('\u{a970}',  &['\u{1106}',  '\u{1103}']),
        ('\u{a971}',  &['\u{1106}',  '\u{1109}']), ('\u{a972}',  &['\u{1107}',  '\u{1109}',
         '\u{1110}']), ('\u{a973}',  &['\u{1107}',  '\u{110f}']), ('\u{a974}',  &['\u{1107}',
         '\u{1112}']), ('\u{a975}',  &['\u{1109}',  '\u{1109}',  '\u{1107}']), ('\u{a976}',
         &['\u{110b}',  '\u{1105}']), ('\u{a977}',  &['\u{110b}',  '\u{1112}']), ('\u{a978}',
         &['\u{110c}',  '\u{110c}',  '\u{1112}']), ('\u{a979}',  &['\u{1110}',  '\u{1110}']),
        ('\u{a97a}',  &['\u{1111}',  '\u{1112}']), ('\u{a97b}',  &['\u{1112}',  '\u{1109}']),
        ('\u{a97c}',  &['\u{1159}',  '\u{1159}']), ('\u{a992}',  &['\u{2c3f}']), ('\u{a9a3}',
         &['\u{a99d}']), ('\u{a9c6}',  &['\u{a9d0}']), ('\u{a9cf}',  &['\u{662}']), ('\u{aa53}',
         &['\u{aa01}']), ('\u{aa56}',  &['\u{aa23}']), ('\u{ab32}',  &['\u{65}']), ('\u{ab35}',
         &['\u{66}']), ('\u{ab3d}',  &['\u{6f}']), ('\u{ab3e}',  &['\u{6f}',  '\u{338}']),
        ('\u{ab3f}',  &['\u{254}',  '\u{338}']), ('\u{ab41}',  &['\u{1dd}',  '\u{6f}',  '\u{338}']),
        ('\u{ab42}',  &['\u{1dd}',  '\u{6f}',  '\u{335}']), ('\u{ab47}',  &['\u{72}']), ('\u{ab48}',
         &['\u{72}']), ('\u{ab4d}',  &['\u{283}']), ('\u{ab4e}',  &['\u{75}']), ('\u{ab52}',
         &['\u{75}']), ('\u{ab53}',  &['\u{3c7}']), ('\u{ab55}',  &['\u{3c7}']), ('\u{ab5a}',
         &['\u{79}']), ('\u{ab60}',  &['\u{459}']), ('\u{ab62}',  &['\u{254}',  '\u{65}']),
        ('\u{ab63}',  &['\u{75}',  '\u{6f}']), ('\u{ab70}',  &['\u{1d05}']), ('\u{ab71}',
         &['\u{280}']), ('\u{ab72}',  &['\u{1d1b}']), ('\u{ab74}',  &['\u{6f}',  '\u{31b}']),
        ('\u{ab75}',  &['\u{69}']), ('\u{ab7a}',  &['\u{1d00}']), ('\u{ab7b}',  &['\u{1d0a}']),
        ('\u{ab7c}',  &['\u{1d07}']), ('\u{ab7e}',  &['\u{242}']), ('\u{ab80}',  &['\u{2c76}']),
        ('\u{ab81}',  &['\u{72}']), ('\u{ab83}',  &['\u{77}']), ('\u{ab87}',  &['\u{28d}']),
        ('\u{ab8b}',  &['\u{29c}']), ('\u{ab8e}',  &['\u{6f}',  '\u{335}']), ('\u{ab90}',
         &['\u{262}']), ('\u{ab93}',  &['\u{7a}']), ('\u{ab9b}',  &['\u{a793}']), ('\u{ab9c}',
         &['\u{75}',  '\u{335}']), ('\u{ab9f}',  &['\u{185}']), ('\u{aba2}',  &['\u{280}']),
        ('\u{aba9}',  &['\u{76}']), ('\u{abaa}',  &['\u{73}']), ('\u{abae}',  &['\u{29f}']),
        ('\u{abaf}',  &['\u{63}']), ('\u{abb2}',  &['\u{1d18}']), ('\u{abb6}',  &['\u{138}']),
        ('\u{abbb}',  &['\u{6f}',  '\u{335}']), ('\u{d7b0}',  &['\u{1169}',  '\u{1167}']),
        ('\u{d7b1}',  &['\u{1169}',  '\u{1169}',  '\u{4e28}']), ('\u{d7b2}',  &['\u{116d}',
         '\u{1161}']), ('\u{d7b3}',  &['\u{116d}',  '\u{1161}',  '\u{4e28}']), ('\u{d7b4}',
         &['\u{116d}',  '\u{1165}']), ('\u{d7b5}',  &['\u{116e}',  '\u{1167}']), ('\u{d7b6}',
         &['\u{116e}',  '\u{4e28}',  '\u{4e28}']), ('\u{d7b7}',  &['\u{1172}',  '\u{1161}',
         '\u{4e28}']), ('\u{d7b8}',  &['\u{1172}',  '\u{1169}']), ('\u{d7b9}',  &['\u{30fc}',
         '\u{1161}']), ('\u{d7ba}',  &['\u{30fc}',  '\u{1165}']), ('\u{d7bb}',  &['\u{30fc}',
         '\u{1165}',  '\u{4e28}']), ('\u{d7bc}',  &['\u{30fc}',  '\u{1169}']), ('\u{d7bd}',
         &['\u{4e28}',  '\u{1163}',  '\u{1169}']), ('\u{d7be}',  &['\u{4e28}',  '\u{1163}',
         '\u{4e28}']), ('\u{d7bf}',  &['\u{4e28}',  '\u{1167}']), ('\u{d7c0}',  &['\u{4e28}',
         '\u{1167}',  '\u{4e28}']), ('\u{d7c1}',  &['\u{4e28}',  '\u{1169}',  '\u{4e28}']),
        ('\u{d7c2}',  &['\u{4e28}',  '\u{116d}']), ('\u{d7c3}',  &['\u{4e28}',  '\u{1172}']),
        ('\u{d7c4}',  &['\u{4e28}',  '\u{4e28}']), ('\u{d7c5}',  &['\u{119e}',  '\u{1161}']),
        ('\u{d7c6}',  &['\u{119e}',  '\u{1165}',  '\u{4e28}']), ('\u{d7cb}',  &['\u{1102}',
         '\u{1105}']), ('\u{d7cc}',  &['\u{1102}',  '\u{110e}']), ('\u{d7cd}',  &['\u{1103}',
         '\u{1103}']), ('\u{d7ce}',  &['\u{1103}',  '\u{1103}',  '\u{1107}']), ('\u{d7cf}',
         &['\u{1103}',  '\u{1107}']), ('\u{d7d0}',  &['\u{1103}',  '\u{1109}']), ('\u{d7d1}',
         &['\u{1103}',  '\u{1109}',  '\u{1100}']), ('\u{d7d2}',  &['\u{1103}',  '\u{110c}']),
        ('\u{d7d3}',  &['\u{1103}',  '\u{110e}']), ('\u{d7d4}',  &['\u{1103}',  '\u{1110}']),
        ('\u{d7d5}',  &['\u{1105}',  '\u{1100}',  '\u{1100}']), ('\u{d7d6}',  &['\u{1105}',
         '\u{1100}',  '\u{1112}']), ('\u{d7d7}',  &['\u{1105}',  '\u{1105}',  '\u{110f}']),
        ('\u{d7d8}',  &['\u{1105}',  '\u{1106}',  '\u{1112}']), ('\u{d7d9}',  &['\u{1105}',
         '\u{1107}',  '\u{1103}']), ('\u{d7da}',  &['\u{1105}',  '\u{1107}',  '\u{1111}']),
        ('\u{d7db}',  &['\u{1105}',  '\u{114c}']), ('\u{d7dc}',  &['\u{1105}',  '\u{1159}',
         '\u{1112}']), ('\u{d7dd}',  &['\u{1105}',  '\u{110b}']), ('\u{d7de}',  &['\u{1106}',
         '\u{1102}']), ('\u{d7df}',  &['\u{1106}',  '\u{1102}',  '\u{1102}']), ('\u{d7e0}',
         &['\u{1106}',  '\u{1106}']), ('\u{d7e1}',  &['\u{1106}',  '\u{1107}',  '\u{1109}']),
        ('\u{d7e2}',  &['\u{1106}',  '\u{110c}']), ('\u{d7e3}',  &['\u{1107}',  '\u{1103}']),
        ('\u{d7e4}',  &['\u{1107}',  '\u{1105}',  '\u{1111}']), ('\u{d7e5}',  &['\u{1107}',
         '\u{1106}']), ('\u{d7e6}',  &['\u{1107}',  '\u{1107}']), ('\u{d7e7}',  &['\u{1107}',
         '\u{1109}',  '\u{1103}']), ('\u{d7e8}',  &['\u{1107}',  '\u{110c}']), ('\u{d7e9}',
         &['\u{1107}',  '\u{110e}']), ('\u{d7ea}',  &['\u{1109}',  '\u{1106}']), ('\u{d7eb}',
         &['\u{1109}',  '\u{1107}',  '\u{110b}']), ('\u{d7ec}',  &['\u{1109}',  '\u{1109}',
         '\u{1100}']), ('\u{d7ed}',  &['\u{1109}',  '\u{1109}',  '\u{1103}']), ('\u{d7ee}',
         &['\u{1109}',  '\u{1140}']), ('\u{d7ef}',  &['\u{1109}',  '\u{110c}']), ('\u{d7f0}',
         &['\u{1109}',  '\u{110e}']), ('\u{d7f1}',  &['\u{1109}',  '\u{1110}']), ('\u{d7f2}',
         &['\u{1105}',  '\u{1112}']), ('\u{d7f3}',  &['\u{1140}',  '\u{1107}']), ('\u{d7f4}',
         &['\u{1140}',  '\u{1107}',  '\u{110b}']), ('\u{d7f5}',  &['\u{114c}',  '\u{1106}']),
        ('\u{d7f6}',  &['\u{114c}',  '\u{1112}']), ('\u{d7f7}',  &['\u{110c}',  '\u{1107}']),
        ('\u{d7f8}',  &['\u{110c}',  '\u{1107}',  '\u{1107}']), ('\u{d7f9}',  &['\u{110c}',
         '\u{110c}']), ('\u{d7fa}',  &['\u{1111}',  '\u{1109}']), ('\u{d7fb}',  &['\u{1111}',
         '\u{1110}']), ('\u{f900}',  &['\u{8c48}']), ('\u{f901}',  &['\u{66f4}']), ('\u{f902}',
         &['\u{8eca}']), ('\u{f903}',  &['\u{8cc8}']), ('\u{f904}',  &['\u{6ed1}']), ('\u{f905}',
         &['\u{4e32}']), ('\u{f906}',  &['\u{53e5}']), ('\u{f907}',  &['\u{9f9c}']), ('\u{f908}',
         &['\u{9f9c}']), ('\u{f909}',  &['\u{5951}']), ('\u{f90a}',  &['\u{91d1}']), ('\u{f90b}',
         &['\u{5587}']), ('\u{f90c}',  &['\u{5948}']), ('\u{f90d}',  &['\u{61f6}']), ('\u{f90e}',
         &['\u{7669}']), ('\u{f90f}',  &['\u{7f85}']), ('\u{f910}',  &['\u{863f}']), ('\u{f911}',
         &['\u{87ba}']), ('\u{f912}',  &['\u{88f8}']), ('\u{f913}',  &['\u{908f}']), ('\u{f914}',
         &['\u{6a02}']), ('\u{f915}',  &['\u{6d1b}']), ('\u{f916}',  &['\u{70d9}']), ('\u{f917}',
         &['\u{73de}']), ('\u{f918}',  &['\u{843d}']), ('\u{f919}',  &['\u{916a}']), ('\u{f91a}',
         &['\u{99f1}']), ('\u{f91b}',  &['\u{4e82}']), ('\u{f91c}',  &['\u{5375}']), ('\u{f91d}',
         &['\u{6b04}']), ('\u{f91e}',  &['\u{721b}']), ('\u{f91f}',  &['\u{862d}']), ('\u{f920}',
         &['\u{9e1e}']), ('\u{f921}',  &['\u{5d50}']), ('\u{f922}',  &['\u{6feb}']), ('\u{f923}',
         &['\u{85cd}']), ('\u{f924}',  &['\u{8964}']), ('\u{f925}',  &['\u{62c9}']), ('\u{f926}',
         &['\u{81d8}']), ('\u{f927}',  &['\u{881f}']), ('\u{f928}',  &['\u{5eca}']), ('\u{f929}',
         &['\u{6717}']), ('\u{f92a}',  &['\u{6d6a}']), ('\u{f92b}',  &['\u{72fc}']), ('\u{f92c}',
         &['\u{90ce}']), ('\u{f92d}',  &['\u{4f86}']), ('\u{f92e}',  &['\u{51b7}']), ('\u{f92f}',
         &['\u{52de}']), ('\u{f930}',  &['\u{64c4}']), ('\u{f931}',  &['\u{6ad3}']), ('\u{f932}',
         &['\u{7210}']), ('\u{f933}',  &['\u{76e7}']), ('\u{f934}',  &['\u{8001}']), ('\u{f935}',
         &['\u{8606}']), ('\u{f936}',  &['\u{865c}']), ('\u{f937}',  &['\u{8def}']), ('\u{f938}',
         &['\u{9732}']), ('\u{f939}',  &['\u{9b6f}']), ('\u{f93a}',  &['\u{9dfa}']), ('\u{f93b}',
         &['\u{788c}']), ('\u{f93c}',  &['\u{797f}']), ('\u{f93d}',  &['\u{7da0}']), ('\u{f93e}',
         &['\u{83c9}']), ('\u{f93f}',  &['\u{9304}']), ('\u{f940}',  &['\u{9e7f}']), ('\u{f941}',
         &['\u{8ad6}']), ('\u{f942}',  &['\u{58df}']), ('\u{f943}',  &['\u{5f04}']), ('\u{f944}',
         &['\u{7c60}']), ('\u{f945}',  &['\u{807e}']), ('\u{f946}',  &['\u{7262}']), ('\u{f947}',
         &['\u{78ca}']), ('\u{f948}',  &['\u{8cc2}']), ('\u{f949}',  &['\u{96f7}']), ('\u{f94a}',
         &['\u{58d8}']), ('\u{f94b}',  &['\u{5c62}']), ('\u{f94c}',  &['\u{6a13}']), ('\u{f94d}',
         &['\u{6dda}']), ('\u{f94e}',  &['\u{6f0f}']), ('\u{f94f}',  &['\u{7d2f}']), ('\u{f950}',
         &['\u{7e37}']), ('\u{f951}',  &['\u{964b}']), ('\u{f952}',  &['\u{52d2}']), ('\u{f953}',
         &['\u{808b}']), ('\u{f954}',  &['\u{51dc}']), ('\u{f955}',  &['\u{51cc}']), ('\u{f956}',
         &['\u{7a1c}']), ('\u{f957}',  &['\u{7dbe}']), ('\u{f958}',  &['\u{83f1}']), ('\u{f959}',
         &['\u{9675}']), ('\u{f95a}',  &['\u{8b80}']), ('\u{f95b}',  &['\u{62cf}']), ('\u{f95c}',
         &['\u{6a02}']), ('\u{f95d}',  &['\u{8afe}']), ('\u{f95e}',  &['\u{4e39}']), ('\u{f95f}',
         &['\u{5be7}']), ('\u{f960}',  &['\u{6012}']), ('\u{f961}',  &['\u{7387}']), ('\u{f962}',
         &['\u{7570}']), ('\u{f963}',  &['\u{5317}']), ('\u{f964}',  &['\u{78fb}']), ('\u{f965}',
         &['\u{4fbf}']), ('\u{f966}',  &['\u{5fa9}']), ('\u{f967}',  &['\u{4e0d}']), ('\u{f968}',
         &['\u{6ccc}']), ('\u{f969}',  &['\u{6578}']), ('\u{f96a}',  &['\u{7d22}']), ('\u{f96b}',
         &['\u{53c3}']), ('\u{f96c}',  &['\u{585e}']), ('\u{f96d}',  &['\u{7701}']), ('\u{f96e}',
         &['\u{8449}']), ('\u{f96f}',  &['\u{8aaa}']), ('\u{f970}',  &['\u{6bba}']), ('\u{f971}',
         &['\u{8fb0}']), ('\u{f972}',  &['\u{6c88}']), ('\u{f973}',  &['\u{62fe}']), ('\u{f974}',
         &['\u{82e5}']), ('\u{f975}',  &['\u{63a0}']), ('\u{f976}',  &['\u{7565}']), ('\u{f977}',
         &['\u{4eae}']), ('\u{f978}',  &['\u{5169}']), ('\u{f979}',  &['\u{51c9}']), ('\u{f97a}',
         &['\u{6881}']), ('\u{f97b}',  &['\u{7ce7}']), ('\u{f97c}',  &['\u{826f}']), ('\u{f97d}',
         &['\u{8ad2}']), ('\u{f97e}',  &['\u{91cf}']), ('\u{f97f}',  &['\u{52f5}']), ('\u{f980}',
         &['\u{5442}']), ('\u{f981}',  &['\u{5973}']), ('\u{f982}',  &['\u{5eec}']), ('\u{f983}',
         &['\u{65c5}']), ('\u{f984}',  &['\u{6ffe}']), ('\u{f985}',  &['\u{792a}']), ('\u{f986}',
         &['\u{95ad}']), ('\u{f987}',  &['\u{9a6a}']), ('\u{f988}',  &['\u{9e97}']), ('\u{f989}',
         &['\u{9ece}']), ('\u{f98a}',  &['\u{529b}']), ('\u{f98b}',  &['\u{66c6}']), ('\u{f98c}',
         &['\u{6b77}']), ('\u{f98d}',  &['\u{8f62}']), ('\u{f98e}',  &['\u{5e74}']), ('\u{f98f}',
         &['\u{6190}']), ('\u{f990}',  &['\u{6200}']), ('\u{f991}',  &['\u{649a}']), ('\u{f992}',
         &['\u{6f23}']), ('\u{f993}',  &['\u{7149}']), ('\u{f994}',  &['\u{7489}']), ('\u{f995}',
         &['\u{79ca}']), ('\u{f996}',  &['\u{7df4}']), ('\u{f997}',  &['\u{806f}']), ('\u{f998}',
         &['\u{8f26}']), ('\u{f999}',  &['\u{84ee}']), ('\u{f99a}',  &['\u{9023}']), ('\u{f99b}',
         &['\u{934a}']), ('\u{f99c}',  &['\u{5217}']), ('\u{f99d}',  &['\u{52a3}']), ('\u{f99e}',
         &['\u{54bd}']), ('\u{f99f}',  &['\u{70c8}']), ('\u{f9a0}',  &['\u{88c2}']), ('\u{f9a1}',
         &['\u{8aaa}']), ('\u{f9a2}',  &['\u{5ec9}']), ('\u{f9a3}',  &['\u{5ff5}']), ('\u{f9a4}',
         &['\u{637b}']), ('\u{f9a5}',  &['\u{6bae}']), ('\u{f9a6}',  &['\u{7c3e}']), ('\u{f9a7}',
         &['\u{7375}']), ('\u{f9a8}',  &['\u{4ee4}']), ('\u{f9a9}',  &['\u{56f9}']), ('\u{f9aa}',
         &['\u{5be7}']), ('\u{f9ab}',  &['\u{5dba}']), ('\u{f9ac}',  &['\u{601c}']), ('\u{f9ad}',
         &['\u{73b2}']), ('\u{f9ae}',  &['\u{7469}']), ('\u{f9af}',  &['\u{7f9a}']), ('\u{f9b0}',
         &['\u{8046}']), ('\u{f9b1}',  &['\u{9234}']), ('\u{f9b2}',  &['\u{96f6}']), ('\u{f9b3}',
         &['\u{9748}']), ('\u{f9b4}',  &['\u{9818}']), ('\u{f9b5}',  &['\u{4f8b}']), ('\u{f9b6}',
         &['\u{79ae}']), ('\u{f9b7}',  &['\u{91b4}']), ('\u{f9b8}',  &['\u{96b7}']), ('\u{f9b9}',
         &['\u{60e1}']), ('\u{f9ba}',  &['\u{4e86}']), ('\u{f9bb}',  &['\u{50da}']), ('\u{f9bc}',
         &['\u{5bee}']), ('\u{f9bd}',  &['\u{5c3f}']), ('\u{f9be}',  &['\u{6599}']), ('\u{f9bf}',
         &['\u{6a02}']), ('\u{f9c0}',  &['\u{71ce}']), ('\u{f9c1}',  &['\u{7642}']), ('\u{f9c2}',
         &['\u{84fc}']), ('\u{f9c3}',  &['\u{907c}']), ('\u{f9c4}',  &['\u{9f8d}']), ('\u{f9c5}',
         &['\u{6688}']), ('\u{f9c6}',  &['\u{962e}']), ('\u{f9c7}',  &['\u{5289}']), ('\u{f9c8}',
         &['\u{677b}']), ('\u{f9c9}',  &['\u{67f3}']), ('\u{f9ca}',  &['\u{6d41}']), ('\u{f9cb}',
         &['\u{6e9c}']), ('\u{f9cc}',  &['\u{7409}']), ('\u{f9cd}',  &['\u{7559}']), ('\u{f9ce}',
         &['\u{786b}']), ('\u{f9cf}',  &['\u{7d10}']), ('\u{f9d0}',  &['\u{985e}']), ('\u{f9d1}',
         &['\u{516d}']), ('\u{f9d2}',  &['\u{622e}']), ('\u{f9d3}',  &['\u{9678}']), ('\u{f9d4}',
         &['\u{502b}']), ('\u{f9d5}',  &['\u{5d19}']), ('\u{f9d6}',  &['\u{6dea}']), ('\u{f9d7}',
         &['\u{8f2a}']), ('\u{f9d8}',  &['\u{5f8b}']), ('\u{f9d9}',  &['\u{6144}']), ('\u{f9da}',
         &['\u{6817}']), ('\u{f9db}',  &['\u{7387}']), ('\u{f9dc}',  &['\u{9686}']), ('\u{f9dd}',
         &['\u{5229}']), ('\u{f9de}',  &['\u{540f}']), ('\u{f9df}',  &['\u{5c65}']), ('\u{f9e0}',
         &['\u{6613}']), ('\u{f9e1}',  &['\u{674e}']), ('\u{f9e2}',  &['\u{68a8}']), ('\u{f9e3}',
         &['\u{6ce5}']), ('\u{f9e4}',  &['\u{7406}']), ('\u{f9e5}',  &['\u{75e2}']), ('\u{f9e6}',
         &['\u{7f79}']), ('\u{f9e7}',  &['\u{88cf}']), ('\u{f9e8}',  &['\u{88e1}']), ('\u{f9e9}',
         &['\u{91cc}']), ('\u{f9ea}',  &['\u{96e2}']), ('\u{f9eb}',  &['\u{533f}']), ('\u{f9ec}',
         &['\u{6eba}']), ('\u{f9ed}',  &['\u{541d}']), ('\u{f9ee}',  &['\u{71d0}']), ('\u{f9ef}',
         &['\u{7498}']), ('\u{f9f0}',  &['\u{85fa}']), ('\u{f9f1}',  &['\u{96a3}']), ('\u{f9f2}',
         &['\u{9c57}']), ('\u{f9f3}',  &['\u{9e9f}']), ('\u{f9f4}',  &['\u{6797}']), ('\u{f9f5}',
         &['\u{6dcb}']), ('\u{f9f6}',  &['\u{81e8}']), ('\u{f9f7}',  &['\u{7acb}']), ('\u{f9f8}',
         &['\u{7b20}']), ('\u{f9f9}',  &['\u{7c92}']), ('\u{f9fa}',  &['\u{72c0}']), ('\u{f9fb}',
         &['\u{7099}']), ('\u{f9fc}',  &['\u{8b58}']), ('\u{f9fd}',  &['\u{4ec0}']), ('\u{f9fe}',
         &['\u{8336}']), ('\u{f9ff}',  &['\u{523a}']), ('\u{fa00}',  &['\u{5207}']), ('\u{fa01}',
         &['\u{5ea6}']), ('\u{fa02}',  &['\u{62d3}']), ('\u{fa03}',  &['\u{7cd6}']), ('\u{fa04}',
         &['\u{5b85}']), ('\u{fa05}',  &['\u{6d1e}']), ('\u{fa06}',  &['\u{66b4}']), ('\u{fa07}',
         &['\u{8f3b}']), ('\u{fa08}',  &['\u{884c}']), ('\u{fa09}',  &['\u{964d}']), ('\u{fa0a}',
         &['\u{898b}']), ('\u{fa0b}',  &['\u{5ed3}']), ('\u{fa0c}',  &['\u{5140}']), ('\u{fa0d}',
         &['\u{55c0}']), ('\u{fa10}',  &['\u{585a}']), ('\u{fa12}',  &['\u{6674}']), ('\u{fa15}',
         &['\u{51de}']), ('\u{fa16}',  &['\u{732a}']), ('\u{fa17}',  &['\u{76ca}']), ('\u{fa18}',
         &['\u{793c}']), ('\u{fa19}',  &['\u{795e}']), ('\u{fa1a}',  &['\u{7965}']), ('\u{fa1b}',
         &['\u{798f}']), ('\u{fa1c}',  &['\u{9756}']), ('\u{fa1d}',  &['\u{7cbe}']), ('\u{fa1e}',
         &['\u{7fbd}']), ('\u{fa20}',  &['\u{8612}']), ('\u{fa22}',  &['\u{8af8}']), ('\u{fa25}',
         &['\u{9038}']), ('\u{fa26}',  &['\u{90fd}']), ('\u{fa2a}',  &['\u{98ef}']), ('\u{fa2b}',
         &['\u{98fc}']), ('\u{fa2c}',  &['\u{9928}']), ('\u{fa2d}',  &['\u{9db4}']), ('\u{fa2e}',
         &['\u{90ce}']), ('\u{fa2f}',  &['\u{96b7}']), ('\u{fa30}',  &['\u{4fae}']), ('\u{fa31}',
         &['\u{50e7}']), ('\u{fa32}',  &['\u{514d}']), ('\u{fa33}',  &['\u{52c9}']), ('\u{fa34}',
         &['\u{52e4}']), ('\u{fa35}',  &['\u{5351}']), ('\u{fa36}',  &['\u{559d}']), ('\u{fa37}',
         &['\u{5606}']), ('\u{fa38}',  &['\u{5668}']), ('\u{fa39}',  &['\u{5840}']), ('\u{fa3a}',
         &['\u{58a8}']), ('\u{fa3b}',  &['\u{5c64}']), ('\u{fa3c}',  &['\u{5c6e}']), ('\u{fa3d}',
         &['\u{6094}']), ('\u{fa3e}',  &['\u{6168}']), ('\u{fa3f}',  &['\u{618e}']), ('\u{fa40}',
         &['\u{61f2}']), ('\u{fa41}',  &['\u{654f}']), ('\u{fa42}',  &['\u{65e2}']), ('\u{fa43}',
         &['\u{6691}']), ('\u{fa44}',  &['\u{6885}']), ('\u{fa45}',  &['\u{6d77}']), ('\u{fa46}',
         &['\u{6e1a}']), ('\u{fa47}',  &['\u{6f22}']), ('\u{fa48}',  &['\u{716e}']), ('\u{fa49}',
         &['\u{722b}']), ('\u{fa4a}',  &['\u{7422}']), ('\u{fa4b}',  &['\u{7891}']), ('\u{fa4c}',
         &['\u{793e}']), ('\u{fa4d}',  &['\u{7949}']), ('\u{fa4e}',  &['\u{7948}']), ('\u{fa4f}',
         &['\u{7950}']), ('\u{fa50}',  &['\u{7956}']), ('\u{fa51}',  &['\u{795d}']), ('\u{fa52}',
         &['\u{798d}']), ('\u{fa53}',  &['\u{798e}']), ('\u{fa54}',  &['\u{7a40}']), ('\u{fa55}',
         &['\u{7a81}']), ('\u{fa56}',  &['\u{7bc0}']), ('\u{fa57}',  &['\u{7df4}']), ('\u{fa58}',
         &['\u{7e09}']), ('\u{fa59}',  &['\u{7e41}']), ('\u{fa5a}',  &['\u{7f72}']), ('\u{fa5b}',
         &['\u{8005}']), ('\u{fa5c}',  &['\u{81ed}']), ('\u{fa5d}',  &['\u{8279}']), ('\u{fa5e}',
         &['\u{8279}']), ('\u{fa5f}',  &['\u{8457}']), ('\u{fa60}',  &['\u{8910}']), ('\u{fa61}',
         &['\u{8996}']), ('\u{fa62}',  &['\u{8b01}']), ('\u{fa63}',  &['\u{8b39}']), ('\u{fa64}',
         &['\u{8cd3}']), ('\u{fa65}',  &['\u{8d08}']), ('\u{fa66}',  &['\u{8fb6}']), ('\u{fa67}',
         &['\u{9038}']), ('\u{fa68}',  &['\u{96e3}']), ('\u{fa69}',  &['\u{97ff}']), ('\u{fa6a}',
         &['\u{983b}']), ('\u{fa6b}',  &['\u{6075}']), ('\u{fa6c}',  &['\u{242ee}']), ('\u{fa6d}',
         &['\u{8218}']), ('\u{fa70}',  &['\u{4e26}']), ('\u{fa71}',  &['\u{51b5}']), ('\u{fa72}',
         &['\u{5168}']), ('\u{fa73}',  &['\u{4f80}']), ('\u{fa74}',  &['\u{5145}']), ('\u{fa75}',
         &['\u{5180}']), ('\u{fa76}',  &['\u{52c7}']), ('\u{fa77}',  &['\u{52fa}']), ('\u{fa78}',
         &['\u{559d}']), ('\u{fa79}',  &['\u{5555}']), ('\u{fa7a}',  &['\u{5599}']), ('\u{fa7b}',
         &['\u{55e2}']), ('\u{fa7c}',  &['\u{585a}']), ('\u{fa7d}',  &['\u{58b3}']), ('\u{fa7e}',
         &['\u{5944}']), ('\u{fa7f}',  &['\u{5954}']), ('\u{fa80}',  &['\u{5a62}']), ('\u{fa81}',
         &['\u{5b28}']), ('\u{fa82}',  &['\u{5ed2}']), ('\u{fa83}',  &['\u{5ed9}']), ('\u{fa84}',
         &['\u{5f69}']), ('\u{fa85}',  &['\u{5fad}']), ('\u{fa86}',  &['\u{60d8}']), ('\u{fa87}',
         &['\u{614e}']), ('\u{fa88}',  &['\u{6108}']), ('\u{fa89}',  &['\u{618e}']), ('\u{fa8a}',
         &['\u{6160}']), ('\u{fa8b}',  &['\u{61f2}']), ('\u{fa8c}',  &['\u{6234}']), ('\u{fa8d}',
         &['\u{63c4}']), ('\u{fa8e}',  &['\u{641c}']), ('\u{fa8f}',  &['\u{6452}']), ('\u{fa90}',
         &['\u{6556}']), ('\u{fa91}',  &['\u{6674}']), ('\u{fa92}',  &['\u{6717}']), ('\u{fa93}',
         &['\u{671b}']), ('\u{fa94}',  &['\u{6756}']), ('\u{fa95}',  &['\u{6b79}']), ('\u{fa96}',
         &['\u{6bba}']), ('\u{fa97}',  &['\u{6d41}']), ('\u{fa98}',  &['\u{6edb}']), ('\u{fa99}',
         &['\u{6ecb}']), ('\u{fa9a}',  &['\u{6f22}']), ('\u{fa9b}',  &['\u{701e}']), ('\u{fa9c}',
         &['\u{716e}']), ('\u{fa9d}',  &['\u{77a7}']), ('\u{fa9e}',  &['\u{7235}']), ('\u{fa9f}',
         &['\u{72af}']), ('\u{faa0}',  &['\u{732a}']), ('\u{faa1}',  &['\u{7471}']), ('\u{faa2}',
         &['\u{7506}']), ('\u{faa3}',  &['\u{753b}']), ('\u{faa4}',  &['\u{761d}']), ('\u{faa5}',
         &['\u{761f}']), ('\u{faa6}',  &['\u{76ca}']), ('\u{faa7}',  &['\u{76db}']), ('\u{faa8}',
         &['\u{76f4}']), ('\u{faa9}',  &['\u{774a}']), ('\u{faaa}',  &['\u{7740}']), ('\u{faab}',
         &['\u{78cc}']), ('\u{faac}',  &['\u{7ab1}']), ('\u{faad}',  &['\u{7bc0}']), ('\u{faae}',
         &['\u{7c7b}']), ('\u{faaf}',  &['\u{7d5b}']), ('\u{fab0}',  &['\u{7df4}']), ('\u{fab1}',
         &['\u{7f3e}']), ('\u{fab2}',  &['\u{8005}']), ('\u{fab3}',  &['\u{8352}']), ('\u{fab4}',
         &['\u{83ef}']), ('\u{fab5}',  &['\u{8779}']), ('\u{fab6}',  &['\u{8941}']), ('\u{fab7}',
         &['\u{8986}']), ('\u{fab8}',  &['\u{8996}']), ('\u{fab9}',  &['\u{8abf}']), ('\u{faba}',
         &['\u{8af8}']), ('\u{fabb}',  &['\u{8acb}']), ('\u{fabc}',  &['\u{8b01}']), ('\u{fabd}',
         &['\u{8afe}']), ('\u{fabe}',  &['\u{8aed}']), ('\u{fabf}',  &['\u{8b39}']), ('\u{fac0}',
         &['\u{8b8a}']), ('\u{fac1}',  &['\u{8d08}']), ('\u{fac2}',  &['\u{8f38}']), ('\u{fac3}',
         &['\u{9072}']), ('\u{fac4}',  &['\u{9199}']), ('\u{fac5}',  &['\u{9276}']), ('\u{fac6}',
         &['\u{967c}']), ('\u{fac7}',  &['\u{96e3}']), ('\u{fac8}',  &['\u{9756}']), ('\u{fac9}',
         &['\u{97db}']), ('\u{faca}',  &['\u{97ff}']), ('\u{facb}',  &['\u{980b}']), ('\u{facc}',
         &['\u{983b}']), ('\u{facd}',  &['\u{9b12}']), ('\u{face}',  &['\u{9f9c}']), ('\u{facf}',
         &['\u{2284a}']), ('\u{fad0}',  &['\u{22844}']), ('\u{fad1}',  &['\u{233d5}']), ('\u{fad2}',
         &['\u{3b9d}']), ('\u{fad3}',  &['\u{4018}']), ('\u{fad4}',  &['\u{4039}']), ('\u{fad5}',
         &['\u{25249}']), ('\u{fad6}',  &['\u{25cd0}']), ('\u{fad7}',  &['\u{27ed3}']), ('\u{fad8}',
         &['\u{9f43}']), ('\u{fad9}',  &['\u{9f8e}']), ('\u{fb00}',  &['\u{66}',  '\u{66}']),
        ('\u{fb01}',  &['\u{66}',  '\u{69}']), ('\u{fb02}',  &['\u{66}',  '\u{6c}']), ('\u{fb03}',
         &['\u{66}',  '\u{66}',  '\u{69}']), ('\u{fb04}',  &['\u{66}',  '\u{66}',  '\u{6c}']),
        ('\u{fb06}',  &['\u{73}',  '\u{74}']), ('\u{fb13}',  &['\u{574}',  '\u{576}']), ('\u{fb14}',
         &['\u{574}',  '\u{565}']), ('\u{fb15}',  &['\u{574}',  '\u{56b}']), ('\u{fb16}',
         &['\u{57e}',  '\u{576}']), ('\u{fb17}',  &['\u{574}',  '\u{56d}']), ('\u{fb20}',
         &['\u{5e2}']), ('\u{fb21}',  &['\u{5d0}']), ('\u{fb22}',  &['\u{5d3}']), ('\u{fb23}',
         &['\u{5d4}']), ('\u{fb24}',  &['\u{5db}']), ('\u{fb25}',  &['\u{5dc}']), ('\u{fb26}',
         &['\u{5dd}']), ('\u{fb27}',  &['\u{5e8}']), ('\u{fb28}',  &['\u{5ea}']), ('\u{fb29}',
         &['\u{2d}',  '\u{307}']), ('\u{fb2b}',  &['\u{fb2a}']), ('\u{fb2d}',  &['\u{fb2c}']),
        ('\u{fb2f}',  &['\u{fb2e}']), ('\u{fb30}',  &['\u{fb2e}']), ('\u{fb39}',  &['\u{fb1d}']),
        ('\u{fb49}',  &['\u{fb2a}']), ('\u{fb4f}',  &['\u{5d0}',  '\u{5dc}']), ('\u{fb50}',
         &['\u{671}']), ('\u{fb51}',  &['\u{671}']), ('\u{fb52}',  &['\u{67b}']), ('\u{fb53}',
         &['\u{67b}']), ('\u{fb54}',  &['\u{67b}']), ('\u{fb55}',  &['\u{67b}']), ('\u{fb56}',
         &['\u{649}',  '\u{6db}']), ('\u{fb57}',  &['\u{649}',  '\u{6db}']), ('\u{fb58}',
         &['\u{649}',  '\u{6db}']), ('\u{fb59}',  &['\u{649}',  '\u{6db}']), ('\u{fb5a}',
         &['\u{680}']), ('\u{fb5b}',  &['\u{680}']), ('\u{fb5c}',  &['\u{680}']), ('\u{fb5d}',
         &['\u{680}']), ('\u{fb5e}',  &['\u{67a}']), ('\u{fb5f}',  &['\u{67a}']), ('\u{fb60}',
         &['\u{67a}']), ('\u{fb61}',  &['\u{67a}']), ('\u{fb62}',  &['\u{67f}']), ('\u{fb63}',
         &['\u{67f}']), ('\u{fb64}',  &['\u{67f}']), ('\u{fb65}',  &['\u{67f}']), ('\u{fb66}',
         &['\u{649}',  '\u{615}']), ('\u{fb67}',  &['\u{649}',  '\u{615}']), ('\u{fb68}',
         &['\u{649}',  '\u{615}']), ('\u{fb69}',  &['\u{649}',  '\u{615}']), ('\u{fb6a}',
         &['\u{6a1}',  '\u{6db}']), ('\u{fb6b}',  &['\u{6a1}',  '\u{6db}']), ('\u{fb6c}',
         &['\u{6a1}',  '\u{6db}']), ('\u{fb6d}',  &['\u{6a1}',  '\u{6db}']), ('\u{fb6e}',
         &['\u{6a6}']), ('\u{fb6f}',  &['\u{6a6}']), ('\u{fb70}',  &['\u{6a6}']), ('\u{fb71}',
         &['\u{6a6}']), ('\u{fb72}',  &['\u{684}']), ('\u{fb73}',  &['\u{684}']), ('\u{fb74}',
         &['\u{684}']), ('\u{fb75}',  &['\u{684}']), ('\u{fb76}',  &['\u{683}']), ('\u{fb77}',
         &['\u{683}']), ('\u{fb78}',  &['\u{683}']), ('\u{fb79}',  &['\u{683}']), ('\u{fb7a}',
         &['\u{686}']), ('\u{fb7b}',  &['\u{686}']), ('\u{fb7c}',  &['\u{686}']), ('\u{fb7d}',
         &['\u{686}']), ('\u{fb7e}',  &['\u{687}']), ('\u{fb7f}',  &['\u{687}']), ('\u{fb80}',
         &['\u{687}']), ('\u{fb81}',  &['\u{687}']), ('\u{fb82}',  &['\u{68d}']), ('\u{fb83}',
         &['\u{68d}']), ('\u{fb84}',  &['\u{68c}']), ('\u{fb85}',  &['\u{68c}']), ('\u{fb86}',
         &['\u{62f}',  '\u{6db}']), ('\u{fb87}',  &['\u{62f}',  '\u{6db}']), ('\u{fb88}',
         &['\u{62f}',  '\u{615}']), ('\u{fb89}',  &['\u{62f}',  '\u{615}']), ('\u{fb8a}',
         &['\u{631}',  '\u{6db}']), ('\u{fb8b}',  &['\u{631}',  '\u{6db}']), ('\u{fb8c}',
         &['\u{631}',  '\u{615}']), ('\u{fb8d}',  &['\u{631}',  '\u{615}']), ('\u{fb8e}',
         &['\u{643}']), ('\u{fb8f}',  &['\u{643}']), ('\u{fb90}',  &['\u{643}']), ('\u{fb91}',
         &['\u{643}']), ('\u{fb92}',  &['\u{6af}']), ('\u{fb93}',  &['\u{6af}']), ('\u{fb94}',
         &['\u{6af}']), ('\u{fb95}',  &['\u{6af}']), ('\u{fb96}',  &['\u{6b3}']), ('\u{fb97}',
         &['\u{6b3}']), ('\u{fb98}',  &['\u{6b3}']), ('\u{fb99}',  &['\u{6b3}']), ('\u{fb9a}',
         &['\u{6b1}']), ('\u{fb9b}',  &['\u{6b1}']), ('\u{fb9c}',  &['\u{6b1}']), ('\u{fb9d}',
         &['\u{6b1}']), ('\u{fb9e}',  &['\u{649}']), ('\u{fb9f}',  &['\u{649}']), ('\u{fba0}',
         &['\u{649}',  '\u{615}']), ('\u{fba1}',  &['\u{649}',  '\u{615}']), ('\u{fba2}',
         &['\u{649}',  '\u{615}']), ('\u{fba3}',  &['\u{649}',  '\u{615}']), ('\u{fba4}',
         &['\u{6c0}']), ('\u{fba5}',  &['\u{6c0}']), ('\u{fba6}',  &['\u{6f}']), ('\u{fba7}',
         &['\u{6f}']), ('\u{fba8}',  &['\u{6f}']), ('\u{fba9}',  &['\u{6f}']), ('\u{fbaa}',
         &['\u{6f}']), ('\u{fbab}',  &['\u{6f}']), ('\u{fbac}',  &['\u{6f}']), ('\u{fbad}',
         &['\u{6f}']), ('\u{fbae}',  &['\u{649}']), ('\u{fbaf}',  &['\u{649}']), ('\u{fbb0}',
         &['\u{6d3}']), ('\u{fbb1}',  &['\u{6d3}']), ('\u{fbd3}',  &['\u{643}',  '\u{6db}']),
        ('\u{fbd4}',  &['\u{643}',  '\u{6db}']), ('\u{fbd5}',  &['\u{643}',  '\u{6db}']),
        ('\u{fbd6}',  &['\u{643}',  '\u{6db}']), ('\u{fbd7}',  &['\u{648}',  '\u{313}']),
        ('\u{fbd8}',  &['\u{648}',  '\u{313}']), ('\u{fbd9}',  &['\u{648}',  '\u{306}']),
        ('\u{fbda}',  &['\u{648}',  '\u{306}']), ('\u{fbdb}',  &['\u{648}',  '\u{670}']),
        ('\u{fbdc}',  &['\u{648}',  '\u{670}']), ('\u{fbdd}',  &['\u{648}',  '\u{313}',
         '\u{674}']), ('\u{fbde}',  &['\u{648}',  '\u{6db}']), ('\u{fbdf}',  &['\u{648}',
         '\u{6db}']), ('\u{fbe0}',  &['\u{6c5}']), ('\u{fbe1}',  &['\u{6c5}']), ('\u{fbe2}',
         &['\u{648}',  '\u{302}']), ('\u{fbe3}',  &['\u{648}',  '\u{302}']), ('\u{fbe4}',
         &['\u{67b}']), ('\u{fbe5}',  &['\u{67b}']), ('\u{fbe6}',  &['\u{67b}']), ('\u{fbe7}',
         &['\u{67b}']), ('\u{fbe8}',  &['\u{649}']), ('\u{fbe9}',  &['\u{649}']), ('\u{fbea}',
         &['\u{649}',  '\u{674}',  '\u{6c}']), ('\u{fbeb}',  &['\u{649}',  '\u{674}',  '\u{6c}']),
        ('\u{fbec}',  &['\u{649}',  '\u{674}',  '\u{6f}']), ('\u{fbed}',  &['\u{649}',  '\u{674}',
         '\u{6f}']), ('\u{fbee}',  &['\u{649}',  '\u{674}',  '\u{648}']), ('\u{fbef}',  &['\u{649}',
         '\u{674}',  '\u{648}']), ('\u{fbf0}',  &['\u{649}',  '\u{674}',  '\u{648}',  '\u{313}']),
        ('\u{fbf1}',  &['\u{649}',  '\u{674}',  '\u{648}',  '\u{313}']), ('\u{fbf2}',  &['\u{649}',
         '\u{674}',  '\u{648}',  '\u{306}']), ('\u{fbf3}',  &['\u{649}',  '\u{674}',  '\u{648}',
         '\u{306}']), ('\u{fbf4}',  &['\u{649}',  '\u{674}',  '\u{648}',  '\u{670}']), ('\u{fbf5}',
         &['\u{649}',  '\u{674}',  '\u{648}',  '\u{670}']), ('\u{fbf6}',  &['\u{649}',  '\u{674}',
         '\u{67b}']), ('\u{fbf7}',  &['\u{649}',  '\u{674}',  '\u{67b}']), ('\u{fbf8}',
         &['\u{649}',  '\u{674}',  '\u{67b}']), ('\u{fbf9}',  &['\u{649}',  '\u{674}',  '\u{649}']),
        ('\u{fbfa}',  &['\u{649}',  '\u{674}',  '\u{649}']), ('\u{fbfb}',  &['\u{649}',  '\u{674}',
         '\u{649}']), ('\u{fbfc}',  &['\u{649}']), ('\u{fbfd}',  &['\u{649}']), ('\u{fbfe}',
         &['\u{649}']), ('\u{fbff}',  &['\u{649}']), ('\u{fc00}',  &['\u{649}',  '\u{674}',
         '\u{62c}']), ('\u{fc01}',  &['\u{649}',  '\u{674}',  '\u{62d}']), ('\u{fc02}',
         &['\u{649}',  '\u{674}',  '\u{645}']), ('\u{fc03}',  &['\u{649}',  '\u{674}',  '\u{649}']),
        ('\u{fc04}',  &['\u{649}',  '\u{674}',  '\u{649}']), ('\u{fc05}',  &['\u{628}',
         '\u{62c}']), ('\u{fc06}',  &['\u{628}',  '\u{62d}']), ('\u{fc07}',  &['\u{628}',
         '\u{62e}']), ('\u{fc08}',  &['\u{628}',  '\u{645}']), ('\u{fc09}',  &['\u{628}',
         '\u{649}']), ('\u{fc0a}',  &['\u{628}',  '\u{649}']), ('\u{fc0b}',  &['\u{62a}',
         '\u{62c}']), ('\u{fc0c}',  &['\u{62a}',  '\u{62d}']), ('\u{fc0d}',  &['\u{62a}',
         '\u{62e}']), ('\u{fc0e}',  &['\u{62a}',  '\u{645}']), ('\u{fc0f}',  &['\u{62a}',
         '\u{649}']), ('\u{fc10}',  &['\u{62a}',  '\u{649}']), ('\u{fc11}',  &['\u{649}',
         '\u{6db}',  '\u{62c}']), ('\u{fc12}',  &['\u{649}',  '\u{6db}',  '\u{645}']), ('\u{fc13}',
         &['\u{649}',  '\u{6db}',  '\u{649}']), ('\u{fc14}',  &['\u{649}',  '\u{6db}',  '\u{649}']),
        ('\u{fc15}',  &['\u{62c}',  '\u{62d}']), ('\u{fc16}',  &['\u{62c}',  '\u{645}']),
        ('\u{fc17}',  &['\u{62d}',  '\u{62c}']), ('\u{fc18}',  &['\u{62d}',  '\u{645}']),
        ('\u{fc19}',  &['\u{62e}',  '\u{62c}']), ('\u{fc1a}',  &['\u{62e}',  '\u{62d}']),
        ('\u{fc1b}',  &['\u{62e}',  '\u{645}']), ('\u{fc1c}',  &['\u{633}',  '\u{62c}']),
        ('\u{fc1d}',  &['\u{633}',  '\u{62d}']), ('\u{fc1e}',  &['\u{633}',  '\u{62e}']),
        ('\u{fc1f}',  &['\u{633}',  '\u{645}']), ('\u{fc20}',  &['\u{635}',  '\u{62d}']),
        ('\u{fc21}',  &['\u{635}',  '\u{645}']), ('\u{fc22}',  &['\u{636}',  '\u{62c}']),
        ('\u{fc23}',  &['\u{636}',  '\u{62d}']), ('\u{fc24}',  &['\u{636}',  '\u{62e}']),
        ('\u{fc25}',  &['\u{636}',  '\u{645}']), ('\u{fc26}',  &['\u{637}',  '\u{62d}']),
        ('\u{fc27}',  &['\u{637}',  '\u{645}']), ('\u{fc28}',  &['\u{638}',  '\u{645}']),
        ('\u{fc29}',  &['\u{639}',  '\u{62c}']), ('\u{fc2a}',  &['\u{639}',  '\u{645}']),
        ('\u{fc2b}',  &['\u{63a}',  '\u{62c}']), ('\u{fc2c}',  &['\u{63a}',  '\u{645}']),
        ('\u{fc2d}',  &['\u{641}',  '\u{62c}']), ('\u{fc2e}',  &['\u{641}',  '\u{62d}']),
        ('\u{fc2f}',  &['\u{641}',  '\u{62e}']), ('\u{fc30}',  &['\u{641}',  '\u{645}']),
        ('\u{fc31}',  &['\u{641}',  '\u{649}']), ('\u{fc32}',  &['\u{641}',  '\u{649}']),
        ('\u{fc33}',  &['\u{642}',  '\u{62d}']), ('\u{fc34}',  &['\u{642}',  '\u{645}']),
        ('\u{fc35}',  &['\u{642}',  '\u{649}']), ('\u{fc36}',  &['\u{642}',  '\u{649}']),
        ('\u{fc37}',  &['\u{643}',  '\u{6c}']), ('\u{fc38}',  &['\u{643}',  '\u{62c}']),
        ('\u{fc39}',  &['\u{643}',  '\u{62d}']), ('\u{fc3a}',  &['\u{643}',  '\u{62e}']),
        ('\u{fc3b}',  &['\u{643}',  '\u{644}']), ('\u{fc3c}',  &['\u{643}',  '\u{645}']),
        ('\u{fc3d}',  &['\u{643}',  '\u{649}']), ('\u{fc3e}',  &['\u{643}',  '\u{649}']),
        ('\u{fc3f}',  &['\u{644}',  '\u{62c}']), ('\u{fc40}',  &['\u{644}',  '\u{62d}']),
        ('\u{fc41}',  &['\u{644}',  '\u{62e}']), ('\u{fc42}',  &['\u{644}',  '\u{645}']),
        ('\u{fc43}',  &['\u{644}',  '\u{649}']), ('\u{fc44}',  &['\u{644}',  '\u{649}']),
        ('\u{fc45}',  &['\u{645}',  '\u{62c}']), ('\u{fc46}',  &['\u{645}',  '\u{62d}']),
        ('\u{fc47}',  &['\u{645}',  '\u{62e}']), ('\u{fc48}',  &['\u{645}',  '\u{645}']),
        ('\u{fc49}',  &['\u{645}',  '\u{649}']), ('\u{fc4a}',  &['\u{645}',  '\u{649}']),
        ('\u{fc4b}',  &['\u{628}',  '\u{62e}']), ('\u{fc4c}',  &['\u{646}',  '\u{62d}']),
        ('\u{fc4d}',  &['\u{646}',  '\u{62e}']), ('\u{fc4e}',  &['\u{646}',  '\u{645}']),
        ('\u{fc4f}',  &['\u{646}',  '\u{649}']), ('\u{fc50}',  &['\u{646}',  '\u{649}']),
        ('\u{fc51}',  &['\u{6f}',  '\u{62c}']), ('\u{fc52}',  &['\u{6f}',  '\u{645}']), ('\u{fc53}',
         &['\u{6f}',  '\u{649}']), ('\u{fc54}',  &['\u{6f}',  '\u{649}']), ('\u{fc55}',
         &['\u{649}',  '\u{62c}']), ('\u{fc56}',  &['\u{649}',  '\u{62d}']), ('\u{fc57}',
         &['\u{649}',  '\u{62e}']), ('\u{fc58}',  &['\u{649}',  '\u{645}']), ('\u{fc59}',
         &['\u{649}',  '\u{649}']), ('\u{fc5a}',  &['\u{649}',  '\u{649}']), ('\u{fc5b}',
         &['\u{630}',  '\u{670}']), ('\u{fc5c}',  &['\u{631}',  '\u{670}']), ('\u{fc5d}',
         &['\u{649}',  '\u{670}']), ('\u{fc5e}',  &['\u{fe72}',  '\u{651}']), ('\u{fc5f}',
         &['\u{fe74}',  '\u{651}']), ('\u{fc60}',  &['\u{fe76}',  '\u{651}']), ('\u{fc61}',
         &['\u{fe78}',  '\u{651}']), ('\u{fc62}',  &['\u{fe7a}',  '\u{651}']), ('\u{fc63}',
         &['\u{fe7c}',  '\u{670}']), ('\u{fc64}',  &['\u{649}',  '\u{674}',  '\u{631}']),
        ('\u{fc65}',  &['\u{649}',  '\u{674}',  '\u{632}']), ('\u{fc66}',  &['\u{649}',  '\u{674}',
         '\u{645}']), ('\u{fc67}',  &['\u{649}',  '\u{674}',  '\u{646}']), ('\u{fc68}',
         &['\u{649}',  '\u{674}',  '\u{649}']), ('\u{fc69}',  &['\u{649}',  '\u{674}',  '\u{649}']),
        ('\u{fc6a}',  &['\u{628}',  '\u{631}']), ('\u{fc6b}',  &['\u{628}',  '\u{632}']),
        ('\u{fc6c}',  &['\u{628}',  '\u{645}']), ('\u{fc6d}',  &['\u{628}',  '\u{646}']),
        ('\u{fc6e}',  &['\u{628}',  '\u{649}']), ('\u{fc6f}',  &['\u{628}',  '\u{649}']),
        ('\u{fc70}',  &['\u{62a}',  '\u{631}']), ('\u{fc71}',  &['\u{62a}',  '\u{632}']),
        ('\u{fc72}',  &['\u{62a}',  '\u{645}']), ('\u{fc73}',  &['\u{62a}',  '\u{646}']),
        ('\u{fc74}',  &['\u{62a}',  '\u{649}']), ('\u{fc75}',  &['\u{62a}',  '\u{649}']),
        ('\u{fc76}',  &['\u{649}',  '\u{6db}',  '\u{631}']), ('\u{fc77}',  &['\u{649}',  '\u{6db}',
         '\u{632}']), ('\u{fc78}',  &['\u{649}',  '\u{6db}',  '\u{645}']), ('\u{fc79}',
         &['\u{649}',  '\u{6db}',  '\u{646}']), ('\u{fc7a}',  &['\u{649}',  '\u{6db}',  '\u{649}']),
        ('\u{fc7b}',  &['\u{649}',  '\u{6db}',  '\u{649}']), ('\u{fc7c}',  &['\u{641}',
         '\u{649}']), ('\u{fc7d}',  &['\u{641}',  '\u{649}']), ('\u{fc7e}',  &['\u{642}',
         '\u{649}']), ('\u{fc7f}',  &['\u{642}',  '\u{649}']), ('\u{fc80}',  &['\u{643}',
         '\u{6c}']), ('\u{fc81}',  &['\u{643}',  '\u{644}']), ('\u{fc82}',  &['\u{643}',
         '\u{645}']), ('\u{fc83}',  &['\u{643}',  '\u{649}']), ('\u{fc84}',  &['\u{643}',
         '\u{649}']), ('\u{fc85}',  &['\u{644}',  '\u{645}']), ('\u{fc86}',  &['\u{644}',
         '\u{649}']), ('\u{fc87}',  &['\u{644}',  '\u{649}']), ('\u{fc88}',  &['\u{645}',
         '\u{6c}']), ('\u{fc89}',  &['\u{645}',  '\u{645}']), ('\u{fc8a}',  &['\u{646}',
         '\u{631}']), ('\u{fc8b}',  &['\u{646}',  '\u{632}']), ('\u{fc8c}',  &['\u{646}',
         '\u{645}']), ('\u{fc8d}',  &['\u{646}',  '\u{646}']), ('\u{fc8e}',  &['\u{646}',
         '\u{649}']), ('\u{fc8f}',  &['\u{646}',  '\u{649}']), ('\u{fc90}',  &['\u{649}',
         '\u{670}']), ('\u{fc91}',  &['\u{649}',  '\u{631}']), ('\u{fc92}',  &['\u{649}',
         '\u{632}']), ('\u{fc93}',  &['\u{649}',  '\u{645}']), ('\u{fc94}',  &['\u{649}',
         '\u{646}']), ('\u{fc95}',  &['\u{649}',  '\u{649}']), ('\u{fc96}',  &['\u{649}',
         '\u{649}']), ('\u{fc97}',  &['\u{649}',  '\u{674}',  '\u{62c}']), ('\u{fc98}',
         &['\u{649}',  '\u{674}',  '\u{62d}']), ('\u{fc99}',  &['\u{649}',  '\u{674}',  '\u{62e}']),
        ('\u{fc9a}',  &['\u{649}',  '\u{674}',  '\u{645}']), ('\u{fc9b}',  &['\u{649}',  '\u{674}',
         '\u{6f}']), ('\u{fc9c}',  &['\u{628}',  '\u{62c}']), ('\u{fc9d}',  &['\u{628}',
         '\u{62d}']), ('\u{fc9e}',  &['\u{628}',  '\u{62e}']), ('\u{fc9f}',  &['\u{628}',
         '\u{645}']), ('\u{fca0}',  &['\u{628}',  '\u{6f}']), ('\u{fca1}',  &['\u{62a}',
         '\u{62c}']), ('\u{fca2}',  &['\u{62a}',  '\u{62d}']), ('\u{fca3}',  &['\u{62a}',
         '\u{62e}']), ('\u{fca4}',  &['\u{62a}',  '\u{645}']), ('\u{fca5}',  &['\u{62a}',
         '\u{6f}']), ('\u{fca6}',  &['\u{649}',  '\u{6db}',  '\u{645}']), ('\u{fca7}',  &['\u{62c}',
         '\u{62d}']), ('\u{fca8}',  &['\u{62c}',  '\u{645}']), ('\u{fca9}',  &['\u{62d}',
         '\u{62c}']), ('\u{fcaa}',  &['\u{62d}',  '\u{645}']), ('\u{fcab}',  &['\u{62e}',
         '\u{62c}']), ('\u{fcac}',  &['\u{62e}',  '\u{645}']), ('\u{fcad}',  &['\u{633}',
         '\u{62c}']), ('\u{fcae}',  &['\u{633}',  '\u{62d}']), ('\u{fcaf}',  &['\u{633}',
         '\u{62e}']), ('\u{fcb0}',  &['\u{633}',  '\u{645}']), ('\u{fcb1}',  &['\u{635}',
         '\u{62d}']), ('\u{fcb2}',  &['\u{635}',  '\u{62e}']), ('\u{fcb3}',  &['\u{635}',
         '\u{645}']), ('\u{fcb4}',  &['\u{636}',  '\u{62c}']), ('\u{fcb5}',  &['\u{636}',
         '\u{62d}']), ('\u{fcb6}',  &['\u{636}',  '\u{62e}']), ('\u{fcb7}',  &['\u{636}',
         '\u{645}']), ('\u{fcb8}',  &['\u{637}',  '\u{62d}']), ('\u{fcb9}',  &['\u{638}',
         '\u{645}']), ('\u{fcba}',  &['\u{639}',  '\u{62c}']), ('\u{fcbb}',  &['\u{639}',
         '\u{645}']), ('\u{fcbc}',  &['\u{63a}',  '\u{62c}']), ('\u{fcbd}',  &['\u{63a}',
         '\u{645}']), ('\u{fcbe}',  &['\u{641}',  '\u{62c}']), ('\u{fcbf}',  &['\u{641}',
         '\u{62d}']), ('\u{fcc0}',  &['\u{641}',  '\u{62e}']), ('\u{fcc1}',  &['\u{641}',
         '\u{645}']), ('\u{fcc2}',  &['\u{642}',  '\u{62d}']), ('\u{fcc3}',  &['\u{642}',
         '\u{645}']), ('\u{fcc4}',  &['\u{643}',  '\u{62c}']), ('\u{fcc5}',  &['\u{643}',
         '\u{62d}']), ('\u{fcc6}',  &['\u{643}',  '\u{62e}']), ('\u{fcc7}',  &['\u{643}',
         '\u{644}']), ('\u{fcc8}',  &['\u{643}',  '\u{645}']), ('\u{fcc9}',  &['\u{644}',
         '\u{62c}']), ('\u{fcca}',  &['\u{644}',  '\u{62d}']), ('\u{fccb}',  &['\u{644}',
         '\u{62e}']), ('\u{fccc}',  &['\u{644}',  '\u{645}']), ('\u{fccd}',  &['\u{644}',
         '\u{6f}']), ('\u{fcce}',  &['\u{645}',  '\u{62c}']), ('\u{fccf}',  &['\u{645}',
         '\u{62d}']), ('\u{fcd0}',  &['\u{645}',  '\u{62e}']), ('\u{fcd1}',  &['\u{645}',
         '\u{645}']), ('\u{fcd2}',  &['\u{628}',  '\u{62e}']), ('\u{fcd3}',  &['\u{646}',
         '\u{62d}']), ('\u{fcd4}',  &['\u{646}',  '\u{62e}']), ('\u{fcd5}',  &['\u{646}',
         '\u{645}']), ('\u{fcd6}',  &['\u{646}',  '\u{6f}']), ('\u{fcd7}',  &['\u{6f}',
         '\u{62c}']), ('\u{fcd8}',  &['\u{6f}',  '\u{645}']), ('\u{fcd9}',  &['\u{6f}',
         '\u{670}']), ('\u{fcda}',  &['\u{649}',  '\u{62c}']), ('\u{fcdb}',  &['\u{649}',
         '\u{62d}']), ('\u{fcdc}',  &['\u{649}',  '\u{62e}']), ('\u{fcdd}',  &['\u{649}',
         '\u{645}']), ('\u{fcde}',  &['\u{649}',  '\u{6f}']), ('\u{fcdf}',  &['\u{649}',  '\u{674}',
         '\u{645}']), ('\u{fce0}',  &['\u{649}',  '\u{674}',  '\u{6f}']), ('\u{fce1}',  &['\u{628}',
         '\u{645}']), ('\u{fce2}',  &['\u{628}',  '\u{6f}']), ('\u{fce3}',  &['\u{62a}',
         '\u{645}']), ('\u{fce4}',  &['\u{62a}',  '\u{6f}']), ('\u{fce5}',  &['\u{649}',  '\u{6db}',
         '\u{645}']), ('\u{fce6}',  &['\u{649}',  '\u{6db}',  '\u{6f}']), ('\u{fce7}',  &['\u{633}',
         '\u{645}']), ('\u{fce8}',  &['\u{633}',  '\u{6f}']), ('\u{fce9}',  &['\u{633}',  '\u{6db}',
         '\u{645}']), ('\u{fcea}',  &['\u{633}',  '\u{6db}',  '\u{6f}']), ('\u{fceb}',  &['\u{643}',
         '\u{644}']), ('\u{fcec}',  &['\u{643}',  '\u{645}']), ('\u{fced}',  &['\u{644}',
         '\u{645}']), ('\u{fcee}',  &['\u{646}',  '\u{645}']), ('\u{fcef}',  &['\u{646}',
         '\u{6f}']), ('\u{fcf0}',  &['\u{649}',  '\u{645}']), ('\u{fcf1}',  &['\u{649}',
         '\u{6f}']), ('\u{fcf2}',  &['\u{fe77}',  '\u{651}']), ('\u{fcf3}',  &['\u{fe79}',
         '\u{651}']), ('\u{fcf4}',  &['\u{fe7b}',  '\u{651}']), ('\u{fcf5}',  &['\u{637}',
         '\u{649}']), ('\u{fcf6}',  &['\u{637}',  '\u{649}']), ('\u{fcf7}',  &['\u{639}',
         '\u{649}']), ('\u{fcf8}',  &['\u{639}',  '\u{649}']), ('\u{fcf9}',  &['\u{63a}',
         '\u{649}']), ('\u{fcfa}',  &['\u{63a}',  '\u{649}']), ('\u{fcfb}',  &['\u{633}',
         '\u{649}']), ('\u{fcfc}',  &['\u{633}',  '\u{649}']), ('\u{fcfd}',  &['\u{633}',
         '\u{6db}',  '\u{649}']), ('\u{fcfe}',  &['\u{633}',  '\u{6db}',  '\u{649}']), ('\u{fcff}',
         &['\u{62d}',  '\u{649}']), ('\u{fd00}',  &['\u{62d}',  '\u{649}']), ('\u{fd01}',
         &['\u{62c}',  '\u{649}']), ('\u{fd02}',  &['\u{62c}',  '\u{649}']), ('\u{fd03}',
         &['\u{62e}',  '\u{649}']), ('\u{fd04}',  &['\u{62e}',  '\u{649}']), ('\u{fd05}',
         &['\u{635}',  '\u{649}']), ('\u{fd06}',  &['\u{635}',  '\u{649}']), ('\u{fd07}',
         &['\u{636}',  '\u{649}']), ('\u{fd08}',  &['\u{636}',  '\u{649}']), ('\u{fd09}',
         &['\u{633}',  '\u{6db}',  '\u{62c}']), ('\u{fd0a}',  &['\u{633}',  '\u{6db}',  '\u{62d}']),
        ('\u{fd0b}',  &['\u{633}',  '\u{6db}',  '\u{62e}']), ('\u{fd0c}',  &['\u{633}',  '\u{6db}',
         '\u{645}']), ('\u{fd0d}',  &['\u{633}',  '\u{6db}',  '\u{631}']), ('\u{fd0e}',
         &['\u{633}',  '\u{631}']), ('\u{fd0f}',  &['\u{635}',  '\u{631}']), ('\u{fd10}',
         &['\u{636}',  '\u{631}']), ('\u{fd11}',  &['\u{637}',  '\u{649}']), ('\u{fd12}',
         &['\u{637}',  '\u{649}']), ('\u{fd13}',  &['\u{639}',  '\u{649}']), ('\u{fd14}',
         &['\u{639}',  '\u{649}']), ('\u{fd15}',  &['\u{63a}',  '\u{649}']), ('\u{fd16}',
         &['\u{63a}',  '\u{649}']), ('\u{fd17}',  &['\u{633}',  '\u{649}']), ('\u{fd18}',
         &['\u{633}',  '\u{649}']), ('\u{fd19}',  &['\u{633}',  '\u{6db}',  '\u{649}']),
        ('\u{fd1a}',  &['\u{633}',  '\u{6db}',  '\u{649}']), ('\u{fd1b}',  &['\u{62d}',
         '\u{649}']), ('\u{fd1c}',  &['\u{62d}',  '\u{649}']), ('\u{fd1d}',  &['\u{62c}',
         '\u{649}']), ('\u{fd1e}',  &['\u{62c}',  '\u{649}']), ('\u{fd1f}',  &['\u{62e}',
         '\u{649}']), ('\u{fd20}',  &['\u{62e}',  '\u{649}']), ('\u{fd21}',  &['\u{635}',
         '\u{649}']), ('\u{fd22}',  &['\u{635}',  '\u{649}']), ('\u{fd23}',  &['\u{636}',
         '\u{649}']), ('\u{fd24}',  &['\u{636}',  '\u{649}']), ('\u{fd25}',  &['\u{633}',
         '\u{6db}',  '\u{62c}']), ('\u{fd26}',  &['\u{633}',  '\u{6db}',  '\u{62d}']), ('\u{fd27}',
         &['\u{633}',  '\u{6db}',  '\u{62e}']), ('\u{fd28}',  &['\u{633}',  '\u{6db}',  '\u{645}']),
        ('\u{fd29}',  &['\u{633}',  '\u{6db}',  '\u{631}']), ('\u{fd2a}',  &['\u{633}',
         '\u{631}']), ('\u{fd2b}',  &['\u{635}',  '\u{631}']), ('\u{fd2c}',  &['\u{636}',
         '\u{631}']), ('\u{fd2d}',  &['\u{633}',  '\u{6db}',  '\u{62c}']), ('\u{fd2e}',
         &['\u{633}',  '\u{6db}',  '\u{62d}']), ('\u{fd2f}',  &['\u{633}',  '\u{6db}',  '\u{62e}']),
        ('\u{fd30}',  &['\u{633}',  '\u{6db}',  '\u{645}']), ('\u{fd31}',  &['\u{633}',  '\u{6f}']),
        ('\u{fd32}',  &['\u{633}',  '\u{6db}',  '\u{6f}']), ('\u{fd33}',  &['\u{637}',  '\u{645}']),
        ('\u{fd34}',  &['\u{633}',  '\u{62c}']), ('\u{fd35}',  &['\u{633}',  '\u{62d}']),
        ('\u{fd36}',  &['\u{633}',  '\u{62e}']), ('\u{fd37}',  &['\u{633}',  '\u{6db}',
         '\u{62c}']), ('\u{fd38}',  &['\u{633}',  '\u{6db}',  '\u{62d}']), ('\u{fd39}',
         &['\u{633}',  '\u{6db}',  '\u{62e}']), ('\u{fd3a}',  &['\u{637}',  '\u{645}']),
        ('\u{fd3b}',  &['\u{638}',  '\u{645}']), ('\u{fd3c}',  &['\u{6c}',  '\u{30b}']),
        ('\u{fd3d}',  &['\u{6c}',  '\u{30b}']), ('\u{fd3e}',  &['\u{28}']), ('\u{fd3f}',
         &['\u{29}']), ('\u{fd50}',  &['\u{62a}',  '\u{62c}',  '\u{645}']), ('\u{fd51}',
         &['\u{62a}',  '\u{62d}',  '\u{62c}']), ('\u{fd52}',  &['\u{62a}',  '\u{62d}',  '\u{62c}']),
        ('\u{fd53}',  &['\u{62a}',  '\u{62d}',  '\u{645}']), ('\u{fd54}',  &['\u{62a}',  '\u{62e}',
         '\u{645}']), ('\u{fd55}',  &['\u{62a}',  '\u{645}',  '\u{62c}']), ('\u{fd56}',
         &['\u{62a}',  '\u{645}',  '\u{62d}']), ('\u{fd57}',  &['\u{62a}',  '\u{645}',  '\u{62e}']),
        ('\u{fd58}',  &['\u{62c}',  '\u{645}',  '\u{62d}']), ('\u{fd59}',  &['\u{62c}',  '\u{645}',
         '\u{62d}']), ('\u{fd5a}',  &['\u{62d}',  '\u{645}',  '\u{649}']), ('\u{fd5b}',
         &['\u{62d}',  '\u{645}',  '\u{649}']), ('\u{fd5c}',  &['\u{633}',  '\u{62d}',  '\u{62c}']),
        ('\u{fd5d}',  &['\u{633}',  '\u{62c}',  '\u{62d}']), ('\u{fd5e}',  &['\u{633}',  '\u{62c}',
         '\u{649}']), ('\u{fd5f}',  &['\u{633}',  '\u{645}',  '\u{62d}']), ('\u{fd60}',
         &['\u{633}',  '\u{645}',  '\u{62d}']), ('\u{fd61}',  &['\u{633}',  '\u{645}',  '\u{62c}']),
        ('\u{fd62}',  &['\u{633}',  '\u{645}',  '\u{645}']), ('\u{fd63}',  &['\u{633}',  '\u{645}',
         '\u{645}']), ('\u{fd64}',  &['\u{635}',  '\u{62d}',  '\u{62d}']), ('\u{fd65}',
         &['\u{635}',  '\u{62d}',  '\u{62d}']), ('\u{fd66}',  &['\u{635}',  '\u{645}',  '\u{645}']),
        ('\u{fd67}',  &['\u{633}',  '\u{6db}',  '\u{62d}',  '\u{645}']), ('\u{fd68}',  &['\u{633}',
         '\u{6db}',  '\u{62d}',  '\u{645}']), ('\u{fd69}',  &['\u{633}',  '\u{6db}',  '\u{62c}',
         '\u{649}']), ('\u{fd6a}',  &['\u{633}',  '\u{6db}',  '\u{645}',  '\u{62e}']), ('\u{fd6b}',
         &['\u{633}',  '\u{6db}',  '\u{645}',  '\u{62e}']), ('\u{fd6c}',  &['\u{633}',  '\u{6db}',
         '\u{645}',  '\u{645}']), ('\u{fd6d}',  &['\u{633}',  '\u{6db}',  '\u{645}',  '\u{645}']),
        ('\u{fd6e}',  &['\u{636}',  '\u{62d}',  '\u{649}']), ('\u{fd6f}',  &['\u{636}',  '\u{62e}',
         '\u{645}']), ('\u{fd70}',  &['\u{636}',  '\u{62e}',  '\u{645}']), ('\u{fd71}',
         &['\u{637}',  '\u{645}',  '\u{62d}']), ('\u{fd72}',  &['\u{637}',  '\u{645}',  '\u{62d}']),
        ('\u{fd73}',  &['\u{637}',  '\u{645}',  '\u{645}']), ('\u{fd74}',  &['\u{637}',  '\u{645}',
         '\u{649}']), ('\u{fd75}',  &['\u{639}',  '\u{62c}',  '\u{645}']), ('\u{fd76}',
         &['\u{639}',  '\u{645}',  '\u{645}']), ('\u{fd77}',  &['\u{639}',  '\u{645}',  '\u{645}']),
        ('\u{fd78}',  &['\u{639}',  '\u{645}',  '\u{649}']), ('\u{fd79}',  &['\u{63a}',  '\u{645}',
         '\u{645}']), ('\u{fd7a}',  &['\u{63a}',  '\u{645}',  '\u{649}']), ('\u{fd7b}',
         &['\u{63a}',  '\u{645}',  '\u{649}']), ('\u{fd7c}',  &['\u{641}',  '\u{62e}',  '\u{645}']),
        ('\u{fd7d}',  &['\u{641}',  '\u{62e}',  '\u{645}']), ('\u{fd7e}',  &['\u{642}',  '\u{645}',
         '\u{62d}']), ('\u{fd7f}',  &['\u{642}',  '\u{645}',  '\u{645}']), ('\u{fd80}',
         &['\u{644}',  '\u{62d}',  '\u{645}']), ('\u{fd81}',  &['\u{644}',  '\u{62d}',  '\u{649}']),
        ('\u{fd82}',  &['\u{644}',  '\u{62d}',  '\u{649}']), ('\u{fd83}',  &['\u{644}',  '\u{62c}',
         '\u{62c}']), ('\u{fd84}',  &['\u{644}',  '\u{62c}',  '\u{62c}']), ('\u{fd85}',
         &['\u{644}',  '\u{62e}',  '\u{645}']), ('\u{fd86}',  &['\u{644}',  '\u{62e}',  '\u{645}']),
        ('\u{fd87}',  &['\u{644}',  '\u{645}',  '\u{62d}']), ('\u{fd88}',  &['\u{644}',  '\u{645}',
         '\u{62d}']), ('\u{fd89}',  &['\u{645}',  '\u{62d}',  '\u{62c}']), ('\u{fd8a}',
         &['\u{645}',  '\u{62d}',  '\u{645}']), ('\u{fd8b}',  &['\u{645}',  '\u{62d}',  '\u{649}']),
        ('\u{fd8c}',  &['\u{645}',  '\u{62c}',  '\u{62d}']), ('\u{fd8d}',  &['\u{645}',  '\u{62c}',
         '\u{645}']), ('\u{fd8e}',  &['\u{645}',  '\u{62e}',  '\u{62c}']), ('\u{fd8f}',
         &['\u{645}',  '\u{62e}',  '\u{645}']), ('\u{fd92}',  &['\u{645}',  '\u{62c}',  '\u{62e}']),
        ('\u{fd93}',  &['\u{6f}',  '\u{645}',  '\u{62c}']), ('\u{fd94}',  &['\u{6f}',  '\u{645}',
         '\u{645}']), ('\u{fd95}',  &['\u{646}',  '\u{62d}',  '\u{645}']), ('\u{fd96}',
         &['\u{646}',  '\u{62d}',  '\u{649}']), ('\u{fd97}',  &['\u{646}',  '\u{62c}',  '\u{645}']),
        ('\u{fd98}',  &['\u{646}',  '\u{62c}',  '\u{645}']), ('\u{fd99}',  &['\u{646}',  '\u{62c}',
         '\u{649}']), ('\u{fd9a}',  &['\u{646}',  '\u{645}',  '\u{649}']), ('\u{fd9b}',
         &['\u{646}',  '\u{645}',  '\u{649}']), ('\u{fd9c}',  &['\u{649}',  '\u{645}',  '\u{645}']),
        ('\u{fd9d}',  &['\u{649}',  '\u{645}',  '\u{645}']), ('\u{fd9e}',  &['\u{628}',  '\u{62e}',
         '\u{649}']), ('\u{fd9f}',  &['\u{62a}',  '\u{62c}',  '\u{649}']), ('\u{fda0}',
         &['\u{62a}',  '\u{62c}',  '\u{649}']), ('\u{fda1}',  &['\u{62a}',  '\u{62e}',  '\u{649}']),
        ('\u{fda2}',  &['\u{62a}',  '\u{62e}',  '\u{649}']), ('\u{fda3}',  &['\u{62a}',  '\u{645}',
         '\u{649}']), ('\u{fda4}',  &['\u{62a}',  '\u{645}',  '\u{649}']), ('\u{fda5}',
         &['\u{62c}',  '\u{645}',  '\u{649}']), ('\u{fda6}',  &['\u{62c}',  '\u{62d}',  '\u{649}']),
        ('\u{fda7}',  &['\u{62c}',  '\u{645}',  '\u{649}']), ('\u{fda8}',  &['\u{633}',  '\u{62e}',
         '\u{649}']), ('\u{fda9}',  &['\u{635}',  '\u{62d}',  '\u{649}']), ('\u{fdaa}',
         &['\u{633}',  '\u{6db}',  '\u{62d}',  '\u{649}']), ('\u{fdab}',  &['\u{636}',  '\u{62d}',
         '\u{649}']), ('\u{fdac}',  &['\u{644}',  '\u{62c}',  '\u{649}']), ('\u{fdad}',
         &['\u{644}',  '\u{645}',  '\u{649}']), ('\u{fdae}',  &['\u{649}',  '\u{62d}',  '\u{649}']),
        ('\u{fdaf}',  &['\u{649}',  '\u{62c}',  '\u{649}']), ('\u{fdb0}',  &['\u{649}',  '\u{645}',
         '\u{649}']), ('\u{fdb1}',  &['\u{645}',  '\u{645}',  '\u{649}']), ('\u{fdb2}',
         &['\u{642}',  '\u{645}',  '\u{649}']), ('\u{fdb3}',  &['\u{646}',  '\u{62d}',  '\u{649}']),
        ('\u{fdb4}',  &['\u{642}',  '\u{645}',  '\u{62d}']), ('\u{fdb5}',  &['\u{644}',  '\u{62d}',
         '\u{645}']), ('\u{fdb6}',  &['\u{639}',  '\u{645}',  '\u{649}']), ('\u{fdb7}',
         &['\u{643}',  '\u{645}',  '\u{649}']), ('\u{fdb8}',  &['\u{646}',  '\u{62c}',  '\u{62d}']),
        ('\u{fdb9}',  &['\u{645}',  '\u{62e}',  '\u{649}']), ('\u{fdba}',  &['\u{644}',  '\u{62c}',
         '\u{645}']), ('\u{fdbb}',  &['\u{643}',  '\u{645}',  '\u{645}']), ('\u{fdbc}',
         &['\u{644}',  '\u{62c}',  '\u{645}']), ('\u{fdbd}',  &['\u{646}',  '\u{62c}',  '\u{62d}']),
        ('\u{fdbe}',  &['\u{62c}',  '\u{62d}',  '\u{649}']), ('\u{fdbf}',  &['\u{62d}',  '\u{62c}',
         '\u{649}']), ('\u{fdc0}',  &['\u{645}',  '\u{62c}',  '\u{649}']), ('\u{fdc1}',
         &['\u{641}',  '\u{645}',  '\u{649}']), ('\u{fdc2}',  &['\u{628}',  '\u{62d}',  '\u{649}']),
        ('\u{fdc3}',  &['\u{643}',  '\u{645}',  '\u{645}']), ('\u{fdc4}',  &['\u{639}',  '\u{62c}',
         '\u{645}']), ('\u{fdc5}',  &['\u{635}',  '\u{645}',  '\u{645}']), ('\u{fdc6}',
         &['\u{633}',  '\u{62e}',  '\u{649}']), ('\u{fdc7}',  &['\u{646}',  '\u{62c}',  '\u{649}']),
        ('\u{fdf0}',  &['\u{635}',  '\u{644}',  '\u{649}']), ('\u{fdf1}',  &['\u{642}',  '\u{644}',
         '\u{649}']), ('\u{fdf2}',  &['\u{6c}',  '\u{644}',  '\u{644}',  '\u{651}',  '\u{670}',
         '\u{6f}']), ('\u{fdf3}',  &['\u{6c}',  '\u{643}',  '\u{628}',  '\u{631}']), ('\u{fdf4}',
         &['\u{645}',  '\u{62d}',  '\u{645}',  '\u{62f}']), ('\u{fdf5}',  &['\u{635}',  '\u{644}',
         '\u{639}',  '\u{645}']), ('\u{fdf6}',  &['\u{631}',  '\u{633}',  '\u{648}',  '\u{644}']),
        ('\u{fdf7}',  &['\u{639}',  '\u{644}',  '\u{649}',  '\u{6f}']), ('\u{fdf8}',  &['\u{648}',
         '\u{633}',  '\u{644}',  '\u{645}']), ('\u{fdf9}',  &['\u{635}',  '\u{644}',  '\u{649}']),
        ('\u{fdfa}',  &['\u{635}',  '\u{644}',  '\u{649}',  '\u{20}',  '\u{6c}',  '\u{644}',
         '\u{644}',  '\u{6f}',  '\u{20}',  '\u{639}',  '\u{644}',  '\u{649}',  '\u{6f}',  '\u{20}',
         '\u{648}',  '\u{633}',  '\u{644}',  '\u{645}']), ('\u{fdfb}',  &['\u{62c}',  '\u{644}',
         '\u{20}',  '\u{62c}',  '\u{644}',  '\u{6c}',  '\u{644}',  '\u{6f}']), ('\u{fdfc}',
         &['\u{631}',  '\u{649}',  '\u{6c}',  '\u{644}']), ('\u{fe19}',  &['\u{2d57}']),
        ('\u{fe30}',  &['\u{3a}']), ('\u{fe31}',  &['\u{2502}']), ('\u{fe34}',  &['\u{2307}']),
        ('\u{fe35}',  &['\u{23dc}']), ('\u{fe36}',  &['\u{23dd}']), ('\u{fe37}',  &['\u{23de}']),
        ('\u{fe38}',  &['\u{23df}']), ('\u{fe39}',  &['\u{23e0}']), ('\u{fe3a}',  &['\u{23e1}']),
        ('\u{fe49}',  &['\u{2c9}']), ('\u{fe4a}',  &['\u{2c9}']), ('\u{fe4b}',  &['\u{2c9}']),
        ('\u{fe4c}',  &['\u{2c9}']), ('\u{fe4d}',  &['\u{5f}']), ('\u{fe4e}',  &['\u{5f}']),
        ('\u{fe4f}',  &['\u{5f}']), ('\u{fe58}',  &['\u{2d}']), ('\u{fe68}',  &['\u{5c}']),
        ('\u{fe80}',  &['\u{621}']), ('\u{fe81}',  &['\u{622}']), ('\u{fe82}',  &['\u{622}']),
        ('\u{fe83}',  &['\u{6c}',  '\u{674}']), ('\u{fe84}',  &['\u{6c}',  '\u{674}']), ('\u{fe85}',
         &['\u{648}',  '\u{674}']), ('\u{fe86}',  &['\u{648}',  '\u{674}']), ('\u{fe87}',
         &['\u{6c}',  '\u{655}']), ('\u{fe88}',  &['\u{6c}',  '\u{655}']), ('\u{fe89}',
         &['\u{649}',  '\u{674}']), ('\u{fe8a}',  &['\u{649}',  '\u{674}']), ('\u{fe8b}',
         &['\u{649}',  '\u{674}']), ('\u{fe8c}',  &['\u{649}',  '\u{674}']), ('\u{fe8d}',
         &['\u{6c}']), ('\u{fe8e}',  &['\u{6c}']), ('\u{fe8f}',  &['\u{628}']), ('\u{fe90}',
         &['\u{628}']), ('\u{fe91}',  &['\u{628}']), ('\u{fe92}',  &['\u{628}']), ('\u{fe93}',
         &['\u{629}']), ('\u{fe94}',  &['\u{629}']), ('\u{fe95}',  &['\u{62a}']), ('\u{fe96}',
         &['\u{62a}']), ('\u{fe97}',  &['\u{62a}']), ('\u{fe98}',  &['\u{62a}']), ('\u{fe99}',
         &['\u{649}',  '\u{6db}']), ('\u{fe9a}',  &['\u{649}',  '\u{6db}']), ('\u{fe9b}',
         &['\u{649}',  '\u{6db}']), ('\u{fe9c}',  &['\u{649}',  '\u{6db}']), ('\u{fe9d}',
         &['\u{62c}']), ('\u{fe9e}',  &['\u{62c}']), ('\u{fe9f}',  &['\u{62c}']), ('\u{fea0}',
         &['\u{62c}']), ('\u{fea1}',  &['\u{62d}']), ('\u{fea2}',  &['\u{62d}']), ('\u{fea3}',
         &['\u{62d}']), ('\u{fea4}',  &['\u{62d}']), ('\u{fea5}',  &['\u{62e}']), ('\u{fea6}',
         &['\u{62e}']), ('\u{fea7}',  &['\u{62e}']), ('\u{fea8}',  &['\u{62e}']), ('\u{fea9}',
         &['\u{62f}']), ('\u{feaa}',  &['\u{62f}']), ('\u{feab}',  &['\u{630}']), ('\u{feac}',
         &['\u{630}']), ('\u{fead}',  &['\u{631}']), ('\u{feae}',  &['\u{631}']), ('\u{feaf}',
         &['\u{632}']), ('\u{feb0}',  &['\u{632}']), ('\u{feb1}',  &['\u{633}']), ('\u{feb2}',
         &['\u{633}']), ('\u{feb3}',  &['\u{633}']), ('\u{feb4}',  &['\u{633}']), ('\u{feb5}',
         &['\u{633}',  '\u{6db}']), ('\u{feb6}',  &['\u{633}',  '\u{6db}']), ('\u{feb7}',
         &['\u{633}',  '\u{6db}']), ('\u{feb8}',  &['\u{633}',  '\u{6db}']), ('\u{feb9}',
         &['\u{635}']), ('\u{feba}',  &['\u{635}']), ('\u{febb}',  &['\u{635}']), ('\u{febc}',
         &['\u{635}']), ('\u{febd}',  &['\u{636}']), ('\u{febe}',  &['\u{636}']), ('\u{febf}',
         &['\u{636}']), ('\u{fec0}',  &['\u{636}']), ('\u{fec1}',  &['\u{637}']), ('\u{fec2}',
         &['\u{637}']), ('\u{fec3}',  &['\u{637}']), ('\u{fec4}',  &['\u{637}']), ('\u{fec5}',
         &['\u{638}']), ('\u{fec6}',  &['\u{638}']), ('\u{fec7}',  &['\u{638}']), ('\u{fec8}',
         &['\u{638}']), ('\u{fec9}',  &['\u{639}']), ('\u{feca}',  &['\u{639}']), ('\u{fecb}',
         &['\u{639}']), ('\u{fecc}',  &['\u{639}']), ('\u{fecd}',  &['\u{63a}']), ('\u{fece}',
         &['\u{63a}']), ('\u{fecf}',  &['\u{63a}']), ('\u{fed0}',  &['\u{63a}']), ('\u{fed1}',
         &['\u{641}']), ('\u{fed2}',  &['\u{641}']), ('\u{fed3}',  &['\u{641}']), ('\u{fed4}',
         &['\u{641}']), ('\u{fed5}',  &['\u{642}']), ('\u{fed6}',  &['\u{642}']), ('\u{fed7}',
         &['\u{642}']), ('\u{fed8}',  &['\u{642}']), ('\u{fed9}',  &['\u{643}']), ('\u{feda}',
         &['\u{643}']), ('\u{fedb}',  &['\u{643}']), ('\u{fedc}',  &['\u{643}']), ('\u{fedd}',
         &['\u{644}']), ('\u{fede}',  &['\u{644}']), ('\u{fedf}',  &['\u{644}']), ('\u{fee0}',
         &['\u{644}']), ('\u{fee1}',  &['\u{645}']), ('\u{fee2}',  &['\u{645}']), ('\u{fee3}',
         &['\u{645}']), ('\u{fee4}',  &['\u{645}']), ('\u{fee5}',  &['\u{646}']), ('\u{fee6}',
         &['\u{646}']), ('\u{fee7}',  &['\u{646}']), ('\u{fee8}',  &['\u{646}']), ('\u{fee9}',
         &['\u{6f}']), ('\u{feea}',  &['\u{6f}']), ('\u{feeb}',  &['\u{6f}']), ('\u{feec}',
         &['\u{6f}']), ('\u{feed}',  &['\u{648}']), ('\u{feee}',  &['\u{648}']), ('\u{feef}',
         &['\u{649}']), ('\u{fef0}',  &['\u{649}']), ('\u{fef1}',  &['\u{649}']), ('\u{fef2}',
         &['\u{649}']), ('\u{fef3}',  &['\u{649}']), ('\u{fef4}',  &['\u{649}']), ('\u{fef5}',
         &['\u{644}',  '\u{622}']), ('\u{fef6}',  &['\u{644}',  '\u{622}']), ('\u{fef7}',
         &['\u{644}',  '\u{6c}',  '\u{674}']), ('\u{fef8}',  &['\u{644}',  '\u{6c}',  '\u{674}']),
        ('\u{fef9}',  &['\u{644}',  '\u{6c}',  '\u{655}']), ('\u{fefa}',  &['\u{644}',  '\u{6c}',
         '\u{655}']), ('\u{fefb}',  &['\u{644}',  '\u{6c}']), ('\u{fefc}',  &['\u{644}',
         '\u{6c}']), ('\u{ff01}',  &['\u{21}']), ('\u{ff02}',  &['\u{27}',  '\u{27}']), ('\u{ff07}',
         &['\u{27}']), ('\u{ff0d}',  &['\u{30fc}']), ('\u{ff1a}',  &['\u{3a}']), ('\u{ff21}',
         &['\u{41}']), ('\u{ff22}',  &['\u{42}']), ('\u{ff23}',  &['\u{43}']), ('\u{ff25}',
         &['\u{45}']), ('\u{ff28}',  &['\u{48}']), ('\u{ff29}',  &['\u{6c}']), ('\u{ff2a}',
         &['\u{4a}']), ('\u{ff2b}',  &['\u{4b}']), ('\u{ff2d}',  &['\u{4d}']), ('\u{ff2e}',
         &['\u{4e}']), ('\u{ff2f}',  &['\u{4f}']), ('\u{ff30}',  &['\u{50}']), ('\u{ff33}',
         &['\u{53}']), ('\u{ff34}',  &['\u{54}']), ('\u{ff38}',  &['\u{58}']), ('\u{ff39}',
         &['\u{59}']), ('\u{ff3a}',  &['\u{5a}']), ('\u{ff3b}',  &['\u{28}']), ('\u{ff3c}',
         &['\u{5c}']), ('\u{ff3d}',  &['\u{29}']), ('\u{ff3e}',  &['\u{fe3f}']), ('\u{ff40}',
         &['\u{27}']), ('\u{ff41}',  &['\u{61}']), ('\u{ff43}',  &['\u{63}']), ('\u{ff45}',
         &['\u{65}']), ('\u{ff47}',  &['\u{67}']), ('\u{ff48}',  &['\u{68}']), ('\u{ff49}',
         &['\u{69}']), ('\u{ff4a}',  &['\u{6a}']), ('\u{ff4c}',  &['\u{6c}']), ('\u{ff4f}',
         &['\u{6f}']), ('\u{ff50}',  &['\u{70}']), ('\u{ff53}',  &['\u{73}']), ('\u{ff56}',
         &['\u{76}']), ('\u{ff58}',  &['\u{78}']), ('\u{ff59}',  &['\u{79}']), ('\u{ff5c}',
         &['\u{2502}']), ('\u{ff5e}',  &['\u{301c}']), ('\u{ff65}',  &['\u{b7}']), ('\u{ffe3}',
         &['\u{2c9}']), ('\u{ffe8}',  &['\u{6c}']), ('\u{ffed}',  &['\u{25aa}']), ('\u{10101}',
         &['\u{b7}']), ('\u{1018e}',  &['\u{4e}',  '\u{30a}']), ('\u{10196}',  &['\u{58}',
         '\u{335}']), ('\u{10197}',  &['\u{56}',  '\u{335}']), ('\u{10198}',  &['\u{6c}',
         '\u{335}',  '\u{6c}',  '\u{335}',  '\u{53}',  '\u{335}']), ('\u{10199}',  &['\u{6c}',
         '\u{335}',  '\u{6c}',  '\u{335}']), ('\u{101a0}',  &['\u{2ce8}']), ('\u{10282}',
         &['\u{42}']), ('\u{10285}',  &['\u{394}']), ('\u{10286}',  &['\u{45}']), ('\u{10287}',
         &['\u{46}']), ('\u{1028a}',  &['\u{6c}']), ('\u{1028d}',  &['\u{245}']), ('\u{10290}',
         &['\u{58}']), ('\u{10292}',  &['\u{4f}']), ('\u{10294}',  &['\u{16dc}']), ('\u{10295}',
         &['\u{50}']), ('\u{10296}',  &['\u{53}']), ('\u{10297}',  &['\u{54}']), ('\u{1029b}',
         &['\u{2b}']), ('\u{102a0}',  &['\u{41}']), ('\u{102a1}',  &['\u{42}']), ('\u{102a2}',
         &['\u{43}']), ('\u{102a3}',  &['\u{394}']), ('\u{102a5}',  &['\u{46}']), ('\u{102ab}',
         &['\u{4f}']), ('\u{102ad}',  &['\u{3d8}']), ('\u{102b0}',  &['\u{4d}']), ('\u{102b1}',
         &['\u{54}']), ('\u{102b2}',  &['\u{59}']), ('\u{102b3}',  &['\u{3a6}']), ('\u{102b4}',
         &['\u{58}']), ('\u{102b5}',  &['\u{3a8}']), ('\u{102b6}',  &['\u{3a9}']), ('\u{102b8}',
         &['\u{2d40}']), ('\u{102cf}',  &['\u{48}']), ('\u{102e1}',  &['\u{62f}']), ('\u{102e4}',
         &['\u{648}']), ('\u{102e8}',  &['\u{637}']), ('\u{102f2}',  &['\u{635}']), ('\u{102f5}',
         &['\u{5a}']), ('\u{10301}',  &['\u{42}']), ('\u{10302}',  &['\u{43}']), ('\u{10309}',
         &['\u{6c}']), ('\u{10311}',  &['\u{4d}']), ('\u{10312}',  &['\u{3d8}']), ('\u{10315}',
         &['\u{54}']), ('\u{10317}',  &['\u{58}']), ('\u{1031a}',  &['\u{38}']), ('\u{1031f}',
         &['\u{2a}']), ('\u{10320}',  &['\u{6c}']), ('\u{10322}',  &['\u{58}']), ('\u{103d1}',
         &['\u{10382}']), ('\u{103d3}',  &['\u{10393}']), ('\u{10401}',  &['\u{190}']),
        ('\u{10404}',  &['\u{4f}']), ('\u{10411}',  &['\u{a4f6}']), ('\u{10415}',  &['\u{43}']),
        ('\u{1041b}',  &['\u{4c}']), ('\u{1041f}',  &['\u{2c70}']), ('\u{10420}',  &['\u{53}']),
        ('\u{10423}',  &['\u{186}']), ('\u{10425}',  &['\u{418}']), ('\u{10429}',  &['\u{a793}']),
        ('\u{1042a}',  &['\u{29a}']), ('\u{1042c}',  &['\u{6f}']), ('\u{1043d}',  &['\u{63}']),
        ('\u{1043f}',  &['\u{277}']), ('\u{10442}',  &['\u{25e}']), ('\u{10443}',  &['\u{29f}']),
        ('\u{10448}',  &['\u{73}']), ('\u{1044b}',  &['\u{254}']), ('\u{1044d}',  &['\u{1d0e}']),
        ('\u{104a0}',  &['\u{10486}']), ('\u{104b0}',  &['\u{245}']), ('\u{104b4}',  &['\u{52}']),
        ('\u{104bc}',  &['\u{4c3}']), ('\u{104c2}',  &['\u{4f}']), ('\u{104c3}',  &['\u{298}']),
        ('\u{104c4}',  &['\u{de}']), ('\u{104cd}',  &['\u{40b}']), ('\u{104ce}',  &['\u{55}']),
        ('\u{104d0}',  &['\u{16e6}']), ('\u{104d1}',  &['\u{3a8}']), ('\u{104d2}',  &['\u{37}']),
        ('\u{104d8}',  &['\u{28c}']), ('\u{104db}',  &['\u{3bb}']), ('\u{104ea}',  &['\u{6f}']),
        ('\u{104eb}',  &['\u{a669}']), ('\u{104f6}',  &['\u{75}']), ('\u{104f9}',  &['\u{3c8}']),
        ('\u{10513}',  &['\u{4e}']), ('\u{10516}',  &['\u{4f}']), ('\u{10518}',  &['\u{4b}']),
        ('\u{1051c}',  &['\u{43}']), ('\u{1051d}',  &['\u{56}']), ('\u{10525}',  &['\u{46}']),
        ('\u{10526}',  &['\u{4c}']), ('\u{10527}',  &['\u{58}']), ('\u{10a3a}',  &['\u{323}']),
        ('\u{10a50}',  &['\u{2e}']), ('\u{10a57}',  &['\u{10a56}',  '\u{10a56}']), ('\u{10cfa}',
         &['\u{10ca5}']), ('\u{10cfc}',  &['\u{10c82}']), ('\u{110bb}',  &['\u{970}']),
        ('\u{111c7}',  &['\u{970}']), ('\u{111ca}',  &['\u{323}']), ('\u{111cb}',  &['\u{93a}']),
        ('\u{111db}',  &['\u{a8fc}']), ('\u{111dc}',  &['\u{a8fb}']), ('\u{111de}',  &['\u{2248}']),
        ('\u{11300}',  &['\u{30a}']), ('\u{11413}',  &['\u{11434}',  '\u{11442}',  '\u{11412}']),
        ('\u{11419}',  &['\u{11434}',  '\u{11442}',  '\u{11418}']), ('\u{11424}',  &['\u{11434}',
         '\u{11442}',  '\u{11423}']), ('\u{1142a}',  &['\u{11434}',  '\u{11442}',  '\u{11429}']),
        ('\u{1142d}',  &['\u{11434}',  '\u{11442}',  '\u{1142c}']), ('\u{1142f}',  &['\u{11434}',
         '\u{11442}',  '\u{1142e}']), ('\u{1144c}',  &['\u{1144b}',  '\u{1144b}']), ('\u{11492}',
         &['\u{998}']), ('\u{11494}',  &['\u{99a}']), ('\u{11496}',  &['\u{99c}']), ('\u{11498}',
         &['\u{99e}']), ('\u{11499}',  &['\u{99f}']), ('\u{1149b}',  &['\u{9a1}']), ('\u{1149d}',
         &['\u{9b2}']), ('\u{1149e}',  &['\u{9a4}']), ('\u{1149f}',  &['\u{9a5}']), ('\u{114a0}',
         &['\u{9a6}']), ('\u{114a1}',  &['\u{9a7}']), ('\u{114a2}',  &['\u{9a8}']), ('\u{114a3}',
         &['\u{9aa}']), ('\u{114a7}',  &['\u{9ae}']), ('\u{114a8}',  &['\u{9af}']), ('\u{114a9}',
         &['\u{9ac}']), ('\u{114aa}',  &['\u{9a3}']), ('\u{114ab}',  &['\u{9b0}']), ('\u{114ad}',
         &['\u{9b7}']), ('\u{114ae}',  &['\u{9b8}']), ('\u{114b0}',  &['\u{9be}']), ('\u{114b1}',
         &['\u{9bf}']), ('\u{114b9}',  &['\u{9c7}']), ('\u{114bc}',  &['\u{9cb}']), ('\u{114bd}',
         &['\u{9d7}']), ('\u{114be}',  &['\u{9cc}']), ('\u{114bf}',  &['\u{306}',  '\u{307}']),
        ('\u{114c1}',  &['\u{983}']), ('\u{114c2}',  &['\u{9cd}']), ('\u{114c3}',  &['\u{323}']),
        ('\u{114c4}',  &['\u{9bd}']), ('\u{114c5}',  &['\u{77}',  '\u{307}']), ('\u{114d0}',
         &['\u{4f}']), ('\u{114d1}',  &['\u{9e7}']), ('\u{114d2}',  &['\u{9e8}']), ('\u{114d6}',
         &['\u{9ec}']), ('\u{115d8}',  &['\u{11582}']), ('\u{115d9}',  &['\u{11582}']),
        ('\u{115da}',  &['\u{11583}']), ('\u{115db}',  &['\u{11584}']), ('\u{115dc}',
         &['\u{115b2}']), ('\u{115dd}',  &['\u{115b3}']), ('\u{11642}',  &['\u{11641}',
         '\u{11641}']), ('\u{11700}',  &['\u{72}',  '\u{6e}']), ('\u{11706}',  &['\u{76}']),
        ('\u{1170a}',  &['\u{77}']), ('\u{1170e}',  &['\u{77}']), ('\u{1170f}',  &['\u{77}']),
        ('\u{118a0}',  &['\u{56}']), ('\u{118a2}',  &['\u{46}']), ('\u{118a3}',  &['\u{4c}']),
        ('\u{118a4}',  &['\u{59}']), ('\u{118a6}',  &['\u{45}']), ('\u{118a8}',  &['\u{2207}']),
        ('\u{118a9}',  &['\u{5a}']), ('\u{118ac}',  &['\u{39}']), ('\u{118ae}',  &['\u{45}']),
        ('\u{118af}',  &['\u{34}']), ('\u{118b2}',  &['\u{4c}']), ('\u{118b5}',  &['\u{4f}']),
        ('\u{118b7}',  &['\u{16dc}']), ('\u{118b8}',  &['\u{55}']), ('\u{118bb}',  &['\u{35}']),
        ('\u{118bc}',  &['\u{54}']), ('\u{118c0}',  &['\u{76}']), ('\u{118c1}',  &['\u{73}']),
        ('\u{118c2}',  &['\u{46}']), ('\u{118c3}',  &['\u{69}']), ('\u{118c4}',  &['\u{7a}']),
        ('\u{118c6}',  &['\u{37}']), ('\u{118c8}',  &['\u{6f}']), ('\u{118ca}',  &['\u{33}']),
        ('\u{118cc}',  &['\u{39}']), ('\u{118ce}',  &['\u{a793}']), ('\u{118d5}',  &['\u{36}']),
        ('\u{118d6}',  &['\u{39}']), ('\u{118d7}',  &['\u{6f}']), ('\u{118d8}',  &['\u{75}']),
        ('\u{118dc}',  &['\u{79}']), ('\u{118e0}',  &['\u{4f}']), ('\u{118e3}',  &['\u{72}',
         '\u{6e}']), ('\u{118e4}',  &['\u{669}']), ('\u{118e5}',  &['\u{5a}']), ('\u{118e6}',
         &['\u{57}']), ('\u{118e9}',  &['\u{43}']), ('\u{118ec}',  &['\u{58}']), ('\u{118ef}',
         &['\u{57}']), ('\u{118f2}',  &['\u{43}']), ('\u{11ae6}',  &['\u{11ae5}',  '\u{11aef}']),
        ('\u{11ae7}',  &['\u{11ae5}',  '\u{11af0}']), ('\u{11ae8}',  &['\u{11ae5}',  '\u{11ae5}']),
        ('\u{11ae9}',  &['\u{11ae5}',  '\u{11ae5}',  '\u{11aef}']), ('\u{11aea}',  &['\u{11ae5}',
         '\u{11ae5}',  '\u{11af0}']), ('\u{11aec}',  &['\u{11aeb}',  '\u{11aef}']), ('\u{11aed}',
         &['\u{11aeb}',  '\u{11aeb}']), ('\u{11aee}',  &['\u{11aeb}',  '\u{11aeb}',  '\u{11aef}']),
        ('\u{11af4}',  &['\u{11af3}',  '\u{11aef}']), ('\u{11af5}',  &['\u{11af3}',  '\u{11af0}']),
        ('\u{11af6}',  &['\u{11af3}',  '\u{11af3}']), ('\u{11af7}',  &['\u{11af3}',  '\u{11af3}',
         '\u{11aef}']), ('\u{11af8}',  &['\u{11af3}',  '\u{11af3}',  '\u{11af0}']), ('\u{11c42}',
         &['\u{11c41}',  '\u{11c41}']), ('\u{11cb2}',  &['\u{11caa}']), ('\u{12038}',
         &['\u{1039a}']), ('\u{132f9}',  &['\u{1099e}']), ('\u{16f07}',  &['\u{393}']),
        ('\u{16f08}',  &['\u{56}']), ('\u{16f0a}',  &['\u{54}']), ('\u{16f16}',  &['\u{4c}']),
        ('\u{16f1a}',  &['\u{394}']), ('\u{16f1c}',  &['\u{a658}']), ('\u{16f26}',  &['\u{a4f6}']),
        ('\u{16f28}',  &['\u{6c}']), ('\u{16f2d}',  &['\u{190}']), ('\u{16f35}',  &['\u{52}']),
        ('\u{16f3a}',  &['\u{53}']), ('\u{16f3b}',  &['\u{33}']), ('\u{16f3d}',  &['\u{245}']),
        ('\u{16f3f}',  &['\u{3e}']), ('\u{16f40}',  &['\u{41}']), ('\u{16f42}',  &['\u{55}']),
        ('\u{16f43}',  &['\u{59}']), ('\u{16f51}',  &['\u{27}']), ('\u{16f52}',  &['\u{27}']),
        ('\u{1ccd6}',  &['\u{41}']), ('\u{1ccd7}',  &['\u{42}']), ('\u{1ccd8}',  &['\u{43}']),
        ('\u{1ccd9}',  &['\u{44}']), ('\u{1ccda}',  &['\u{45}']), ('\u{1ccdb}',  &['\u{46}']),
        ('\u{1ccdc}',  &['\u{47}']), ('\u{1ccdd}',  &['\u{48}']), ('\u{1ccde}',  &['\u{6c}']),
        ('\u{1ccdf}',  &['\u{4a}']), ('\u{1cce0}',  &['\u{4b}']), ('\u{1cce1}',  &['\u{4c}']),
        ('\u{1cce2}',  &['\u{4d}']), ('\u{1cce3}',  &['\u{4e}']), ('\u{1cce4}',  &['\u{4f}']),
        ('\u{1cce5}',  &['\u{50}']), ('\u{1cce6}',  &['\u{51}']), ('\u{1cce7}',  &['\u{52}']),
        ('\u{1cce8}',  &['\u{53}']), ('\u{1cce9}',  &['\u{54}']), ('\u{1ccea}',  &['\u{55}']),
        ('\u{1cceb}',  &['\u{56}']), ('\u{1ccec}',  &['\u{57}']), ('\u{1cced}',  &['\u{58}']),
        ('\u{1ccee}',  &['\u{59}']), ('\u{1ccef}',  &['\u{5a}']), ('\u{1ccf0}',  &['\u{4f}']),
        ('\u{1ccf1}',  &['\u{6c}']), ('\u{1ccf2}',  &['\u{32}']), ('\u{1ccf3}',  &['\u{33}']),
        ('\u{1ccf4}',  &['\u{34}']), ('\u{1ccf5}',  &['\u{35}']), ('\u{1ccf6}',  &['\u{36}']),
        ('\u{1ccf7}',  &['\u{37}']), ('\u{1ccf8}',  &['\u{38}']), ('\u{1ccf9}',  &['\u{39}']),
        ('\u{1d114}',  &['\u{7b}']), ('\u{1d16d}',  &['\u{2e}']), ('\u{1d202}',  &['\u{4fe}']),
        ('\u{1d206}',  &['\u{33}']), ('\u{1d20b}',  &['\u{418}']), ('\u{1d20d}',  &['\u{56}']),
        ('\u{1d20f}',  &['\u{5c}']), ('\u{1d212}',  &['\u{37}']), ('\u{1d213}',  &['\u{46}']),
        ('\u{1d214}',  &['\u{102bc}']), ('\u{1d215}',  &['\u{a4f6}']), ('\u{1d216}',  &['\u{52}']),
        ('\u{1d217}',  &['\u{2c6f}']), ('\u{1d21a}',  &['\u{4f}',  '\u{335}']), ('\u{1d21b}',
         &['\u{2144}']), ('\u{1d21c}',  &['\u{a4d5}']), ('\u{1d221}',  &['\u{190}']), ('\u{1d222}',
         &['\u{460}']), ('\u{1d22a}',  &['\u{4c}']), ('\u{1d22b}',  &['\u{a4f6}']), ('\u{1d230}',
         &['\u{a7fb}']), ('\u{1d236}',  &['\u{3c}']), ('\u{1d237}',  &['\u{3e}']), ('\u{1d238}',
         &['\u{228f}']), ('\u{1d239}',  &['\u{2290}']), ('\u{1d23a}',  &['\u{2f}']), ('\u{1d23b}',
         &['\u{5c}']), ('\u{1d23f}',  &['\u{16cb}']), ('\u{1d245}',  &['\u{548}']), ('\u{1d400}',
         &['\u{41}']), ('\u{1d401}',  &['\u{42}']), ('\u{1d402}',  &['\u{43}']), ('\u{1d403}',
         &['\u{44}']), ('\u{1d404}',  &['\u{45}']), ('\u{1d405}',  &['\u{46}']), ('\u{1d406}',
         &['\u{47}']), ('\u{1d407}',  &['\u{48}']), ('\u{1d408}',  &['\u{6c}']), ('\u{1d409}',
         &['\u{4a}']), ('\u{1d40a}',  &['\u{4b}']), ('\u{1d40b}',  &['\u{4c}']), ('\u{1d40c}',
         &['\u{4d}']), ('\u{1d40d}',  &['\u{4e}']), ('\u{1d40e}',  &['\u{4f}']), ('\u{1d40f}',
         &['\u{50}']), ('\u{1d410}',  &['\u{51}']), ('\u{1d411}',  &['\u{52}']), ('\u{1d412}',
         &['\u{53}']), ('\u{1d413}',  &['\u{54}']), ('\u{1d414}',  &['\u{55}']), ('\u{1d415}',
         &['\u{56}']), ('\u{1d416}',  &['\u{57}']), ('\u{1d417}',  &['\u{58}']), ('\u{1d418}',
         &['\u{59}']), ('\u{1d419}',  &['\u{5a}']), ('\u{1d41a}',  &['\u{61}']), ('\u{1d41b}',
         &['\u{62}']), ('\u{1d41c}',  &['\u{63}']), ('\u{1d41d}',  &['\u{64}']), ('\u{1d41e}',
         &['\u{65}']), ('\u{1d41f}',  &['\u{66}']), ('\u{1d420}',  &['\u{67}']), ('\u{1d421}',
         &['\u{68}']), ('\u{1d422}',  &['\u{69}']), ('\u{1d423}',  &['\u{6a}']), ('\u{1d424}',
         &['\u{6b}']), ('\u{1d425}',  &['\u{6c}']), ('\u{1d426}',  &['\u{72}',  '\u{6e}']),
        ('\u{1d427}',  &['\u{6e}']), ('\u{1d428}',  &['\u{6f}']), ('\u{1d429}',  &['\u{70}']),
        ('\u{1d42a}',  &['\u{71}']), ('\u{1d42b}',  &['\u{72}']), ('\u{1d42c}',  &['\u{73}']),
        ('\u{1d42d}',  &['\u{74}']), ('\u{1d42e}',  &['\u{75}']), ('\u{1d42f}',  &['\u{76}']),
        ('\u{1d430}',  &['\u{77}']), ('\u{1d431}',  &['\u{78}']), ('\u{1d432}',  &['\u{79}']),
        ('\u{1d433}',  &['\u{7a}']), ('\u{1d434}',  &['\u{41}']), ('\u{1d435}',  &['\u{42}']),
        ('\u{1d436}',  &['\u{43}']), ('\u{1d437}',  &['\u{44}']), ('\u{1d438}',  &['\u{45}']),
        ('\u{1d439}',  &['\u{46}']), ('\u{1d43a}',  &['\u{47}']), ('\u{1d43b}',  &['\u{48}']),
        ('\u{1d43c}',  &['\u{6c}']), ('\u{1d43d}',  &['\u{4a}']), ('\u{1d43e}',  &['\u{4b}']),
        ('\u{1d43f}',  &['\u{4c}']), ('\u{1d440}',  &['\u{4d}']), ('\u{1d441}',  &['\u{4e}']),
        ('\u{1d442}',  &['\u{4f}']), ('\u{1d443}',  &['\u{50}']), ('\u{1d444}',  &['\u{51}']),
        ('\u{1d445}',  &['\u{52}']), ('\u{1d446}',  &['\u{53}']), ('\u{1d447}',  &['\u{54}']),
        ('\u{1d448}',  &['\u{55}']), ('\u{1d449}',  &['\u{56}']), ('\u{1d44a}',  &['\u{57}']),
        ('\u{1d44b}',  &['\u{58}']), ('\u{1d44c}',  &['\u{59}']), ('\u{1d44d}',  &['\u{5a}']),
        ('\u{1d44e}',  &['\u{61}']), ('\u{1d44f}',  &['\u{62}']), ('\u{1d450}',  &['\u{63}']),
        ('\u{1d451}',  &['\u{64}']), ('\u{1d452}',  &['\u{65}']), ('\u{1d453}',  &['\u{66}']),
        ('\u{1d454}',  &['\u{67}']), ('\u{1d456}',  &['\u{69}']), ('\u{1d457}',  &['\u{6a}']),
        ('\u{1d458}',  &['\u{6b}']), ('\u{1d459}',  &['\u{6c}']), ('\u{1d45a}',  &['\u{72}',
         '\u{6e}']), ('\u{1d45b}',  &['\u{6e}']), ('\u{1d45c}',  &['\u{6f}']), ('\u{1d45d}',
         &['\u{70}']), ('\u{1d45e}',  &['\u{71}']), ('\u{1d45f}',  &['\u{72}']), ('\u{1d460}',
         &['\u{73}']), ('\u{1d461}',  &['\u{74}']), ('\u{1d462}',  &['\u{75}']), ('\u{1d463}',
         &['\u{76}']), ('\u{1d464}',  &['\u{77}']), ('\u{1d465}',  &['\u{78}']), ('\u{1d466}',
         &['\u{79}']), ('\u{1d467}',  &['\u{7a}']), ('\u{1d468}',  &['\u{41}']), ('\u{1d469}',
         &['\u{42}']), ('\u{1d46a}',  &['\u{43}']), ('\u{1d46b}',  &['\u{44}']), ('\u{1d46c}',
         &['\u{45}']), ('\u{1d46d}',  &['\u{46}']), ('\u{1d46e}',  &['\u{47}']), ('\u{1d46f}',
         &['\u{48}']), ('\u{1d470}',  &['\u{6c}']), ('\u{1d471}',  &['\u{4a}']), ('\u{1d472}',
         &['\u{4b}']), ('\u{1d473}',  &['\u{4c}']), ('\u{1d474}',  &['\u{4d}']), ('\u{1d475}',
         &['\u{4e}']), ('\u{1d476}',  &['\u{4f}']), ('\u{1d477}',  &['\u{50}']), ('\u{1d478}',
         &['\u{51}']), ('\u{1d479}',  &['\u{52}']), ('\u{1d47a}',  &['\u{53}']), ('\u{1d47b}',
         &['\u{54}']), ('\u{1d47c}',  &['\u{55}']), ('\u{1d47d}',  &['\u{56}']), ('\u{1d47e}',
         &['\u{57}']), ('\u{1d47f}',  &['\u{58}']), ('\u{1d480}',  &['\u{59}']), ('\u{1d481}',
         &['\u{5a}']), ('\u{1d482}',  &['\u{61}']), ('\u{1d483}',  &['\u{62}']), ('\u{1d484}',
         &['\u{63}']), ('\u{1d485}',  &['\u{64}']), ('\u{1d486}',  &['\u{65}']), ('\u{1d487}',
         &['\u{66}']), ('\u{1d488}',  &['\u{67}']), ('\u{1d489}',  &['\u{68}']), ('\u{1d48a}',
         &['\u{69}']), ('\u{1d48b}',  &['\u{6a}']), ('\u{1d48c}',  &['\u{6b}']), ('\u{1d48d}',
         &['\u{6c}']), ('\u{1d48e}',  &['\u{72}',  '\u{6e}']), ('\u{1d48f}',  &['\u{6e}']),
        ('\u{1d490}',  &['\u{6f}']), ('\u{1d491}',  &['\u{70}']), ('\u{1d492}',  &['\u{71}']),
        ('\u{1d493}',  &['\u{72}']), ('\u{1d494}',  &['\u{73}']), ('\u{1d495}',  &['\u{74}']),
        ('\u{1d496}',  &['\u{75}']), ('\u{1d497}',  &['\u{76}']), ('\u{1d498}',  &['\u{77}']),
        ('\u{1d499}',  &['\u{78}']), ('\u{1d49a}',  &['\u{79}']), ('\u{1d49b}',  &['\u{7a}']),
        ('\u{1d49c}',  &['\u{41}']), ('\u{1d49e}',  &['\u{43}']), ('\u{1d49f}',  &['\u{44}']),
        ('\u{1d4a2}',  &['\u{47}']), ('\u{1d4a5}',  &['\u{4a}']), ('\u{1d4a6}',  &['\u{4b}']),
        ('\u{1d4a9}',  &['\u{4e}']), ('\u{1d4aa}',  &['\u{4f}']), ('\u{1d4ab}',  &['\u{50}']),
        ('\u{1d4ac}',  &['\u{51}']), ('\u{1d4ae}',  &['\u{53}']), ('\u{1d4af}',  &['\u{54}']),
        ('\u{1d4b0}',  &['\u{55}']), ('\u{1d4b1}',  &['\u{56}']), ('\u{1d4b2}',  &['\u{57}']),
        ('\u{1d4b3}',  &['\u{58}']), ('\u{1d4b4}',  &['\u{59}']), ('\u{1d4b5}',  &['\u{5a}']),
        ('\u{1d4b6}',  &['\u{61}']), ('\u{1d4b7}',  &['\u{62}']), ('\u{1d4b8}',  &['\u{63}']),
        ('\u{1d4b9}',  &['\u{64}']), ('\u{1d4bb}',  &['\u{66}']), ('\u{1d4bd}',  &['\u{68}']),
        ('\u{1d4be}',  &['\u{69}']), ('\u{1d4bf}',  &['\u{6a}']), ('\u{1d4c0}',  &['\u{6b}']),
        ('\u{1d4c1}',  &['\u{6c}']), ('\u{1d4c2}',  &['\u{72}',  '\u{6e}']), ('\u{1d4c3}',
         &['\u{6e}']), ('\u{1d4c5}',  &['\u{70}']), ('\u{1d4c6}',  &['\u{71}']), ('\u{1d4c7}',
         &['\u{72}']), ('\u{1d4c8}',  &['\u{73}']), ('\u{1d4c9}',  &['\u{74}']), ('\u{1d4ca}',
         &['\u{75}']), ('\u{1d4cb}',  &['\u{76}']), ('\u{1d4cc}',  &['\u{77}']), ('\u{1d4cd}',
         &['\u{78}']), ('\u{1d4ce}',  &['\u{79}']), ('\u{1d4cf}',  &['\u{7a}']), ('\u{1d4d0}',
         &['\u{41}']), ('\u{1d4d1}',  &['\u{42}']), ('\u{1d4d2}',  &['\u{43}']), ('\u{1d4d3}',
         &['\u{44}']), ('\u{1d4d4}',  &['\u{45}']), ('\u{1d4d5}',  &['\u{46}']), ('\u{1d4d6}',
         &['\u{47}']), ('\u{1d4d7}',  &['\u{48}']), ('\u{1d4d8}',  &['\u{6c}']), ('\u{1d4d9}',
         &['\u{4a}']), ('\u{1d4da}',  &['\u{4b}']), ('\u{1d4db}',  &['\u{4c}']), ('\u{1d4dc}',
         &['\u{4d}']), ('\u{1d4dd}',  &['\u{4e}']), ('\u{1d4de}',  &['\u{4f}']), ('\u{1d4df}',
         &['\u{50}']), ('\u{1d4e0}',  &['\u{51}']), ('\u{1d4e1}',  &['\u{52}']), ('\u{1d4e2}',
         &['\u{53}']), ('\u{1d4e3}',  &['\u{54}']), ('\u{1d4e4}',  &['\u{55}']), ('\u{1d4e5}',
         &['\u{56}']), ('\u{1d4e6}',  &['\u{57}']), ('\u{1d4e7}',  &['\u{58}']), ('\u{1d4e8}',
         &['\u{59}']), ('\u{1d4e9}',  &['\u{5a}']), ('\u{1d4ea}',  &['\u{61}']), ('\u{1d4eb}',
         &['\u{62}']), ('\u{1d4ec}',  &['\u{63}']), ('\u{1d4ed}',  &['\u{64}']), ('\u{1d4ee}',
         &['\u{65}']), ('\u{1d4ef}',  &['\u{66}']), ('\u{1d4f0}',  &['\u{67}']), ('\u{1d4f1}',
         &['\u{68}']), ('\u{1d4f2}',  &['\u{69}']), ('\u{1d4f3}',  &['\u{6a}']), ('\u{1d4f4}',
         &['\u{6b}']), ('\u{1d4f5}',  &['\u{6c}']), ('\u{1d4f6}',  &['\u{72}',  '\u{6e}']),
        ('\u{1d4f7}',  &['\u{6e}']), ('\u{1d4f8}',  &['\u{6f}']), ('\u{1d4f9}',  &['\u{70}']),
        ('\u{1d4fa}',  &['\u{71}']), ('\u{1d4fb}',  &['\u{72}']), ('\u{1d4fc}',  &['\u{73}']),
        ('\u{1d4fd}',  &['\u{74}']), ('\u{1d4fe}',  &['\u{75}']), ('\u{1d4ff}',  &['\u{76}']),
        ('\u{1d500}',  &['\u{77}']), ('\u{1d501}',  &['\u{78}']), ('\u{1d502}',  &['\u{79}']),
        ('\u{1d503}',  &['\u{7a}']), ('\u{1d504}',  &['\u{41}']), ('\u{1d505}',  &['\u{42}']),
        ('\u{1d507}',  &['\u{44}']), ('\u{1d508}',  &['\u{45}']), ('\u{1d509}',  &['\u{46}']),
        ('\u{1d50a}',  &['\u{47}']), ('\u{1d50d}',  &['\u{4a}']), ('\u{1d50e}',  &['\u{4b}']),
        ('\u{1d50f}',  &['\u{4c}']), ('\u{1d510}',  &['\u{4d}']), ('\u{1d511}',  &['\u{4e}']),
        ('\u{1d512}',  &['\u{4f}']), ('\u{1d513}',  &['\u{50}']), ('\u{1d514}',  &['\u{51}']),
        ('\u{1d516}',  &['\u{53}']), ('\u{1d517}',  &['\u{54}']), ('\u{1d518}',  &['\u{55}']),
        ('\u{1d519}',  &['\u{56}']), ('\u{1d51a}',  &['\u{57}']), ('\u{1d51b}',  &['\u{58}']),
        ('\u{1d51c}',  &['\u{59}']), ('\u{1d51e}',  &['\u{61}']), ('\u{1d51f}',  &['\u{62}']),
        ('\u{1d520}',  &['\u{63}']), ('\u{1d521}',  &['\u{64}']), ('\u{1d522}',  &['\u{65}']),
        ('\u{1d523}',  &['\u{66}']), ('\u{1d524}',  &['\u{67}']), ('\u{1d525}',  &['\u{68}']),
        ('\u{1d526}',  &['\u{69}']), ('\u{1d527}',  &['\u{6a}']), ('\u{1d528}',  &['\u{6b}']),
        ('\u{1d529}',  &['\u{6c}']), ('\u{1d52a}',  &['\u{72}',  '\u{6e}']), ('\u{1d52b}',
         &['\u{6e}']), ('\u{1d52c}',  &['\u{6f}']), ('\u{1d52d}',  &['\u{70}']), ('\u{1d52e}',
         &['\u{71}']), ('\u{1d52f}',  &['\u{72}']), ('\u{1d530}',  &['\u{73}']), ('\u{1d531}',
         &['\u{74}']), ('\u{1d532}',  &['\u{75}']), ('\u{1d533}',  &['\u{76}']), ('\u{1d534}',
         &['\u{77}']), ('\u{1d535}',  &['\u{78}']), ('\u{1d536}',  &['\u{79}']), ('\u{1d537}',
         &['\u{7a}']), ('\u{1d538}',  &['\u{41}']), ('\u{1d539}',  &['\u{42}']), ('\u{1d53b}',
         &['\u{44}']), ('\u{1d53c}',  &['\u{45}']), ('\u{1d53d}',  &['\u{46}']), ('\u{1d53e}',
         &['\u{47}']), ('\u{1d540}',  &['\u{6c}']), ('\u{1d541}',  &['\u{4a}']), ('\u{1d542}',
         &['\u{4b}']), ('\u{1d543}',  &['\u{4c}']), ('\u{1d544}',  &['\u{4d}']), ('\u{1d546}',
         &['\u{4f}']), ('\u{1d54a}',  &['\u{53}']), ('\u{1d54b}',  &['\u{54}']), ('\u{1d54c}',
         &['\u{55}']), ('\u{1d54d}',  &['\u{56}']), ('\u{1d54e}',  &['\u{57}']), ('\u{1d54f}',
         &['\u{58}']), ('\u{1d550}',  &['\u{59}']), ('\u{1d552}',  &['\u{61}']), ('\u{1d553}',
         &['\u{62}']), ('\u{1d554}',  &['\u{63}']), ('\u{1d555}',  &['\u{64}']), ('\u{1d556}',
         &['\u{65}']), ('\u{1d557}',  &['\u{66}']), ('\u{1d558}',  &['\u{67}']), ('\u{1d559}',
         &['\u{68}']), ('\u{1d55a}',  &['\u{69}']), ('\u{1d55b}',  &['\u{6a}']), ('\u{1d55c}',
         &['\u{6b}']), ('\u{1d55d}',  &['\u{6c}']), ('\u{1d55e}',  &['\u{72}',  '\u{6e}']),
        ('\u{1d55f}',  &['\u{6e}']), ('\u{1d560}',  &['\u{6f}']), ('\u{1d561}',  &['\u{70}']),
        ('\u{1d562}',  &['\u{71}']), ('\u{1d563}',  &['\u{72}']), ('\u{1d564}',  &['\u{73}']),
        ('\u{1d565}',  &['\u{74}']), ('\u{1d566}',  &['\u{75}']), ('\u{1d567}',  &['\u{76}']),
        ('\u{1d568}',  &['\u{77}']), ('\u{1d569}',  &['\u{78}']), ('\u{1d56a}',  &['\u{79}']),
        ('\u{1d56b}',  &['\u{7a}']), ('\u{1d56c}',  &['\u{41}']), ('\u{1d56d}',  &['\u{42}']),
        ('\u{1d56e}',  &['\u{43}']), ('\u{1d56f}',  &['\u{44}']), ('\u{1d570}',  &['\u{45}']),
        ('\u{1d571}',  &['\u{46}']), ('\u{1d572}',  &['\u{47}']), ('\u{1d573}',  &['\u{48}']),
        ('\u{1d574}',  &['\u{6c}']), ('\u{1d575}',  &['\u{4a}']), ('\u{1d576}',  &['\u{4b}']),
        ('\u{1d577}',  &['\u{4c}']), ('\u{1d578}',  &['\u{4d}']), ('\u{1d579}',  &['\u{4e}']),
        ('\u{1d57a}',  &['\u{4f}']), ('\u{1d57b}',  &['\u{50}']), ('\u{1d57c}',  &['\u{51}']),
        ('\u{1d57d}',  &['\u{52}']), ('\u{1d57e}',  &['\u{53}']), ('\u{1d57f}',  &['\u{54}']),
        ('\u{1d580}',  &['\u{55}']), ('\u{1d581}',  &['\u{56}']), ('\u{1d582}',  &['\u{57}']),
        ('\u{1d583}',  &['\u{58}']), ('\u{1d584}',  &['\u{59}']), ('\u{1d585}',  &['\u{5a}']),
        ('\u{1d586}',  &['\u{61}']), ('\u{1d587}',  &['\u{62}']), ('\u{1d588}',  &['\u{63}']),
        ('\u{1d589}',  &['\u{64}']), ('\u{1d58a}',  &['\u{65}']), ('\u{1d58b}',  &['\u{66}']),
        ('\u{1d58c}',  &['\u{67}']), ('\u{1d58d}',  &['\u{68}']), ('\u{1d58e}',  &['\u{69}']),
        ('\u{1d58f}',  &['\u{6a}']), ('\u{1d590}',  &['\u{6b}']), ('\u{1d591}',  &['\u{6c}']),
        ('\u{1d592}',  &['\u{72}',  '\u{6e}']), ('\u{1d593}',  &['\u{6e}']), ('\u{1d594}',
         &['\u{6f}']), ('\u{1d595}',  &['\u{70}']), ('\u{1d596}',  &['\u{71}']), ('\u{1d597}',
         &['\u{72}']), ('\u{1d598}',  &['\u{73}']), ('\u{1d599}',  &['\u{74}']), ('\u{1d59a}',
         &['\u{75}']), ('\u{1d59b}',  &['\u{76}']), ('\u{1d59c}',  &['\u{77}']), ('\u{1d59d}',
         &['\u{78}']), ('\u{1d59e}',  &['\u{79}']), ('\u{1d59f}',  &['\u{7a}']), ('\u{1d5a0}',
         &['\u{41}']), ('\u{1d5a1}',  &['\u{42}']), ('\u{1d5a2}',  &['\u{43}']), ('\u{1d5a3}',
         &['\u{44}']), ('\u{1d5a4}',  &['\u{45}']), ('\u{1d5a5}',  &['\u{46}']), ('\u{1d5a6}',
         &['\u{47}']), ('\u{1d5a7}',  &['\u{48}']), ('\u{1d5a8}',  &['\u{6c}']), ('\u{1d5a9}',
         &['\u{4a}']), ('\u{1d5aa}',  &['\u{4b}']), ('\u{1d5ab}',  &['\u{4c}']), ('\u{1d5ac}',
         &['\u{4d}']), ('\u{1d5ad}',  &['\u{4e}']), ('\u{1d5ae}',  &['\u{4f}']), ('\u{1d5af}',
         &['\u{50}']), ('\u{1d5b0}',  &['\u{51}']), ('\u{1d5b1}',  &['\u{52}']), ('\u{1d5b2}',
         &['\u{53}']), ('\u{1d5b3}',  &['\u{54}']), ('\u{1d5b4}',  &['\u{55}']), ('\u{1d5b5}',
         &['\u{56}']), ('\u{1d5b6}',  &['\u{57}']), ('\u{1d5b7}',  &['\u{58}']), ('\u{1d5b8}',
         &['\u{59}']), ('\u{1d5b9}',  &['\u{5a}']), ('\u{1d5ba}',  &['\u{61}']), ('\u{1d5bb}',
         &['\u{62}']), ('\u{1d5bc}',  &['\u{63}']), ('\u{1d5bd}',  &['\u{64}']), ('\u{1d5be}',
         &['\u{65}']), ('\u{1d5bf}',  &['\u{66}']), ('\u{1d5c0}',  &['\u{67}']), ('\u{1d5c1}',
         &['\u{68}']), ('\u{1d5c2}',  &['\u{69}']), ('\u{1d5c3}',  &['\u{6a}']), ('\u{1d5c4}',
         &['\u{6b}']), ('\u{1d5c5}',  &['\u{6c}']), ('\u{1d5c6}',  &['\u{72}',  '\u{6e}']),
        ('\u{1d5c7}',  &['\u{6e}']), ('\u{1d5c8}',  &['\u{6f}']), ('\u{1d5c9}',  &['\u{70}']),
        ('\u{1d5ca}',  &['\u{71}']), ('\u{1d5cb}',  &['\u{72}']), ('\u{1d5cc}',  &['\u{73}']),
        ('\u{1d5cd}',  &['\u{74}']), ('\u{1d5ce}',  &['\u{75}']), ('\u{1d5cf}',  &['\u{76}']),
        ('\u{1d5d0}',  &['\u{77}']), ('\u{1d5d1}',  &['\u{78}']), ('\u{1d5d2}',  &['\u{79}']),
        ('\u{1d5d3}',  &['\u{7a}']), ('\u{1d5d4}',  &['\u{41}']), ('\u{1d5d5}',  &['\u{42}']),
        ('\u{1d5d6}',  &['\u{43}']), ('\u{1d5d7}',  &['\u{44}']), ('\u{1d5d8}',  &['\u{45}']),
        ('\u{1d5d9}',  &['\u{46}']), ('\u{1d5da}',  &['\u{47}']), ('\u{1d5db}',  &['\u{48}']),
        ('\u{1d5dc}',  &['\u{6c}']), ('\u{1d5dd}',  &['\u{4a}']), ('\u{1d5de}',  &['\u{4b}']),
        ('\u{1d5df}',  &['\u{4c}']), ('\u{1d5e0}',  &['\u{4d}']), ('\u{1d5e1}',  &['\u{4e}']),
        ('\u{1d5e2}',  &['\u{4f}']), ('\u{1d5e3}',  &['\u{50}']), ('\u{1d5e4}',  &['\u{51}']),
        ('\u{1d5e5}',  &['\u{52}']), ('\u{1d5e6}',  &['\u{53}']), ('\u{1d5e7}',  &['\u{54}']),
        ('\u{1d5e8}',  &['\u{55}']), ('\u{1d5e9}',  &['\u{56}']), ('\u{1d5ea}',  &['\u{57}']),
        ('\u{1d5eb}',  &['\u{58}']), ('\u{1d5ec}',  &['\u{59}']), ('\u{1d5ed}',  &['\u{5a}']),
        ('\u{1d5ee}',  &['\u{61}']), ('\u{1d5ef}',  &['\u{62}']), ('\u{1d5f0}',  &['\u{63}']),
        ('\u{1d5f1}',  &['\u{64}']), ('\u{1d5f2}',  &['\u{65}']), ('\u{1d5f3}',  &['\u{66}']),
        ('\u{1d5f4}',  &['\u{67}']), ('\u{1d5f5}',  &['\u{68}']), ('\u{1d5f6}',  &['\u{69}']),
        ('\u{1d5f7}',  &['\u{6a}']), ('\u{1d5f8}',  &['\u{6b}']), ('\u{1d5f9}',  &['\u{6c}']),
        ('\u{1d5fa}',  &['\u{72}',  '\u{6e}']), ('\u{1d5fb}',  &['\u{6e}']), ('\u{1d5fc}',
         &['\u{6f}']), ('\u{1d5fd}',  &['\u{70}']), ('\u{1d5fe}',  &['\u{71}']), ('\u{1d5ff}',
         &['\u{72}']), ('\u{1d600}',  &['\u{73}']), ('\u{1d601}',  &['\u{74}']), ('\u{1d602}',
         &['\u{75}']), ('\u{1d603}',  &['\u{76}']), ('\u{1d604}',  &['\u{77}']), ('\u{1d605}',
         &['\u{78}']), ('\u{1d606}',  &['\u{79}']), ('\u{1d607}',  &['\u{7a}']), ('\u{1d608}',
         &['\u{41}']), ('\u{1d609}',  &['\u{42}']), ('\u{1d60a}',  &['\u{43}']), ('\u{1d60b}',
         &['\u{44}']), ('\u{1d60c}',  &['\u{45}']), ('\u{1d60d}',  &['\u{46}']), ('\u{1d60e}',
         &['\u{47}']), ('\u{1d60f}',  &['\u{48}']), ('\u{1d610}',  &['\u{6c}']), ('\u{1d611}',
         &['\u{4a}']), ('\u{1d612}',  &['\u{4b}']), ('\u{1d613}',  &['\u{4c}']), ('\u{1d614}',
         &['\u{4d}']), ('\u{1d615}',  &['\u{4e}']), ('\u{1d616}',  &['\u{4f}']), ('\u{1d617}',
         &['\u{50}']), ('\u{1d618}',  &['\u{51}']), ('\u{1d619}',  &['\u{52}']), ('\u{1d61a}',
         &['\u{53}']), ('\u{1d61b}',  &['\u{54}']), ('\u{1d61c}',  &['\u{55}']), ('\u{1d61d}',
         &['\u{56}']), ('\u{1d61e}',  &['\u{57}']), ('\u{1d61f}',  &['\u{58}']), ('\u{1d620}',
         &['\u{59}']), ('\u{1d621}',  &['\u{5a}']), ('\u{1d622}',  &['\u{61}']), ('\u{1d623}',
         &['\u{62}']), ('\u{1d624}',  &['\u{63}']), ('\u{1d625}',  &['\u{64}']), ('\u{1d626}',
         &['\u{65}']), ('\u{1d627}',  &['\u{66}']), ('\u{1d628}',  &['\u{67}']), ('\u{1d629}',
         &['\u{68}']), ('\u{1d62a}',  &['\u{69}']), ('\u{1d62b}',  &['\u{6a}']), ('\u{1d62c}',
         &['\u{6b}']), ('\u{1d62d}',  &['\u{6c}']), ('\u{1d62e}',  &['\u{72}',  '\u{6e}']),
        ('\u{1d62f}',  &['\u{6e}']), ('\u{1d630}',  &['\u{6f}']), ('\u{1d631}',  &['\u{70}']),
        ('\u{1d632}',  &['\u{71}']), ('\u{1d633}',  &['\u{72}']), ('\u{1d634}',  &['\u{73}']),
        ('\u{1d635}',  &['\u{74}']), ('\u{1d636}',  &['\u{75}']), ('\u{1d637}',  &['\u{76}']),
        ('\u{1d638}',  &['\u{77}']), ('\u{1d639}',  &['\u{78}']), ('\u{1d63a}',  &['\u{79}']),
        ('\u{1d63b}',  &['\u{7a}']), ('\u{1d63c}',  &['\u{41}']), ('\u{1d63d}',  &['\u{42}']),
        ('\u{1d63e}',  &['\u{43}']), ('\u{1d63f}',  &['\u{44}']), ('\u{1d640}',  &['\u{45}']),
        ('\u{1d641}',  &['\u{46}']), ('\u{1d642}',  &['\u{47}']), ('\u{1d643}',  &['\u{48}']),
        ('\u{1d644}',  &['\u{6c}']), ('\u{1d645}',  &['\u{4a}']), ('\u{1d646}',  &['\u{4b}']),
        ('\u{1d647}',  &['\u{4c}']), ('\u{1d648}',  &['\u{4d}']), ('\u{1d649}',  &['\u{4e}']),
        ('\u{1d64a}',  &['\u{4f}']), ('\u{1d64b}',  &['\u{50}']), ('\u{1d64c}',  &['\u{51}']),
        ('\u{1d64d}',  &['\u{52}']), ('\u{1d64e}',  &['\u{53}']), ('\u{1d64f}',  &['\u{54}']),
        ('\u{1d650}',  &['\u{55}']), ('\u{1d651}',  &['\u{56}']), ('\u{1d652}',  &['\u{57}']),
        ('\u{1d653}',  &['\u{58}']), ('\u{1d654}',  &['\u{59}']), ('\u{1d655}',  &['\u{5a}']),
        ('\u{1d656}',  &['\u{61}']), ('\u{1d657}',  &['\u{62}']), ('\u{1d658}',  &['\u{63}']),
        ('\u{1d659}',  &['\u{64}']), ('\u{1d65a}',  &['\u{65}']), ('\u{1d65b}',  &['\u{66}']),
        ('\u{1d65c}',  &['\u{67}']), ('\u{1d65d}',  &['\u{68}']), ('\u{1d65e}',  &['\u{69}']),
        ('\u{1d65f}',  &['\u{6a}']), ('\u{1d660}',  &['\u{6b}']), ('\u{1d661}',  &['\u{6c}']),
        ('\u{1d662}',  &['\u{72}',  '\u{6e}']), ('\u{1d663}',  &['\u{6e}']), ('\u{1d664}',
         &['\u{6f}']), ('\u{1d665}',  &['\u{70}']), ('\u{1d666}',  &['\u{71}']), ('\u{1d667}',
         &['\u{72}']), ('\u{1d668}',  &['\u{73}']), ('\u{1d669}',  &['\u{74}']), ('\u{1d66a}',
         &['\u{75}']), ('\u{1d66b}',  &['\u{76}']), ('\u{1d66c}',  &['\u{77}']), ('\u{1d66d}',
         &['\u{78}']), ('\u{1d66e}',  &['\u{79}']), ('\u{1d66f}',  &['\u{7a}']), ('\u{1d670}',
         &['\u{41}']), ('\u{1d671}',  &['\u{42}']), ('\u{1d672}',  &['\u{43}']), ('\u{1d673}',
         &['\u{44}']), ('\u{1d674}',  &['\u{45}']), ('\u{1d675}',  &['\u{46}']), ('\u{1d676}',
         &['\u{47}']), ('\u{1d677}',  &['\u{48}']), ('\u{1d678}',  &['\u{6c}']), ('\u{1d679}',
         &['\u{4a}']), ('\u{1d67a}',  &['\u{4b}']), ('\u{1d67b}',  &['\u{4c}']), ('\u{1d67c}',
         &['\u{4d}']), ('\u{1d67d}',  &['\u{4e}']), ('\u{1d67e}',  &['\u{4f}']), ('\u{1d67f}',
         &['\u{50}']), ('\u{1d680}',  &['\u{51}']), ('\u{1d681}',  &['\u{52}']), ('\u{1d682}',
         &['\u{53}']), ('\u{1d683}',  &['\u{54}']), ('\u{1d684}',  &['\u{55}']), ('\u{1d685}',
         &['\u{56}']), ('\u{1d686}',  &['\u{57}']), ('\u{1d687}',  &['\u{58}']), ('\u{1d688}',
         &['\u{59}']), ('\u{1d689}',  &['\u{5a}']), ('\u{1d68a}',  &['\u{61}']), ('\u{1d68b}',
         &['\u{62}']), ('\u{1d68c}',  &['\u{63}']), ('\u{1d68d}',  &['\u{64}']), ('\u{1d68e}',
         &['\u{65}']), ('\u{1d68f}',  &['\u{66}']), ('\u{1d690}',  &['\u{67}']), ('\u{1d691}',
         &['\u{68}']), ('\u{1d692}',  &['\u{69}']), ('\u{1d693}',  &['\u{6a}']), ('\u{1d694}',
         &['\u{6b}']), ('\u{1d695}',  &['\u{6c}']), ('\u{1d696}',  &['\u{72}',  '\u{6e}']),
        ('\u{1d697}',  &['\u{6e}']), ('\u{1d698}',  &['\u{6f}']), ('\u{1d699}',  &['\u{70}']),
        ('\u{1d69a}',  &['\u{71}']), ('\u{1d69b}',  &['\u{72}']), ('\u{1d69c}',  &['\u{73}']),
        ('\u{1d69d}',  &['\u{74}']), ('\u{1d69e}',  &['\u{75}']), ('\u{1d69f}',  &['\u{76}']),
        ('\u{1d6a0}',  &['\u{77}']), ('\u{1d6a1}',  &['\u{78}']), ('\u{1d6a2}',  &['\u{79}']),
        ('\u{1d6a3}',  &['\u{7a}']), ('\u{1d6a4}',  &['\u{69}']), ('\u{1d6a5}',  &['\u{237}']),
        ('\u{1d6a8}',  &['\u{41}']), ('\u{1d6a9}',  &['\u{42}']), ('\u{1d6aa}',  &['\u{393}']),
        ('\u{1d6ab}',  &['\u{394}']), ('\u{1d6ac}',  &['\u{45}']), ('\u{1d6ad}',  &['\u{5a}']),
        ('\u{1d6ae}',  &['\u{48}']), ('\u{1d6af}',  &['\u{4f}',  '\u{335}']), ('\u{1d6b0}',
         &['\u{6c}']), ('\u{1d6b1}',  &['\u{4b}']), ('\u{1d6b2}',  &['\u{245}']), ('\u{1d6b3}',
         &['\u{4d}']), ('\u{1d6b4}',  &['\u{4e}']), ('\u{1d6b5}',  &['\u{39e}']), ('\u{1d6b6}',
         &['\u{4f}']), ('\u{1d6b7}',  &['\u{3a0}']), ('\u{1d6b8}',  &['\u{50}']), ('\u{1d6b9}',
         &['\u{4f}',  '\u{335}']), ('\u{1d6ba}',  &['\u{1a9}']), ('\u{1d6bb}',  &['\u{54}']),
        ('\u{1d6bc}',  &['\u{59}']), ('\u{1d6bd}',  &['\u{3a6}']), ('\u{1d6be}',  &['\u{58}']),
        ('\u{1d6bf}',  &['\u{3a8}']), ('\u{1d6c0}',  &['\u{3a9}']), ('\u{1d6c1}',  &['\u{2207}']),
        ('\u{1d6c2}',  &['\u{61}']), ('\u{1d6c3}',  &['\u{df}']), ('\u{1d6c4}',  &['\u{79}']),
        ('\u{1d6c5}',  &['\u{1e9f}']), ('\u{1d6c6}',  &['\u{a793}']), ('\u{1d6c7}',  &['\u{3b6}']),
        ('\u{1d6c8}',  &['\u{6e}',  '\u{329}']), ('\u{1d6c9}',  &['\u{4f}',  '\u{335}']),
        ('\u{1d6ca}',  &['\u{69}']), ('\u{1d6cb}',  &['\u{138}']), ('\u{1d6cc}',  &['\u{3bb}']),
        ('\u{1d6cd}',  &['\u{3bc}']), ('\u{1d6ce}',  &['\u{76}']), ('\u{1d6cf}',  &['\u{3be}']),
        ('\u{1d6d0}',  &['\u{6f}']), ('\u{1d6d1}',  &['\u{3c0}']), ('\u{1d6d2}',  &['\u{70}']),
        ('\u{1d6d3}',  &['\u{3c2}']), ('\u{1d6d4}',  &['\u{6f}']), ('\u{1d6d5}',  &['\u{1d1b}']),
        ('\u{1d6d6}',  &['\u{75}']), ('\u{1d6d7}',  &['\u{278}']), ('\u{1d6d8}',  &['\u{3c7}']),
        ('\u{1d6d9}',  &['\u{3c8}']), ('\u{1d6da}',  &['\u{3c9}']), ('\u{1d6db}',  &['\u{2202}']),
        ('\u{1d6dc}',  &['\u{a793}']), ('\u{1d6dd}',  &['\u{4f}',  '\u{335}']), ('\u{1d6de}',
         &['\u{138}']), ('\u{1d6df}',  &['\u{278}']), ('\u{1d6e0}',  &['\u{70}']), ('\u{1d6e1}',
         &['\u{3c0}']), ('\u{1d6e2}',  &['\u{41}']), ('\u{1d6e3}',  &['\u{42}']), ('\u{1d6e4}',
         &['\u{393}']), ('\u{1d6e5}',  &['\u{394}']), ('\u{1d6e6}',  &['\u{45}']), ('\u{1d6e7}',
         &['\u{5a}']), ('\u{1d6e8}',  &['\u{48}']), ('\u{1d6e9}',  &['\u{4f}',  '\u{335}']),
        ('\u{1d6ea}',  &['\u{6c}']), ('\u{1d6eb}',  &['\u{4b}']), ('\u{1d6ec}',  &['\u{245}']),
        ('\u{1d6ed}',  &['\u{4d}']), ('\u{1d6ee}',  &['\u{4e}']), ('\u{1d6ef}',  &['\u{39e}']),
        ('\u{1d6f0}',  &['\u{4f}']), ('\u{1d6f1}',  &['\u{3a0}']), ('\u{1d6f2}',  &['\u{50}']),
        ('\u{1d6f3}',  &['\u{4f}',  '\u{335}']), ('\u{1d6f4}',  &['\u{1a9}']), ('\u{1d6f5}',
         &['\u{54}']), ('\u{1d6f6}',  &['\u{59}']), ('\u{1d6f7}',  &['\u{3a6}']), ('\u{1d6f8}',
         &['\u{58}']), ('\u{1d6f9}',  &['\u{3a8}']), ('\u{1d6fa}',  &['\u{3a9}']), ('\u{1d6fb}',
         &['\u{2207}']), ('\u{1d6fc}',  &['\u{61}']), ('\u{1d6fd}',  &['\u{df}']), ('\u{1d6fe}',
         &['\u{79}']), ('\u{1d6ff}',  &['\u{1e9f}']), ('\u{1d700}',  &['\u{a793}']), ('\u{1d701}',
         &['\u{3b6}']), ('\u{1d702}',  &['\u{6e}',  '\u{329}']), ('\u{1d703}',  &['\u{4f}',
         '\u{335}']), ('\u{1d704}',  &['\u{69}']), ('\u{1d705}',  &['\u{138}']), ('\u{1d706}',
         &['\u{3bb}']), ('\u{1d707}',  &['\u{3bc}']), ('\u{1d708}',  &['\u{76}']), ('\u{1d709}',
         &['\u{3be}']), ('\u{1d70a}',  &['\u{6f}']), ('\u{1d70b}',  &['\u{3c0}']), ('\u{1d70c}',
         &['\u{70}']), ('\u{1d70d}',  &['\u{3c2}']), ('\u{1d70e}',  &['\u{6f}']), ('\u{1d70f}',
         &['\u{1d1b}']), ('\u{1d710}',  &['\u{75}']), ('\u{1d711}',  &['\u{278}']), ('\u{1d712}',
         &['\u{3c7}']), ('\u{1d713}',  &['\u{3c8}']), ('\u{1d714}',  &['\u{3c9}']), ('\u{1d715}',
         &['\u{2202}']), ('\u{1d716}',  &['\u{a793}']), ('\u{1d717}',  &['\u{4f}',  '\u{335}']),
        ('\u{1d718}',  &['\u{138}']), ('\u{1d719}',  &['\u{278}']), ('\u{1d71a}',  &['\u{70}']),
        ('\u{1d71b}',  &['\u{3c0}']), ('\u{1d71c}',  &['\u{41}']), ('\u{1d71d}',  &['\u{42}']),
        ('\u{1d71e}',  &['\u{393}']), ('\u{1d71f}',  &['\u{394}']), ('\u{1d720}',  &['\u{45}']),
        ('\u{1d721}',  &['\u{5a}']), ('\u{1d722}',  &['\u{48}']), ('\u{1d723}',  &['\u{4f}',
         '\u{335}']), ('\u{1d724}',  &['\u{6c}']), ('\u{1d725}',  &['\u{4b}']), ('\u{1d726}',
         &['\u{245}']), ('\u{1d727}',  &['\u{4d}']), ('\u{1d728}',  &['\u{4e}']), ('\u{1d729}',
         &['\u{39e}']), ('\u{1d72a}',  &['\u{4f}']), ('\u{1d72b}',  &['\u{3a0}']), ('\u{1d72c}',
         &['\u{50}']), ('\u{1d72d}',  &['\u{4f}',  '\u{335}']), ('\u{1d72e}',  &['\u{1a9}']),
        ('\u{1d72f}',  &['\u{54}']), ('\u{1d730}',  &['\u{59}']), ('\u{1d731}',  &['\u{3a6}']),
        ('\u{1d732}',  &['\u{58}']), ('\u{1d733}',  &['\u{3a8}']), ('\u{1d734}',  &['\u{3a9}']),
        ('\u{1d735}',  &['\u{2207}']), ('\u{1d736}',  &['\u{61}']), ('\u{1d737}',  &['\u{df}']),
        ('\u{1d738}',  &['\u{79}']), ('\u{1d739}',  &['\u{1e9f}']), ('\u{1d73a}',  &['\u{a793}']),
        ('\u{1d73b}',  &['\u{3b6}']), ('\u{1d73c}',  &['\u{6e}',  '\u{329}']), ('\u{1d73d}',
         &['\u{4f}',  '\u{335}']), ('\u{1d73e}',  &['\u{69}']), ('\u{1d73f}',  &['\u{138}']),
        ('\u{1d740}',  &['\u{3bb}']), ('\u{1d741}',  &['\u{3bc}']), ('\u{1d742}',  &['\u{76}']),
        ('\u{1d743}',  &['\u{3be}']), ('\u{1d744}',  &['\u{6f}']), ('\u{1d745}',  &['\u{3c0}']),
        ('\u{1d746}',  &['\u{70}']), ('\u{1d747}',  &['\u{3c2}']), ('\u{1d748}',  &['\u{6f}']),
        ('\u{1d749}',  &['\u{1d1b}']), ('\u{1d74a}',  &['\u{75}']), ('\u{1d74b}',  &['\u{278}']),
        ('\u{1d74c}',  &['\u{3c7}']), ('\u{1d74d}',  &['\u{3c8}']), ('\u{1d74e}',  &['\u{3c9}']),
        ('\u{1d74f}',  &['\u{2202}']), ('\u{1d750}',  &['\u{a793}']), ('\u{1d751}',  &['\u{4f}',
         '\u{335}']), ('\u{1d752}',  &['\u{138}']), ('\u{1d753}',  &['\u{278}']), ('\u{1d754}',
         &['\u{70}']), ('\u{1d755}',  &['\u{3c0}']), ('\u{1d756}',  &['\u{41}']), ('\u{1d757}',
         &['\u{42}']), ('\u{1d758}',  &['\u{393}']), ('\u{1d759}',  &['\u{394}']), ('\u{1d75a}',
         &['\u{45}']), ('\u{1d75b}',  &['\u{5a}']), ('\u{1d75c}',  &['\u{48}']), ('\u{1d75d}',
         &['\u{4f}',  '\u{335}']), ('\u{1d75e}',  &['\u{6c}']), ('\u{1d75f}',  &['\u{4b}']),
        ('\u{1d760}',  &['\u{245}']), ('\u{1d761}',  &['\u{4d}']), ('\u{1d762}',  &['\u{4e}']),
        ('\u{1d763}',  &['\u{39e}']), ('\u{1d764}',  &['\u{4f}']), ('\u{1d765}',  &['\u{3a0}']),
        ('\u{1d766}',  &['\u{50}']), ('\u{1d767}',  &['\u{4f}',  '\u{335}']), ('\u{1d768}',
         &['\u{1a9}']), ('\u{1d769}',  &['\u{54}']), ('\u{1d76a}',  &['\u{59}']), ('\u{1d76b}',
         &['\u{3a6}']), ('\u{1d76c}',  &['\u{58}']), ('\u{1d76d}',  &['\u{3a8}']), ('\u{1d76e}',
         &['\u{3a9}']), ('\u{1d76f}',  &['\u{2207}']), ('\u{1d770}',  &['\u{61}']), ('\u{1d771}',
         &['\u{df}']), ('\u{1d772}',  &['\u{79}']), ('\u{1d773}',  &['\u{1e9f}']), ('\u{1d774}',
         &['\u{a793}']), ('\u{1d775}',  &['\u{3b6}']), ('\u{1d776}',  &['\u{6e}',  '\u{329}']),
        ('\u{1d777}',  &['\u{4f}',  '\u{335}']), ('\u{1d778}',  &['\u{69}']), ('\u{1d779}',
         &['\u{138}']), ('\u{1d77a}',  &['\u{3bb}']), ('\u{1d77b}',  &['\u{3bc}']), ('\u{1d77c}',
         &['\u{76}']), ('\u{1d77d}',  &['\u{3be}']), ('\u{1d77e}',  &['\u{6f}']), ('\u{1d77f}',
         &['\u{3c0}']), ('\u{1d780}',  &['\u{70}']), ('\u{1d781}',  &['\u{3c2}']), ('\u{1d782}',
         &['\u{6f}']), ('\u{1d783}',  &['\u{1d1b}']), ('\u{1d784}',  &['\u{75}']), ('\u{1d785}',
         &['\u{278}']), ('\u{1d786}',  &['\u{3c7}']), ('\u{1d787}',  &['\u{3c8}']), ('\u{1d788}',
         &['\u{3c9}']), ('\u{1d789}',  &['\u{2202}']), ('\u{1d78a}',  &['\u{a793}']), ('\u{1d78b}',
         &['\u{4f}',  '\u{335}']), ('\u{1d78c}',  &['\u{138}']), ('\u{1d78d}',  &['\u{278}']),
        ('\u{1d78e}',  &['\u{70}']), ('\u{1d78f}',  &['\u{3c0}']), ('\u{1d790}',  &['\u{41}']),
        ('\u{1d791}',  &['\u{42}']), ('\u{1d792}',  &['\u{393}']), ('\u{1d793}',  &['\u{394}']),
        ('\u{1d794}',  &['\u{45}']), ('\u{1d795}',  &['\u{5a}']), ('\u{1d796}',  &['\u{48}']),
        ('\u{1d797}',  &['\u{4f}',  '\u{335}']), ('\u{1d798}',  &['\u{6c}']), ('\u{1d799}',
         &['\u{4b}']), ('\u{1d79a}',  &['\u{245}']), ('\u{1d79b}',  &['\u{4d}']), ('\u{1d79c}',
         &['\u{4e}']), ('\u{1d79d}',  &['\u{39e}']), ('\u{1d79e}',  &['\u{4f}']), ('\u{1d79f}',
         &['\u{3a0}']), ('\u{1d7a0}',  &['\u{50}']), ('\u{1d7a1}',  &['\u{4f}',  '\u{335}']),
        ('\u{1d7a2}',  &['\u{1a9}']), ('\u{1d7a3}',  &['\u{54}']), ('\u{1d7a4}',  &['\u{59}']),
        ('\u{1d7a5}',  &['\u{3a6}']), ('\u{1d7a6}',  &['\u{58}']), ('\u{1d7a7}',  &['\u{3a8}']),
        ('\u{1d7a8}',  &['\u{3a9}']), ('\u{1d7a9}',  &['\u{2207}']), ('\u{1d7aa}',  &['\u{61}']),
        ('\u{1d7ab}',  &['\u{df}']), ('\u{1d7ac}',  &['\u{79}']), ('\u{1d7ad}',  &['\u{1e9f}']),
        ('\u{1d7ae}',  &['\u{a793}']), ('\u{1d7af}',  &['\u{3b6}']), ('\u{1d7b0}',  &['\u{6e}',
         '\u{329}']), ('\u{1d7b1}',  &['\u{4f}',  '\u{335}']), ('\u{1d7b2}',  &['\u{69}']),
        ('\u{1d7b3}',  &['\u{138}']), ('\u{1d7b4}',  &['\u{3bb}']), ('\u{1d7b5}',  &['\u{3bc}']),
        ('\u{1d7b6}',  &['\u{76}']), ('\u{1d7b7}',  &['\u{3be}']), ('\u{1d7b8}',  &['\u{6f}']),
        ('\u{1d7b9}',  &['\u{3c0}']), ('\u{1d7ba}',  &['\u{70}']), ('\u{1d7bb}',  &['\u{3c2}']),
        ('\u{1d7bc}',  &['\u{6f}']), ('\u{1d7bd}',  &['\u{1d1b}']), ('\u{1d7be}',  &['\u{75}']),
        ('\u{1d7bf}',  &['\u{278}']), ('\u{1d7c0}',  &['\u{3c7}']), ('\u{1d7c1}',  &['\u{3c8}']),
        ('\u{1d7c2}',  &['\u{3c9}']), ('\u{1d7c3}',  &['\u{2202}']), ('\u{1d7c4}',  &['\u{a793}']),
        ('\u{1d7c5}',  &['\u{4f}',  '\u{335}']), ('\u{1d7c6}',  &['\u{138}']), ('\u{1d7c7}',
         &['\u{278}']), ('\u{1d7c8}',  &['\u{70}']), ('\u{1d7c9}',  &['\u{3c0}']), ('\u{1d7ca}',
         &['\u{46}']), ('\u{1d7cb}',  &['\u{3dd}']), ('\u{1d7ce}',  &['\u{4f}']), ('\u{1d7cf}',
         &['\u{6c}']), ('\u{1d7d0}',  &['\u{32}']), ('\u{1d7d1}',  &['\u{33}']), ('\u{1d7d2}',
         &['\u{34}']), ('\u{1d7d3}',  &['\u{35}']), ('\u{1d7d4}',  &['\u{36}']), ('\u{1d7d5}',
         &['\u{37}']), ('\u{1d7d6}',  &['\u{38}']), ('\u{1d7d7}',  &['\u{39}']), ('\u{1d7d8}',
         &['\u{4f}']), ('\u{1d7d9}',  &['\u{6c}']), ('\u{1d7da}',  &['\u{32}']), ('\u{1d7db}',
         &['\u{33}']), ('\u{1d7dc}',  &['\u{34}']), ('\u{1d7dd}',  &['\u{35}']), ('\u{1d7de}',
         &['\u{36}']), ('\u{1d7df}',  &['\u{37}']), ('\u{1d7e0}',  &['\u{38}']), ('\u{1d7e1}',
         &['\u{39}']), ('\u{1d7e2}',  &['\u{4f}']), ('\u{1d7e3}',  &['\u{6c}']), ('\u{1d7e4}',
         &['\u{32}']), ('\u{1d7e5}',  &['\u{33}']), ('\u{1d7e6}',  &['\u{34}']), ('\u{1d7e7}',
         &['\u{35}']), ('\u{1d7e8}',  &['\u{36}']), ('\u{1d7e9}',  &['\u{37}']), ('\u{1d7ea}',
         &['\u{38}']), ('\u{1d7eb}',  &['\u{39}']), ('\u{1d7ec}',  &['\u{4f}']), ('\u{1d7ed}',
         &['\u{6c}']), ('\u{1d7ee}',  &['\u{32}']), ('\u{1d7ef}',  &['\u{33}']), ('\u{1d7f0}',
         &['\u{34}']), ('\u{1d7f1}',  &['\u{35}']), ('\u{1d7f2}',  &['\u{36}']), ('\u{1d7f3}',
         &['\u{37}']), ('\u{1d7f4}',  &['\u{38}']), ('\u{1d7f5}',  &['\u{39}']), ('\u{1d7f6}',
         &['\u{4f}']), ('\u{1d7f7}',  &['\u{6c}']), ('\u{1d7f8}',  &['\u{32}']), ('\u{1d7f9}',
         &['\u{33}']), ('\u{1d7fa}',  &['\u{34}']), ('\u{1d7fb}',  &['\u{35}']), ('\u{1d7fc}',
         &['\u{36}']), ('\u{1d7fd}',  &['\u{37}']), ('\u{1d7fe}',  &['\u{38}']), ('\u{1d7ff}',
         &['\u{39}']), ('\u{1e8c7}',  &['\u{6c}']), ('\u{1e8c8}',  &['\u{2220}']), ('\u{1e8c9}',
         &['\u{663}']), ('\u{1e8cb}',  &['\u{38}']), ('\u{1e8cc}',  &['\u{2202}']), ('\u{1e8cd}',
         &['\u{2202}',  '\u{335}']), ('\u{1ee00}',  &['\u{6c}']), ('\u{1ee01}',  &['\u{628}']),
        ('\u{1ee02}',  &['\u{62c}']), ('\u{1ee03}',  &['\u{62f}']), ('\u{1ee05}',  &['\u{648}']),
        ('\u{1ee06}',  &['\u{632}']), ('\u{1ee07}',  &['\u{62d}']), ('\u{1ee08}',  &['\u{637}']),
        ('\u{1ee09}',  &['\u{649}']), ('\u{1ee0a}',  &['\u{643}']), ('\u{1ee0b}',  &['\u{644}']),
        ('\u{1ee0c}',  &['\u{645}']), ('\u{1ee0d}',  &['\u{646}']), ('\u{1ee0e}',  &['\u{633}']),
        ('\u{1ee0f}',  &['\u{639}']), ('\u{1ee10}',  &['\u{641}']), ('\u{1ee11}',  &['\u{635}']),
        ('\u{1ee12}',  &['\u{642}']), ('\u{1ee13}',  &['\u{631}']), ('\u{1ee14}',  &['\u{633}',
         '\u{6db}']), ('\u{1ee15}',  &['\u{62a}']), ('\u{1ee16}',  &['\u{649}',  '\u{6db}']),
        ('\u{1ee17}',  &['\u{62e}']), ('\u{1ee18}',  &['\u{630}']), ('\u{1ee19}',  &['\u{636}']),
        ('\u{1ee1a}',  &['\u{638}']), ('\u{1ee1b}',  &['\u{63a}']), ('\u{1ee1c}',  &['\u{649}']),
        ('\u{1ee1d}',  &['\u{649}']), ('\u{1ee1e}',  &['\u{6a1}']), ('\u{1ee1f}',  &['\u{6a1}']),
        ('\u{1ee21}',  &['\u{628}']), ('\u{1ee22}',  &['\u{62c}']), ('\u{1ee24}',  &['\u{6f}']),
        ('\u{1ee27}',  &['\u{62d}']), ('\u{1ee29}',  &['\u{649}']), ('\u{1ee2a}',  &['\u{643}']),
        ('\u{1ee2b}',  &['\u{644}']), ('\u{1ee2c}',  &['\u{645}']), ('\u{1ee2d}',  &['\u{646}']),
        ('\u{1ee2e}',  &['\u{633}']), ('\u{1ee2f}',  &['\u{639}']), ('\u{1ee30}',  &['\u{641}']),
        ('\u{1ee31}',  &['\u{635}']), ('\u{1ee32}',  &['\u{642}']), ('\u{1ee34}',  &['\u{633}',
         '\u{6db}']), ('\u{1ee35}',  &['\u{62a}']), ('\u{1ee36}',  &['\u{649}',  '\u{6db}']),
        ('\u{1ee37}',  &['\u{62e}']), ('\u{1ee39}',  &['\u{636}']), ('\u{1ee3b}',  &['\u{63a}']),
        ('\u{1ee42}',  &['\u{62c}']), ('\u{1ee47}',  &['\u{62d}']), ('\u{1ee49}',  &['\u{649}']),
        ('\u{1ee4b}',  &['\u{644}']), ('\u{1ee4d}',  &['\u{646}']), ('\u{1ee4e}',  &['\u{633}']),
        ('\u{1ee4f}',  &['\u{639}']), ('\u{1ee51}',  &['\u{635}']), ('\u{1ee52}',  &['\u{642}']),
        ('\u{1ee54}',  &['\u{633}',  '\u{6db}']), ('\u{1ee57}',  &['\u{62e}']), ('\u{1ee59}',
         &['\u{636}']), ('\u{1ee5b}',  &['\u{63a}']), ('\u{1ee5d}',  &['\u{649}']), ('\u{1ee5f}',
         &['\u{6a1}']), ('\u{1ee61}',  &['\u{628}']), ('\u{1ee62}',  &['\u{62c}']), ('\u{1ee64}',
         &['\u{6f}']), ('\u{1ee67}',  &['\u{62d}']), ('\u{1ee68}',  &['\u{637}']), ('\u{1ee69}',
         &['\u{649}']), ('\u{1ee6a}',  &['\u{643}']), ('\u{1ee6c}',  &['\u{645}']), ('\u{1ee6d}',
         &['\u{646}']), ('\u{1ee6e}',  &['\u{633}']), ('\u{1ee6f}',  &['\u{639}']), ('\u{1ee70}',
         &['\u{641}']), ('\u{1ee71}',  &['\u{635}']), ('\u{1ee72}',  &['\u{642}']), ('\u{1ee74}',
         &['\u{633}',  '\u{6db}']), ('\u{1ee75}',  &['\u{62a}']), ('\u{1ee76}',  &['\u{649}',
         '\u{6db}']), ('\u{1ee77}',  &['\u{62e}']), ('\u{1ee79}',  &['\u{636}']), ('\u{1ee7a}',
         &['\u{638}']), ('\u{1ee7b}',  &['\u{63a}']), ('\u{1ee7c}',  &['\u{649}']), ('\u{1ee7e}',
         &['\u{6a1}']), ('\u{1ee80}',  &['\u{6c}']), ('\u{1ee81}',  &['\u{628}']), ('\u{1ee82}',
         &['\u{62c}']), ('\u{1ee83}',  &['\u{62f}']), ('\u{1ee84}',  &['\u{6f}']), ('\u{1ee85}',
         &['\u{648}']), ('\u{1ee86}',  &['\u{632}']), ('\u{1ee87}',  &['\u{62d}']), ('\u{1ee88}',
         &['\u{637}']), ('\u{1ee89}',  &['\u{649}']), ('\u{1ee8b}',  &['\u{644}']), ('\u{1ee8c}',
         &['\u{645}']), ('\u{1ee8d}',  &['\u{646}']), ('\u{1ee8e}',  &['\u{633}']), ('\u{1ee8f}',
         &['\u{639}']), ('\u{1ee90}',  &['\u{641}']), ('\u{1ee91}',  &['\u{635}']), ('\u{1ee92}',
         &['\u{642}']), ('\u{1ee93}',  &['\u{631}']), ('\u{1ee94}',  &['\u{633}',  '\u{6db}']),
        ('\u{1ee95}',  &['\u{62a}']), ('\u{1ee96}',  &['\u{649}',  '\u{6db}']), ('\u{1ee97}',
         &['\u{62e}']), ('\u{1ee98}',  &['\u{630}']), ('\u{1ee99}',  &['\u{636}']), ('\u{1ee9a}',
         &['\u{638}']), ('\u{1ee9b}',  &['\u{63a}']), ('\u{1eea1}',  &['\u{628}']), ('\u{1eea2}',
         &['\u{62c}']), ('\u{1eea3}',  &['\u{62f}']), ('\u{1eea5}',  &['\u{648}']), ('\u{1eea6}',
         &['\u{632}']), ('\u{1eea7}',  &['\u{62d}']), ('\u{1eea8}',  &['\u{637}']), ('\u{1eea9}',
         &['\u{649}']), ('\u{1eeab}',  &['\u{644}']), ('\u{1eeac}',  &['\u{645}']), ('\u{1eead}',
         &['\u{646}']), ('\u{1eeae}',  &['\u{633}']), ('\u{1eeaf}',  &['\u{639}']), ('\u{1eeb0}',
         &['\u{641}']), ('\u{1eeb1}',  &['\u{635}']), ('\u{1eeb2}',  &['\u{642}']), ('\u{1eeb3}',
         &['\u{631}']), ('\u{1eeb4}',  &['\u{633}',  '\u{6db}']), ('\u{1eeb5}',  &['\u{62a}']),
        ('\u{1eeb6}',  &['\u{649}',  '\u{6db}']), ('\u{1eeb7}',  &['\u{62e}']), ('\u{1eeb8}',
         &['\u{630}']), ('\u{1eeb9}',  &['\u{636}']), ('\u{1eeba}',  &['\u{638}']), ('\u{1eebb}',
         &['\u{63a}']), ('\u{1f100}',  &['\u{4f}',  '\u{2e}']), ('\u{1f101}',  &['\u{4f}',
         '\u{2c}']), ('\u{1f102}',  &['\u{6c}',  '\u{2c}']), ('\u{1f103}',  &['\u{32}',  '\u{2c}']),
        ('\u{1f104}',  &['\u{33}',  '\u{2c}']), ('\u{1f105}',  &['\u{34}',  '\u{2c}']),
        ('\u{1f106}',  &['\u{35}',  '\u{2c}']), ('\u{1f107}',  &['\u{36}',  '\u{2c}']),
        ('\u{1f108}',  &['\u{37}',  '\u{2c}']), ('\u{1f109}',  &['\u{38}',  '\u{2c}']),
        ('\u{1f10a}',  &['\u{39}',  '\u{2c}']), ('\u{1f10f}',  &['\u{24}',  '\u{20e0}']),
        ('\u{1f110}',  &['\u{28}',  '\u{41}',  '\u{29}']), ('\u{1f111}',  &['\u{28}',  '\u{42}',
         '\u{29}']), ('\u{1f112}',  &['\u{28}',  '\u{43}',  '\u{29}']), ('\u{1f113}',  &['\u{28}',
         '\u{44}',  '\u{29}']), ('\u{1f114}',  &['\u{28}',  '\u{45}',  '\u{29}']), ('\u{1f115}',
         &['\u{28}',  '\u{46}',  '\u{29}']), ('\u{1f116}',  &['\u{28}',  '\u{47}',  '\u{29}']),
        ('\u{1f117}',  &['\u{28}',  '\u{48}',  '\u{29}']), ('\u{1f118}',  &['\u{28}',  '\u{6c}',
         '\u{29}']), ('\u{1f119}',  &['\u{28}',  '\u{4a}',  '\u{29}']), ('\u{1f11a}',  &['\u{28}',
         '\u{4b}',  '\u{29}']), ('\u{1f11b}',  &['\u{28}',  '\u{4c}',  '\u{29}']), ('\u{1f11c}',
         &['\u{28}',  '\u{4d}',  '\u{29}']), ('\u{1f11d}',  &['\u{28}',  '\u{4e}',  '\u{29}']),
        ('\u{1f11e}',  &['\u{28}',  '\u{4f}',  '\u{29}']), ('\u{1f11f}',  &['\u{28}',  '\u{50}',
         '\u{29}']), ('\u{1f120}',  &['\u{28}',  '\u{51}',  '\u{29}']), ('\u{1f121}',  &['\u{28}',
         '\u{52}',  '\u{29}']), ('\u{1f122}',  &['\u{28}',  '\u{53}',  '\u{29}']), ('\u{1f123}',
         &['\u{28}',  '\u{54}',  '\u{29}']), ('\u{1f124}',  &['\u{28}',  '\u{55}',  '\u{29}']),
        ('\u{1f125}',  &['\u{28}',  '\u{56}',  '\u{29}']), ('\u{1f126}',  &['\u{28}',  '\u{57}',
         '\u{29}']), ('\u{1f127}',  &['\u{28}',  '\u{58}',  '\u{29}']), ('\u{1f128}',  &['\u{28}',
         '\u{59}',  '\u{29}']), ('\u{1f129}',  &['\u{28}',  '\u{5a}',  '\u{29}']), ('\u{1f12a}',
         &['\u{28}',  '\u{53}',  '\u{29}']), ('\u{1f16d}',  &['\u{33c4}',  '\u{9}',  '\u{20dd}']),
        ('\u{1f16e}',  &['\u{43}',  '\u{20e0}']), ('\u{1f240}',  &['\u{28}',  '\u{672c}',
         '\u{29}']), ('\u{1f241}',  &['\u{28}',  '\u{4e09}',  '\u{29}']), ('\u{1f242}',  &['\u{28}',
         '\u{4e8c}',  '\u{29}']), ('\u{1f243}',  &['\u{28}',  '\u{5b89}',  '\u{29}']), ('\u{1f244}',
         &['\u{28}',  '\u{70b9}',  '\u{29}']), ('\u{1f245}',  &['\u{28}',  '\u{6253}',  '\u{29}']),
        ('\u{1f246}',  &['\u{28}',  '\u{76d7}',  '\u{29}']), ('\u{1f247}',  &['\u{28}',  '\u{52dd}',
         '\u{29}']), ('\u{1f248}',  &['\u{28}',  '\u{6557}',  '\u{29}']), ('\u{1f312}',
         &['\u{263d}']), ('\u{1f318}',  &['\u{263e}']), ('\u{1f319}',  &['\u{263d}']), ('\u{1f700}',
         &['\u{51}',  '\u{45}']), ('\u{1f701}',  &['\u{a658}']), ('\u{1f702}',  &['\u{394}']),
        ('\u{1f704}',  &['\u{102bc}']), ('\u{1f707}',  &['\u{41}',  '\u{52}']), ('\u{1f708}',
         &['\u{56}',  '\u{1de4}']), ('\u{1f70a}',  &['\u{2629}']), ('\u{1f714}',  &['\u{4f}',
         '\u{335}']), ('\u{1f728}',  &['\u{102a8}']), ('\u{1f73a}',  &['\u{29df}']), ('\u{1f74c}',
         &['\u{43}']), ('\u{1f754}',  &['\u{16dc}']), ('\u{1f755}',  &['\u{22a1}']), ('\u{1f75c}',
         &['\u{73}',  '\u{73}',  '\u{73}']), ('\u{1f75e}',  &['\u{224f}']), ('\u{1f768}',
         &['\u{54}']), ('\u{1f76b}',  &['\u{4d}',  '\u{42}']), ('\u{1f76c}',  &['\u{56}',
         '\u{42}']), ('\u{1f771}',  &['\u{22a0}']), ('\u{1fbf0}',  &['\u{4f}']), ('\u{1fbf1}',
         &['\u{6c}']), ('\u{1fbf2}',  &['\u{32}']), ('\u{1fbf3}',  &['\u{33}']), ('\u{1fbf4}',
         &['\u{34}']), ('\u{1fbf5}',  &['\u{35}']), ('\u{1fbf6}',  &['\u{36}']), ('\u{1fbf7}',
         &['\u{37}']), ('\u{1fbf8}',  &['\u{38}']), ('\u{1fbf9}',  &['\u{39}']), ('\u{21fe8}',
         &['\u{276c}']), ('\u{2f800}',  &['\u{4e3d}']), ('\u{2f801}',  &['\u{4e38}']), ('\u{2f802}',
         &['\u{4e41}']), ('\u{2f803}',  &['\u{20122}']), ('\u{2f804}',  &['\u{4f60}']),
        ('\u{2f805}',  &['\u{4fae}']), ('\u{2f806}',  &['\u{4fbb}']), ('\u{2f807}',  &['\u{4f75}']),
        ('\u{2f808}',  &['\u{507a}']), ('\u{2f809}',  &['\u{5099}']), ('\u{2f80a}',  &['\u{50e7}']),
        ('\u{2f80b}',  &['\u{50cf}']), ('\u{2f80c}',  &['\u{349e}']), ('\u{2f80d}',
         &['\u{2063a}']), ('\u{2f80e}',  &['\u{514d}']), ('\u{2f80f}',  &['\u{5154}']),
        ('\u{2f810}',  &['\u{5164}']), ('\u{2f811}',  &['\u{5177}']), ('\u{2f812}',
         &['\u{2051c}']), ('\u{2f813}',  &['\u{34b9}']), ('\u{2f814}',  &['\u{5167}']),
        ('\u{2f815}',  &['\u{518d}']), ('\u{2f816}',  &['\u{2054b}']), ('\u{2f817}',
         &['\u{5197}']), ('\u{2f818}',  &['\u{51a4}']), ('\u{2f819}',  &['\u{4ecc}']), ('\u{2f81a}',
         &['\u{51ac}']), ('\u{2f81b}',  &['\u{51b5}']), ('\u{2f81c}',  &['\u{291df}']),
        ('\u{2f81d}',  &['\u{51f5}']), ('\u{2f81e}',  &['\u{5203}']), ('\u{2f81f}',  &['\u{34df}']),
        ('\u{2f820}',  &['\u{523b}']), ('\u{2f821}',  &['\u{5246}']), ('\u{2f822}',  &['\u{5272}']),
        ('\u{2f823}',  &['\u{5277}']), ('\u{2f824}',  &['\u{3515}']), ('\u{2f825}',  &['\u{52c7}']),
        ('\u{2f826}',  &['\u{52c9}']), ('\u{2f827}',  &['\u{52e4}']), ('\u{2f828}',  &['\u{52fa}']),
        ('\u{2f829}',  &['\u{5305}']), ('\u{2f82a}',  &['\u{5306}']), ('\u{2f82b}',  &['\u{5317}']),
        ('\u{2f82c}',  &['\u{5349}']), ('\u{2f82d}',  &['\u{5351}']), ('\u{2f82e}',  &['\u{535a}']),
        ('\u{2f82f}',  &['\u{5373}']), ('\u{2f830}',  &['\u{537d}']), ('\u{2f831}',  &['\u{537f}']),
        ('\u{2f832}',  &['\u{537f}']), ('\u{2f833}',  &['\u{537f}']), ('\u{2f834}',
         &['\u{20a2c}']), ('\u{2f835}',  &['\u{7070}']), ('\u{2f836}',  &['\u{53ca}']),
        ('\u{2f837}',  &['\u{53df}']), ('\u{2f838}',  &['\u{20b63}']), ('\u{2f839}',
         &['\u{53eb}']), ('\u{2f83a}',  &['\u{53f1}']), ('\u{2f83b}',  &['\u{5406}']), ('\u{2f83c}',
         &['\u{549e}']), ('\u{2f83d}',  &['\u{5438}']), ('\u{2f83e}',  &['\u{5448}']), ('\u{2f83f}',
         &['\u{5468}']), ('\u{2f840}',  &['\u{54a2}']), ('\u{2f841}',  &['\u{54f6}']), ('\u{2f842}',
         &['\u{5510}']), ('\u{2f843}',  &['\u{5553}']), ('\u{2f844}',  &['\u{5563}']), ('\u{2f845}',
         &['\u{5584}']), ('\u{2f846}',  &['\u{5584}']), ('\u{2f847}',  &['\u{5599}']), ('\u{2f848}',
         &['\u{55ab}']), ('\u{2f849}',  &['\u{55b3}']), ('\u{2f84a}',  &['\u{55c2}']), ('\u{2f84b}',
         &['\u{5716}']), ('\u{2f84c}',  &['\u{5606}']), ('\u{2f84d}',  &['\u{5717}']), ('\u{2f84e}',
         &['\u{5651}']), ('\u{2f84f}',  &['\u{5674}']), ('\u{2f850}',  &['\u{5207}']), ('\u{2f851}',
         &['\u{58ee}']), ('\u{2f852}',  &['\u{57ce}']), ('\u{2f853}',  &['\u{57f4}']), ('\u{2f854}',
         &['\u{580d}']), ('\u{2f855}',  &['\u{578b}']), ('\u{2f856}',  &['\u{5832}']), ('\u{2f857}',
         &['\u{5831}']), ('\u{2f858}',  &['\u{58ac}']), ('\u{2f859}',  &['\u{214e4}']),
        ('\u{2f85a}',  &['\u{58f2}']), ('\u{2f85b}',  &['\u{58f7}']), ('\u{2f85c}',  &['\u{5906}']),
        ('\u{2f85d}',  &['\u{591a}']), ('\u{2f85e}',  &['\u{5922}']), ('\u{2f85f}',  &['\u{5962}']),
        ('\u{2f860}',  &['\u{216a8}']), ('\u{2f861}',  &['\u{216ea}']), ('\u{2f862}',
         &['\u{59ec}']), ('\u{2f863}',  &['\u{5a1b}']), ('\u{2f864}',  &['\u{5a27}']), ('\u{2f865}',
         &['\u{59d8}']), ('\u{2f866}',  &['\u{5a66}']), ('\u{2f867}',  &['\u{36ee}']), ('\u{2f868}',
         &['\u{36fc}']), ('\u{2f869}',  &['\u{5b08}']), ('\u{2f86a}',  &['\u{5b3e}']), ('\u{2f86b}',
         &['\u{5b3e}']), ('\u{2f86c}',  &['\u{219c8}']), ('\u{2f86d}',  &['\u{5bc3}']),
        ('\u{2f86e}',  &['\u{5bd8}']), ('\u{2f86f}',  &['\u{5be7}']), ('\u{2f870}',  &['\u{5bf3}']),
        ('\u{2f871}',  &['\u{21b18}']), ('\u{2f872}',  &['\u{5bff}']), ('\u{2f873}',
         &['\u{5c06}']), ('\u{2f874}',  &['\u{5f53}']), ('\u{2f875}',  &['\u{5c22}']), ('\u{2f876}',
         &['\u{3781}']), ('\u{2f877}',  &['\u{5c60}']), ('\u{2f878}',  &['\u{5c6e}']), ('\u{2f879}',
         &['\u{5cc0}']), ('\u{2f87a}',  &['\u{5c8d}']), ('\u{2f87b}',  &['\u{21de4}']),
        ('\u{2f87c}',  &['\u{5d43}']), ('\u{2f87d}',  &['\u{21de6}']), ('\u{2f87e}',
         &['\u{5d6e}']), ('\u{2f87f}',  &['\u{5d6b}']), ('\u{2f880}',  &['\u{5d7c}']), ('\u{2f881}',
         &['\u{5de1}']), ('\u{2f882}',  &['\u{5de2}']), ('\u{2f883}',  &['\u{382f}']), ('\u{2f884}',
         &['\u{5dfd}']), ('\u{2f885}',  &['\u{5e28}']), ('\u{2f886}',  &['\u{5e3d}']), ('\u{2f887}',
         &['\u{5e69}']), ('\u{2f888}',  &['\u{3862}']), ('\u{2f889}',  &['\u{22183}']),
        ('\u{2f88a}',  &['\u{387c}']), ('\u{2f88b}',  &['\u{5eb0}']), ('\u{2f88c}',  &['\u{5eb3}']),
        ('\u{2f88d}',  &['\u{5eb6}']), ('\u{2f88e}',  &['\u{5eca}']), ('\u{2f88f}',
         &['\u{2a392}']), ('\u{2f890}',  &['\u{5efe}']), ('\u{2f891}',  &['\u{22331}']),
        ('\u{2f892}',  &['\u{22331}']), ('\u{2f893}',  &['\u{8201}']), ('\u{2f894}',
         &['\u{5f22}']), ('\u{2f895}',  &['\u{5f22}']), ('\u{2f896}',  &['\u{38c7}']), ('\u{2f897}',
         &['\u{232b8}']), ('\u{2f898}',  &['\u{261da}']), ('\u{2f899}',  &['\u{5f62}']),
        ('\u{2f89a}',  &['\u{5f6b}']), ('\u{2f89b}',  &['\u{38e3}']), ('\u{2f89c}',  &['\u{5f9a}']),
        ('\u{2f89d}',  &['\u{5fcd}']), ('\u{2f89e}',  &['\u{5fd7}']), ('\u{2f89f}',  &['\u{5ff9}']),
        ('\u{2f8a0}',  &['\u{6081}']), ('\u{2f8a1}',  &['\u{393a}']), ('\u{2f8a2}',  &['\u{391c}']),
        ('\u{2f8a3}',  &['\u{6094}']), ('\u{2f8a4}',  &['\u{226d4}']), ('\u{2f8a5}',
         &['\u{60c7}']), ('\u{2f8a6}',  &['\u{6148}']), ('\u{2f8a7}',  &['\u{614c}']), ('\u{2f8a8}',
         &['\u{614e}']), ('\u{2f8a9}',  &['\u{614c}']), ('\u{2f8aa}',  &['\u{617a}']), ('\u{2f8ab}',
         &['\u{618e}']), ('\u{2f8ac}',  &['\u{61b2}']), ('\u{2f8ad}',  &['\u{61a4}']), ('\u{2f8ae}',
         &['\u{61af}']), ('\u{2f8af}',  &['\u{61de}']), ('\u{2f8b0}',  &['\u{61f2}']), ('\u{2f8b1}',
         &['\u{61f6}']), ('\u{2f8b2}',  &['\u{6210}']), ('\u{2f8b3}',  &['\u{621b}']), ('\u{2f8b4}',
         &['\u{625d}']), ('\u{2f8b5}',  &['\u{62b1}']), ('\u{2f8b6}',  &['\u{62d4}']), ('\u{2f8b7}',
         &['\u{6350}']), ('\u{2f8b8}',  &['\u{22b0c}']), ('\u{2f8b9}',  &['\u{633d}']),
        ('\u{2f8ba}',  &['\u{62fc}']), ('\u{2f8bb}',  &['\u{6368}']), ('\u{2f8bc}',  &['\u{6383}']),
        ('\u{2f8bd}',  &['\u{63e4}']), ('\u{2f8be}',  &['\u{22bf1}']), ('\u{2f8bf}',
         &['\u{6422}']), ('\u{2f8c0}',  &['\u{63c5}']), ('\u{2f8c1}',  &['\u{63a9}']), ('\u{2f8c2}',
         &['\u{3a2e}']), ('\u{2f8c3}',  &['\u{6469}']), ('\u{2f8c4}',  &['\u{647e}']), ('\u{2f8c5}',
         &['\u{649d}']), ('\u{2f8c6}',  &['\u{6477}']), ('\u{2f8c7}',  &['\u{3a6c}']), ('\u{2f8c8}',
         &['\u{654f}']), ('\u{2f8c9}',  &['\u{656c}']), ('\u{2f8ca}',  &['\u{2300a}']),
        ('\u{2f8cb}',  &['\u{65e3}']), ('\u{2f8cc}',  &['\u{66f8}']), ('\u{2f8cd}',  &['\u{6649}']),
        ('\u{2f8ce}',  &['\u{3b19}']), ('\u{2f8cf}',  &['\u{6691}']), ('\u{2f8d0}',  &['\u{3b08}']),
        ('\u{2f8d1}',  &['\u{3ae4}']), ('\u{2f8d2}',  &['\u{5192}']), ('\u{2f8d3}',  &['\u{5195}']),
        ('\u{2f8d4}',  &['\u{6700}']), ('\u{2f8d5}',  &['\u{669c}']), ('\u{2f8d6}',  &['\u{80ad}']),
        ('\u{2f8d7}',  &['\u{43d9}']), ('\u{2f8d8}',  &['\u{6717}']), ('\u{2f8d9}',  &['\u{671b}']),
        ('\u{2f8da}',  &['\u{6721}']), ('\u{2f8db}',  &['\u{675e}']), ('\u{2f8dc}',  &['\u{6753}']),
        ('\u{2f8dd}',  &['\u{233c3}']), ('\u{2f8de}',  &['\u{3b49}']), ('\u{2f8df}',
         &['\u{67fa}']), ('\u{2f8e0}',  &['\u{6785}']), ('\u{2f8e1}',  &['\u{6852}']), ('\u{2f8e2}',
         &['\u{6885}']), ('\u{2f8e3}',  &['\u{2346d}']), ('\u{2f8e4}',  &['\u{688e}']),
        ('\u{2f8e5}',  &['\u{681f}']), ('\u{2f8e6}',  &['\u{6914}']), ('\u{2f8e7}',  &['\u{3b9d}']),
        ('\u{2f8e8}',  &['\u{6942}']), ('\u{2f8e9}',  &['\u{69a3}']), ('\u{2f8ea}',  &['\u{69ea}']),
        ('\u{2f8eb}',  &['\u{6aa8}']), ('\u{2f8ec}',  &['\u{236a3}']), ('\u{2f8ed}',
         &['\u{6adb}']), ('\u{2f8ee}',  &['\u{3c18}']), ('\u{2f8ef}',  &['\u{6b21}']), ('\u{2f8f0}',
         &['\u{238a7}']), ('\u{2f8f1}',  &['\u{6b54}']), ('\u{2f8f2}',  &['\u{3c4e}']),
        ('\u{2f8f3}',  &['\u{6b72}']), ('\u{2f8f4}',  &['\u{6b9f}']), ('\u{2f8f5}',  &['\u{6bba}']),
        ('\u{2f8f6}',  &['\u{6bbb}']), ('\u{2f8f7}',  &['\u{23a8d}']), ('\u{2f8f8}',
         &['\u{21d0b}']), ('\u{2f8f9}',  &['\u{23afa}']), ('\u{2f8fa}',  &['\u{6c4e}']),
        ('\u{2f8fb}',  &['\u{23cbc}']), ('\u{2f8fc}',  &['\u{6cbf}']), ('\u{2f8fd}',
         &['\u{6ccd}']), ('\u{2f8fe}',  &['\u{6c67}']), ('\u{2f8ff}',  &['\u{6d16}']), ('\u{2f900}',
         &['\u{6d3e}']), ('\u{2f901}',  &['\u{6d77}']), ('\u{2f902}',  &['\u{6d41}']), ('\u{2f903}',
         &['\u{6d69}']), ('\u{2f904}',  &['\u{6d78}']), ('\u{2f905}',  &['\u{6d85}']), ('\u{2f906}',
         &['\u{23d1e}']), ('\u{2f907}',  &['\u{6d34}']), ('\u{2f908}',  &['\u{6e2f}']),
        ('\u{2f909}',  &['\u{6e6e}']), ('\u{2f90a}',  &['\u{3d33}']), ('\u{2f90b}',  &['\u{6ecb}']),
        ('\u{2f90c}',  &['\u{6ec7}']), ('\u{2f90d}',  &['\u{23ed1}']), ('\u{2f90e}',
         &['\u{6df9}']), ('\u{2f90f}',  &['\u{6f6e}']), ('\u{2f910}',  &['\u{23f5e}']),
        ('\u{2f911}',  &['\u{23f8e}']), ('\u{2f912}',  &['\u{6fc6}']), ('\u{2f913}',
         &['\u{7039}']), ('\u{2f914}',  &['\u{701e}']), ('\u{2f915}',  &['\u{701b}']), ('\u{2f916}',
         &['\u{3d96}']), ('\u{2f917}',  &['\u{704a}']), ('\u{2f918}',  &['\u{707d}']), ('\u{2f919}',
         &['\u{7077}']), ('\u{2f91a}',  &['\u{70ad}']), ('\u{2f91b}',  &['\u{20525}']),
        ('\u{2f91c}',  &['\u{7145}']), ('\u{2f91d}',  &['\u{24263}']), ('\u{2f91e}',
         &['\u{719c}']), ('\u{2f91f}',  &['\u{243ab}']), ('\u{2f920}',  &['\u{7228}']),
        ('\u{2f921}',  &['\u{7235}']), ('\u{2f922}',  &['\u{7250}']), ('\u{2f923}',
         &['\u{24608}']), ('\u{2f924}',  &['\u{7280}']), ('\u{2f925}',  &['\u{7295}']),
        ('\u{2f926}',  &['\u{24735}']), ('\u{2f927}',  &['\u{24814}']), ('\u{2f928}',
         &['\u{737a}']), ('\u{2f929}',  &['\u{738b}']), ('\u{2f92a}',  &['\u{3eac}']), ('\u{2f92b}',
         &['\u{73a5}']), ('\u{2f92c}',  &['\u{3eb8}']), ('\u{2f92d}',  &['\u{3eb8}']), ('\u{2f92e}',
         &['\u{7447}']), ('\u{2f92f}',  &['\u{745c}']), ('\u{2f930}',  &['\u{7471}']), ('\u{2f931}',
         &['\u{7485}']), ('\u{2f932}',  &['\u{74ca}']), ('\u{2f933}',  &['\u{3f1b}']), ('\u{2f934}',
         &['\u{7524}']), ('\u{2f935}',  &['\u{24c36}']), ('\u{2f936}',  &['\u{753e}']),
        ('\u{2f937}',  &['\u{24c92}']), ('\u{2f938}',  &['\u{7570}']), ('\u{2f939}',
         &['\u{2219f}']), ('\u{2f93a}',  &['\u{7610}']), ('\u{2f93b}',  &['\u{24fa1}']),
        ('\u{2f93c}',  &['\u{24fb8}']), ('\u{2f93d}',  &['\u{25044}']), ('\u{2f93e}',
         &['\u{3ffc}']), ('\u{2f93f}',  &['\u{4008}']), ('\u{2f940}',  &['\u{76f4}']), ('\u{2f941}',
         &['\u{250f3}']), ('\u{2f942}',  &['\u{250f2}']), ('\u{2f943}',  &['\u{25119}']),
        ('\u{2f944}',  &['\u{25133}']), ('\u{2f945}',  &['\u{771e}']), ('\u{2f946}',
         &['\u{771f}']), ('\u{2f947}',  &['\u{771f}']), ('\u{2f948}',  &['\u{774a}']), ('\u{2f949}',
         &['\u{4039}']), ('\u{2f94a}',  &['\u{778b}']), ('\u{2f94b}',  &['\u{4046}']), ('\u{2f94c}',
         &['\u{4096}']), ('\u{2f94d}',  &['\u{2541d}']), ('\u{2f94e}',  &['\u{784e}']),
        ('\u{2f94f}',  &['\u{788c}']), ('\u{2f950}',  &['\u{78cc}']), ('\u{2f951}',  &['\u{40e3}']),
        ('\u{2f952}',  &['\u{25626}']), ('\u{2f953}',  &['\u{7956}']), ('\u{2f954}',
         &['\u{2569a}']), ('\u{2f955}',  &['\u{256c5}']), ('\u{2f956}',  &['\u{798f}']),
        ('\u{2f957}',  &['\u{79eb}']), ('\u{2f958}',  &['\u{412f}']), ('\u{2f959}',  &['\u{7a40}']),
        ('\u{2f95a}',  &['\u{7a4a}']), ('\u{2f95b}',  &['\u{7a4f}']), ('\u{2f95c}',
         &['\u{2597c}']), ('\u{2f95d}',  &['\u{25aa7}']), ('\u{2f95e}',  &['\u{25aa7}']),
        ('\u{2f95f}',  &['\u{7aee}']), ('\u{2f960}',  &['\u{4202}']), ('\u{2f961}',
         &['\u{25bab}']), ('\u{2f962}',  &['\u{7bc6}']), ('\u{2f963}',  &['\u{7bc9}']),
        ('\u{2f964}',  &['\u{4227}']), ('\u{2f965}',  &['\u{25c80}']), ('\u{2f966}',
         &['\u{7cd2}']), ('\u{2f967}',  &['\u{42a0}']), ('\u{2f968}',  &['\u{7ce8}']), ('\u{2f969}',
         &['\u{7ce3}']), ('\u{2f96a}',  &['\u{7d00}']), ('\u{2f96b}',  &['\u{25f86}']),
        ('\u{2f96c}',  &['\u{7d63}']), ('\u{2f96d}',  &['\u{4301}']), ('\u{2f96e}',  &['\u{7dc7}']),
        ('\u{2f96f}',  &['\u{7e02}']), ('\u{2f970}',  &['\u{7e45}']), ('\u{2f971}',  &['\u{4334}']),
        ('\u{2f972}',  &['\u{26228}']), ('\u{2f973}',  &['\u{26247}']), ('\u{2f974}',
         &['\u{4359}']), ('\u{2f975}',  &['\u{262d9}']), ('\u{2f976}',  &['\u{7f7a}']),
        ('\u{2f977}',  &['\u{2633e}']), ('\u{2f978}',  &['\u{7f95}']), ('\u{2f979}',
         &['\u{7ffa}']), ('\u{2f97a}',  &['\u{8005}']), ('\u{2f97b}',  &['\u{264da}']),
        ('\u{2f97c}',  &['\u{26523}']), ('\u{2f97d}',  &['\u{8060}']), ('\u{2f97e}',
         &['\u{265a8}']), ('\u{2f97f}',  &['\u{8070}']), ('\u{2f980}',  &['\u{2335f}']),
        ('\u{2f981}',  &['\u{43d5}']), ('\u{2f982}',  &['\u{80b2}']), ('\u{2f983}',  &['\u{8103}']),
        ('\u{2f984}',  &['\u{440b}']), ('\u{2f985}',  &['\u{813e}']), ('\u{2f986}',  &['\u{5ab5}']),
        ('\u{2f987}',  &['\u{267a7}']), ('\u{2f988}',  &['\u{267b5}']), ('\u{2f989}',
         &['\u{23393}']), ('\u{2f98a}',  &['\u{2339c}']), ('\u{2f98b}',  &['\u{8201}']),
        ('\u{2f98c}',  &['\u{8204}']), ('\u{2f98d}',  &['\u{8f9e}']), ('\u{2f98e}',  &['\u{446b}']),
        ('\u{2f98f}',  &['\u{8291}']), ('\u{2f990}',  &['\u{828b}']), ('\u{2f991}',  &['\u{829d}']),
        ('\u{2f992}',  &['\u{52b3}']), ('\u{2f993}',  &['\u{82b1}']), ('\u{2f994}',  &['\u{82b3}']),
        ('\u{2f995}',  &['\u{82bd}']), ('\u{2f996}',  &['\u{82e6}']), ('\u{2f997}',
         &['\u{26b3c}']), ('\u{2f998}',  &['\u{82e5}']), ('\u{2f999}',  &['\u{831d}']),
        ('\u{2f99a}',  &['\u{8363}']), ('\u{2f99b}',  &['\u{83ad}']), ('\u{2f99c}',  &['\u{8323}']),
        ('\u{2f99d}',  &['\u{83bd}']), ('\u{2f99e}',  &['\u{83e7}']), ('\u{2f99f}',  &['\u{8457}']),
        ('\u{2f9a0}',  &['\u{8353}']), ('\u{2f9a1}',  &['\u{83ca}']), ('\u{2f9a2}',  &['\u{83cc}']),
        ('\u{2f9a3}',  &['\u{83dc}']), ('\u{2f9a4}',  &['\u{26c36}']), ('\u{2f9a5}',
         &['\u{26d6b}']), ('\u{2f9a6}',  &['\u{26cd5}']), ('\u{2f9a7}',  &['\u{452b}']),
        ('\u{2f9a8}',  &['\u{84f1}']), ('\u{2f9a9}',  &['\u{84f3}']), ('\u{2f9aa}',  &['\u{8516}']),
        ('\u{2f9ab}',  &['\u{273ca}']), ('\u{2f9ac}',  &['\u{8564}']), ('\u{2f9ad}',
         &['\u{26f2c}']), ('\u{2f9ae}',  &['\u{455d}']), ('\u{2f9af}',  &['\u{4561}']),
        ('\u{2f9b0}',  &['\u{26fb1}']), ('\u{2f9b1}',  &['\u{270d2}']), ('\u{2f9b2}',
         &['\u{456b}']), ('\u{2f9b3}',  &['\u{8650}']), ('\u{2f9b4}',  &['\u{865c}']), ('\u{2f9b5}',
         &['\u{8667}']), ('\u{2f9b6}',  &['\u{8669}']), ('\u{2f9b7}',  &['\u{86a9}']), ('\u{2f9b8}',
         &['\u{8688}']), ('\u{2f9b9}',  &['\u{870e}']), ('\u{2f9ba}',  &['\u{86e2}']), ('\u{2f9bb}',
         &['\u{8779}']), ('\u{2f9bc}',  &['\u{8728}']), ('\u{2f9bd}',  &['\u{876b}']), ('\u{2f9be}',
         &['\u{8786}']), ('\u{2f9bf}',  &['\u{45d7}']), ('\u{2f9c0}',  &['\u{87e1}']), ('\u{2f9c1}',
         &['\u{8801}']), ('\u{2f9c2}',  &['\u{45f9}']), ('\u{2f9c3}',  &['\u{8860}']), ('\u{2f9c4}',
         &['\u{8863}']), ('\u{2f9c5}',  &['\u{27667}']), ('\u{2f9c6}',  &['\u{88d7}']),
        ('\u{2f9c7}',  &['\u{88de}']), ('\u{2f9c8}',  &['\u{4635}']), ('\u{2f9c9}',  &['\u{88fa}']),
        ('\u{2f9ca}',  &['\u{34bb}']), ('\u{2f9cb}',  &['\u{278ae}']), ('\u{2f9cc}',
         &['\u{27966}']), ('\u{2f9cd}',  &['\u{46be}']), ('\u{2f9ce}',  &['\u{46c7}']),
        ('\u{2f9cf}',  &['\u{8aa0}']), ('\u{2f9d0}',  &['\u{8aed}']), ('\u{2f9d1}',  &['\u{8b8a}']),
        ('\u{2f9d2}',  &['\u{8c55}']), ('\u{2f9d3}',  &['\u{27ca8}']), ('\u{2f9d4}',
         &['\u{8cab}']), ('\u{2f9d5}',  &['\u{8cc1}']), ('\u{2f9d6}',  &['\u{8d1b}']), ('\u{2f9d7}',
         &['\u{8d77}']), ('\u{2f9d8}',  &['\u{27f2f}']), ('\u{2f9d9}',  &['\u{20804}']),
        ('\u{2f9da}',  &['\u{8dcb}']), ('\u{2f9db}',  &['\u{8dbc}']), ('\u{2f9dc}',  &['\u{8df0}']),
        ('\u{2f9dd}',  &['\u{208de}']), ('\u{2f9de}',  &['\u{8ed4}']), ('\u{2f9df}',
         &['\u{8f38}']), ('\u{2f9e0}',  &['\u{285d2}']), ('\u{2f9e1}',  &['\u{285ed}']),
        ('\u{2f9e2}',  &['\u{9094}']), ('\u{2f9e3}',  &['\u{90f1}']), ('\u{2f9e4}',  &['\u{9111}']),
        ('\u{2f9e5}',  &['\u{2872e}']), ('\u{2f9e6}',  &['\u{911b}']), ('\u{2f9e7}',
         &['\u{9238}']), ('\u{2f9e8}',  &['\u{92d7}']), ('\u{2f9e9}',  &['\u{92d8}']), ('\u{2f9ea}',
         &['\u{927c}']), ('\u{2f9eb}',  &['\u{93f9}']), ('\u{2f9ec}',  &['\u{9415}']), ('\u{2f9ed}',
         &['\u{28bfa}']), ('\u{2f9ee}',  &['\u{958b}']), ('\u{2f9ef}',  &['\u{4995}']),
        ('\u{2f9f0}',  &['\u{95b7}']), ('\u{2f9f1}',  &['\u{28d77}']), ('\u{2f9f2}',
         &['\u{49e6}']), ('\u{2f9f3}',  &['\u{96c3}']), ('\u{2f9f4}',  &['\u{5db2}']), ('\u{2f9f5}',
         &['\u{9723}']), ('\u{2f9f6}',  &['\u{29145}']), ('\u{2f9f7}',  &['\u{2921a}']),
        ('\u{2f9f8}',  &['\u{4a6e}']), ('\u{2f9f9}',  &['\u{4a76}']), ('\u{2f9fa}',  &['\u{97e0}']),
        ('\u{2f9fb}',  &['\u{2940a}']), ('\u{2f9fc}',  &['\u{4ab2}']), ('\u{2f9fd}',
         &['\u{29496}']), ('\u{2f9fe}',  &['\u{980b}']), ('\u{2f9ff}',  &['\u{980b}']),
        ('\u{2fa00}',  &['\u{9829}']), ('\u{2fa01}',  &['\u{295b6}']), ('\u{2fa02}',
         &['\u{98e2}']), ('\u{2fa03}',  &['\u{4b33}']), ('\u{2fa04}',  &['\u{9929}']), ('\u{2fa05}',
         &['\u{99a7}']), ('\u{2fa06}',  &['\u{99c2}']), ('\u{2fa07}',  &['\u{99fe}']), ('\u{2fa08}',
         &['\u{4bce}']), ('\u{2fa09}',  &['\u{29b30}']), ('\u{2fa0a}',  &['\u{9b12}']),
        ('\u{2fa0b}',  &['\u{9c40}']), ('\u{2fa0c}',  &['\u{9cfd}']), ('\u{2fa0d}',  &['\u{4cce}']),
        ('\u{2fa0e}',  &['\u{4ced}']), ('\u{2fa0f}',  &['\u{9d67}']), ('\u{2fa10}',
         &['\u{2a0ce}']), ('\u{2fa11}',  &['\u{4cf8}']), ('\u{2fa12}',  &['\u{2a105}']),
        ('\u{2fa13}',  &['\u{2a20e}']), ('\u{2fa14}',  &['\u{2a291}']), ('\u{2fa15}',
         &['\u{9ebb}']), ('\u{2fa16}',  &['\u{4d56}']), ('\u{2fa17}',  &['\u{9ef9}']), ('\u{2fa18}',
         &['\u{9efe}']), ('\u{2fa19}',  &['\u{9f05}']), ('\u{2fa1a}',  &['\u{9f0f}']), ('\u{2fa1b}',
         &['\u{9f16}']), ('\u{2fa1c}',  &['\u{9f3b}']), ('\u{2fa1d}',  &['\u{2a600}'])
    ];

}

pub mod potential_mixed_script_confusable {
    #[inline]
    pub fn potential_mixed_script_confusable(c: char) -> bool {
        match c as usize {
            _ => super::util::bsearch_table(c, CONFUSABLES)
        }
    }
    const CONFUSABLES: &'static [char] = &[
        '\u{41}', '\u{42}', '\u{43}', '\u{45}', '\u{48}', '\u{49}', '\u{4a}', '\u{4b}', '\u{4d}',
        '\u{4e}', '\u{4f}', '\u{50}', '\u{53}', '\u{54}', '\u{55}', '\u{56}', '\u{57}', '\u{58}',
        '\u{59}', '\u{5a}', '\u{61}', '\u{62}', '\u{63}', '\u{65}', '\u{66}', '\u{67}', '\u{68}',
        '\u{69}', '\u{6a}', '\u{6c}', '\u{6e}', '\u{6f}', '\u{70}', '\u{71}', '\u{72}', '\u{73}',
        '\u{75}', '\u{76}', '\u{77}', '\u{78}', '\u{79}', '\u{c6}', '\u{c7}', '\u{df}', '\u{e6}',
        '\u{e7}', '\u{f6}', '\u{127}', '\u{131}', '\u{138}', '\u{18f}', '\u{259}', '\u{391}',
        '\u{392}', '\u{393}', '\u{395}', '\u{396}', '\u{397}', '\u{398}', '\u{399}', '\u{39a}',
        '\u{39b}', '\u{39c}', '\u{39d}', '\u{39f}', '\u{3a0}', '\u{3a1}', '\u{3a4}', '\u{3a5}',
        '\u{3a6}', '\u{3a7}', '\u{3b1}', '\u{3b2}', '\u{3b3}', '\u{3b4}', '\u{3b5}', '\u{3b8}',
        '\u{3b9}', '\u{3ba}', '\u{3bd}', '\u{3bf}', '\u{3c0}', '\u{3c1}', '\u{3c3}', '\u{3c4}',
        '\u{3c5}', '\u{3c6}', '\u{404}', '\u{405}', '\u{406}', '\u{408}', '\u{410}', '\u{411}',
        '\u{412}', '\u{413}', '\u{415}', '\u{417}', '\u{41a}', '\u{41b}', '\u{41c}', '\u{41d}',
        '\u{41e}', '\u{41f}', '\u{420}', '\u{421}', '\u{422}', '\u{423}', '\u{424}', '\u{425}',
        '\u{42b}', '\u{42c}', '\u{42e}', '\u{430}', '\u{431}', '\u{433}', '\u{435}', '\u{43a}',
        '\u{43e}', '\u{43f}', '\u{440}', '\u{441}', '\u{442}', '\u{443}', '\u{444}', '\u{445}',
        '\u{454}', '\u{455}', '\u{456}', '\u{458}', '\u{45b}', '\u{48c}', '\u{48d}', '\u{490}',
        '\u{491}', '\u{492}', '\u{493}', '\u{498}', '\u{49e}', '\u{49f}', '\u{4aa}', '\u{4ab}',
        '\u{4ae}', '\u{4af}', '\u{4b0}', '\u{4b1}', '\u{4bb}', '\u{4bd}', '\u{4bf}', '\u{4c0}',
        '\u{4c7}', '\u{4c9}', '\u{4cd}', '\u{4cf}', '\u{4d4}', '\u{4d5}', '\u{4d8}', '\u{4d9}',
        '\u{4e0}', '\u{4e8}', '\u{4e9}', '\u{511}', '\u{51b}', '\u{51c}', '\u{51d}', '\u{53b}',
        '\u{544}', '\u{548}', '\u{54a}', '\u{54c}', '\u{54d}', '\u{54f}', '\u{553}', '\u{555}',
        '\u{561}', '\u{563}', '\u{566}', '\u{56e}', '\u{570}', '\u{571}', '\u{578}', '\u{57a}',
        '\u{57c}', '\u{57d}', '\u{581}', '\u{584}', '\u{585}', '\u{5b4}', '\u{5d5}', '\u{5d8}',
        '\u{5d9}', '\u{5df}', '\u{5e1}', '\u{5f0}', '\u{5f1}', '\u{5f2}', '\u{5f3}', '\u{5f4}',
        '\u{625}', '\u{627}', '\u{629}', '\u{647}', '\u{660}', '\u{661}', '\u{665}', '\u{667}',
        '\u{668}', '\u{669}', '\u{6be}', '\u{6c1}', '\u{6c3}', '\u{6d5}', '\u{6f0}', '\u{6f1}',
        '\u{6f5}', '\u{6f7}', '\u{6f8}', '\u{6f9}', '\u{6ff}', '\u{901}', '\u{902}', '\u{903}',
        '\u{93c}', '\u{93d}', '\u{941}', '\u{942}', '\u{946}', '\u{94d}', '\u{966}', '\u{967}',
        '\u{968}', '\u{969}', '\u{96a}', '\u{96e}', '\u{971}', '\u{981}', '\u{983}', '\u{9bc}',
        '\u{9e6}', '\u{9ea}', '\u{9ed}', '\u{a02}', '\u{a03}', '\u{a3c}', '\u{a4b}', '\u{a4d}',
        '\u{a66}', '\u{a67}', '\u{a6a}', '\u{a81}', '\u{a82}', '\u{a83}', '\u{abc}', '\u{abd}',
        '\u{ac1}', '\u{ac2}', '\u{acd}', '\u{ae6}', '\u{ae8}', '\u{ae9}', '\u{aea}', '\u{aee}',
        '\u{b01}', '\u{b03}', '\u{b20}', '\u{b3c}', '\u{b66}', '\u{b68}', '\u{b82}', '\u{b89}',
        '\u{b90}', '\u{b9c}', '\u{ba3}', '\u{bb4}', '\u{bb6}', '\u{bbf}', '\u{bcd}', '\u{be6}',
        '\u{be8}', '\u{c02}', '\u{c03}', '\u{c05}', '\u{c06}', '\u{c07}', '\u{c12}', '\u{c13}',
        '\u{c14}', '\u{c1c}', '\u{c1e}', '\u{c23}', '\u{c2f}', '\u{c31}', '\u{c32}', '\u{c66}',
        '\u{c67}', '\u{c68}', '\u{c6f}', '\u{c82}', '\u{c83}', '\u{c85}', '\u{c86}', '\u{c87}',
        '\u{c92}', '\u{c93}', '\u{c94}', '\u{c9c}', '\u{c9e}', '\u{ca3}', '\u{caf}', '\u{cb1}',
        '\u{cb2}', '\u{ce6}', '\u{ce7}', '\u{ce8}', '\u{cef}', '\u{d02}', '\u{d03}', '\u{d09}',
        '\u{d1c}', '\u{d20}', '\u{d23}', '\u{d34}', '\u{d36}', '\u{d3a}', '\u{d3f}', '\u{d40}',
        '\u{d4e}', '\u{d66}', '\u{d6d}', '\u{d82}', '\u{d83}', '\u{e08}', '\u{e1a}', '\u{e1b}',
        '\u{e1d}', '\u{e1e}', '\u{e1f}', '\u{e22}', '\u{e34}', '\u{e35}', '\u{e36}', '\u{e37}',
        '\u{e38}', '\u{e39}', '\u{e48}', '\u{e49}', '\u{e4a}', '\u{e4b}', '\u{e4d}', '\u{e50}',
        '\u{e88}', '\u{e8d}', '\u{e9a}', '\u{e9b}', '\u{e9d}', '\u{e9e}', '\u{e9f}', '\u{eb8}',
        '\u{eb9}', '\u{ec8}', '\u{ec9}', '\u{eca}', '\u{ecb}', '\u{ecd}', '\u{ed0}', '\u{f37}',
        '\u{101d}', '\u{1036}', '\u{1038}', '\u{1040}', '\u{10e7}', '\u{10ff}', '\u{1200}',
        '\u{1206}', '\u{1223}', '\u{1240}', '\u{1260}', '\u{1261}', '\u{1294}', '\u{12ae}',
        '\u{12d0}', '\u{1323}', '\u{17b7}', '\u{17b8}', '\u{17b9}', '\u{17ba}', '\u{17c6}',
        '\u{17cb}', '\u{1e9e}', '\u{3007}', '\u{304f}', '\u{3078}', '\u{30a4}', '\u{30a8}',
        '\u{30ab}', '\u{30bf}', '\u{30c8}', '\u{30cb}', '\u{30ce}', '\u{30cf}', '\u{30d8}',
        '\u{30ed}', '\u{3126}', '\u{4e00}', '\u{4e3f}', '\u{4e8c}', '\u{4ebb}', '\u{513f}',
        '\u{516b}', '\u{529b}', '\u{535c}', '\u{53e3}', '\u{56d7}', '\u{5915}', '\u{5de5}',
        '\u{a792}', '\u{a793}', '\u{a7d6}', '\u{21fe8}'
    ];

}

