// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_PAYMENTS_CONTENT_ANDROID_PAYMENT_APP_H_
#define COMPONENTS_PAYMENTS_CONTENT_ANDROID_PAYMENT_APP_H_

#include <memory>
#include <set>
#include <string>

#include "base/memory/weak_ptr.h"
#include "base/optional.h"
#include "components/payments/content/android_app_communication.h"
#include "components/payments/content/payment_app.h"
#include "components/payments/core/android_app_description.h"
#include "url/gurl.h"

namespace payments {

// A cross-platform way to invoke an Android payment app.
class AndroidPaymentApp : public PaymentApp {
 public:
  // Creates an instance of AndroidPaymentApp.
  //
  // The |payment_method_names| is the set of payment method identifiers
  // supported by this app, e.g., ["https://example1.com",
  // "https://example2.com"]. This set should not be empty.
  //
  // The |stringified_method_data| is a mapping from payment method identifiers
  // that this app can handle to the method-specific data provided by the
  // merchant. The set of keys should match exactly the |payment_method_names|.
  // It is the responsibility of the creator of AndroidPaymentApp to filter out
  // the data from merchant that is not in |payment_method_names|.
  AndroidPaymentApp(
      const std::set<std::string>& payment_method_names,
      std::unique_ptr<std::map<std::string, std::set<std::string>>>
          stringified_method_data,
      const GURL& top_level_origin,
      const GURL& payment_request_origin,
      const std::string& payment_request_id,
      std::unique_ptr<AndroidAppDescription> description,
      base::WeakPtr<AndroidAppCommunication> communication);
  ~AndroidPaymentApp() override;

  AndroidPaymentApp(const AndroidPaymentApp& other) = delete;
  AndroidPaymentApp& operator=(const AndroidPaymentApp& other) = delete;

  // PaymentApp implementation.
  void InvokePaymentApp(Delegate* delegate) override;
  bool IsCompleteForPayment() const override;
  uint32_t GetCompletenessScore() const override;
  bool CanPreselect() const override;
  base::string16 GetMissingInfoLabel() const override;
  bool HasEnrolledInstrument() const override;
  void RecordUse() override;
  bool NeedsInstallation() const override;
  std::string GetId() const override;
  base::string16 GetLabel() const override;
  base::string16 GetSublabel() const override;
  const SkBitmap* icon_bitmap() const override;
  bool IsValidForModifier(
      const std::string& method,
      bool supported_networks_specified,
      const std::set<std::string>& supported_networks) const override;
  base::WeakPtr<PaymentApp> AsWeakPtr() override;
  bool HandlesShippingAddress() const override;
  bool HandlesPayerName() const override;
  bool HandlesPayerEmail() const override;
  bool HandlesPayerPhone() const override;
  bool IsWaitingForPaymentDetailsUpdate() const override;
  void UpdateWith(
      mojom::PaymentRequestDetailsUpdatePtr details_update) override;
  void OnPaymentDetailsNotUpdated() override;
  bool IsPreferred() const override;

 private:
  void OnPaymentAppResponse(Delegate* delegate,
                            const base::Optional<std::string>& error_message,
                            bool is_activity_result_ok,
                            const std::string& payment_method_identifier,
                            const std::string& stringified_details);

  const std::unique_ptr<std::map<std::string, std::set<std::string>>>
      stringified_method_data_;
  const GURL top_level_origin_;
  const GURL payment_request_origin_;
  const std::string payment_request_id_;
  const std::unique_ptr<AndroidAppDescription> description_;
  base::WeakPtr<AndroidAppCommunication> communication_;

  base::WeakPtrFactory<AndroidPaymentApp> weak_ptr_factory_{this};
};

}  // namespace payments

#endif  // COMPONENTS_PAYMENTS_CONTENT_ANDROID_PAYMENT_APP_H_
