// Copyright (C) 2016 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only WITH Qt-GPL-exception-1.0

#include "allprojectsfind.h"

#include "editorconfiguration.h"
#include "project.h"
#include "projectexplorer.h"
#include "projectexplorertr.h"
#include "projectmanager.h"

#include <coreplugin/editormanager/editormanager.h>

#include <texteditor/textdocument.h>

#include <utils/algorithm.h>
#include <utils/qtcsettings.h>

#include <QGridLayout>

using namespace TextEditor;
using namespace Utils;

namespace ProjectExplorer::Internal {

AllProjectsFind::AllProjectsFind() :  m_configWidget(nullptr)
{
    connect(ProjectExplorerPlugin::instance(), &ProjectExplorerPlugin::fileListChanged,
            this, &AllProjectsFind::handleFileListChanged);
}

QString AllProjectsFind::id() const
{
    return QLatin1String("All Projects");
}

QString AllProjectsFind::displayName() const
{
    return Tr::tr("All Projects");
}

bool AllProjectsFind::isEnabled() const
{
    return BaseFileFind::isEnabled() && ProjectManager::hasProjects();
}

FileContainerProvider AllProjectsFind::fileContainerProvider() const
{
    return [nameFilters = fileNameFilters(), exclusionFilters = fileExclusionFilters()] {
        return filesForProjects(nameFilters, exclusionFilters, ProjectManager::projects());
    };
}

FileContainer AllProjectsFind::filesForProjects(const QStringList &nameFilters,
                                                const QStringList &exclusionFilters,
                                                const QList<Project *> &projects)
{
    const FilterFilesFunction filterFiles
        = Utils::filterFilesFunction(nameFilters, exclusionFilters);
    const QMap<FilePath, TextEncoding> openEditorEncodings
        = TextDocument::openedTextDocumentEncodings();
    QMap<FilePath, TextEncoding> encodings;
    for (const Project *project : projects) {
        const EditorConfiguration *config = project->editorConfiguration();
        TextEncoding projectEncoding = config->useGlobalSettings()
            ? Core::EditorManager::defaultTextEncoding()
            : config->textEncoding();
        const FilePaths filteredFiles = filterFiles(project->files(Project::SourceFiles));
        for (const FilePath &fileName : filteredFiles) {
            TextEncoding encoding = openEditorEncodings.value(fileName);
            if (!encoding.isValid())
                encoding = projectEncoding;
            encodings.insert(fileName, encoding);
        }
    }
    return FileListContainer(encodings.keys(), encodings.values());
}

QString AllProjectsFind::label() const
{
    return Tr::tr("All Projects:");
}

QString AllProjectsFind::toolTip() const
{
    // last arg is filled by BaseFileFind::runNewSearch
    return Tr::tr("Filter: %1\nExcluding: %2\n%3")
            .arg(fileNameFilters().join(','))
            .arg(fileExclusionFilters().join(','));
}

void AllProjectsFind::handleFileListChanged()
{
    emit enabledChanged(isEnabled());
}

QWidget *AllProjectsFind::createConfigWidget()
{
    if (!m_configWidget) {
        m_configWidget = new QWidget;
        auto gridLayout = new QGridLayout(m_configWidget);
        gridLayout->setContentsMargins(0, 0, 0, 0);
        m_configWidget->setLayout(gridLayout);
        const QList<QPair<QWidget *, QWidget *>> patternWidgets = createPatternWidgets();
        int row = 0;
        for (const QPair<QWidget *, QWidget *> &p : patternWidgets) {
            gridLayout->addWidget(p.first, row, 0, Qt::AlignRight);
            gridLayout->addWidget(p.second, row, 1);
            ++row;
        }
        m_configWidget->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred);
    }
    return m_configWidget;
}

const char kDefaultInclusion[] = "*";
const char kDefaultExclusion[] = "";

Store AllProjectsFind::save() const
{
    Store s;
    writeCommonSettings(s, kDefaultInclusion, kDefaultExclusion);
    return s;
}

void AllProjectsFind::restore(const Utils::Store &s)
{
    readCommonSettings(s, kDefaultInclusion, kDefaultExclusion);
}

QByteArray AllProjectsFind::settingsKey() const
{
    return "AllProjectsFind";
}

} // ProjectExplorer::Internal
