// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System;
using System.Text;

using Debug = System.Diagnostics.Debug;

namespace Internal.TypeSystem
{
    public class DebugNameFormatter : TypeNameFormatter<DebugNameFormatter.Void, DebugNameFormatter.FormatOptions>
    {
        public static readonly DebugNameFormatter Instance = new DebugNameFormatter();

        public override Void AppendName(StringBuilder sb, ArrayType type, FormatOptions options)
        {
            AppendName(sb, type.ElementType, options);

            if (!type.IsSzArray && type.Rank == 1)
            {
                sb.Append("[*]");
            }
            else
            {
                sb.Append('[');
                sb.Append(',', type.Rank - 1);
                sb.Append(']');
            }

            return Void.Value;
        }

        public override Void AppendName(StringBuilder sb, ByRefType type, FormatOptions options)
        {
            AppendName(sb, type.ParameterType, options);
            sb.Append('&');

            return Void.Value;
        }

        public override Void AppendName(StringBuilder sb, PointerType type, FormatOptions options)
        {
            AppendName(sb, type.ParameterType, options);
            sb.Append('*');

            return Void.Value;
        }

        public override Void AppendName(StringBuilder sb, FunctionPointerType type, FormatOptions options)
        {
            MethodSignature signature = type.Signature;

            sb.Append("(*");
            AppendName(sb, signature.ReturnType, options);
            sb.Append(")(");
            for (int i = 0; i < signature.Length; i++)
            {
                if (i > 0)
                    sb.Append(',');
                AppendName(sb, signature[i], options);
            }
            sb.Append(')');

            return Void.Value;
        }

        public override Void AppendName(StringBuilder sb, GenericParameterDesc type, FormatOptions options)
        {
            sb.Append(type.Name);

            return Void.Value;
        }

        public override Void AppendName(StringBuilder sb, SignatureMethodVariable type, FormatOptions options)
        {
            sb.Append("!!");
            sb.Append(type.Index);

            return Void.Value;
        }

        public override Void AppendName(StringBuilder sb, SignatureTypeVariable type, FormatOptions options)
        {
            sb.Append("!");
            sb.Append(type.Index);

            return Void.Value;
        }

        protected override Void AppendNameForNestedType(StringBuilder sb, DefType nestedType, DefType containingType, FormatOptions options)
        {
            if ((options & FormatOptions.NamespaceQualify) != 0)
            {
                AppendName(sb, containingType, options);
                sb.Append('+');
            }

            sb.Append(nestedType.Name);

            return Void.Value;
        }

        protected override Void AppendNameForNamespaceType(StringBuilder sb, DefType type, FormatOptions options)
        {
            // Shortcut some of the well known types
            switch (type.Category)
            {
                case TypeFlags.Void:
                    sb.Append("void");
                    return Void.Value;
                case TypeFlags.Boolean:
                    sb.Append("bool");
                    return Void.Value;
                case TypeFlags.Char:
                    sb.Append("char");
                    return Void.Value;
                case TypeFlags.SByte:
                    sb.Append("int8");
                    return Void.Value;
                case TypeFlags.Byte:
                    sb.Append("uint8");
                    return Void.Value;
                case TypeFlags.Int16:
                    sb.Append("int16");
                    return Void.Value;
                case TypeFlags.UInt16:
                    sb.Append("uint16");
                    return Void.Value;
                case TypeFlags.Int32:
                    sb.Append("int32");
                    return Void.Value;
                case TypeFlags.UInt32:
                    sb.Append("uint32");
                    return Void.Value;
                case TypeFlags.Int64:
                    sb.Append("int64");
                    return Void.Value;
                case TypeFlags.UInt64:
                    sb.Append("uint64");
                    return Void.Value;
                case TypeFlags.IntPtr:
                    sb.Append("native int");
                    return Void.Value;
                case TypeFlags.UIntPtr:
                    sb.Append("native uint");
                    return Void.Value;
                case TypeFlags.Single:
                    sb.Append("float32");
                    return Void.Value;
                case TypeFlags.Double:
                    sb.Append("float64");
                    return Void.Value;
            }

            if (type.IsString)
            {
                sb.Append("string");
                return Void.Value;
            }

            if (type.IsObject)
            {
                sb.Append("object");
                return Void.Value;
            }

            if (((options & FormatOptions.AssemblyQualify) != 0)
                && type is MetadataType mdType
                && mdType.Module is IAssemblyDesc)
            {
                sb.Append('[');

                // Trim the "System.Private." prefix
                string assemblyName = ((IAssemblyDesc)mdType.Module).GetName().Name;
                if (assemblyName.StartsWith("System.Private"))
                    assemblyName = "S.P" + assemblyName.Substring(14);

                sb.Append(assemblyName);
                sb.Append(']');
            }

            if ((options & FormatOptions.NamespaceQualify) != 0)
            {
                string ns = type.Namespace;
                if (!string.IsNullOrEmpty(ns))
                {
                    sb.Append(ns);
                    sb.Append('.');
                }
            }

            sb.Append(type.Name);

            return Void.Value;
        }

        protected override Void AppendNameForInstantiatedType(StringBuilder sb, DefType type, FormatOptions options)
        {
            AppendName(sb, type.GetTypeDefinition(), options);

            FormatOptions parameterOptions = options & ~FormatOptions.AssemblyQualify;

            sb.Append('<');

            for (int i = 0; i < type.Instantiation.Length; i++)
            {
                if (i != 0)
                    sb.Append(',');

                AppendName(sb, type.Instantiation[i], parameterOptions);
            }

            sb.Append('>');

            return Void.Value;
        }

        public struct Void
        {
            public static Void Value => default(Void);
        }

        [Flags]
        public enum FormatOptions
        {
            None = 0,
            AssemblyQualify = 0x1,
            NamespaceQualify = 0x2,

            Default = AssemblyQualify | NamespaceQualify,
        }
    }
}
