`geojson-pydantic` version 1.0 introduced [many breaking changes](../release-notes.md). This
document aims to help with migrating your code to use `geojson-pydantic` 1.0.


## Pydantic 2.0

The biggest update introduced in **1.0** is the new pydantic *major* version requirement [**~2.0**](https://docs.pydantic.dev/2.0/migration/).

In addition to being faster, this new major version has plenty of new API which we used in `geojson-pydantic` (like the new `model_serializer` method).

```python

from geojson_pydantic import Point

# Before
Point.dict()  # serialize to dict object
Point.json()  # serialize to json string

with open("point.geojson") as f:
    Point.parse_file(f)  # parse file content to model

p = {}
Point.parse_obj(obj)  # parse dict object

##################
# Now (geojson-pydantic ~= 1.0)

Point.model_dump()
Point.model_dump_json()

with open("point.geojson") as f:
    Point.model_validate_json(f.read())

p = {}
Point.model_validate(obj)
```

ref: https://github.com/developmentseed/geojson-pydantic/pull/130

## FeatureCollection Generic model

In **1.0** we updated the generic FeatureCollection model to depends only on a generic Feature model.

```python
# Before
FeatureCollection[Geometry, Properties]

# Now (geojson-pydantic ~= 1.0)
FeatureCollection[Feature[Geometry, Properties]]
```

e.g

```python
from pydantic import BaseModel
from geojson_pydantic import Feature, FeatureCollection, Polygon

class CustomProperties(BaseModel):
    id: str
    description: str
    size: int

# Create a new FeatureCollection Model which should only
# Accept Features with Polygon geometry type and matching the properties
MyFc = FeatureCollection[Feature[Polygon, CustomProperties]]
```

ref: https://github.com/developmentseed/geojson-pydantic/issues/134

## Exclude `bbox` and `id` if null

Using the new pydantic `model_serializer` method, we are now able to `customize` JSON output for the models to better match the GeoJSON spec

```python
# Before
Point(type="Point", coordinates=[0, 0]).json()
>> '{"type":"Point","coordinates":[0.0,0.0],"bbox":null}'

# Now (geojson-pydantic ~= 1.0)
Point(type="Point", coordinates=[0, 0]).model_dump_json()
>> '{"type":"Point","coordinates":[0.0,0.0]}'
```

ref: https://github.com/developmentseed/geojson-pydantic/issues/125


## Change in WKT output for Multi* geometries

```python
from geojson_pydantic import MultiPoint

geom = MultiPoint(type='MultiPoint', coordinates=[(1.0, 2.0, 3.0)])

# Before
print(geom.wkt)
>> MULTIPOINT Z (1 2 3)

# Now (geojson-pydantic ~= 1.0)
print(geom.wkt)
>> MULTIPOINT Z ((1 2 3))
```

ref: https://github.com/developmentseed/geojson-pydantic/issues/139
